"""
Module EEPAS0FModel
"""

__version__ = "$Revision$"
__revision__ = "$Id$"

import os

import CSEPFile
from EEPASModel import EEPASModel
from Forecast import Forecast
from CSEPStorage import CSEPStorage


#--------------------------------------------------------------------------------
#
# EEPAS-0F forecast model.
#
# This class is designed to invoke three-months PPE and EEPAS forecast models. 
# It uses 'eepas_0f.par' configuration for the run. It has model weights
# disabled.
#
class EEPAS0FModel (EEPASModel):

    # Static data of the class
    
    # Keyword identifying the type of the forecast
    Type = EEPASModel.Type + "-0F"
    
    # 0F configuration generates two forecast files: PPE and EEPAS
    PPEForecastFile = None
    
    
    #--------------------------------------------------------------------
    #
    # Initialization.
    #
    # Input: 
    #        dir_path - Directory to store forecast file to.
    #        args - Optional arguments for the model. Default is None.
    # 
    def __init__ (self, dir_path, args = None):
        """ Initialization for EEPAS0FModel class"""
        
        EEPASModel.__init__(self, dir_path, args, "eepas_0f.par")
        

    #----------------------------------------------------------------------------
    #
    # Return keyword identifying the model.
    #
    # Input: None.
    #
    # Output:
    #           String identifying the type
    #
    def type (self):
        """ Returns keyword identifying the forecast model type."""
        
        return self.Type


    #----------------------------------------------------------------------------
    #
    # Create forecast.
    # This method is overwritten to create metadata file for the full region PPE 
    # forecast model.
    #
    # Input: 
    #       test_date - datetime object that represents testing date.
    #       catalog_dir - Directory with catalog data
    #       archive_dir - Directory to store intermediate model results if any.
    #                     Default is None.    
    #        
    # Output:
    #        Name of created forecast file.
    #
    def create (self, test_date, catalog_dir, archive_dir=None):
        """ Generate forecast."""
        
        # Invoke the models
        result_files = Forecast.create(self, 
                                       test_date, 
                                       catalog_dir, 
                                       archive_dir)

        # Check if PPE forecast file wasn't generated (EEPAS-0F forecast was staged) 
        # and PPE file is archived ---> stage the file
        file_is_staged = False
        if os.path.exists(os.path.join(self.dir,
                                       EEPAS0FModel.__ppeForecastFile(test_date))) is False and \
           archive_dir is not None:
           file_is_staged = CSEPStorage.stage(self, 
                                              [EEPAS0FModel.__ppeForecastFile(test_date)],
                                              os.path.join(archive_dir,
                                                           Forecast.archiveDir(EEPAS0FModel.__ppeForecastFile(test_date))))


        # PPE model is invoked implicitly by the OF configuration for EEPAS,
        # create metadata file for it.
        if file_is_staged is False:
           
           comment = "PPE forecast file with start date '%s' and end date '%s'" \
                     %(self.start_date, self.end_date)
           
           Forecast.metadata(os.path.join(self.dir, 
                                          EEPAS0FModel.__ppeForecastFile(self.start_date)),
                             comment,
                             archive_dir)
        
        result_files.append(EEPAS0FModel.__ppeForecastFile(test_date))
        
        return result_files
        

    #----------------------------------------------------------------------------
    #
    # Writes forecasts output files to the file handle that are specific to the 
    # model configuration - some configurations generate two forecasts files.
    #
    # Input:
    #        fhandle - Handle to the open control file used to invoke the model.
    #
    # Output:
    #        Handle to the control file.
    #        
    def writeForecastFile (self, fhandle):
        """ Writes forecast output file(s) to the file that are specific to the
            model configuration."""

        fhandle.write("PARAMETERS\n")
        fhandle.write("nfile 27\n")
        fhandle.write("endp/\n")
        fhandle.write("FILES\n")
        
        fhandle.write("array '%s'\n" %os.path.join(self.dir, 
                                                   EEPAS0FModel.__ppeForecastFile(self.start_date)))
        fhandle.write("end files\n")
        
        EEPASModel.writeForecastFile(self, fhandle)
        
        return fhandle
     

    #----------------------------------------------------------------------------
    #
    # Creates filename for PPE foreacast if it does not exist yet.
    #
    # Input:
    #        start_date - Start date of the forecast. Default is None.
    #
    # Output:
    #        Handle to the control file.
    #        
    def __ppeForecastFile (start_date = None):
        """ Creates filename for PPE foreacast."""
        
        if EEPAS0FModel.PPEForecastFile is None:
        
           EEPAS0FModel.PPEForecastFile = "PPE_%s_%s_%s%s" %(start_date.month, 
                                                             start_date.day, 
                                                             start_date.year,
                                                             CSEPFile.Extension.ASCII)        

        return EEPAS0FModel.PPEForecastFile
     
    __ppeForecastFile = staticmethod(__ppeForecastFile)
     
     
     
     