"""
Module DBM_NZ_ThreeMonthModel
"""

__version__ = "$Revision$"
__revision__ = "$Id$"

import os, datetime

import Environment, CSEPFile, CSEPLogging, CSEP, CSEPXMLGeneric, CSEPInputParams
from Forecast import Forecast
from ThreeMonthsForecast import ThreeMonthsForecast
from ReproducibilityFiles import ReproducibilityFiles
from DataSourceFactory import DataSourceFactory
from CatalogDataSource import CatalogDataSource



#--------------------------------------------------------------------------------
#
# Three Months Double Branching forecast model.
#
# This class is designed to invoke a three-months Double Branching
# forecast model. It prepares input catalog data, formats input file with model 
# parameters, and invokes the model. It places forecast file under user 
# specified directory.
#
class DBM_NZ_ThreeMonthModel (ThreeMonthsForecast):

    # Static data of the class
    
    # Keyword identifying type of the class
    Type = 'DBM' 
    
    # Center code path for the model
    __NZRootPath = os.environ["NZHOME"]
    __modelPath = os.path.join(__NZRootPath,
                              'NewZealandCode',
                              'DoubleBranchingModel')
    __executableFile = "DBM_NZ"

    # Input files required and provided by the model
    __Back1File = 'NZ_back1_0.1d_CSEP.txt'
    __Back2File = 'NZ_back2_0.1d_CSEP.txt'


    # randomSeedFile - to read/store random seed from/to the file.
    #                  '0' - draw random seed by the system, save to the file
    #                  'xxx.file' - read the random seed from provided xxx.file
    __randomSeedFileOption = "randomSeedFile"
    
    # Number of simulations for the model - default is 1000
    __numSimulationsOption = "numberSimulations"
    
    # Random seed file base
    __randomSeedFile = "DBMRandomSeed"

    # This data is static for the class - safe because we generate
    # only one forecast per model for any CSEP run.
    __defaultArgs = {__randomSeedFileOption : None,
                     __numSimulationsOption : '1000'}
    
    
    #----------------------------------------------------------------------------
    #
    # Initialization.
    #
    # Input: 
    #        dir_path - Directory to store forecast file to.
    #        args - Optional arguments for the model. Default is None.
    # 
    def __init__ (self, dir_path, args=None):
        """ Initialization for DBM_NZ_ThreeMonthModel class"""
        
        ThreeMonthsForecast.__init__(self, dir_path)

        # Input arguments for the model were provided:
        self.__args = CSEPInputParams.CSEPInputParams.parse(DBM_NZ_ThreeMonthModel.__defaultArgs, 
                                                            args)
                               
        # Name of the input catalog
        self.__modelInputCatalog = CSEPFile.Name.extension(self.inputCatalogFilename(),
                                                           'dat')


        # Model generates xml format forecast in (EQ's per year per km^2) units
        # Will need to specify filename for intermediate XML format file to be
        # generated by the model - to be archived once CSEP converts rates to
        # (EQ's per year per degree^2) units in XML format
        self.__rawXMLFilename = None
        

    #----------------------------------------------------------------------------
    #
    # Return keyword identifying the model.
    #
    # Input: None.
    #
    # Output:
    #           String identifying the type
    #
    def type (self):
        """ Returns keyword identifying the forecast model type."""
        
        return self.Type


    #----------------------------------------------------------------------------
    #
    # Return full path for the result forecast file that is based on XML
    # master template.
    #
    # Input: None.
    #
    # Output:
    #           String identifying the filename.
    #
    def filename (self):
        """ Returns filename of generated forecast."""
        
        return CSEPFile.Name.xml(Forecast.filename(self))


    #----------------------------------------------------------------------------
    #
    # Write input parameter file for the model.
    #
    # Input: None.
    #        
    def writeParameterFile (self):
        """ Format input parameter file for the model.
            Created file will be passed to the model."""

        fhandle = Forecast.writeParameterFile(self)
        
        fhandle.write("%s\t%s\n" %('4.7d0', 'mu')) 
        fhandle.write("%s\t%s\n" %('0.008d0', 'k1'))
        fhandle.write("%s\t%s\n" %('1.18d0', 'p'))   
        fhandle.write("%s\t%s\n" %('0.00003d0', 'c'))
        fhandle.write("%s\t%s\n" %('1.3d0', 'alpha1'))
        fhandle.write("%s\t%s\n" %('4d0', 'd1'))
        fhandle.write("%s\t%s\n" %('1.5d0', 'q1'))
        fhandle.write("%s\t%s\n" %('0.5d0', 'gamma'))
        fhandle.write("%s\t%s\n" %('0.014d0', 'k2'))
        fhandle.write("%s\t%s\n" %('33d0', 'tau'))
        fhandle.write("%s\t%s\n" %('0.0d0', 'alpha2'))
        fhandle.write("%s\t%s\n" %('11d0', 'd2'))
        fhandle.write("%s\t%s\n" %('1.5d0', 'q2'))
        
        # Write number of coordinates sets
        fhandle.write("1 NUMBER OF SETS OF COORDINATES\n")
        
        # Write region coordinates to the file
        fhandle.write("-51.7d0\tXLATMIN\n")
        fhandle.write("-23.2d0\tXLATMAX\n")
        fhandle.write("159.25d0\tXLONMIN\n")
        fhandle.write("185.2d0\tXLONMAX\n")
            
        fhandle.write("0.1d0\tGRID STEP\n")
        fhandle.write("%s\t%s\n" %('0.4d0', '!THRESHOLD PROBABILITY AFTERSHOCKS'))

        fhandle.write("%s\t%s\n" %('1951',
                                   '!START INPUT CATALOG'))


      #  fhandle.write("%s\t%s\n" %(CatalogDataSource.StartDate.year,
      #                             '!START INPUT CATALOG'))

        fhandle.write("%s\t%s\n" %(self.__args[DBM_NZ_ThreeMonthModel.__numSimulationsOption], 
                                   '!NUMBER OF SIMULATIONS'))
        
        # Write forecast period
        fhandle.write("%s.0\t%s\n" %(self.start_date.strftime(CSEP.Time.DateTimeSpaceFormat),
                                  'START DATA FORECASTING'))
        fhandle.write("%s.0\t%s\n" %(self.end_date.strftime(CSEP.Time.DateTimeSpaceFormat), 
                                  'END DATA FORECASTING'))
        fhandle.write("%s.0\t%s\n" %(self.start_date.strftime(CSEP.Time.DateTimeSpaceFormat),
                                  'STARTING DATA SIMULATION'))


        fhandle.write("%s\n" 
                      %os.path.join(DBM_NZ_ThreeMonthModel.__modelPath,
                                    DBM_NZ_ThreeMonthModel.__Back2File))

        fhandle.write("%s\n" %os.path.join(self.catalogDir,
                                           self.__modelInputCatalog))

        fhandle.write("%s\n" 
                      %os.path.join(DBM_NZ_ThreeMonthModel.__modelPath,
                                    DBM_NZ_ThreeMonthModel.__Back1File))

        random_seed_file = self.__args[DBM_NZ_ThreeMonthModel.__randomSeedFileOption]
        if random_seed_file is not None:
           # Specify to the model to read random seed from the file 
           line  = "1 !LOGICAL VALUE FOR RANDOM SEED VALUE (0=READ_RANDOM_SEED_FROM_FILE False; 1=READ_RANDOM_SEED_FROM_FILE True)\n"
           fhandle.write(line)
           
           fhandle.write("%s\n" %random_seed_file)
        
        else:
           # Specify to the model to save random seed to the file
           line  = "0 !LOGICAL VALUE FOR RANDOM SEED VALUE (0=READ_RANDOM_SEED_FROM_FILE False; 1=READ_RANDOM_SEED_FROM_FILE True)\n"
           fhandle.write(line)
           
           random_seed_file = "%s_%s" %(DBM_NZ_ThreeMonthModel.__randomSeedFile, 
                                        self.start_date.date())
           line = "%s\n" %os.path.join(os.getcwd(), random_seed_file)
           fhandle.write(line)
           
           # Record random seed file with reproducibility registry
           info_msg = "DBMN random seed file to generate '%s' forecast file for %s." \
                      %(self.__rawXMLFilename, self.start_date.date())
   
           ReproducibilityFiles.add(self,
                                         random_seed_file, 
                                         info_msg, 
                                         CSEPFile.Format.ASCII)



# the option below was used to use eaxctly the same random seed file as the example provided by
# Anna Maria Lombardi to test the forecast outcome. The file 'random_seed_file.txt' has to be in the
# same directory from which the wrapper is called
#       
#        fhandle.write("%s\n" %('1 !LOGICAL VALUE FOR RANDOM SEED VALUE (0=READ_RANDOM_SEED_FROM_FILE False; 1=READ_RANDOM_SEED_FROM_FILE True)'))#
#        fhandle.write("%s\n" %('random_seed_file.txt'))#
#        



# Write output filename for the forecast in ASCII format
        fhandle.write("%s\n" %self.rawFilename())
        
        # Model generates xml format forecast in (EQ's per year per km^2) units
        self.__rawXMLFilename = self.filename() 
        #self.__rawXMLFilename = self.rawFilename() + CSEPFile.Extension.XML
        fhandle.write("%s\n" %self.__rawXMLFilename)


        fhandle.write("%s\t%s\n" %('5.0d0', '!MINIMUM MAGNITUDE'))
        fhandle.write("%s\t%s\n" %('9.0d0', '!MAXIMUM MAGNITUDE'))
        fhandle.write("%s\t%s\n" %('0.1d0', '!STEP MAGNITUDE'))
        fhandle.write("%s\t%s\n" %('1.0d0', '!B-VALUE'))
        # Close the file
        fhandle.close()


    #----------------------------------------------------------------------------
    #
    # Invoke the model.
    #
    # Input: None
    #        
    def run (self):
        """ Invoke DBM_NZ_ThreeMonthModel forecast."""

        # If modelers output some debug/progress info to stderr, ignore it - 
        # don't trigger an exception 
        ignore_model_stderr = True
        
        # invoke the model:
        Environment.invokeCommand('%s %s' %(os.path.join(DBM_NZ_ThreeMonthModel.__modelPath,
                                                         DBM_NZ_ThreeMonthModel.__executableFile),
                                            self.parameterFile),
                                  ignore_model_stderr)

        # Create metadata file for original forecast under archive directory
        comment = "Raw (generated by model code) %s forecast file with start date '%s' and end date '%s' in ASCII format" \
                  %(self.type(), self.start_date, self.end_date)
                  
        Forecast.metadata(self.rawFilename(), 
                                   comment, 
                                   self.archive)

#        # Create metadata file for original XML forecast under archive directory
#        comment = "Raw (generated by model code) %s forecast file with start date '%s' and end date '%s' in XML format" \
#                  %(self.type(), self.start_date, self.end_date)
#                  
#        Forecast.metadata(self.__rawXMLFilename, 
#                                   comment, 
#                                   self.archive)
#        
#        # Convert rates from (EQ per year per km^2) to (EQ per year per degree^2),
#        # and write new rates to final forecast XML format file
#        CSEPXMLGeneric.Forecast.kmToDegreeXMLRates(self.__rawXMLFilename, 
#                                                   self.filename())
#        
#        if self.archive is not None:
#           
#           # Move raw ASCII forecast file to the archive directory to avoid it
#           # in evaluation tests 
#           CSEPLogging.CSEPLogging.getLogger(DBM_NZ_ThreeMonthModel.__name__).info("run(): moving %s to %s"
#                                                                                 %(self.rawFilename(),
#                                                                                   self.archive))
#                                                                               
#           new_path = os.path.join(self.archive, 
#                                   os.path.basename(self.rawFilename()))
#           os.renames(self.rawFilename(), new_path)
#
#
#           # Move raw XML forecast file to the archive directory to avoid it
#           # in evaluation tests 
#           CSEPLogging.CSEPLogging.getLogger(DBM_NZ_ThreeMonthModel.__name__).info("run(): moving %s to %s"
#                                                                                 %(self.__rawXMLFilename,
#                                                                                   self.archive))
                                                                               
#           new_path = os.path.join(self.archive, 
#                                   os.path.basename(self.__rawXMLFilename))
#           os.renames(self.__rawXMLFilename, new_path)


    #---------------------------------------------------------------------------
    #
    # Return commands that should be used to capture version of external
    # software packages the model is dependent on. 
    #
    # Input: None.
    #
    # Output:
    #           String identifying the type
    #
    @staticmethod
    def externalSoftwareVersions ():
        """ Returns dictionary of command to determine external software 
            version and flag if output of that command is redirected to the
            stderr (True) or not (False) (java -version, for example)."""
        

        return {"gfortran --version" : False} 

