"""
CSEPXMLGeneric module
"""

__version__ = "$Revision: 2284 $"
__revision__ = "$Id: CSEPXMLGeneric.py 2284 2009-05-11 19:49:09Z liukis $"


import os, re, datetime, string, time
from xml.dom import minidom

from decimal import *

import CSEPFile, MatlabLogical, CSEPXML, CSEP, CSEPLogging, CSEPInitFile, \
       CSEPGeneric


#--------------------------------------------------------------------------------
#
# Forecast
#
# This module contains XML generic routines for forecast data.
#
class Forecast (object):
   
   # Static data of the class

   # Element names and attributes for XML format data
   CellMaskAttribute = 'mask'
   
   BinMaskAttribute = 'mk'
   
   # Field separator for the metadata key and value
   __metaKeyValueSeparator = '='
   
   
   #-----------------------------------------------------------------------------
   # This class is designed to represent a bin information: mask bit, rate, and
   # magnitude values.
   #
   # This class was introduced to avoid floating point representations problems 
   # used by Python. When instantiating tuples with already rounded 
   # floating point values to the second precision digit, it was still using
   # internal not "precise" representation. This class provides interface to
   # enforce rounding to the 2-nd precision digit.
   #
   class BinInfo (object):
      
      def __init__(self, magnitude, rate, mask = Decimal('1.0')):
         
         # Decimal object for magnitude
         self.magnitude = magnitude
         
         # String representation of rate
         self.rate = rate
         
         # Decimal object for mask bit
         self.mask = mask.quantize(Decimal('1.0'))
    
    
   #--------------------------------------------------------------------
   #
   # Returns expected filename for the the model data that is
   # based on XML template.
   #
   # Input: 
   #        file_path - Path to the file.
   #        extension - File extension of interest. Default is Matlab file
   #                    extension.
   #
   # Output:
   #         Expected filename
   #
   def fromXMLTemplateFilename (file_path, 
                                extension = CSEPFile.Extension.MATLAB):
      """ Returns filename that is based on XML template"""

        
      # Get rid of FromXMLPostfix (if any) from filename
      filename = re.sub(CSEP.Forecast.FromXMLPostfix,
                        '',
                        file_path)
        
      # Replace extension with 'fromXML.extension'
      file_from_xml = re.sub('([.][^.]+$)', 
                             CSEP.Forecast.FromXMLPostfix + 
                             extension,
                             filename)
      return file_from_xml

   fromXMLTemplateFilename = staticmethod(fromXMLTemplateFilename)


   #-----------------------------------------------------------------------------
   #
   # Returns filename for the the model map ASCII data.
   #
   # Input: 
   #        file_path - Path to the file.
   #        dir_name - Directory to store map file to.
   #        test_name - Name of the test for which forecast map is generated.
   #                    Default is None. 
   #
   # Output:
   #        Full path to the map file.
   #
   def mapFilename (file_path, dir_name, test_name = None):
      """ Returns full path to the filename for the forecast map."""

        
      model_path, model_name = os.path.split(file_path)
      
      prefix = CSEP.Forecast.MapReadyPrefix
      if test_name is not None:
         prefix += test_name
         prefix += "_"
      
      return os.path.join(dir_name, "%s%s" %(prefix, 
                                             CSEPFile.Name.ascii(model_name)))

   mapFilename = staticmethod(mapFilename)


   #-----------------------------------------------------------------------------
   #
   # Parse forecast issue date as provided in ASCII format file into ISO8601 
   # format date and time.
   #
   # Input:
   #        date_string - String representation of the date. 
   #                      For example, "Tue Aug 21 01:03:07 2007".
   #
   # Output:
   #        string - ISO8601 formatted date and time
   #
   def __parseIssueDate(date_string):
      """Parse issue date into ISO8601 format string. For now only
         entry string of default format (as in time.strptime()) 
         "Tue Aug 21 01:03:07 2007" is supported."""
      
      issue_time = datetime.datetime(*(time.strptime(date_string)[0:6]))
      
      return "%sT%s" %(issue_time.date(), issue_time.time())
   
   __parseIssueDate = staticmethod(__parseIssueDate)
   

   #-----------------------------------------------------------------------------
   #
   # Parse forecast start date as provided in ASCII format file in "%Y-%m-%d"
   # format into ISO8601 format date and time.
   #
   # Input:
   #        date_string - String representation of the date. 
   #                      For example, "2007-07-22".
   #   
   # Output:
   #        tuple of string (ISO8601 formatted date and time) and datetime object
   #        that represents the start date (to be used for creation of forecast
   #        end date)       
   #
   def __parseStartDate(date_string):
      """Parse forecast start date into ISO8601 format string.
         For now only entry string of "2007-07-22" format is supported."""
      
      start_date = datetime.datetime(*(time.strptime(date_string, "%Y-%m-%d")[0:6]))
      
      return ("%sT%s" %(start_date.date(), start_date.time()), start_date)
   
   __parseStartDate = staticmethod(__parseStartDate)


   #-----------------------------------------------------------------------------
   #
   # Parse forecast start date as provided in ASCII format file in "%Y-%m-%d"
   # format into ISO8601 format date and time.
   #
   # Input:
   #        duration_string - String representation of the duration in days.
   #                          For example, "1 day(s)".
   #   
   # Output:
   #        string - ISO8601 formatted date and time
   #
   def __parseDuration(duration_string, start_date):
      """Parse forecast duration in days and create ISO8601 format string 
         representation of the forecast end date. For now only entry string of 
         "1 day(s)" format is supported."""
      
      duration = string.replace(duration_string, 'day(s)', '')
      duration_value = int(duration.strip())
      
      date_diff = datetime.timedelta(duration_value)
      end_date = start_date + date_diff
      
      return "%sT%s" %(end_date.date(), end_date.time())
   
   __parseDuration = staticmethod(__parseDuration)
   
   
   XMLModelName = 'modelName'
   
   # Dictionary of ASCII to XML metadata elements: there is one-to-one transition
   # of data for these fields
   __metadata = {'modelname' : XMLModelName,
                 'version' : 'version',
                 'author' : 'author'}

   # The following elements require conversion when interpreted from ASCII
   # to XML format
   # TODO: All conversion functions support only one type of input format for the
   # date string. Use regular expressions if multiple formats are to be supported.
   __asciiIssueDate = 'issue_date'
   XMLIssueDate = 'issueDate'
   
   __asciiStartDate = 'forecast_start_date'
   XMLStartDate = 'forecastStartDate'
   
   __asciiDuration = 'forecast_duration'
   XMLEndDate = 'forecastEndDate'
   
   XMLCellDimension = 'defaultCellDimension'
   XMLLatRange = 'latRange'
   XMLLonRange = 'lonRange'
   
   # Start date of the forecast as provided by ASCII format data
   __startDate = None

   # Flag if metadata about the model was provided in ASCII format file.
   # Default is False.
   __metadataInFile = False
   

   #-----------------------------------------------------------------------------
   #
   # Parse forecast metadata as provided in ASCII format data.
   #
   # Input:
   #        line - Line of ASCII data.
   #        xmldoc - Root element of XML document
   #   
   # Output: None
   #
   def __parseMetadata(line, xmldoc):
      """Parse forecast metadata as provided in ASCII format data."""

      # Set forecast metadata if any is available
      tokens = [token.strip() for token in \
                line.split(Forecast.__metaKeyValueSeparator)]
       
      if len(tokens) != 2:
         return
       
      meta_key, meta_value = tokens 
      
      # metadata has corresponding element in XML format
      if Forecast.__metadata.has_key(meta_key) is True:
         
         Forecast.__metadataInFile = True
         
         # Acquire corresponding XML element, and set it's text to the
         # metadata value
         xml_elements = xmldoc.elements(Forecast.__metadata[meta_key])
         xml_elements[0].text = meta_value
          
      elif meta_key == Forecast.__asciiIssueDate:
          
         Forecast.__metadataInFile = True
                   
         # Convert issue date into XML expected format element
         xml_elements = xmldoc.elements(Forecast.XMLIssueDate)
         xml_elements[0].text = Forecast.__parseIssueDate(meta_value)
          
      elif meta_key == Forecast.__asciiStartDate:
         
         Forecast.__metadataInFile = True
                   
         # Convert forecast start date into XML expected format element
         xml_elements = xmldoc.elements(Forecast.XMLStartDate)
         xml_elements[0].text, Forecast.__startDate = Forecast.__parseStartDate(meta_value)
          
      elif meta_key == Forecast.__asciiDuration:
         
         Forecast.__metadataInFile = True
                  
         # start date must be already known when parsing the duration 
         if Forecast.__startDate is None:
            
            error_msg = "Forecast.__parseMetadata(): expected valid Forecast.__startDate \
object, received None"
            
            CSEPLogging.CSEPLogging.getLogger(__name__).error(error_msg)
            raise RuntimeError, error_msg
         
         # Convert forecast start date and duration into end date XML 
         # format element
         xml_elements = xmldoc.elements(Forecast.XMLEndDate)
         xml_elements[0].text = Forecast.__parseDuration(meta_value, 
                                                         Forecast.__startDate)
         
      return
                
   __parseMetadata = staticmethod(__parseMetadata)      


   #-----------------------------------------------------------------------------
   #
   # toXML
   # 
   # This method converts forecast data from ASCII to the XML format. It
   # uses master forecast template in XML format.
   #
   # Input: 
   #         model_file - Filename for forecast in ASCII format.
   #         template_file - Template file in XML format to use for the
   #                         forecast.   
   #         start_date - datetime object that represents start date for the model
   #         end_date - datetime object that represents end date for the model
   #         name - Name of the model 
   #
   # Output: Filename for XML data.
   # 
   def toXML (model_file, 
              template_file, 
              start_date = None, 
              end_date = None, 
              name = None):
      """ Convert ASCII forecast data into XML format by populating master XML
          template, and save it to the file.
          The function uses the same filename but with XML extension to 
          store XML formatted data."""

      # Read XML template in, and get handler to the root element
      xmldoc = CSEPInitFile.CSEPInitFile(template_file, 
                                         [], 
                                         CSEPXML.FORECAST_NAMESPACE)
      
      print "XML TEMPLATE:", template_file
      # Extract cell dimensions
      cell_dims = xmldoc.elements("defaultCellDimension")
      
      # Set current context for the Decimal numbers
      getcontext.rounding = ROUND_HALF_UP
      two_digit_precision = Decimal('0.01')
      mask_on = Decimal('1')
      
      
      # Compute half of range to get to the center point
      lat_range = Decimal(cell_dims[0].attrib['latRange']) * Decimal('0.5')
      lon_range = Decimal(cell_dims[0].attrib['lonRange']) * Decimal('0.5')
      
      # Extract bin dimensions
      bin_dims = xmldoc.elements("defaultMagBinDimension")
      # Compute half of range to get to the center point      
      mag_range = Decimal(bin_dims[0].text) * Decimal('0.5')
      
      # Extract depth element
      depth_elem = xmldoc.elements("depthLayer")
      
      # Extract top and bottom depth values
      depth_range_top = Decimal(depth_elem[0].attrib['min'])
      depth_range_bottom = Decimal(depth_elem[0].attrib['max'])
      
      # Logger for the module
      logger = CSEPLogging.CSEPLogging.getLogger(__name__)

      # Read model_file into the dictionary that is adjusted to the cell/bin 
      # center point
      model_dict = {}
      fhandle = CSEPFile.openFile(model_file)
      counter = 0
      
      # datetime object that reprsents forecast start date (if present in ASCII
      # format data)
      Forecast.__startDate = None
      Forecast.__metadataInFile = False
      
      for line in fhandle.readlines():
         line  = line.strip()
         
         # Parse lines that begin with strings in case they contain metadata
         # about forecast
         if len(line) != 0 and line[0].isalpha() is True:

            Forecast.__parseMetadata(line, xmldoc)
            # Done processing line that contains metadata
            continue
          

         line_tokens = line.split()
         if len(line_tokens) == 0:
            # Reached end of file
            break
         
         counter += 1
          
         min_lon, max_lon, min_lat, max_lat, depth_top, depth_bottom, \
         min_mag, max_mag, rate, mask_bit = line.split()
         
         
         # Check that depth values are within the range
         if Decimal(depth_top) < depth_range_top or \
            Decimal(depth_bottom) > depth_range_bottom:
            error_msg = "Forecast.toXML(): model '%s' contains bin for the depth \
out of '%s' template range [%s;%s] (%s)." %(model_file, 
                                            template_file,
                                            depth_range_top, 
                                            depth_range_bottom,
                                            line)
            logger.error(error_msg)
            
            raise RuntimeError, error_msg
         
         # To handle floating point number representation by different programs
         # (such as Matlab when used to convert forecast model to ASCII), use
         # round() function to the 2nd point of precision
         lon_value = Decimal(min_lon) + lon_range
         lon_value = lon_value.quantize(two_digit_precision)
         
         lat_value = Decimal(min_lat) + lat_range
         lat_value = lat_value.quantize(two_digit_precision)
         
         mag_value = Decimal(min_mag) + mag_range
         mag_value = mag_value.quantize(two_digit_precision)
         
         #logger.debug("Long=%s, lat=%s, magn=%s" \
         #             %(lon_value, lat_value, mag_value))
         
         # Insert entry into the model dictionary
         if not model_dict.has_key(lon_value):
            # There is no entry for longitude yet - create one
            model_dict[lon_value] = {lat_value : []}
            #logger.debug("Model dict: %s" %model_dict)
            
         elif not model_dict[lon_value].has_key(lat_value):
            # There is no latitude entry for registered longitude yet - create one
            model_dict[lon_value][lat_value] = []
            #logger.debug("Model dict: %s" %model_dict)
      
         # Append tuple of magnitude, rate and mask bit for the bin
         model_dict[lon_value][lat_value].append(Forecast.BinInfo(mag_value, 
                                                                  rate, 
                                                                  Decimal(mask_bit)))
      
      # Set provided input metadata for the model if such was not provided in
      # the ASCII format file
      if Forecast.__metadataInFile is False:
         
         # Model name
         if name is not None:
            xml_elements = xmldoc.elements(Forecast.XMLModelName)
            xml_elements[0].text = name
         
         # Start date of the forecast
         if start_date is not None:
            xml_elements = xmldoc.elements(Forecast.XMLStartDate)
            xml_elements[0].text = "%sT%s" %(start_date.date(), start_date.time())
         
         # End date of the forecast
         if end_date is not None:
            xml_elements = xmldoc.elements(Forecast.XMLEndDate)
            xml_elements[0].text = "%sT%s" %(end_date.date(), end_date.time())
         
         # Capture issue date
         now = datetime.datetime.now()
         xml_elements = xmldoc.elements(Forecast.XMLIssueDate)
         xml_elements[0].text = "%sT%s" %(now.date(), now.time())
         

      #logger.debug("Parsed %s lines of model file %s" %(counter, model_file))
      
      ### DEBUG
      #logger.debug("Lon_keys: %s" %model_dict.keys())
      #for key, value in model_dict.items():
      #   print "Lat keys for ", key, " = ", value.keys()
      
         
      # Step through all cell elements in the template
      for cell in xmldoc.elements("cell"):
         
         lon = Decimal(cell.attrib['lon']).quantize(two_digit_precision)
         lat = Decimal(cell.attrib['lat']).quantize(two_digit_precision)
         
         #logger.debug("Cell: long=%s lat=%s" %(lon, lat))

         if model_dict.has_key(lon) is False or \
            model_dict[lon].has_key(lat) is False:
            
            # The whole cell is missing in the model, set cell mask to False,
            # and continue the 'cell' loop
            logger.warning("Forecast.toXML(): %s model is missing cell for lon=%s lat=%s" \
                             %(model_file, lon, lat))
                  
            if model_dict.has_key(lon) is False:
               logger.warning("Forecast.toXML(): (missing lon=%s)" %lon)
            else:
               logger.warning("Forecast.toXML(): (missing lat=%s)" %lat)
               
            cell.attrib[Forecast.CellMaskAttribute] = MatlabLogical.Boolean[False]
            continue
         
         # There are some/all bins available for the cell in the model
         cell.attrib[Forecast.CellMaskAttribute] = MatlabLogical.Boolean[True]
         
         
         # Step through all bin elements, and populate them with model values
         cell_bins = xmldoc.children(cell, "bin") 
         num_bins = len(cell_bins) # number of bins per cell
         mask_zero_bins = 0 # number of bins that have mask bit set to False
         
         for bin in cell_bins:
            mag = Decimal(bin.attrib['m']).quantize(two_digit_precision)
      
            found_model_bin = None
            
            for bin_info in model_dict[lon][lat]:

               if bin_info.magnitude == mag:
                  found_model_bin = bin_info
                  break
               
            if found_model_bin is not None:
               
               # Populate bin values with model's bin
               #bin.clear()
               bin.text = found_model_bin.rate
               
               # Set model's mask bit if it's not set to "True",
               # otherwise it inherits the 'mask' attribute value from parent
               # 'cell' element
               if found_model_bin.mask != mask_on:
                  bin.attrib[Forecast.BinMaskAttribute] = str(found_model_bin.mask)
                  mask_zero_bins += 1
               
            else:
               # model doesn't have a template bin
               bin.attrib[Forecast.BinMaskAttribute] = MatlabLogical.Boolean[False]
               mask_zero_bins += 1

               
         # All bins within a cell have 'False' mask --->
         if mask_zero_bins == num_bins:
            # Instead set mask attribute to False for the whole cell
            cell.attrib[Forecast.CellMaskAttribute] = MatlabLogical.Boolean[False]
            
            # And remove mask attribute from all bins
            for bin in cell_bins:
               del bin.attrib[Forecast.BinMaskAttribute]
               
         
      # Save populated template with model to the file
      fhandle = CSEPFile.openFile(CSEPFile.Name.xml(model_file), 
                                  CSEPFile.Mode.WRITE)
      xmldoc.write(fhandle)
      fhandle.close()
      
      
      return CSEPFile.Name.xml(model_file)

   toXML = staticmethod(toXML)


   #-----------------------------------------------------------------------------
   #
   # toASCII
   # 
   # This method converts forecast data from XML to the ASCII format. It
   # appends Forecast.FromXMLExtension extension to original file to
   # indicate the origin of the file.
   #
   # Input: 
   #         model_file - Filename for forecast in XML format.
   #
   # Output: Filename for ASCII data.
   # 
   def toASCII (model_file):
      """ Convert XML forecast data into ASCII format, and save it to the file.
          The function uses the same filename but with 
          Forecast.FromXMLExtension.Matlab extension to indicate origin of the 
          data."""
          
      # Read XML template in, and get handler to the root element
      xmldoc = CSEPInitFile.CSEPInitFile(model_file, 
                                         [], 
                                         CSEPXML.FORECAST_NAMESPACE)
      
      
      # Extract cell dimensions
      cell_dims = xmldoc.elements("defaultCellDimension")
      
      # Compute half of range to get to the corner point as for original Matlab
      # template
      lat_range = float(cell_dims[0].attrib['latRange']) * 0.5
      lon_range = float(cell_dims[0].attrib['lonRange']) * 0.5
      
      # Extract bin dimensions
      bin_dims = xmldoc.elements("defaultMagBinDimension")
      # Compute half of range to get to the corner point      
      mag_range = float(bin_dims[0].text) * 0.5

      # Extract depth element
      depth_elem = xmldoc.elements("depthLayer")
      
      # Extract top and bottom depth values
      depth_range_top = depth_elem[0].attrib['min']
      depth_range_bottom = depth_elem[0].attrib['max']
      
      # Extract flag for last magnitude bin - open or not
      mag_bin_elem = xmldoc.elements('lastMagBinOpen')[0]
      mag_bin_is_open = MatlabLogical.Boolean[mag_bin_elem.text.strip()]
      
      # Open ASCII file for writing
      ascii_filename = Forecast.fromXMLTemplateFilename(model_file,
                                                        CSEPFile.Extension.ASCII)
      fhandle = CSEPFile.openFile(ascii_filename,
                                  CSEPFile.Mode.WRITE)
      
      # Step through all cell elements in the model
      for cell in xmldoc.elements("cell"):
         
         lon = float(cell.attrib['lon'])
         lat = float(cell.attrib['lat'])

         cell_mask = "1"
         if cell.attrib.has_key(Forecast.CellMaskAttribute):
            cell_mask = cell.attrib[Forecast.CellMaskAttribute]
         
         # Step through all bin elements, and populate them with model values
         cell_bins = xmldoc.children(cell, "bin") 
         
         last_bin_index = len(cell_bins) - 1
         
         for bin_index, bin in enumerate(cell_bins):
            mag = float(bin.attrib['m'])
            
            bin_mask = cell_mask
            if bin.attrib.has_key(Forecast.BinMaskAttribute):
               bin_mask = bin.attrib[Forecast.BinMaskAttribute]
               
            max_mag = str(mag + mag_range)
            
            if bin_index == last_bin_index and \
               mag_bin_is_open == MatlabLogical.Boolean[True]:
               
               # If last bin magnitude is open, set to the highest value
               max_mag = '10.0'

            # Create a tuple of forecast values for the bin and write it to the
            # file
            values = (str(lon - lon_range), 
                      str(lon + lon_range),
                      str(lat - lat_range),
                      str(lat + lat_range),
                      depth_range_top,
                      depth_range_bottom,
                      str(mag - mag_range),
                      max_mag,
                      bin.text,
                      bin_mask)
            
            fhandle.write('\t'.join(values))
            fhandle.write('\n')
            
      # Close the file
      fhandle.close()
      
      return ascii_filename
      
   toASCII = staticmethod(toASCII)


   #-----------------------------------------------------------------------------
   #
   # toASCIIMap
   # 
   # This method converts forecast data from XML to the ASCII format data that
   # is required to generate a forecast map.
   #
   # Input: 
   #         model_file - Filename for forecast in XML format.
   #         dir_path - Result directory to store map-ready file to.
   #         test_name - Name of the test for each forecast map is generated.
   #                     Default is None.
   #         scale_rate - Scale rate to apply to the forecast rates. 
   #                      Default is 1.0.
   #
   # Output: Filename for ASCII map data.
   # 
   def toASCIIMap (model_file, dir_path, test_name = None, scale_rate = 1.0):
      """ Convert XML forecast data into ASCII format ready for GMT map generation,
          and save it to the file. The function uses the same filename but with 
          Forecast.FromXMLExtension.ASCII extension to indicate origin of the 
          data."""
          
      # Open map ASCII file for writing
      map_filename = Forecast.mapFilename(model_file,
                                          dir_path,
                                          test_name)
      # Check if file already exists
      if os.path.exists(map_filename) is True:
      
         return map_filename


      # Read XML forecast in, and get handler to the root element
      xmldoc = CSEPInitFile.CSEPInitFile(model_file, 
                                         [], 
                                         CSEPXML.FORECAST_NAMESPACE)
      
      fhandle = CSEPFile.openFile(map_filename,
                                  CSEPFile.Mode.WRITE)
      
      # Step through all cell elements in the model
      for cell in xmldoc.elements("cell"):
         
         cell_mask = "1"
         if Forecast.CellMaskAttribute in cell.attrib:
            cell_mask = cell.attrib[Forecast.CellMaskAttribute]
            
         cell_rate = 0   
         
         # Sum up all bin rates within the cell - disregard the masking bit for
         # the bin
         
         # Step through all bin elements, and populate them with model values
         cell_bins = xmldoc.children(cell, "bin") 
         
         for bin in cell_bins:
            
            cell_rate += float(bin.text.strip())*scale_rate
                  
         # Create a tuple of forecast values for the bin and write it to the
         # file
         values = (cell.attrib['lon'], 
                   cell.attrib['lat'],
                   repr(cell_rate),
                   cell_mask)
         
         fhandle.write('\t'.join(values))
         fhandle.write('\n')
            
      # Close the file
      fhandle.close()
      
      return map_filename
      
   toASCIIMap = staticmethod(toASCIIMap)


   #--------------------------------------------------------------------------------
   #
   # Utility to convert forecast rates from (EQ per forecast period per km^2) to 
   # (EQ per forecast period per degree^2) units. It expects input forecast
   # in XML format and generates new XML format forecast file with rates 
   # new units rate and to populate XML master template with
   # converted to new units.
   #
   # Input:
   #        forecast_file - Path to the forecast file in XML format
   #        new_forecast_file - Path to the XML format forecast file with rates
   #                             in new units.
   #
   # Output: None
   #
   def kmToDegreeXMLRates(forecast_file, 
                          new_forecast_file): 
      """ Convert forecast rates from (EQ per forecast period per km^2) to 
          (EQ per forecast period per degree^2) units."""
   
      # Set current context for the Decimal numbers
      getcontext.rounding = ROUND_HALF_UP
      
      # Read XML forecast in, and get handler to the root element
      xmldoc = CSEPInitFile.CSEPInitFile(forecast_file, 
                                         [], 
                                         CSEPXML.FORECAST_NAMESPACE)
      
      cell_dims = xmldoc.elements(Forecast.XMLCellDimension)[0]
      lat_half_range = float(cell_dims.attrib[Forecast.XMLLatRange]) * 0.5
      lon_half_range = float(cell_dims.attrib[Forecast.XMLLonRange]) * 0.5


      # Step through all cell elements in the template
      for cell in xmldoc.elements('cell'):
         
         lon = float(cell.attrib['lon'])
         lat = float(cell.attrib['lat'])

         nw_lon = lon - lon_half_range
         nw_lat = lat + lat_half_range
         se_lon = lon + lon_half_range
         se_lat = lat - lat_half_range

         # There is only one bin element per cell, populate it with model rate
         for bin in xmldoc.children(cell, "bin"):
         
            # Populate bin values with model's bin rate
            rate = float(bin.text.strip()) * CSEPGeneric.GeoUtils.areaOfRectangularRegion(nw_lat, nw_lon,
                                                                                          se_lat, se_lon)
            bin.text = str(rate)
      
      
      # Save model with updated rates to the file
      fhandle = CSEPFile.openFile(new_forecast_file, 
                                  CSEPFile.Mode.WRITE)
      xmldoc.write(fhandle)
      fhandle.close()
      
      return
      
   kmToDegreeXMLRates = staticmethod(kmToDegreeXMLRates)

