"""
Module EEPAS0SModel
"""

__version__ = "$Revision$"
__revision__ = "$Id$"

import os

import CSEPFile, Environment, CSEPLogging

from Forecast import Forecast
from EEPASModel import EEPASModel
from EEPAS0FModel import EEPAS0FModel
from CSEPStorage import CSEPStorage


#--------------------------------------------------------------------------------
#
# EEPAS-0S forecast model.
#
# This class is designed to invoke three-months PPE and EEPAS forecast models for 
# the region of Southern California. 
# It uses 'eepas_0s.par' configuration for the run. It has model weights
# disabled.
# Models for Southern California regions must be combined with '0F' configuration
# forecasts for the whole testing area of California before evaluation tests
# are invoked for them.
#
class STEP+EEPASModel (STEP+EEPAS):

    # Static data of the class
    
    # Keyword identifying the type of the forecast
    Type = STEP+EEPASModel.Type + "STEP+EEPAS"
    
    
    #--------------------------------------------------------------------
    #
    # Initialization.
    #
    # Input: 
    #        dir_path - Directory to store forecast file to.
    #        args - Optional arguments for the model. Default is None.
    # 
    def __init__ (self, dir_path, args = None):
        """ Initialization for EEPAS0SModel class"""
        
        STEP+EEPAS
        

    #----------------------------------------------------------------------------
    #
    # Return keyword identifying the model.
    #
    # Input: None.
    #
    # Output:
    #           String identifying the type
    #
    def type (self):
        """ Returns keyword identifying the forecast model type."""
        
        return self.Type


    #----------------------------------------------------------------------------
    #
    # Return object identifying the model for the whole testing region of
    # California.
    #
    # Input: None.
    #
    # Output:
    #         EEPAS0FModel object.
    #
    def fullRegionModel (self):
        """ Returns object identifying the model for the whole testing region of
            California."""
        
        return self.__fullRegionModels
     

    #----------------------------------------------------------------------------
    #
    # Create forecast.
    # This method is overwritten to invoke the full testing region PPE and
    # EEPAS forecasts models if their forecasts files are not generated yet. And
    # to combine forecasts for the whole testing area with forecasts for the
    # Southern California testing region. Once forecasts are combined, Southern
    # California models are archived and not used for evaluation.
    #
    # Input: 
    #       test_date - datetime object that represents testing date.
    #       catalog_dir - Directory with catalog data
    #       archive_dir - Directory to store intermediate model results if any.
    #                     Default is None.    
    #        
    # Output:
    #        Name of created forecast file.
    #
    def create (self, test_date, catalog_dir, archive_dir=None):
        """ Generate forecast."""

        # Invoke the full testing region models if their forecasts files 
        # don't exist yet (checked by create() method)
        result_files = self.__fullRegionModels.create(test_date, 
                                                      catalog_dir,
                                                      archive_dir)

        # Invoke Southern California region forecasts now 
        Forecast.create(self, test_date, catalog_dir, archive_dir)

        # Check if PPE forecast file wasn't generated (EEPAS-0S forecast was staged)
        # and PPE is archived ---> stage the file
        file_is_staged = False
        if os.path.exists(os.path.join(self.dir,
                                       EEPAS0SModel.__ppeForecastFile(test_date))) is False and \
           archive_dir is not None:
           file_is_staged = CSEPStorage.stage(self, 
                                              [EEPAS0SModel.__ppeForecastFile(test_date)],
                                              os.path.join(archive_dir,
                                                           Forecast.archiveDir(EEPAS0SModel.__ppeForecastFile(test_date))))

        if file_is_staged is False:
        
           # PPE model is invoked implicitly by the OS configuration for EEPAS,
           # create metadata file for it.
           comment = "PPE forecast file for Southern California with start date '%s' and end date '%s'" \
                     %(self.start_date, self.end_date)
           
           Forecast.metadata(os.path.join(self.dir, 
                                          EEPAS0SModel.__ppeForecastFile(self.start_date)), 
                             comment,
                             archive_dir)

        
        # Check if combined PPE forecast is already archived ---> stage the file
        combined_filename = "PPE-S_Combined_%s_%s_%s%s" %(self.start_date.month, 
                                                          self.start_date.day, 
                                                          self.start_date.year,
                                                          CSEPFile.Extension.ASCII)        

        file_is_staged = False
        if archive_dir is not None:
           file_is_staged = CSEPStorage.stage(self, 
                                              [combined_filename],
                                              os.path.join(archive_dir,
                                                           Forecast.archiveDir(combined_filename)))
        
        if file_is_staged is False:
           
           # Combine the forecasts, and archive ones for Southern California region
           command = "ln -s %s fort.10" %os.path.join(self.dir, 
                                                      EEPAS0SModel.PPEForecastFile)
           Environment.invokeCommand(command)
           
           command = "ln -s %s fort.11" %os.path.join(self.dir, 
                                                      EEPAS0FModel.PPEForecastFile)
           Environment.invokeCommand(command)
           
           command = "ln -s %s fort.12" %os.path.join(self.dir, 
                                                      combined_filename)
           Environment.invokeCommand(command)

           # If modelers output some debug/progress info to stderr, ignore it - 
           # don't trigger an exception 
           ignore_model_errors = True
           Environment.invokeCommand(EEPASModel.CombineForecastsExecutable,
                                     ignore_model_errors)
           Environment.invokeCommand("rm -rf fort.*")
        
        
           ### Create metadata file for PPE combined forecast
           comment = "PPE forecast for Southern California combined with PPE \
forecast for the whole testing region of California with start date '%s' and end date '%s'" \
                     %(self.start_date, self.end_date)
           
           Forecast.metadata(os.path.join(self.dir, 
                                          combined_filename), 
                             comment,
                             archive_dir)

        result_files.append(combined_filename)
        

        # Check if combined forecast file is archived ---> stage the file
        combined_filename = "%s_Combined_%s_%s_%s%s" %(EEPAS0SModel.Type,
                                                       self.start_date.month, 
                                                       self.start_date.day, 
                                                       self.start_date.year,
                                                       CSEPFile.Extension.ASCII)        
        
        file_is_staged = False
        if archive_dir is not None:
           file_is_staged = CSEPStorage.stage(self, 
                                              [combined_filename],
                                              os.path.join(archive_dir,
                                                           Forecast.archiveDir(combined_filename)))
        
        if file_is_staged is False:
           
           command = "ln -s %s fort.10" %self.filename()
           Environment.invokeCommand(command)
           
           command = "ln -s %s fort.11" %self.__fullRegionModels.filename()
           Environment.invokeCommand(command)
           
           command = "ln -s %s fort.12" %os.path.join(self.dir, 
                                                      combined_filename)
           Environment.invokeCommand(command)
           
           result_files.append(combined_filename)
                   
           Environment.invokeCommand(EEPASModel.CombineForecastsExecutable,
                                     ignore_model_errors)
           Environment.invokeCommand("rm -rf fort.*")


           ### Create metadata file for EEPAS combined forecast
           comment = "EEPAS forecast for Southern California combined with EEPAS \
   forecast for the whole testing region of California with start date '%s' and end date '%s'" \
                     %(self.start_date, self.end_date)
           
           Forecast.metadata(os.path.join(self.dir, 
                                          combined_filename), 
                             comment,
                             archive_dir)
        
        result_files.append(combined_filename)
        
        
        # Archive Southern California forecasts - don't evaluate them
        if archive_dir is not None:
           
           # EEPAS file
           model_path = self.filename()
           
           # Check if file is a soft link - it was staged ===>
           # remove the link, since forecast file is already archived
           if os.path.islink(model_path) is True:

              msg = "create(): removing soft link %s to already archived data file" \
                    %model_path 
              CSEPLogging.CSEPLogging.getLogger(EEPAS0SModel.__name__).info(msg)

              os.remove(model_path)
           else:   
           
              new_path = os.path.join(archive_dir, 
                                      os.path.basename(model_path))
              os.renames(model_path, new_path)
           
           
           # PPE file
           model_path = os.path.join(self.dir,
                                     EEPAS0SModel.PPEForecastFile)
           
           # Check if file is a soft link - it was staged ===>
           # remove the link, since forecast file is already archived
           if os.path.islink(model_path) is True:

              msg = "create(): removing soft link %s to already archived data file" \
                    %model_path 
              CSEPLogging.CSEPLogging.getLogger(EEPAS0SModel.__name__).info(msg)

              os.remove(model_path)
              
           else:   
           
              new_path = os.path.join(archive_dir,
                                      EEPAS0SModel.PPEForecastFile)
              os.renames(model_path, new_path)

        return result_files
     

    #----------------------------------------------------------------------------
    #
    # Writes forecasts output files to the file handle that are specific to the 
    # model configuration - some configurations generate two forecasts files.
    #
    # Input:
    #        fhandle - Handle to the open control file used to invoke the model.
    #
    # Output:
    #        Handle to the control file.
    #        
    def writeForecastFile (self, fhandle):
        """ Writes forecast output file(s) to the file that are specific to the
            model configuration."""

        fhandle.write("PARAMETERS\n")
        fhandle.write("nfile 27\n")
        fhandle.write("endp/\n")
        fhandle.write("FILES\n")
        
        fhandle.write("array '%s'\n" %os.path.join(self.dir, 
                                                   EEPAS0SModel.__ppeForecastFile(self.start_date)))
        fhandle.write("end files\n")
        
        EEPASModel.writeForecastFile(self, fhandle)
        
        return fhandle
     

    #----------------------------------------------------------------------------
    #
    # Writes grid information that are specific to the model configuration.
    #
    # Input:
    #        fhandle - Handle to the open control file used to invoke the model.
    #
    # Output:
    #        Handle to the control file.
    #        
    def writeGrids (self, fhandle):
        """ Writes grid information that are specific to the model configuration."""

        fhandle.write("PPEGRID\n")
        EEPASModel.writeGrids(self, fhandle)
        
        return fhandle
     
        
    #----------------------------------------------------------------------------
    #
    # Returns name of the Southern California area polygon file that is 
    # specific to the model configuration.
    #
    # Input: None
    #
    # Output:
    #        Name of the file.
    #        
    def polygonFile (self):
        """ Returns name of the testing area polygon file that is specific to the 
            model configuration."""

        return 'scalpoly1.dat'
        
        
    #----------------------------------------------------------------------------
    #
    # Returns value for nfile parameter.
    #
    # Input: None
    #
    # Output:
    #        Name of the file.
    #        
    def nfile (self):
        """ Returns value for nfile parameter."""

        return 27


    #----------------------------------------------------------------------------
    #
    # Creates filename for PPE foreacast if it does not exist yet.
    #
    # Input:
    #        start_date - Start date of the forecast.
    #
    # Output:
    #        Handle to the control file.
    #        
    def __ppeForecastFile (start_date):
        """ Creates filename for PPE foreacast."""
        
        if EEPAS0SModel.PPEForecastFile is None:
        
           EEPAS0SModel.PPEForecastFile = "PPE-S_%s_%s_%s%s" %(start_date.month, 
                                                               start_date.day, 
                                                               start_date.year,
                                                               CSEPFile.Extension.ASCII)
        return EEPAS0SModel.PPEForecastFile
     
    __ppeForecastFile = staticmethod(__ppeForecastFile)
     
     
        