"""
Module EEPASModel
"""

__version__ = "$Revision$"
__revision__ = "$Id$"


import os, datetime

import Environment, CSEPFile, CSEPLogging, CSEP
from Forecast import Forecast
from ThreeMonthsForecast import ThreeMonthsForecast
from ReproducibilityFiles import ReproducibilityFiles
from DataSourceFactory import DataSourceFactory
from CatalogDataSource import CatalogDataSource



#--------------------------------------------------------------------------------
#
#
# This class is designed to invoke three-months PPE and EEPAS forecast model. 
# It prepares input catalog data, formats control file with model parameters,
# and invokes the model. It places forecast file under user specified directory.
#
class EEPASModel (ThreeMonthsForecast):

    # Static data of the class
    
    # Keyword identifying the type of the forecast
    Type = "EEPAS"

    # 30-day lag in input catalog - specific to the EEPAS models
    __catalogLagDays = 30
    

    # configuration generates two forecast files: PPE and EEPAS
    PPEForecastFile = None
    
    # Top-level path for the model executable and configuration files
    __NZRootPath = os.environ["NZHOME"]
    _execPath = os.path.join(__NZRootPath,
                              'NewZealandCode',
                              'EEPAS')
    __catalogCoversionExecutable = "reformatscec"
    __executable = "eepasof20ug77"
    __grid="EEPGRID"

    
    CombineForecastsExecutable = os.path.join(_execPath, 'combineforecasts')



    
    
    
    
    #----------------------------------------------------------------------------
    #
    # Initialization.
    #
    # Input: 
    #        dir_path - Directory to store forecast file to.
    #        args - Optional arguments for the model. EEPAS models don't 
    #               require any configurable inputs by the CSEP system.
    #        param_file - Parameter file for a specific model configuration.
    #        use_weights - Flag if model shouuld use weights. Default is 0 (False).
    # 
    def __init__ (self, dir_path, args, param_file, use_weights ):
        """ Initialization for EEPASModel class"""
        
        ThreeMonthsForecast.__init__(self, dir_path)
        
        # Make sure that user is not providing any inputs to the model
        if args is not None:
            error_msg = "%s forecast model does not support any input arguments. \
'%s' input string is provided." %(EEPASModel.Type, args)

            CSEPLogging.CSEPLogging.getLogger(EEPASModel.__name__).error(error_msg)
            raise RuntimeError, error_msg

        # Flag for model weights 
        self.__weights = use_weights
        
        # Filename for input catalog in a model expected format
        self.__modelInputCatalog = CSEPFile.Name.extension(self.inputCatalogFilename(),
                                                           'ref')
        
        # Parameter file for configuration
        self.__paramFile = param_file
 
         

    #----------------------------------------------------------------------------
    #
    # Return keyword identifying the model.
    #
    # Input: None.
    #
    # Output:
    #           String identifying the type
    #
    def type (self):
        """ Returns keyword identifying the forecast model type."""
        
        return self.Type
    #define executable file

    def executableFile (self):
        """ Returns the executable file of the EEPAS model."""

        return self.__executable

    def gridFile (self):
        """ Returns the grid file of the EEPAS model."""

        return self.__grid

    def writeGrids (self, fhandle):
        """ Writes grid information that are specific to the model configuration."""

        fhandle.write("%s\n" %self.__grid)

        return fhandle



    #----------------------------------------------------------------------------
    #
    # Return filename for the input catalog data. It overwrites parent's
    # method because EEPAS is using ASCII format catalog file.
    #
    # Input: None.
    #
    # Output:
    #        String identifying the filename.
    #
    def inputCatalogFilename (self):
        """ Returns filename of input catalog data."""
        
        return CSEPFile.Name.ascii(Forecast.inputCatalogFilename(self))

        
    #----------------------------------------------------------------------------
    #
    # Write input parameter file for the model.
    #
    # Input: None
    #        
    def writeParameterFile (self):
        """ Format input parameter file for the model.
            Created file will be used by Fortran executable that invokes the
            model."""

        fhandle = Forecast.writeParameterFile(self)
        
        # Write generic part of input parameters at the top of the file:
        
        fhandle.write("FILES\n")
        
        fhandle.write("licence '%s'\n" \
                      %os.path.join(EEPASModel._execPath, 'licencekey.dat'))
        
        fhandle.write("normint '%s'\n" \
                      %os.path.join(EEPASModel._execPath, 'normalintegral.dat'))
        
        fhandle.write("grdm '%s'\n" \
                      %os.path.join(EEPASModel._execPath, 'relmmagsnz.dat'))
        
        fhandle.write("grdl '%s'\n" \
                      %os.path.join(EEPASModel._execPath, 'relmblocksnz.dat'))

        fhandle.write("catalog '%s'\n" \
                      %os.path.join(self.catalogDir, self.__modelInputCatalog))
        
        fhandle.write("polygon '%s'\n" %os.path.join(EEPASModel._execPath,
                                                     self.polygonFile()))
        fhandle.write("end files\n")
        
        fhandle.write("POLY/GON  reads in target polygon from file\n")
         
        fhandle.write("CATA/LOGUE selects input data from catalogue file\n")
        
        fhandle.write("all /latmin,latmax,longmin,longmx\n")

#        data_class = DataSourceFactory().classReference(CSEP.DataSource.Type)
        
	fhandle.write("1951, 01, 01, 0, 0, 0.0\n")
#        fhandle.write("%s, %s, %s, %s, %s, %s,\n" \
#                      %(ThreeMonthsModelInputPostProcess.CatalogStartDate.year,
#                        ThreeMonthsModelInputPostProcess.CatalogStartDate.month,
#                        ThreeMonthsModelInputPostProcess.CatalogStartDate.day,
#                        ThreeMonthsModelInputPostProcess.CatalogStartDate.hour,
#                        ThreeMonthsModelInputPostProcess.CatalogStartDate.minute,
#                        ThreeMonthsModelInputPostProcess.CatalogStartDate.second))
        
        catalog_end_date = self.start_date - \
                           datetime.timedelta(days=EEPASModel.__catalogLagDays)
        fhandle.write("%s, %s, %s, %s, %s, %s,\n"
                       %(catalog_end_date.year,
                         catalog_end_date.month,
                         catalog_end_date.day,
                         catalog_end_date.hour,
                         catalog_end_date.minute,
                         catalog_end_date.second))
#        fhandle.write("%s,0.0, 40.0, 5.0 / min mag,depmin, depmax, depmar (depth range in km)\n" % data_class.MinMagnitude)
        fhandle.write("2.95,0.0, 40.0, 5.0 / min mag,depmin, depmax, depmar (depth range in km)\n")
        
        self.writeLocSeis(fhandle)

        fhandle.write("PARAMETERS\n")

        fhandle.write("'%s' /\n" %os.path.join(EEPASModel._execPath,
                                               self.__paramFile))
        
	fhandle.write("PARAMETERS\n")
        fhandle.write("nfile 27\n")
        fhandle.write("endp/\n")
        fhandle.write("FILES\n")
        
        fhandle.write("array '%s'\n" %os.path.join(self.dir, 
                                                   self.ppeForecastFile(self, self.start_date)))
        fhandle.write("end files\n")
        
	self.writeForecastFile(fhandle)
	
        fhandle.write("PERIOD\n")
        fhandle.write("1987, 01, 01, 2006, 03,31/ y1,m1,d1,y2,m2,d2\n")
        fhandle.write("WEIGHTS\n")
        fhandle.write("%s\n" %self.__weights)
        fhandle.write("COMPUTE WEIGHTS\n")
        fhandle.write("LIKELIHOODS\n")
        fhandle.write("PARAMETERS\n")
        fhandle.write("dely %s\n" %EEPASModel.__catalogLagDays)
        fhandle.write("nfile %s\n" %self.nfile())
        
        # Write forecast period
        fhandle.write("gyr1 %s\n" %self.start_date.year)
        fhandle.write("gmo1 %s\n" %self.start_date.month)
        fhandle.write("gdy1 %s\n" %self.start_date.day)
        fhandle.write("ghr1 %s\n" %self.start_date.hour)
        fhandle.write("gmn1 %s\n" %self.start_date.minute)
        fhandle.write("gsc1 %s\n" %self.start_date.second)

        fhandle.write("gyr2 %s\n" %self.end_date.year)
        fhandle.write("gmo2 %s\n" %self.end_date.month)
        fhandle.write("gdy2 %s\n" %self.end_date.day)
        fhandle.write("ghr2 %s\n" %self.end_date.hour)
        fhandle.write("gmn2 %s\n" %self.end_date.minute)
        fhandle.write("gsc2 %s\n" %self.end_date.second)
        fhandle.write("endp/\n")

        self.writeGrids(fhandle)
        fhandle.write("STOP\n")

        # Close the file
        fhandle.close()


    #----------------------------------------------------------------------------
    #
    # Prepare R execution script and invoke the model.
    #
    # Input: None
    #        
    def run (self):
        """ Run EEPAS forecast."""

        # If modelers output some debug/progress info to stderr, ignore it - 
        # don't trigger an exception 
        ignore_model_errors = True


        # Create link for CSEP input catalog and model formatted catalog files if
        # one doesn't exist yet
        if os.path.exists(os.path.join(self.catalogDir, 
                                       self.__modelInputCatalog)) is False:
           
           command = "ln -s %s fort.10" %os.path.join(self.catalogDir,
                                                      self.inputCatalogFilename())
           Environment.invokeCommand(command)
           
           command = "ln -s %s fort.11" %os.path.join(self.catalogDir, 
                                                      self.__modelInputCatalog)
           Environment.invokeCommand(command)
           
           
           # Reformat input catalog into model expected one
           Environment.invokeCommand(os.path.join(EEPASModel._execPath, 
                                                  EEPASModel.__catalogCoversionExecutable),
                                     ignore_model_errors) 
   
           # Clean up temporary links to the catalog files
           Environment.invokeCommand("rm fort.*")
        
        
        # Invoke the model
        command = "%s %s" %(os.path.join(EEPASModel._execPath,
                                         self.executableFile()),
                            self.parameterFile)

        Environment.invokeCommand(command, 
                                  ignore_model_errors)

        # Clean up temporary files generated by the run
        command = "rm -rf *.[0-9][0-9] *.ou* fort.*"
        Environment.invokeCommand(command)


#----------------------------------------------------------------------------
    #
    # Creates filename for PPE foreacast if it does not exist yet.
    #
    # Input:
    #        start_date - Start date of the forecast. Default is None.
    #
    # Output:
    #        Handle to the control file.
    #        
    def ppeForecastFile (self, start_date = None):
        """ Creates filename for PPE foreacast."""
	EEPASModel.PPEForecastFile = "PPE%s_%s_%s_%s%s" %(self.type(),
	                                                     start_date.month, 
                                                             start_date.day, 
                                                             start_date.year,
                                                             CSEPFile.Extension.ASCII)        

        return EEPASModel.PPEForecastFile
     
    ppeForecastFile = staticmethod(ppeForecastFile)

    #----------------------------------------------------------------------------
    #
    # Writes forecasts output files to the file handle that are specific to the 
    # model configuration - some configurations generate two forecasts files.
    #
    # Input:
    #        fhandle - Handle to the open control file used to invoke the model.
    #
    # Output:
    #        Handle to the control file.
    #        
    def writeForecastFile (self, fhandle):
        """ Writes forecast output file(s) to the file that are specific to the
            model configuration."""

        fhandle.write("PARAMETERS\n")
        
        fhandle.write("nfile 28\n")
        fhandle.write("endp/\n")
        fhandle.write("FILES\n")
        fhandle.write("array '%s'\n" %self.filename())
        fhandle.write("end files\n")
        
        return fhandle


    #----------------------------------------------------------------------------
    #
    # Writes grid information that are specific to the model configuration.
    #
    # Input:
    #        fhandle - Handle to the open control file used to invoke the model.
    #
    # Output:
    #        Handle to the control file.
    #        
    def writeGrids (self, fhandle):
        """ Writes grid information that are specific to the model configuration."""

        fhandle.write("%s\n" %self.gridFile())
       
        return fhandle

    def writeLocSeis (self, fhandle):
        """ Writes local seismicity rate, only needed in ERDEEP."""

        return fhandle



    #----------------------------------------------------------------------------
    #
    # Returns name of the testing area polygon file that is specific to the 
    # model configuration.
    #
    # Input: None
    #
    # Output:
    #        Name of the file.
    #        
    def polygonFile (self):
        """ Returns name of the testing area polygon file that is specific to the 
            model configuration."""

        return 'nztestpoly.dat'


    #----------------------------------------------------------------------------
    #
    # Returns value for nfile parameter. Default is 26.
    #
    # Input: None
    #
    # Output:
    #        Name of the file.
    #        
    def nfile (self):
        """ Returns value for nfile parameter. Default is 26."""

        return 26


    #---------------------------------------------------------------------------
    #
    # Return commands that should be used to capture version of external
    # software packages the model is dependent on. 
    #
    # Input: None.
    #
    # Output:
    #           String identifying the type
    #
    @staticmethod
    def externalSoftwareVersions ():
        """ Returns dictionary of command to determine external software 
            version and flag if output of that command is redirected to the
            stderr (True) or not (False) (java -version, for example)."""
        

        return {} 

