"""
Module M8Model
"""

__version__ = "$Revision$"
__revision__ = "$Id$"

import sys, os, datetime

import Environment, CSEPGeneric, CSEPFile, CSEPInputParams
from Forecast import Forecast
from SixMonthsForecast import SixMonthsForecast
from CSEPLogging import CSEPLogging


#-------------------------------------------------------------------------------------
#
# M8 forecast model.
#
# This class is designed to invoke a three-month M8 forecast model for New Zealand
# natural laboratory.
# It prepares input catalog data, formats input file with model parameters, 
# and invokes the M8 model. It places forecast file under user specified 
# directory.
#
class M8Model (SixMonthsForecast):

    # Static data of the class
    
    # Keyword identifying type of the class
    Type = "M8"
    
    # Name of the rateGrid file used for plotting
    #__GridRateFile = "rateGrid_gmt.txt"
    
    __RPackagePath = os.environ['M8_R_PACKAGE']
        
    # Name of the RELM_poly file
    __RELMPolyFile = "mArea.txt"
 
    # Center code path for the model
    #__modelPath = os.path.join(Forecast.CodePath, 
    #                           'NewZealand',
    #                           'M8',
    #                'M8')
    __NZRootPath = os.environ["NZHOME"]
    __modelPath = os.path.join(__NZRootPath, 
                                'NewZealandCode',
                                'M8',
                                'M8')
    
    #--------------------------------------------------------------------
    #
    # Initialization.
    #
    # Input: 
    #        dir_path - Directory to store forecast file to.
    #        args - Optional arguments for the model. Default is None.
    # 
    def __init__ (self, dir_path, args=None):
        """ Initialization for M8Model class"""
        
        SixMonthsForecast.__init__(self, dir_path)

        # Name of the input catalog
        self.__modelInputCatalog = CSEPFile.Name.extension(self.inputCatalogFilename(),
                                                           'dat')


    #--------------------------------------------------------------------
    #
    # Return keyword identifying the model.
    #
    # Input: None.
    #
    # Output:
    #           String identifying the type
    #
    def type (self):
        """ Returns keyword identifying the forecast model type."""
        
        return self.Type

    #--------------------------------------------------------------------
    #
    # Return full path for the result forecast file.
    #
    # Input: None.
    #
    # Output:
    #           String identifying the filename.
    #
    def filename (self):
        """ Returns filename of generated forecast."""

        # Model is generating XML format forecast, change extension for the
        # result filename
        return CSEPFile.Name.xml(Forecast.filename(self))
     
        
    #--------------------------------------------------------------------
    #
    # Write input parameter file for the model.
    #
    # Input: None.
    #        
    def writeParameterFile (self):
        """ Format input parameter file for the model.
            Created file will be used by Matlab script that invokes the
            model."""

        fhandle = Forecast.writeParameterFile(self)
        
        # Start time of the data - to make STEP code happy use later than
        # 1985/1/1 - 1992/1/1
        line = "1 1 1948 0 0 0\n"
        #print "In writeParam"
        fhandle.write(line)
        
        # Test date - append start "hour minute second" as all zero's 
        line = "%s %s %s 0 0 0\n" %(self.start_date.day, 
                                    self.start_date.month, 
                                    self.start_date.year)
        fhandle.write(line)
        
        # Duration in days
        numDays = self.numDays(self.start_date,self.end_date)
        line = "%s\n" %numDays
        fhandle.write(line)
        
        # Path to the input catalog file
        fhandle.write(os.path.join(self.catalogDir, self.__modelInputCatalog) + "\n")
        #print self.__modelInputCatalog 
        
        # Path to the output forecast file
        fhandle.write(self.filename() + "\n")
        
        # Path to the rateGrid file - for plotting
        #fhandle.write(os.path.join(self.catalogDir, self.__GridRateFile) + "\n")

        # Path to the RELM_poly.txt file
        fhandle.write(os.path.join(self.__modelPath, self.__RELMPolyFile) + "\n")
        
        # Close the file
        fhandle.close()


    #----------------------------------------------------------------------------
    #
    # Return filename for the input catalog data. It overwrites parent's
    # method because M8 is using ASCII format catalog file.
    #
    # Input: None.
    #
    # Output:
    #        String identifying the filename.
    #
    def inputCatalogFilename (self):
       """ Returns filename of input catalog data."""

       return CSEPFile.Name.ascii(Forecast.inputCatalogFilename(self))




    #--------------------------------------------------------------------
    #
    # Invoke the model.
    #
    # Input: None
    #        
    def run (self):
        """ Run M8 forecast."""

        # If modelers output some debug/progress info to stderr, ignore it -
        # don't trigger an exception
        ignore_model_errors = False

        
        # Put the path to the parameter file in an environment variable so the M8 code can pick it up
        #os.environ["CONTROLFILE"] = self.parameterFile
        os.environ["CONTROLFILE"] = os.path.join(self.catalogDir, self.parameterFile)
        #print "M8 FILE",os.path.join(self.catalogDir, self.parameterFile)

        # Invoke the model
        command =  "%s CMD BATCH %s/testnew.r\n" %(os.path.join(M8Model.__RPackagePath,
                                                               'bin',
                                                               'R'),
                                                   M8Model.__modelPath) 
        #print command
        
        Environment.invokeCommand(command,
                                  ignore_model_errors)


    #---------------------------------------------------------------------------
    #
    # Return commands that should be used to capture version of external
    # software packages the model is dependent on. 
    #
    # Input: None.
    #
    # Output:
    #           String identifying the type
    #
    @staticmethod
    def externalSoftwareVersions ():
        """ Returns dictionary of command to determine external software 
            version and flag if output of that command is redirected to the
            stderr (True) or not (False) (java -version, for example)."""
        
        return {"%s --version" %os.path.join(M8Model.__RPackagePath,
                                             'bin',
                                             'R') : False} #  R version
