"""
Module STEPJAVAModel
"""

__version__ = "$Revision$"
__revision__ = "$Id$"

import os, datetime

import CSEPFile, CSEPLogging, Environment
from Forecast import Forecast
from OneDayForecast import OneDayForecast


#-------------------------------------------------------------------------------------
#
# STEPJAVA forecast model.
#
# This class is designed to invoke a one-day STEP forecast model. It prepares 
# input catalog data, formats input file with model parameters, and invokes
# the STEP model. It places forecast file under user specified directory.
#
class STEPJAVAModel (OneDayForecast):

    # Static data of the class

    # Keyword identifying type of the class
    Type = "STEPJAVA"

    __backgroundFile = "NZdailyRates.txt"

    # Center code path for the model
    __NZRootPath = os.environ["NZHOME"]
    __modelPath = os.path.join('/data2/csep/AdditionalNZCode/'
                               'STEP_Java',
                               'OpenSHA')

    #__catalogDir = os.path.join(__NZRootPath,
    #                           'NewZealandCode',
    #                           'tmpCatalog')


    #--------------------------------------------------------------------
    #
    # Initialization.
    #
    # Input: 
    #        dir_path - Directory to store forecast file to.
    #        args - Optional arguments for the model. Default is None.
    # 
    def __init__ (self, dir_path, args=None):
        """ Initialization for STEPJAVAModel class"""

        # Use later than actual download start date for the input catalog
        # to make STEP code happy
        OneDayForecast.__init__(self, dir_path)

        # Input arguments for the model are not supported
        if args is not None:

           error_msg = "Input parameters are not supported by '%s' model, provided: %s." \
                       %(self.type(),
                         args)

           CSEPLogging.CSEPLogging.getLogger(STEPJAVAModel.__name__).error(error_msg)
           raise RuntimeError, error_msg


    #--------------------------------------------------------------------
    #
    # Return keyword identifying the model.
    #
    # Input: None.
    #
    # Output:
    #           String identifying the type
    #
    def type (self):
        """ Returns keyword identifying the forecast model type."""

        return self.Type


    #--------------------------------------------------------------------
    #
    # Write input parameter file for the model.
    #
    # Input: None.
    #        
    def writeParameterFile (self):
        """ Format input parameter file for the model."""


        fhandle = Forecast.writeParameterFile(self)

        # Start time of the data - to make STEP code happy use later than
        # 1985/1/1 - 1992/1/1
        line = "1 1 1990 0 0 0\n"
        fhandle.write(line)


        # Test date - append start "hour minute second" as all zero's 
        line = "%s %s %s 0 0 0\n" \
               %(self.start_date.day,
                 self.start_date.month,
                 self.start_date.year)
        fhandle.write(line)

        # Duration in days
        line = "1\n"
        fhandle.write(line)

        # Path to the input catalog file
        fhandle.write(os.path.join(self.catalogDir,
                                   CSEPFile.Name.extension(self.inputCatalogFilename(),
                                                           'dat')+ "\n"))


#        # Path to the input catalog file
#        fhandle.write(os.path.join(self.catalogDir,
#                                   self.inputCatalogFilename()) + "\n")

#         # Path to the output forecast file
#         fhandle.write(self.filename() + "\n")


        # Path to the output forecast file
        fhandle.write(self.filename() + "\n")

        # Path to the CSEP_background file
        fhandle.write(os.path.join(self.__modelPath,
                                   self.__backgroundFile) + "\n")

        # Close the file
        fhandle.close()



    #---------------------------------------------------------------------------
    #
    # Invoke the model.
    #
    # Input: None
    #        
    def run (self):
        """ Run STEP forecast."""

        cwd = os.getcwd()
        os.chdir(os.path.join(STEPJAVAModel.__modelPath,
                              'build'))

        try:
            # Model requests to change to the model installation directory
            # since it's using relative paths to locate other files
            __command = "java -jar lib/step-aftershock.jar 0 -f %s" %os.path.join(cwd,
                                                                                  self.parameterFile)

            Environment.invokeCommand(__command)

        finally:
            # Go back to the directory
            os.chdir(cwd)


    #---------------------------------------------------------------------------
    #
    # Return commands that should be used to capture version of external
    # software packages the model is dependent on. 
    #
    # Input: None.
    #
    # Output:
    #           String identifying the type
    #
    @staticmethod
    def externalSoftwareVersions ():
        """ Returns dictionary of command to determine external software 
            version and flag if output of that command is redirected to the
            stderr (True) or not (False) (java -version, for example)."""


        return {"java -version" : True} # Java output info to stderr


