`M8` <-
function (catalogue, M0, centrelong, centrelat, radius = radius.M8default(M0), 
    minday, start.series, training = "user", end.training, time.breaks,
    running.total = 12, smoother = 6, TIP.length = 10, plotit = TRUE,
    title = "M8 Series and TIPs") 
{
    attributes(minday) <- attributes(catalogue[, "time"])
    attributes(start.series) <- attributes(catalogue[, "time"])
    attributes(end.training) <- attributes(catalogue[, "time"])
    if (!is.data.frame(catalogue) || !any(names(catalogue) == 
        "n.aftershocks")) {
        stop("Please use decluster.M8() function to get mainshocks\n")
    }
    if (minday != min(catalogue[, "time"])) 
        catalogue <- catalogue[catalogue[, "time"] >= minday, ]
    series <- M8.series(catalogue = catalogue, M0 = M0,
        centrelong = centrelong, centrelat = centrelat, radius = radius,
        minday = minday, start.series = start.series, training = training,
        end.training = end.training, time.breaks = time.breaks)
    if (!is.character(series)) {
        res <- M8.TIP(series, M0, training = training,
            end.training = end.training, smoother = smoother,
            TIP.length = TIP.length)
        if (plotit) {
            plot.M8(res, title = title)
        }
        return(res)
    }
    else {
        return("Result not available.")
    }
}

`M8.TIP` <-
function (series, M0, training = "user", end.training, smoother = 6, 
    TIP.length = 10) 
{
    x <- series$series
    n <- nrow(x)
    if (n/2 < 7) 
        stop("Not enough years of data to generate TIP!!\n")
    alpha <- c(0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.25)
    max.mag <- series$max.events[, "magnitude"]
    tops <- matrix(NA, nrow = n, ncol = 7)
    attributes(tops) <- attributes(x[, 1:7])
    exceeds <- matrix(FALSE, nrow = n, ncol = 7)
    attributes(exceeds) <- attributes(x[, 1:7])
    percent <- matrix(NA, nrow = n, ncol = 7)
    attributes(percent) <- attributes(x[, 1:7])
    percent.3years <- matrix(NA, nrow = n, ncol = 7)
    attributes(percent.3years) <- attributes(x[, 1:7])
    percent6 <- rep(NA, n)
    attributes(percent6) <- attributes(x[, 1])
    percent.all <- rep(NA, n)
    attributes(percent.all) <- attributes(x[, 1])
    g <- rep(0, n)
    attributes(g) <- attributes(x[, 1])
    h <- rep(0, n)
    attributes(h) <- attributes(x[, 1])
    TIP <- rep(FALSE, n)
    attributes(TIP) <- attributes(x[, 1])
    TIP.type <- rep("", n)
    for (j in 1:7) {
        xj <- as.vector(x[, j])
        if (training == "user") {
            i <- which(x[, "time"] >= end.training)[1]
            tops[1:n, j] <- quantile(xj[1:i], probs = 1 - alpha[j])
            exceeds[1:n, j] <- (x[1:n, j] >= tops[1:n, j])
	    if (all(x[1:i, j] == 0)) {
		percent[1:n, j] <- 0
	    } else {
                percent[1:n, j] <- emp.cdf(x[1:i, j], x[1:n, j])
	    }
            for (ii in 1:n) percent.3years[ii, j] <- max(percent[max(1, 
                (ii - (smoother - 1))):ii, j])
        }
        else {
            stop(paste("Unknown value for training: ", training))
        }
    }
    i.TIP <- 1
    TIP[1] <- NA
    TIP.type[1] <- ""
    for (i in i.TIP:n) {
	savoptions <- options()
        options(warn = -1)
        g[i] <- ((sum(exceeds[max(i - 5, 1):i, 1:2]) > 0) + 
            (sum(exceeds[max(i - 5, 1):i, 3:4]) > 0) + (sum(exceeds[max(i - 
            5, 1):i, 5:6]) > 0) + (sum(exceeds[max(i - 5, 
            1):i, 7]) > 0))
        h[i] <- sum(apply(exceeds[max(i - 5, 1):i, , drop = FALSE], 
            2, sum) > 0)
        percent6[i] <- sort(as.vector(percent.3years[i, 1:6]))[2]
        percent.all[i] <- min(percent6[i] - (1 - alpha[1]), 
            percent.3years[i, 7] - (1 - alpha[7]))
        options(savoptions)
        if (i > i.TIP && g[i] >= 4 && h[i] >= 6 && g[i - 1] >= 
            4 && h[i - 1] >= 6) {
            TIP[i:min(i + (TIP.length - 1), n)] <- TRUE
            if (max(max.mag[(i - 1):i], na.rm=TRUE) >= M0) {
                TIP.type[i:min(i + (TIP.length - 1), n)] <- "c.e."
            }
            else if (i < n && max(max.mag[(i + 1):min(n, i + TIP.length)], 
	          na.rm=TRUE) >= M0) {
                TIP.type[i:min(i + (TIP.length - 1), n)] <- "STIP"
	    }
            else if (i < n && max(max.mag[(i + 1):min(n, i + 
                TIP.length)], na.rm=TRUE) >= M0 - 0.5) {
                TIP.type[i:min(i + (TIP.length - 1), n)] <- "STIP-"
            }
            else if (i + (TIP.length - 1) >= n) {
                TIP.type[i:min(i + (TIP.length - 1), n)] <- "CTIP"
            }
            else {
                TIP.type[i:(i + (TIP.length - 1))] <- "FTIP"
            }
            if (TIP.type[i] == "c.e.") {
                tmp.i <- ((i - 1):i)[series$max.events[(i - 1):i, 
                  "magnitude"] >= M0]
            }
            else if (TIP.type[i] == "STIP") {
                tmp.i <- ((i + 1):min(i + 10, n))[series$max.events[(i + 
                  1):min(i + TIP.length, n), "magnitude"] >= 
                  M0]
            }
            else if (TIP.type[i] == "STIP-") {
                tmp.i <- ((i + 1):min(i + 10, n))[series$max.events[(i + 
                  1):min(i + TIP.length, n), "magnitude"] >= 
                  M0 - 0.5]
            }
        }
    }
    x <- list(series = series$series, max.events = series$max.events, 
        tops = tops, exceeds = exceeds, TIP = TIP, TIP.type = TIP.type, 
        TIP.level = percent.all, training = training, M0 = M0, 
        g = g, h = h)
    class(x) <- "M8"
    return(x)
}

`M8.series` <-
function (catalogue, M0, centrelong = mean(catalogue[, "longitude"]), 
    centrelat = mean(catalogue[, "latitude"]), radius = radius.M8default(M0), 
    minday, start.series, training = "user", end.training, time.breaks,
    running.total = 12) 
{
    if (!is.data.frame(catalogue) || !any(names(catalogue) == 
        "n.aftershocks")) 
        stop("Please use decluster.M8() function to get mainshocks\n")
    if (years1(max(catalogue[, "time"])) - years1(min(catalogue[, 
        "time"])) < 12) 
        stop("Not enough years of data avalable in catalogue.\n")
    if (years1(start.series) - years1(min(catalogue[, "time"])) < 
        10) 
        stop("Series generation start too early.\n")
    interval.start <- start.series
    sel <- in.circle(catalogue, centrelong = centrelong, centrelat = centrelat, 
        radius = radius)
    catalogue <- catalogue[sel, ]
    n <- nrow(catalogue)
    data.start <- catalogue[1, "time"]
    data.end <- catalogue[n, "time"]
    if (training == "user") {
        in.training <- (catalogue[, "time"] >= start.series & 
            catalogue[, "time"] < end.training)
        n.training <- sum(in.training)
        data.length <- (end.training - start.series)/365.2422
        n.CAT20 <- data.length * 20
        n.CAT10 <- data.length * 10
        if (n.CAT20 > n.training) {
            return("Result not available")
        }
        ordered.mags <- rev(sort(catalogue[in.training, "magnitude"]))
        cut.CAT20 <- ordered.mags[n.CAT20]
        cut.CAT10 <- ordered.mags[n.CAT10]
    }
    in.CAT20 <- (catalogue[, "magnitude"] >= cut.CAT20)
    in.CAT10 <- (catalogue[, "magnitude"] >= cut.CAT10)
    in.CAT20a <- in.CAT20 & (catalogue[, "magnitude"] < M0 - 0.5)
    in.CAT10a <- in.CAT10 & (catalogue[, "magnitude"] < M0 - 0.5)
    in.CATMS <- (catalogue[, "magnitude"] >= M0 - 2 &
        catalogue[, "magnitude"] < M0 - 0.2)
    n <- length(time.breaks)
    if (time.breaks[n] < end.training)
        return("Result not available")
    cover.breaks <- (time.breaks[1] <= catalogue[, "time"] & 
        catalogue[, "time"] <= time.breaks[n])
    catalogue <- catalogue[cover.breaks, ]
    n.group <- length(time.breaks) - 1
    group.labels <- format(time.breaks)
    group.labels.end <- format(time.breaks - 1)
    group.labels <- paste(substring(group.labels[-length(group.labels)], 
        1, 9), "-", substring(group.labels.end[-1], 1, 9), sep = "")
    group <- cut(catalogue[, "time"], time.breaks, labels = group.labels, 
        include.lowest = TRUE)
    adj.first <- 1
    adj.last <- 1
    basic.sum20 <- c(table(group[in.CAT20]))
    basic.sum20[1] <- round(basic.sum20[1] * adj.first)
    basic.sum20[n.group] <- round(basic.sum20[n.group] * adj.last)
    basic.sum10 <- c(table(group[in.CAT10]))
    basic.sum10[1] <- round(basic.sum10[1] * adj.first)
    basic.sum10[n.group] <- round(basic.sum10[n.group] * adj.last)
    basic.sum20a <- c(table(group[in.CAT20a]))
    basic.sum10a <- c(table(group[in.CAT10a]))
    basic.exp20a <- tapply(catalogue[in.CAT20a, "magnitude"], 
        list(group[in.CAT20a]), "M8.wgt.sum")
    basic.exp10a <- tapply(catalogue[in.CAT10a, "magnitude"], 
        list(group[in.CAT10a]), "M8.wgt.sum")
    basic.exp20a[is.na(basic.exp20a)] <- 0
    basic.exp10a[is.na(basic.exp10a)] <- 0
    basic.maxB <- tapply(catalogue[in.CATMS, "n.aftershocks"], 
        list(group[in.CATMS]), "max")
    basic.maxB[is.na(basic.maxB)] <- 0
    i0 <- which.min(abs(time.breaks - start.series))
    time.breaks <- time.breaks[-1]
    basic.maxM <- group.max.mag(catalogue, group)
    cumsum.1 <- cumsum(basic.sum20)
    series.1 <- cumsum.1[i0:n.group] - cumsum.1[(i0 - running.total):(n.group - 
        running.total)]
    cumsum.2 <- cumsum(basic.sum10)
    series.2 <- cumsum.2[i0:n.group] - cumsum.2[(i0 - running.total):(n.group - 
        running.total)]
    series.3 <- series.1 - cumsum.1[(i0 - running.total):(n.group - 
        12)]/(((i0 - running.total):(n.group - running.total))/running.total)
    series.4 <- series.2 - cumsum.2[(i0 - running.total):(n.group - 
        12)]/(((i0 - running.total):(n.group - running.total))/running.total)
    cumsum.5 <- cumsum(basic.sum20a)
    cumsum.5e <- cumsum(basic.exp20a)
    series.5 <- ((cumsum.5e[i0:n.group] - cumsum.5e[(i0 - running.total):
	(n.group - running.total)])/(cumsum.5[i0:n.group] - cumsum.5[
	(i0 - running.total):(n.group - running.total)])^(0.67))
    cumsum.6 <- cumsum(basic.sum10a)
    cumsum.6e <- cumsum(basic.exp10a)
    series.6 <- ((cumsum.6e[i0:n.group] - cumsum.6e[(i0 - running.total):
	(n.group - running.total)])/(cumsum.6[i0:n.group] - cumsum.6[
	(i0 - running.total):(n.group - running.total)])^(0.67))
    series.7 <- as.vector(apply(cbind(basic.maxB[i0:n.group], 
        basic.maxB[(i0 - 1):(n.group - 1)]), 1, "max"))
    series <- ts(cbind(F1 = series.1, F2 = series.2, F3 = series.3, 
        F4 = series.4, F5 = series.5, F6 = series.6, F7 = series.7, 
        time = time.breaks[-(1:(i0 - 1))]), start = years1(start.series), 
        frequency = 2)
    dimnames(series)[[2]] <- c("F1", "F2", "F3", "F4", "F5", 
        "F6", "F7", "time")
    return(list(series = series, max.events = basic.maxM[-(1:(i0 - 1)), ]))
}

`decluster.M8` <-
function (catalogue, cutoff.mag = 4, minday = min(catalogue$time), 
    mindepth = 0, maxdepth = Inf, decluster.name, 
    magn.window = c(3, 3.5, 4, 4.5, 5, 5.5, 6, 6.5, 7, 7.5, 8), 
    space.window = c(30, 40, 40, 40, 50, 50, 50, 100, 100, 150, 200),
    time.window = c(6, 11, 23, 46, 91, 183, 274, 365, 730, 913, 1096)) 
{
    start.compute <- proc.time()[3]
    del <- (is.na(catalogue[, "time"]) | is.na(catalogue[, "magnitude"]) | 
        is.na(catalogue[, "longitude"]) | is.na(catalogue[, "latitude"]) | 
        is.na(catalogue[, "depth"]))
    new.catalogue <- catalogue[!del, c("time", "magnitude", "longitude", 
        "latitude")]
    del2 <- (new.catalogue[, "magnitude"] < cutoff.mag | catalogue[!del, 
        "depth"] < mindepth | catalogue[!del, "depth"] >= maxdepth)
    new.catalogue <- new.catalogue[!del2, ]
    del[!del] <- del2
    n0 <- nrow(new.catalogue)
    n.aftershocks <- rep(0, n0)
    order.magnitude <- order(-new.catalogue[, "magnitude"], new.catalogue[, 
        "time"])
    sel <- as.integer(row.names(new.catalogue))
    res <- .Fortran("recat", as.integer(n0),
        as.double(as.matrix(new.catalogue)), 
        as.integer(order.magnitude),
        sel = as.integer(sel),
        n.aftershocks = as.integer(n.aftershocks), 
        magn.window = as.single(magn.window),
        space.window = as.single(space.window), 
        time.window = as.single(time.window),
        as.integer(length(magn.window)), 
        NAOK = TRUE, PACKAGE = "ssM8")
    is.main <- (res$sel == 0)
    catalogue.main <- catalogue
    catalogue.main[, "n.aftershocks"] <- 0
    catalogue.main[!del, "n.aftershocks"] <- res$n.aftershocks
    catalogue.main[del, "n.aftershocks"] <- NA
    catalogue.main <- catalogue.main[!del, ][is.main, ]
    catalogue.main <- catalogue.main[catalogue.main[, "time"] >= 
        minday, ]
    if (minday != min(catalogue$time)) {
        class(minday) <- "datetimes"
    }
    attr(catalogue.main, "catname") <- decluster.name
    assign(decluster.name, catalogue.main, envir = globalenv())
    return(invisible(res$sel))
}

`get.synoptic.events` <-
function (centres, critical, M1, R1, maxdepth = Inf, mindepth = 0, lag = 1) 
{
    as.catalogue(subsetrect(NZ, minmag = M1, mindepth = mindepth, 
        minday = critical$time.breaks[lag + 1], maxdepth = maxdepth, 
        maxday = max(critical$time.breaks), report.count = FALSE), 
        "temp")
    centres1 <- makeborder(centres)
    d <- d1  <- rep(NA, nrow(temp))
    for (i in 1:nrow(temp)) {
	d[i] <- which.min(arcdist(c(temp$longitude[i], 
            temp$latitude[i]), cbind(centres$longitude, centres$latitude)))
	d1[i] <- which.min(arcdist(c(temp$longitude[i], 
            temp$latitude[i]), cbind(centres1$longitude, centres1$latitude)))
    }
    temp <- temp[d == d1, ]
    temp$circle.num <- rep(NA, nrow(temp))
    for (i in 1:nrow(temp))
        temp$circle.num[i] <- order(arcdist(c(temp$longitude[i], 
            temp$latitude[i]), cbind(centres$longitude, centres$latitude)))[1]
    temp$time.int <- rep(NA, nrow(temp))
    for (i in 1:nrow(temp))
        temp$time.int[i] <- seq(1, length(critical$time.breaks))[
	    (temp$time[i] < critical$time.breaks)][1] - 1
    n <- length(centres$longitude)
    series <- matrix(0, nrow = length(critical$time.breaks) - 1, ncol = n)
    for (i in 1:nrow(temp))
        series[temp$time.int[i], temp$circle.num[i]] <- 1
    active.centres <- seq(1, n)[!is.na(critical$series[1, ])]
    series <- series[((lag + 1):(length(critical$time.breaks) - 1)),
        active.centres]
    return(list(series = series, cat = temp))
}

`getcrit` <-
function (minday, mindepth, maxdepth, minmag, M0, start.series, 
    end.training, centres) 
{
    n <- length(centres$longitude)
    as.catalogue(subsetrect(NZ, minday = minday, maxdepth = maxdepth, 
        minmag = minmag, mindepth = mindepth, maxday = end.training,
	report.count = FALSE), "temp")
    temp <- get("temp")
    decluster.M8(temp, cutoff.mag = minmag, decluster.name = "NZ.mainshocks")
    NZ.mainshocks <- get("NZ.mainshocks")
    timeints <- rev(seq(end.training, minday, -183))
    attributes(timeints) <- attributes(temp[, "time"])
    time.breaks <- as.numeric(timeints)

    analysis <- function(i, M0) {
        M8res <- M8(NZ.mainshocks, M0 = M0, plotit = FALSE,
            centrelong = centres$longitude[i], 
            centrelat = centres$latitude[i], minday = minday, 
            start.series = start.series, end.training = end.training,
	    time.breaks = time.breaks)
        if (is.atomic(M8res)) {
            TIPlevel <- NULL
        }
        else {
            TIPlevel <- M8res$TIP.level
        }
        assign(paste("temp", i, sep = ""), TIPlevel, pos = 1)
    }
    for (i in 1:n) analysis(i, M0)

    timeints <- timeints[which.min(abs(timeints - start.series)):
	length(timeints)]
    class(timeints) <- "datetimes"
    time.breaks <- as.numeric(timeints)
    timeints <- paste(substring(format(timeints[-length(timeints)]), 
        1, 9), "-", substring(format(timeints[-1] - 1), 1, 9), sep = "")
    m <- length(timeints)

    series <- matrix(NA, nrow = m, ncol = n)
    for (i in 1:n) {
        x <- get(paste("temp", i, sep = ""))
        if (is.null(x)) 
            series[, i] <- rep(NA, m)
        else series[, i] <- x
    }
    dimnames(series) <- list(timeints, NULL)
    rm(list = paste("temp", seq(1, n), sep = ""), pos = 1)
    rm(NZ.mainshocks, pos = 1)
    rm(temp, pos = 1)
    return(list(series = series, time.breaks = time.breaks, M0 = M0, 
        start.series = start.series, end.training = end.training, 
        mindepth = mindepth, maxdepth = maxdepth, minday = minday, 
        maxday = end.training, minmag = minmag))
}

`makeborder` <-
function (centres) {
  border <- list(longitude = numeric(0), latitude = numeric(0))
  deltalat <- min(diff(lats <- sort(unique(centres$latitude))))
  for (i in 1:length(lats)) {
    deltalong <- min(diff(longs <-
      sort(centres$longitude[centres$latitude == lats[i]])))
    if (i == 1) {	# minimum latitude
      lat <- lats[1] - deltalat
      border$longitude <- c(border$longitude, longs[1] - deltalong,
        longs, max(longs) + deltalong)
      border$latitude <- c(border$latitude, rep(lat, length(longs) + 2))
    } else if (i == length(lats)) { # maximum latitude
      lat <- lats[i] + deltalat
      border$longitude <- c(border$longitude, longs[1] - deltalong,
        longs, max(longs) + deltalong)
      border$latitude <- c(border$latitude, rep(lat, length(longs) + 2))
    }
    border$longitude <- c(border$longitude, range(longs) + c(-1, 1)*
	deltalong)
    border$latitude <- c(border$latitude, rep(lats[i], 2))
  }
  return(list(longitude = c(centres$longitude, border$longitude),
    latitude =  c(centres$latitude, border$latitude)))
}

`plotforecast` <-
function (centres, critical, events, forecast, pdffile, title1 = TRUE,
    title2 = "", bounds = c(0, .01, .02, .05, .07, .09, 1)) 
{
    require(maps, quietly = TRUE)
    time.breaks <- critical$time.breaks
    xlim <- c(165, 180)
    ylim <- c(-48, -33.9)
    colour.brightness <- 2
    colours <- c("blue", "cyan", "green", "yellow", "tan", "red")
    graphics.off()
    palette(c("black", "gray35", paste(colours, colour.brightness, 
        sep = ""), "white"))
    if (!missing(pdffile))
      pdf(pdffile, width=8.25, height=11.667)
    par(mar = c(7.1, 4.1, 4.1, 2.1))
    par(font.main=1, font.axis=1, font.lab=1, font.sub=1,
      cex.main=1.7, cex.axis=0.9, cex.lab=1, cex.sub=1)
    code <- parse(text = paste(c(bounds[1], paste(rep("<= {", 6),
        colours, rep("< {", 6), bounds[-1]), rep("}", 12)), collapse = " "))
    latitude <- centres$latitude
    longitude <- centres$longitude
    map("nz", fill = FALSE, col = 1, xlim = xlim, ylim = ylim, resolution = 1)
    axis(1, at = seq(165, 180, 5))
    axis(2, at = seq(-47, -35, 3), las = 0)
    box()
    forecast <- forecast*ndays/183
    for (k in seq(2, length(bounds))) {
        a <- (forecast >= bounds[k - 1] & forecast < bounds[k])
	if (sum(a, na.rm = T) > 0) {
	    symbols(x = longitude[a], y = latitude[a], fg = k + 1,
		bg = k + 1, add = TRUE, inches = FALSE,
		rectangles = matrix(0.09, length(longitude[a]), 2))
	}
    }
    map("nz", resolution = 1, add=TRUE)
    if (title1 == TRUE) 
        title1a <- paste("M8 Synoptic Forecast:", ForecastPeriod)
#            dimnames(critical$series)[[1]][length(time.breaks) - 1])
    if (title1 == FALSE) 
        title1a <- ""
    mtext(title1a, outer = FALSE, side = 3, line = 3.5,
      cex = par()$cex.main/1.2)
    mtext(title2, outer = FALSE, side = 3, line = 2, cex = par()$cex.main/1.5)
    title(sub = code, line = 4)
    if (!missing(pdffile))
        dev.off()
}

`runsynop` <-
function(centres, M0 = 7, M1 = 5.7, mindepth = 0,
    maxdepth = Inf, minmag = 3.0, minday = julian(1, 1, 1987),
    start.series = yearsago(10, from = catend), end.training = catend,
    plotit = FALSE, plotfile, generatecritical = TRUE,
    savecritical = FALSE) {
  require(ssM8, quietly = TRUE)
  if (generatecritical) {
    crit <- getcrit(minday = minday, mindepth = mindepth,
      maxdepth = maxdepth, minmag = minmag, M0 = M0, start.series =
      start.series, end.training = end.training, centres = centres)
    if (savecritical) save(crit, file="crit.rda")
  } else
    load("crit.rda")

  synopevents <- get.synoptic.events(centres, crit, M1 = M1,
    R1 = radius.M8default(M1), maxdepth = 39.99, lag = 1)

  a <- !is.na(crit$series[1, ])
  xbar <- synoptic.forecast(centres, crit, radius.M8default(M1),
    list(series = crit$series[, a]), weight = "uniform", sigma = 1)
  if (is.null(xbar)) {
    print("No Forecast available")
    break
  } else {
    z <- data.frame(explanatory = as.numeric(xbar$series[-1, ]),
      response = as.numeric(synopevents$series > 0))
    model <- glm(cbind(response, 1 - response) ~ explanatory, data = z,
      family = binomial(link = logit), control = glm.control(maxit = 30))
    print(paste("P-Value =", 1 - pchisq(-model$deviance + model$null.deviance, 1)))

    synop <- list(series=matrix(fitted(model), ncol=sum(a)))
    next6m <- rep(NA, length(crit$series[1, ]))
    next6m[a] <- synop$series[nrow(synop$series), ]

    if (plotit) {
      if (missing(plotfile)) plotfile <- "temp.pdf"
      plotforecast(centres, crit, NULL, next6m, plotfile,
	title2 = substitute(list(M[0]==M0, M[1]==M1), list(M0=M0, M1=M1)),
        bounds = c(0, 2.0e-5, 3.0e-5, 5.0e-5, 1.0e-4, 5.0e-4, 1))
    }
    return(next6m)
  }
}

`synoptic.forecast` <-
function (centres, critical, R1, forecast, sigma = NA, weight = "cone") 
{
    if ((weight != "cone") & (weight != "uniform")) 
        weight <- "cone"
    latitude <- centres$latitude[(!is.na(critical$series[1, ]))]
    longitude <- centres$longitude[(!is.na(critical$series[1, ]))]
    pts <- cbind(longitude, latitude)
    n <- nrow(pts)
    if (n > 0) {
      if (is.na(sigma)) 
          sigma <- centres$d^2/(pi * R1^2)
      w <- matrix(NA, nrow = n, ncol = n)
      for (i in 1:n) {
          if (weight == "cone") {
              a <- 1 - arcdist(pts[i, ], pts)/R1
              a[a < 0] <- 0
          }
          if (weight == "uniform") 
              a <- (arcdist(pts[i, ], pts) <= R1)
          w[i, ] <- sigma * a/sum(a)
      }
      z <- forecast$series %*% t(w)
      return(list(series = z))
  } else return(NULL)
}

`today` <-
function() {
  thisdate <- as.POSIXlt(Sys.time())
  thisyear <- thisdate$year + 1900
  thismonth <- thisdate$mon + 1
  thisday <- thisdate$mday
  return(julian(x = thismonth, d = thisday, y = thisyear))
}

`yearsago` <-
function(n = 10, from) {
  if (missing(from)) from <- today()
  return(julian(x = months1(from), d = days1(from), y = years1(from) - round(n)))
}

