"distance.globe" <-
function(a1, a2, earthradius = 6371)
{
	#   Distance on earth given longitude and latitude
	RAD <- pi/180
	x1 <- cos(a1$latitude * RAD) * cos(a1$longitude * RAD)
	y1 <- cos(a1$latitude * RAD) * sin(a1$longitude * RAD)
	z1 <- sin(a1$latitude * RAD)
	x2 <- cos(a2$latitude * RAD) * cos(a2$longitude * RAD)
	y2 <- cos(a2$latitude * RAD) * sin(a2$longitude * RAD)
	z2 <- sin(a2$latitude * RAD)	
	# the above is globe coordinate (1, longit, latit) to
	# Decarsian coordinate (x, y, z)
	linearDist <- sqrt((x1 - x2)^2 + (y1 - y2)^2 + (z1 - z2)^2)	
	# distance in linear way on unit globe
	globeDist <- 2 * earthradius * asin(linearDist/2)	
	# linear distance /2 = earthradius * sin(angle/2)
	return(globeDist)
}


"firstGE" <-
function(vec, value)
{
	#   Find index of the first element which is >= a give value
	vec <- c(vec)
	vec[is.na(vec)] <-  - Inf
	return(((1:length(vec))[vec >= value])[1])
}


"firstLE" <-
function(vec, value)
{
	#   Find index of the first element which is <= a give value
	vec <- c(vec)
	vec[is.na(vec)] <- Inf
	return(((1:length(vec))[vec <= value])[1])
}


"group.max.mag" <-
function(catalogue, group)
{
	#--------------------------------------------------
	#     group.max.mag()
	#  Get records with maximum magnitudes in each group
	#  Input:
	#    catalogue --- mainshocks catalogue, data frame
	#    group  --- catalogue grouping, sorted ascendingly
	#--------------------------------------------------
	n <- nrow(catalogue)
	ng <- max(as.numeric(group))
	#  first.recs <- c(1, diff(group))
	#  group.start <- (1:n)[as.logical(first.recs)]
	group.start <- tapply(1:n, list(group), min)
	inds <- tapply(catalogue[, "magnitude"], list(group), maxIndex)
	catalogue[group.start + inds - 1,  ]
}


"in.circle" <-
function(c, centrelong = mean(c[, "longitude"]), centrelat = mean(c[, 
	"latitude"]), radius, M0)
{
	#------------------------------------------------------------
	#     Creates a boolean array for events in catalogue c satisfying
	#     circular restriction and other criteria. Use it to
	#     get the indece of a CI from a whole catalogue
	#------------------------------------------------------------
	if(missing(radius)) {
		if(!missing(M0)) {
			radius <- radius.M8default(M0)
		}
		else {
			stop("in.circle(): radius not defined\n")
		}
	}
	dist <- distance.globe(list(longitude = centrelong, latitude = 
		centrelat), list(longitude = c[, "longitude"], latitude = c[, 
		"latitude"]))
	restrict <- (dist <= radius)
	restrict[is.na(restrict)] <- FALSE
	return(restrict)
}


"lastGE" <-
function(vec, value)
{
	#   Find index of the last element which is >= a give value
	vec <- c(vec)
	vec[is.na(vec)] <-  - Inf
	return((rev(1:length(vec))[rev(vec >= value)])[1])
}


"lastLE" <-
function(vec, value)
{
	#   Find index of the last element which is <= a given value
	vec <- c(vec)
	vec[is.na(vec)] <- Inf
	return((rev(1:length(vec))[rev(vec <= value)])[1])
}


"M8.wgt.sum" <-
function(x)
{
	#  Compute M8 weighted magnitude sum: sum( 10^{x} )
	if(length(x) > 0) return(sum(10^(0.46 * x))) else 
			return(0)
}


"maxIndex" <-
function(vec)
{
	#   Find index of the maximum value in a vector
	vec <- c(vec)
	vec[is.na(vec)] <-  - Inf
	ind <- ((1:length(vec))[vec == max(vec, na.rm = TRUE)])[1]
	return(ind)
}


"percentile" <-
function(x)
{
	# Calculate percentile of each value of a vector
	n <- length(x)
	pct <- 1:n	#!! this is extremely inefficient
	for(i in 1:n) {
		pct[i] <- sum(x > x[i])
	}
	# number of values larger(>) than this value
	pct <- (n - pct)/n
	# this is the strict empirical distribution function
	# pct of x[i] == (percent of x <= x[i])
}


"prt.mainshocks" <-
function(c, add.title = TRUE)
{
	if(add.title) {
		cat("time Magnitu (longit, latit) after\n")
	}
	cat(paste(format(c[, "time"]), "  Mag", format(c[, "magnitude"], digits
		 = 3), " (", format(c[, "longitude"], digits = 5), ", ",
		 format(c[, "latitude"], digits = 5), ") ", format(c[,
		 "n.aftershocks"], digits = 4), "\n", sep = ""), sep = "")
}


"radius.M8default" <-
function(magnitudeLarge)
{
	#------------------------------------------------------------
	#   Default CI(circle of investigation) radius as by M8 manual
	#------------------------------------------------------------
	return(55.5 * (exp(magnitudeLarge - 5.6) + 1))
}


"sumrow" <-
function(x)
{
	x <- as.matrix(x)
	apply(x, 2, "sum")
}


"timegrid" <-
function(reference, start.datetime, end.datetime)
{
	#------------------------------------------------------------
	# Generate time grid with fixed months interval
	#   given reference, start and end time.
	# Input:
	#   reference: 3 element (year, month, day) giving a date
	#              that must appear as one of the grids
	#   start.datetime: a datetime which must be inside the first
	#              grid.
	#   end.datetime: a datetime which must be inside the last grid.
	# Output: 
	#   Returns a matrix with 2 columns containing
	#       year, month of the generated time grid
	#------------------------------------------------------------
	if(reference["day"] != 1) {
		cat("timegrid(): Warning:\n")
		cat("  day not equal 1\n")
	}
	if((reference["month"] - 1) %% 6 != 0) {
		cat("timegrid(): Warning:\n")
		cat("  Month not equal 1 or 7\n")
	}
	dt0 <- datetime(year = reference["year"], month = reference["month"], 
		day = reference["day"])
	n <- ceiling((end.datetime - dt0)/365.2422) * 2 + 2	
	# the second half of the grid
	grid2 <- matrix(-1, nrow = n, ncol = 2)
	i <- 1
	y <- reference["year"]
	m <- reference["month"]
	d <- reference["day"]
	dt <- dt0
	last.grid <- 0
	while(last.grid != 2) {
		grid2[i, 1] <- y
		grid2[i, 2] <- m
		m <- m + 6
		if(m > 12) {
			m <- m - 12
			y <- y + 1
		}
		dt <- datetime(year = y, month = m, day = reference["day"])
		i <- i + 1
		if(dt < end.datetime) {
			last.grid <- 0
		}
		else {
			last.grid <- last.grid + 1
		}
		#browser()
	}
	grid2 <- grid2[grid2[, 2] > 0,  ]	
	# this grid start from the reference grid, end grid
	#   at just >= end.datetime.
	n <- ceiling((dt0 - start.datetime)/365.2422) * 2 + 2	
	# the first half of the grid
	grid1 <- matrix(-1, nrow = n, ncol = 2)
	i <- n
	y <- reference["year"]
	m <- reference["month"]
	d <- reference["day"]
	dt <- dt0
	last.grid <- 0
	while(last.grid != 2) {
		grid1[i, 1] <- y
		grid1[i, 2] <- m
		m <- m - 6
		if(m < 1) {
			m <- m + 12
			y <- y - 1
		}
		dt <- datetime(year = y, month = m, day = reference["day"])
		i <- i - 1
		if(dt > start.datetime) {
			last.grid <- 0
		}
		else {
			last.grid <- last.grid + 1
		}
		#browser()
	}
	grid1 <- grid1[grid1[, 2] > 0,  ]	
	# Now bind grid1 and grid2 vertically, and droping the
	#   duplicated reference grid
	grid <- cbind(year = c(grid1[, 1], grid2[-1, 1]), month = c(grid1[, 2], 
		grid2[-1, 2]))
	grid
}


"zeros" <-
function(n)
{
	#      Generate vector of zeros
	return(c(matrix(0, nrow = n, ncol = 1)))
}
