\name{M8.TIP}
\alias{M8.TIP}
\title{
M8 Times of Increased Probability
}
\description{
Calculates the times of increased probability for the M8 Algorithm.
}

\usage{
M8.TIP(series, M0, training="user", end.training=NA, smoother=6,
       TIP.length=10, debug=FALSE)
}
\arguments{
\item{series}{
a list object produced by the function \code{\link{M8.series}}.
}
\item{M0}{
numeric. The aim of the algorithm is to predict earthquakes with a magnitude greater than or equal value to this value.
}
\item{training}{
has three possible modes: \code{"moving"}, \code{"user"}, and \code{"all"}. See Details in \code{\link{M8.series}} for further explanation.
}
\item{end.training}{
Julian date for the end of the training period. This date should be the end point of a bin. This is only used when the training mode is \code{"user"}.
}
\item{smoother}{
integer, denoting the length of the smoothing window. The series are smoothed at a given point by taking their maximum value over previous values within this window. The default value is 6 (i.e. three years).
}
\item{TIP.length}{
is the duration of the TIP or \dQuote{Time of Increased Probability}. The M8 default is 5 years (i.e. 10 time intervals), and is independent of \code{M0}.
}
\item{debug}{
logical. The default is \code{FALSE}.
}}
\value{
A list object with the same components as that returned by \code{\link{M8.series}}, but with the additional variables:

\item{tops}{
is a rts with n rows and 7 columns. It contains the historical top 10 percent (for the first 6 series) or 25 percent of values (for the 7th series) for each 6 month interval. The percentile depends on the training mode.
}
\item{exceeds}{
is a rts with n rows and 7 columns. It is logical, indicating whether each series exceeds the relevant historical percentiles for each six month interval.
}
\item{TIP}{
is a logical variable indicating whether a TIP is declared in each 6 month interval, see \eqn{Y(J_i)} in \dQuote{Details}.
}
\item{TIP.type}{
classifies the TIPS with the following character strings:
\describe{
\item{\code{"c.e."}}{earthquake caused, that is, large earthquake \eqn{\ge M_0}{>= M0} in the preceding year of the TIP declaration.}
\item{\code{"STIP"}}{successful warning, large event occurred in the 5 year period after declaration.}
\item{\code{"FTIP"}}{failed warning, large event did not occur in the 5 year period after declaration.}
\item{\code{"CTIP"}}{current warning, not 5 years past since declaration.}
\item{\code{"STIP-"}}{between STIP and FTIP, strictly speaking is FTIP, i.e., event with magnitude greater than or equal to \eqn{M_0-0.5}{M0-0.5} occurred in the 5 year period.}
}}
\item{TIP.level}{
is a number between \eqn{-0.9} and \eqn{0.1}, and is also referred to as the \emph{critical series}, see \eqn{W(J_i)} in \dQuote{Details}. Two consecutive positive values of \code{TIP.level} is equivalent to a TIP.
}
\item{g}{is the number of different types of measures exceeding their critical value in the last 3 years.
}
\item{h}{is the number of measures exceeding their critical value in the last 3 years.
}}

\details{
The empirical distribution of each series is initially calculated. The way this is done depends on the training mode. If the training mode is \code{"moving"}, only the threshold percentiles for declaring a TIP are calculated. This is because the \code{"moving"} mode emulates the position of someone making real-time predictions - the empirical percentiles are updated with each new data point. Before the empirical series are used for prediction they are smoothed by taking the maximum value over the past 3 years (argument \code{smoother} = 6). 

If the training mode is \code{"all"}, the empirical distribution of the series is calculated using all available values of the series. If the mode is \code{"user"}, only values of series during the training period are used to calculate the empirical distribution. The \code{rts} objects \code{tops} and \code{exceeds} are computed. The object \code{tops} gives the threshold percentiles for each series relevant at time \eqn{i}, and \code{exceeds} is a logical \code{rts} indicating whether the series exceeds the threshold percentile in the time interval \eqn{J_i}.

\code{\link{M8.TIP}} then looks at the decision rule for declaring a TIP. For a TIP to be declared, 5 out of the first 6 series must cross their 90th percentile (at least once in the last 3 years), and the 7th series must also cross its 75th percentile (also at least once in the last 3 years). This was formalised by Harte et al (2003) as follows. Denote the \eqn{m}th series in interval \eqn{J_i} as \eqn{F_m(J_i)}.
\enumerate{
\item Let
\deqn{
U_m(J_i) = G_m[F_m(J_i)]\,,
}{
U_m(J_i) = G_m[F_m(J_i)],
}
where \eqn{G_m} is the empirical distribution function of the \eqn{m}th series. Then \eqn{100\,U_m(J_i)}{100*U_m(J_i)} is the percentile value corresponding to the series value \eqn{F_m(J_i)}.

\item Find the maximum for each in the preceding 3 years, i.e.
\deqn{
V_m(J_i) = \max \big\{ U_m(J_i), U_m(J_{i-1}), \cdots, U_m(J_{t-5}) \big\}.
}{
V_m(J_i) = \max\{ U_m(J_i), U_m(J_{i-1}), \dots, U_m(J_{t-5}) \}.
}

\item Now let \eqn{V_{[2]}(J_i)} denote the second smallest of the six values \eqn{V_1(J_i),\cdots,V_6(J_i)}{V_1(J_i),\dots,V_6(J_i)}. Then the \code{TIP.level} is defined as
\deqn{
W(J_i) = \min\left\{V_{[2]}(J_i) - 0.9, V_7(J_i)-0.75\right\}.
}{
W(J_i) = \min\{ V_{[2]}(J_i) - 0.9, V_7(J_i) - 0.75 \}.
}
We refer to \eqn{W(J_i)} as the \emph{critical series} (see Harte et al, 2003). Values are listed in the output object as \code{TIP.level}.
}
A TIP is declared if \eqn{W(J_i) \ge 0}{W(J_i) >= 0} for \emph{two consecutive intervals}. Hence, by letting
\deqn{
Y(J_i) = \min\{W(J_{i-1}), W(J_i)\}\,,
}{
Y(J_i) = \min\{ W(J_{i-1}), W(J_i) \},
}
a TIP is declared in time interval \eqn{J_i} if \eqn{Y(J_i) \ge 0}{Y(J_i) >= 0}, and then the value of \code{TIP} in the output object is \code{TRUE}, otherwise \code{FALSE}. The duration of a TIP is specified by the argument \code{TIP.length}, which is 5 years by default.

If an earthquake with magnitude greater than or equal to the target magnitude occurs, the character variable \code{TIP.type} is set to \code{"STIP"} (successful TIP). If an earthquake with magnitude smaller than \eqn{M_0}{M0}, but \eqn{\ge M_0-0.5}{>= M0-0.5} occurs, \code{TIP.type} is set to \code{"STIP-"} (nearly successful TIP). If the TIP is triggered by an earthquake with magnitude greater than or equal to the target magnitude, \code{TIP.type} is set to \code{"c.e."}. 
If no earthquake with magnitude \eqn{\ge M_0}{>= M0} occurs, \code{TIP.type} is set to \code{"FTIP"} (false TIP), and if the 5 year duration of the TIP has not come to an end (and no earthquake \eqn{\ge M_0}{>= M0} has occurred) \code{TIP.type} is set to \code{"CTIP"} (continuing TIP). 
}

\author{
Li Dongfeng, 1997; modified by Maaike Vreede, 1998.
}

\seealso{
\code{\link{decluster.M8}}, \code{\link{M8}}, \code{\link{M8.series}}
}
\examples{
#   Requires the package ssNZ
require(ssNZ)

as.catalogue(subsetrect(NZ, minday=julian(1,1,1965),
                        maxday=julian(1,1,2000), minmag=4.5),
             catname="NZ45")

decluster.M8(NZ45, cutoff.mag=4.5, decluster.name="NZ.mainshocks")

temp1 <- M8.series(NZ.mainshocks, M0=7.0, minday=julian(y=1965, x=1, d=1),
                   start.series=julian(y=1975, x=1, d=1), centrelong=176, 
                   centrelat=-39, end.training=julian(y=2000, x=1, d=1))

temp2 <- M8.TIP(temp1, M0=7.0, end.training=julian(y=2000, x=1, d=1))

print(cbind(time=temp1$series[,"time"], TIP=temp2$TIP,
            TIP.type=temp2$TIP.type,
            TIP.level=signif(temp2$TIP.level, digits=3)),
      quote=FALSE)
}

\keyword{ts}
