\name{epicentres}
\alias{epicentres}
\title{
Epicentral Plot of Selected Events
}
\description{
Plots earthquake epicentres selected from a given catalogue. The colour and size of each point can be made to represent the depth and the magnitude of the event, respectively.
}
\usage{
epicentres(events, usr=NA, magnitude=c(-Inf, Inf), cex=NA,
           criteria = TRUE, depth=c(0, Inf),
           colours=c("red2", "yellow2", "green2", "cyan2", "blue2"),
           mapname="world2")}

\arguments{
\item{events}{
defines the events to be plotted. It is an object of class \code{"subset"}, \code{"catalogue"}, or \code{"matrix"}. If of class \code{"subset"}, it will generally be created by \code{\link[ssBase]{subsetcircle}}, \code{\link[ssBase]{subsetpolygon}}, \code{\link[ssBase]{subsetrect}} or \code{\link[ssBase]{subsetsphere}}. If of class \code{"matrix"}, it must contain the named columns: \code{longitude}, \code{latitude}, \code{depth}, and \code{magnitude}.
}
\item{usr}{
a vector of length 4 than defines boundary positions of the axes, i.e. \code{c(xmin, xmax, ymin, ymax)}. The default is selected so that the extreme events are just included.
}
\item{magnitude}{
vector of magnitudes, where \code{magnitude[i]} is less than or equal to \code{magnitude[i+1]}. If \code{length(magnitude)==2}, then events with magnitude greater than or equal to \code{magnitude[1]} and less than \code{magnitude[2]} are selected and events are plotted as points. If \code{length(magnitude)} is greater than 2, magnitude defines a sequence of intervals, for which events are plotted as circles of increasing size by for increasing magnitude. The size of the circles is determined by the \code{cex} parameter. Default is \code{c(-Inf, Inf)}. 
}
\item{cex}{
vector containing the sizes of the plotted points or circles. The length of the vector should correspond to the number of magnitude intervals defined by the magnitude vector. If \code{length(magnitude)==2}, then events will be plotted as points and cex will be a scalar denoting the size of the points. If \code{length(magnitude)} is greater than 2, then the events are plotted as circles of increasing radius for increasing magnitude. The circle radii are proportional to the values specified in the \code{cex} vector. Default values of \code{cex} are \code{seq(0.2, length(magnitude)-1, 0.5)}. These values are appropriate for plots with the order of a thousand points, but may be too small for 100 or less.
}
\item{criteria}{
logical variable. If \code{TRUE} (default) the subsetting parameters are printed below the graph.
}
\item{depth}{
vector of depths, where \code{depth[i]} is less than or equal to \code{depth[i+1]}. If \code{length(depth)==2}, then events with depth greater than or equal to \code{depth[1]} and less than \code{depth[2]} are selected. If \code{length(depth)} is greater than 2, depth defines a sequence of intervals, for which events are plotted in the colours specified by the parameter colours. Default is \code{c(0, Inf)}.
}
\item{colours}{
vector of colours for the different depth intervals.
}
\item{mapname}{
character string giving the map name. Low resolution maps contained in package \pkg{maps} are \code{"nz"} and \code{"world2"} (default). High resolution maps contained in the package \pkg{mapdata} are \code{"nzHires"}, \code{"world2Hires"} and \code{"chinaHires"}. The \dQuote{2} on \code{"world2"} denotes the version where the longitudes are all positive.
}}

\details{
The plot is placed into the current graphics device. If a graphics device is not open, one of the default type is opened.

The plot uses a rectangular projection. The aspect of the plot (\eqn{y/x} ratio) is calculated within the function, and is set so that the use of the available area within the graphics device is maximised. The graphics device remains open at the end of the function execution so that additional annotation and points can be added to the plot, if required. The margins and aspect ratio will also remain in effect at the end of the execution of the function, and should be reset if multiple plots are to be written to the same output file; see \dQuote{Examples} below.
}

\seealso{
\code{\link{depth.hist}}, \code{\link{freq.cusum}},
\code{\link{freq.magnitude}}, \code{\link{magnitude.cusum}},
\code{\link{magnitude.time}}, \code{\link{threeD}}, \code{\link{timeplot}},
\code{\link{multigraph}}
}
\examples{
#   Remember the parameters of the graphics device

par.reset <- par(no.readonly=TRUE)

#--------------------------------------------------------------------

#    Here the data are contained in a matrix

data <- cbind(latitude=c( -41.70, -39.30, -40.50, -30.62, -28.42, -32.31),
              longitude=c(172.20,  177.00, 175.50, 178.42, 179.97, 181.21),
              depth=c(20, 30, 45, 12, 300, 339),
              magnitude=c(7.8, 7.8, 7.6, 7.4, 7.4, 7.9))

epicentres(data, usr=c(172, 182, -42, -28), cex=2, mapname="nz")
par(par.reset)

#--------------------------------------------------------------------

#    Here the data are contained in a catalogue

data(NZ55)

epicentres(NZ55, mapname="nz")
title(main="Some Large NZ Events")
par(par.reset)


#--------------------------------------------------------------------

#    Requires the NZ Catalogue
require(ssNZ)

#   Make the Wellington Catalogue
as.catalogue(subsetrect(NZ, minlat=-42.2, maxlat=-40.5,
             minlong=173.6, maxlong=176.0, minday=julian(1,1,1978)),
             catname="Wellington")

b <- subsetrect(Wellington, minlong=173.6, maxlong=176.0, minlat=-42.1,
                 maxlat=-40.5, mindepth=36, minday=julian(1,1,1988),
                 maxday=julian(1,1,1993))

#   Note that usr specifies the area in the plot
epicentres(b, usr=c(b$minlong-0.01, b$maxlong+0.01,
                    b$minlat-0.005, b$maxlat+0.005),
           depth=c(36, 50, 70, 100, 150, Inf), criteria=FALSE,
           magnitude=c(2, 3, 4, 5, 6, Inf), mapname="nz")
title(sub=expression(paste("Depth (km):  ", 36 <= {red < {50 <= 
      {yellow < {70 <= {green < {100 <= {cyan < {150 <= {blue <
      infinity}}}}}}}}})), line=3)
title(main="Wellington Catalogue:  Deep Events")
par(par.reset)

#--------------------------------------------------------------------

#    Requires the NZ Catalogue
require(ssNZ)

b <- subsetrect(Wellington, minlong=173.6, maxlong=176.0, minlat=-42.1,
                 maxlat=-40.5, maxdepth=35.99, minday=julian(1,1,1988),
                 maxday=julian(1,1,1993))

#   Note that usr specifies the area in the plot
epicentres(b, usr=c(b$minlong-0.01, b$maxlong+0.01,
                    b$minlat-0.005, b$maxlat+0.005),
           depth=c(0, 10, 15, 20, 25, 36), criteria=FALSE,
           magnitude=c(2, 3, 4, 5, 6, Inf), mapname="nz")
title(sub=expression(paste("Depth (km):  ", 0 <= {red < {10 <= 
      {yellow < {15 <= {green < {20 <= {cyan < {25 <= {blue < 36}}}}}}}}})),
      line=3)
title(main="Wellington Catalogue:  Shallow Events")
par(par.reset)

#--------------------------------------------------------------------

#    Requires the NZ Catalogue
require(ssNZ)

b <- subsetrect(NZ, minlong=166, maxlong=180, minlat=-48,
                 maxlat=-34, minmag=3, minday=julian(1,1,1970),
                 maxday=julian(1,1,1993), mindepth=36)

epicentres(b, depth=c(36, 70, 100, 140, 200, Inf), criteria=FALSE,
           usr=c(b$minlong, b$maxlong, b$minlat, b$maxlat), mapname="nz")
title(sub=expression(paste("Depth (km):  ", 36 <= {red < {70 <= 
      {yellow < {100 <= {green < {140 <= {cyan < {200 <= {blue <
      infinity}}}}}}}}})), line=3, cex.sub=0.95)
title(main="NZ Catalogue:  Deep Events")
par(par.reset)

#--------------------------------------------------------------------

#    Requires the NZ Catalogue
require(ssNZ)

b <- subsetcircle(NZ, centrelong=176.8, centrelat=-37.8,
                   minmag=3, minday=julian(1,1,1987),
                   maxday=julian(1,1,1988), maxradius=100)

epicentres(b, cex=1.3, mapname="nz")
title(main="Edgecumbe Earthquake")
par(par.reset)

#--------------------------------------------------------------------

#    Requires the NZ Catalogue
require(ssNZ)

b <- subsetcircle(NZ, centrelong=177.5, centrelat=-37,
                   minmag=3, minday=julian(1,1,1984),
                   maxday=julian(1,1,1986), maxradius=100)

epicentres(b, cex=1.5, mapname="nz")
title(main="Bay of Plenty Swarm")
par(par.reset)
    
#--------------------------------------------------------------------

#    Requires the NZ Catalogue
require(ssNZ)

b <- subsetcircle(NZ, centrelong=176.4, centrelat=-40.4,
                   minmag=3, minday=julian(1,1,1990),
                   maxday=julian(1,1,1991), maxradius=50)

epicentres(b, cex=1.5, mapname="nz")
title(main="Weber (Dannevirke) Earthquake")
par(par.reset)
}

\keyword{hplot}

