import java.io.FileOutputStream;
import java.io.BufferedOutputStream;
import java.io.OutputStreamWriter;
import java.io.BufferedWriter;
import java.util.Calendar;
import java.text.SimpleDateFormat;
import java.util.Random;

/**
 *
 * @author jzechar
 *
 */
public class MolchanTools {

    /**
     * Compute area skill score: 1 - normalized area under the Molchan trajectory up to given tau; that is, 1 - the area/tau
     * 
     * @param molchanTrajectory sorted jump values
     * @param tau value up to which we want to evaluate the ASS
     * @return area skill score for the given trajectory up to the given tau
     */
    private static float ass(float[] molchanTrajectory, float tau) {
        float ass = 1.0f - MolchanTools.areaUnderMolchanTrajectory(molchanTrajectory, tau);
        return ass;
    }

    /**
     * Compute normalized area under the Molchan trajectory up to given tau; that is, get the area and divide it by tau
     * 
     * @param taus sorted jump values
     * @param tau value up to which we want to evaluate the ASS (value by which we'll normalize the area)
     * @return normalized area under the specified trajectory up to given tau
     */
    private static float areaUnderMolchanTrajectory(float[] taus, float tau) {
        // there are always N+1 jumps
        int N = taus.length - 1;

        // determine on which leg of the error trajectory tau falls
        int n = MolchanTools.whichLegAreWeOn(taus, tau);
        float[] nus = new float[N + 1];
        // set the nu values
        for (int j = 0; j < nus.length; j++) {
            nus[j] = (float) (N - j) / (float) N;
        }

        float area = 0.0f;
        // Sum the area under the trajectory up until the nth leg
        for (int i = 0; i <= n - 1; i++) {
            area += nus[i] * (taus[i + 1] - taus[i]);
        }
        // add the final bit of area under the trajecory
        area += nus[n] * (tau - taus[n]);

        // normalize the area by dividing by tau
        area /= tau;
        return area;
    }

    /**
     * For the specified predictor, cost map, and target eqks, generate Matlab script to plot ASS trajectory as a function of tau
     *
     * @param predictorValues predictor alarm function values
     * @param costMapValues cost map alarm function values
     * @param targetEqks distribution of target eqks
     * @param matlabFile file to which data is written
     */
    /*
    public static void trajectoryMatlab(float[] predictorValues, float[] costMapValues, short[] targetEqks, String matlabFile) {
    int N = ArrayUtil.sum(targetEqks);
    float[] trajectory = MolchanTools.trajectory(predictorValues, costMapValues, targetEqks);
    //        float[] trajectory = MolchanTools.sortedJumps(fullTrajectory, N);
    
    try {
    FileOutputStream oOutFIS = new FileOutputStream(matlabFile);
    BufferedOutputStream oOutBIS = new BufferedOutputStream(oOutFIS);
    BufferedWriter oWriter = new BufferedWriter(new OutputStreamWriter(oOutBIS));
    oWriter.write("% This file was automatically generated by MolchanTools.generateMatlabForASSTrajectory\n");
    oWriter.write("tau = [0:0.02:1];\n");
    oWriter.write("ass = [0;");
    for (int i = 2; i < 101; i += 2) {
    float tau = i * 0.01f;
    float ass = MolchanTools.ass(trajectory, tau);
    oWriter.write(ass + ";");
    }
    oWriter.write("];");
    
    oWriter.close();
    oOutBIS.close();
    oOutFIS.close();
    } catch (Exception ex) {
    System.err.println("Error in MolchanTools.generateMatlabForASSTrajectory()");
    ex.printStackTrace();
    System.exit(-1);
    }
    }
     */
    /**
     * For the specified predictor, cost map, and target eqks, compute ASS trajectory as a function of tau
     *
     * @param predictorValues predictor alarm function values
     * @param costMapValues cost map alarm function values
     * @param targetEqks distribution of target eqks
     * @return ass trajectory as a function of tau
     */
    /*
    public static float[] trajectory(float[] predictorValues, float[] costMapValues, short[] targetEqks) {
    int N = ArrayUtil.sum(targetEqks);
    float[] trajectory = MolchanTools.trajectory(predictorValues, costMapValues, targetEqks);
    //        float[] trajectory = MolchanTools.sortedJumps(fullTrajectory, N);
    float[] assTrajectory = new float[100];
    
    for (int i = 1; i < 101; i++) {
    float tau = i * 0.01f;
    float ass = MolchanTools.ass(trajectory, tau);
    assTrajectory[i - 1] = ass;
    }
    return assTrajectory;
    } 
     */
    /**
     * For the specified Molchan trajectory, compute ASS trajectory as a function of tau
     *
     * @param molchanTrajectory the Molchan trajectory jumps of interest
     * @return ass trajectory as a function of tau
     */
    private static float[] assTrajectory(float[] molchanTrajectory) {
//        float[] trajectory = MolchanTools.sortedJumps(fullTrajectory, N);
        float[] assTrajectory = new float[100];

        for (int i = 1; i < 101; i++) {
            float tau = i * 0.01f;
            float ass = MolchanTools.ass(molchanTrajectory, tau);
            assTrajectory[i - 1] = ass;
        }
        return assTrajectory;
    }

    /**
     * Compute the desired significance bound of area skill score for unskilled predictions for the specified target eqk distribution as a function of tau from 0 to 1, step 0.01.
     * We simulate many unskilled trajectories and compute the area skill score trajectory for each.  The confidence bound is then estimated by picking out the ASS value that 
     * corresponds to the desired percentile.  We save the resultant confidence bound in the specified Matlab script file.
     *
     * @param eqkDistribution number of target eqks per cell in the experiment
     * @param alpha desired significance value
     * @param matlabFile file to which data is written
     */
    /*
    public static void confidenceBoundsMatlab(short[] eqkDistribution, float alpha, String matlabFile) {
    // For the given eqk distribution, generate random trajectories and compute area under each trajectory
    int numberOfRandomizations = 10000;
    
    // make room for storing the area skill score values for each random trajectory for each of the values of tau
    float[][] pathAreaSkillScore = new float[100][numberOfRandomizations];
    
    // make room for storing the confidence bounds
    float[] confidenceBound = new float[100];
    
    // get all of our random trajectories
    for (int pathCounter = 0; pathCounter < numberOfRandomizations; pathCounter++) {
    float[] randomTrajectory = MolchanTools.randomTrajectory(eqkDistribution);
    for (int i = 1; i < 101; i++) {
    float tau = i * 0.01f;
    pathAreaSkillScore[i - 1][pathCounter] = MolchanTools.ass(randomTrajectory, tau);
    }
    }
    
    for (int i = 1; i < 101; i++) {
    float[] bounds = MathUtil.confidenceBounds(pathAreaSkillScore[i - 1], alpha);
    confidenceBound[i - 1] = bounds[0]; // alpha bound
    }
    
    try {
    FileOutputStream oOutFIS = new FileOutputStream(matlabFile);
    BufferedOutputStream oOutBIS = new BufferedOutputStream(oOutFIS);
    BufferedWriter oWriter = new BufferedWriter(new OutputStreamWriter(oOutBIS));
    oWriter.write("% This file was automatically generated by MolchanTools.generateMatlabForASSTrajectoryConfidenceBounds w/ alpha = " + alpha + "\n");
    oWriter.write("tau = 0:0.01:1;\n");
    
    oWriter.write("bound = [");
    for (int i = 1; i < 101; i++) {
    oWriter.write(confidenceBound[i - 1] + ";");
    }
    oWriter.write("];\n");
    
    oWriter.close();
    oOutBIS.close();
    oOutFIS.close();
    } catch (Exception ex) {
    System.err.println("Error in MolchanTools.generateMatlabForASSTrajectory()");
    ex.printStackTrace();
    System.exit(-1);
    }
    }
     */
    /**
     * Compute ASS(1) for the specified lat/lon/mag AlarmFunction file and specified target eqk catalog file.
     *
     * Bin the target catalog
     * Create a AlarmFunction map from the AlarmFunction file
     * Use the AlarmFunction map to predict the target catalog
     * Generate a Molchan trajectory for this strategy, using the specified cost map.
     * Compute ASS from the trajectory
     *
     * @param predictorName name of AlarmFunction (e.g., PI, RI, NSHMP)
     * @param predictorFile path to file containing AlarmFunction values in lat lon mag rpfValue format
     * @param targetEqkCatalog path to target eqk catalog file
     * @param matlabFile path to file to which we'll be writing the matlab data
     * @param costMapFile path to cost map file (specifies "cost" of each box)
     * @param enforceMinWeightRequirement Should we require that every box cost some minimum, nonzero amount?
     * @param experimentParameterFile path to file containing experiment parameters (e.g., min/max lat/lon/mag and grid spacing)
     * @return the area skill score for the given alarm function, cost map, and target eqk distribution
     */
    /*
    public static float ass(String predictorFile, String targetEqkCatalog, String costMapFile, boolean enforceMinWeightRequirement) {
    
    //        System.out.println("initializing cost map and predictor functions");
    CSEPForecast cost = new CSEPForecast(costMapFile);
    CSEPForecast predictor = new CSEPForecast(predictorFile);
    //        System.out.println("done");
    
    float[] costMap = cost.values();
    float[] predictorMap = predictor.values();
    float minLat = predictor.minLat();
    float maxLat = predictor.maxLat();
    float minLon = predictor.minLon();
    float maxLon = predictor.maxLon();
    float boxSize = predictor.boxSize();
    
    //        System.out.println("counting events in the catalog");
    int N = Utility.numberOfEventsInCatalog(targetEqkCatalog);
    //        System.out.println("done (there are " + N + ")");
    
    //        System.out.println("binning target eqks");
    short[] targetEqksMap = Utility.eventMapFromCatalog(targetEqkCatalog, minLat, maxLat, minLon, maxLon, boxSize);
    //        System.out.println("done");
    
    float[] trajectory = MolchanTools.molchan(predictorMap, costMap, targetEqksMap);
    //        float[] trajectory = MolchanTools.sortedJumps(rpfTrajectory, N);
    //        System.out.println("computing ass");
    float ass = MolchanTools.ass(trajectory, 1.0f);
    //        System.out.println("done");
    return ass;
    }
     */
    /**
     * Compute the commonly-used significance bounds of area skill score for unskilled predictions for the specified target eqk distribution as  a function of tau from 0 to 1, 
     * step 0.01.  We simulate many unskilled trajectories and compute the area skill score trajectory for each.  The confidence bounds are then estimated by picking out the 
     * ASS values that corresponds to the desired percentiles.  We save the resultant confidence bounds in the specified Matlab script file.
     *
     * @param eqkDistribution number of target eqks per cell in the experiment
     * @param matlabFile file to which data is written
     */
    /*
    public static void confidenceBoundsMatlab(short[] eqkDistribution, String matlabFile) {
    // For the given eqk distribution, generate random trajectories and compute area under each trajectory
    int numberOfRandomizations = 10000;
    
    // make room for storing the area skill score values for each random trajectory for each of the values of tau
    float[][] pathAreaSkillScore = new float[100][numberOfRandomizations];
    
    // make room for storing the confidence bounds
    float[] confidenceBound01 = new float[100];
    float[] confidenceBound05 = new float[100];
    float[] confidenceBound10 = new float[100];
    
    // get all of our random trajectories
    for (int pathCounter = 0; pathCounter < numberOfRandomizations; pathCounter++) {
    float[] randomTrajectory = MolchanTools.randomTrajectory(eqkDistribution);
    for (int i = 1; i < 101; i++) {
    float tau = i * 0.01f;
    pathAreaSkillScore[i - 1][pathCounter] = MolchanTools.ass(randomTrajectory, tau);
    }
    }
    
    for (int i = 1; i < 101; i++) {
    float[] bounds01 = MathUtil.confidenceBounds(pathAreaSkillScore[i - 1], 0.99f);
    float[] bounds05 = MathUtil.confidenceBounds(pathAreaSkillScore[i - 1], 0.95f);
    float[] bounds10 = MathUtil.confidenceBounds(pathAreaSkillScore[i - 1], 0.9f);
    confidenceBound01[i - 1] = bounds01[0];
    confidenceBound05[i - 1] = bounds05[0];
    confidenceBound10[i - 1] = bounds10[0];
    }
    
    try {
    FileOutputStream oOutFIS = new FileOutputStream(matlabFile);
    BufferedOutputStream oOutBIS = new BufferedOutputStream(oOutFIS);
    BufferedWriter oWriter = new BufferedWriter(new OutputStreamWriter(oOutBIS));
    oWriter.write("% This file was automatically generated by MolchanTools.generateMatlabForASSTrajectoryConfidenceBounds w/ alpha = 0.99, 0.95, 0.9\n");
    oWriter.write("tau = 0.0:0.01:1;");
    
    oWriter.write("bound01 = [0;");
    for (int i = 0; i < confidenceBound01.length; i++) {
    oWriter.write(confidenceBound01[i] + ";");
    }
    oWriter.write("];\n");
    
    oWriter.write("bound05 = [0;");
    for (int i = 0; i < confidenceBound05.length; i++) {
    oWriter.write(confidenceBound05[i] + ";");
    }
    oWriter.write("];\n");
    
    oWriter.write("bound10 = [0;");
    for (int i = 0; i < confidenceBound10.length; i++) {
    oWriter.write(confidenceBound10[i] + ";");
    }
    oWriter.write("];\n");
    
    oWriter.close();
    oOutBIS.close();
    oOutFIS.close();
    } catch (Exception ex) {
    System.err.println("Error in MolchanTools.generateMatlabForASSTrajectoryConfidenceBound()");
    ex.printStackTrace();
    System.exit(-1);
    }
    }
     */
    /**
     * Compute ASS(1) for the specified AlarmFunction and specified target eqk distribution.
     *
     * Use the AlarmFunction map to predict the target catalog
     * Generate a Molchan trajectory for this strategy, using the specified cost map.
     * Compute ASS from the trajectory
     *
     * @param predictor AlarmFunction values in lat lon mag rpfValue format
     * @param targetEqkCatalog path to target eqk catalog file
     * @param matlabFile path to file to which we'll be writing the matlab data
     * @param costMapFile path to cost map file (specifies "cost" of each box)
     * @param enforceMinWeightRequirement Should we require that every box cost some minimum, nonzero amount?
     * @param experimentParameterFile path to file containing experiment parameters (e.g., min/max lat/lon/mag and grid spacing)
     * @return area skill score for the specified alarm function, cost map, and observed target eqk distribution
     */
    /*
    public static float ass(float[] predictor, short[] targetEqks, float[] cost, boolean enforceMinWeightRequirement) {
    short N = ArrayUtil.sum(targetEqks);
    
    float[] trajectory = MolchanTools.molchan(predictor, cost, targetEqks);
    //        float[] trajectory = MolchanTools.sortedJumps(fullTrajectory, N);
    float ass = MolchanTools.ass(trajectory, 1.0f);
    return ass;
    }
     */
    /**
     * Compute the exact area skill score probability density for the specified number of target eqks, using the distribution of the sum of uniform random variables.  
     * Regardless of N, we want to compute the density from 0 to 1 at increments of 0.001, yielding 1001 points.
     * 
     * @int N the number of observed target eqks
     * @return an array containing exact area skill score probability density values 
     */
    /*
    public double[] exactASSPDF(int N) {
    // the sum of N uniform random variables ranges from 0 to N, so we need to do some rescaling.
    double[] exactPDF = new double[1001];
    
    for (int i = 0; i < exactPDF.length; i++) {
    double valueAtWhichToEvaluatePDF = i * 0.001 * N;
    exactPDF[i] = MathUtil.uniformSumDensity(N, valueAtWhichToEvaluatePDF) * N;
    }
    
    return exactPDF;
    }
     */
    /**
     * For the specified Molchan trajectory generate Matlab  script to plot ASS trajectory as a function of tau
     *
     * @param trajectory Molchan trajectory
     * @param matlabFile file to which data is written
     */
    /*
    public static void trajectoryMatlab(float[] trajectory, String matlabFile) {
    try {
    FileOutputStream oOutFIS = new FileOutputStream(matlabFile);
    BufferedOutputStream oOutBIS = new BufferedOutputStream(oOutFIS);
    BufferedWriter oWriter = new BufferedWriter(new OutputStreamWriter(oOutBIS));
    oWriter.write("% This file was automatically generated by MolchanTools.generateMatlabForASSTrajectory\n");
    oWriter.write("tau = [0:0.02:1];\n");
    oWriter.write("ass = [0;");
    for (int i = 2; i < 101; i += 2) {
    float tau = i * 0.01f;
    float ass = MolchanTools.ass(trajectory, tau);
    oWriter.write(ass + ";");
    }
    oWriter.write("];");
    
    oWriter.close();
    oOutBIS.close();
    oOutFIS.close();
    } catch (Exception ex) {
    System.err.println("Error in MolchanTools.generateMatlabForASSTrajectory()");
    ex.printStackTrace();
    System.exit(-1);
    }
    }
     */
    /**
     * Compute the desired significance bound of area skill score for unskilled predictions for the specified target eqk distribution as a function of tau from 0 to 1, step 0.01.
     * We simulate many unskilled trajectories and compute the area skill score trajectory for each.  The confidence bound is then estimated by picking out the ASS value that 
     * corresponds to the desired percentile.
     *
     * @param eqkDistribution number of target eqks per cell in the experiment
     * @param alpha desired significance value
     * @return the desired confidence bounds as a function of tau
     */
    /*
    public static float[] confidenceBounds(short[] eqkDistribution, float alpha) {
    // For the given eqk distribution, generate random trajectories and compute area under each trajectory
    int numberOfRandomizations = 10000;
    
    // make room for storing the area skill score values for each random trajectory for each of the values of tau
    float[][] pathAreaSkillScore = new float[100][numberOfRandomizations];
    
    // make room for storing the confidence bounds
    float[] confidenceBound = new float[100];
    
    // get all of our random trajectories
    for (int i = 0; i < numberOfRandomizations; i++) {
    float[] randomTrajectory = MolchanTools.randomTrajectory(eqkDistribution);
    for (int j = 1; j < 101; j++) {
    float tau = j * 0.01f;
    pathAreaSkillScore[j - 1][i] = MolchanTools.ass(randomTrajectory, tau);
    }
    }
    
    for (int i = 0; i < 100; i++) {
    float[] bounds = MathUtil.confidenceBounds(pathAreaSkillScore[i], alpha);
    confidenceBound[i] = bounds[0]; // alpha bound
    }
    
    return confidenceBound;
    }
     */
    /**
     * Compute the confidence bounds for unskilled alarm functions'  ASS trajectories for the specified target eqk distribution and reference model.  The confidence bounds 
     * are obtained by simulating many random alarm functions, computing the corresponding ASS trajectories and determining the bound on ASS at each 
     * value of tau; that is, if I want the 10% bound at a given value of tau, I find the value of ASS above which 90% of the simulated trajectories fall.
     *
     * @param eqkMap number of target eqks in each bin
     * @param costMap reference model to use in computing Molchan trajectories
     * @param desiredAlpha significance value
     * @return array containing, for each value of tau, the ASS which solves alpha = desiredAlpha at tau
     */
    /*
    public static float[] confidenceBounds(short[] eqkMap, float[] costMap, float desiredAlpha) {
    int numberOfSimulations = 1000;
    
    // make room for storing the area skill score values for each random trajectory for each of the values of tau
    float[][] pathAreaSkillScore = new float[100][numberOfSimulations];
    
    // make room for storing the confidence bounds
    float[] confidenceBound = new float[100];
    
    // get all of our random trajectories
    for (int i = 0; i < numberOfSimulations; i++) {
    float[] randomAlarmFunctionValues = ArrayUtil.randomFloats(eqkMap.length);
    float[] randomTrajectory = MolchanTools.molchan(randomAlarmFunctionValues, costMap, eqkMap);
    for (int j = 1; j < 101; j++) {
    float tau = j * 0.01f;
    pathAreaSkillScore[j - 1][i] = MolchanTools.ass(randomTrajectory, tau);
    }
    }
    
    for (int i = 0; i < 100; i++) {
    float[] bounds = MathUtil.confidenceBounds(pathAreaSkillScore[i], desiredAlpha);
    confidenceBound[i] = bounds[0]; // alpha bound
    }
    
    return confidenceBound;        
    }
     */
    /**
     * Given the specified trajectory, determine on which leg of the trajectory tau falls.  In particular, this answers the question: How many jumps have occurred up to tau?
     * 
     * @param trajectory sorted list of tau values at which there's been a jump
     * @param tau value up to which we're evaluating
     * @return the leg of the specified trajectory at the specified tau
     */
    private static int whichLegAreWeOn(float[] trajectory, float tau) {
        // By default, we'll start on the 0th leg
        int n = 0;

        // If tau > jump_i, then we've obtained at least i hits, keep looking
        for (int i = 1; i < trajectory.length; i++) {
            if (trajectory[i] < tau) {
                n = i;
            } else {
                return i - 1;
            }
        }
        return n;
    }

    /**
     * Compute the two-sided confidence bounds for unskilled predictions Molchan and ASS trajectories for the specified target eqk distribution and reference model.  
     * The confidence bounds are obtained by simulating many random alarm functions, computing the corresponding Molchan/ASS trajectories and computing 
     * the empirical bounds.  For Molchan bounds, we're interested in the empirical bound on tau at each  value of nu_i and for ASS bounds, we're interested
     * in the empirical bound on ASS at each value of tau; that is,  if I want the 10% Molchan bound at a given value of nu, I find the value of tau below which 
     * and above which 10% of the simulated trajectories obtain miss rate nu_i.  
     * 
     * @param eqkMap array describing the number of observed target eqks in each forecast bin
     * @param costMap reference model to use in computing Molchan trajectories
     * @param alpha significance value of interest
     * @param seed value w/ which to seed the random number generator
     * @return array containing Molchan and ASS confidence bounds; the first N values are the Molchan bounds, the remaining 100 values are the ASS bounds
     */
    private static float[][] confidenceBoundsMolchanAndASS(short[] eqkMap, float[] referenceForecast, float alpha, long seed) {
        int N = ArrayUtil.sum(eqkMap);

        // These will hold the upper and lower confidence bound points; N+1 Molchan bounds and 100 ASS bounds
        float[][] bounds = new float[N + 101][2];

        // How many random alarm functions should we simulate?
        int numberOfSimulations = 1000;

        // Make room to temporarily store each Molchan trajectory and each ASS trajectory
        float[][] randomTrajectories = new float[numberOfSimulations][N + 1];
        float[][] randomASSTrajectories = new float[numberOfSimulations][100];

        Random rndgen = new Random(seed);
        for (int i = 0; i < numberOfSimulations; i++) {
            long nextSeed = rndgen.nextLong();
            float[] randomAlarmFunctionValues = Utility.randomForecast(referenceForecast, nextSeed); // Simulate a random alarm function

            randomTrajectories[i] = MolchanTools.molchanTrajectory(randomAlarmFunctionValues, referenceForecast, eqkMap); // Compute the Molchan trajectory

            randomASSTrajectories[i] = MolchanTools.assTrajectory(randomTrajectories[i]); // Compute the corresponding ASS trajectory

        }

        // For each nu_i for i = 0 to N, construct an array containing the simulated tau_i and determine the empirical confidence bounds for this nu_i
        float[] tausAtGivenNu = new float[numberOfSimulations];
        for (int i = 0; i <= N; i++) {
            for (int j = 0; j < numberOfSimulations; j++) {
                tausAtGivenNu[j] = randomTrajectories[j][i];
            }
            bounds[i] = MathUtil.confidenceBounds(tausAtGivenNu, alpha);
        }

        // For each tau in 0.01:0.01:1, construct an array containing the simulated tau_i and determine the empirical confidence bounds for this nu_i
        float[] assesAtGivenTau = new float[numberOfSimulations];
        // For each value of tau, compute the corresponding confidence value
        for (int i = 0; i < 100; i++) {
            for (int j = 0; j < numberOfSimulations; j++) {
                assesAtGivenTau[j] = randomASSTrajectories[j][i];
            }
            bounds[N + 1 + i] = MathUtil.confidenceBounds(assesAtGivenTau, alpha);
        }

        return bounds;
    }

    /**
     * Calculate the error trajecory for the given target eqk catalog using the specified alarm function and cost map; each of these is provided as array that represents 
     * a 1-D mapping of the gridded study region.  The idea here is to sort the alarm function values in descending order and keep the target eqks and cost map 
     * synchronized (so the bins match). Then we use the target eqks and the cost map directly to determine the trajectory point associated w/ each threshold.  This may 
     * result in some equal threshold values having different tau/nu values, so we take another pass at the trajectory, starting from the end; if two thresholds are equal, they 
     * are assigned the maximum tau/minimum nu value associated w/ this threshold.  The final step is to reduce the full trajectory to the simple jumps-only representation.
     *
     * @param afv alarm function values
     * @param costMap representation of the "cost" for occupying each box with an alarm.
     * @param eqkDistribution representation of target eqk distribution, each entry contains the number of target epicenters contained w/i this bin
     * @return error trajectory containing the jumps; the minimum value of tau at which each miss rate is obtained
     */
    private static float[] molchanTrajectory(float[] afv, float[] costMap, short[] eqkDistribution) {
        // copy the forecast values so that we're not changing the forecast itself when we sort below
        float[] alarmFunctionValues = new float[afv.length];
        System.arraycopy(afv, 0, alarmFunctionValues, 0, afv.length);

        // normalize a copy of the cost map values so that we're not changing the cost map itself when we sort below
        float[] costMapValues = ArrayUtil.normalizeIgnoreNegativeValues(costMap);

        // copy the eqk distribution so that we're not changing the distribution itself when we sort below
        short[] eqkMap = new short[eqkDistribution.length];
        System.arraycopy(eqkDistribution, 0, eqkMap, 0, eqkDistribution.length);

        short N = ArrayUtil.sum(eqkMap);
        float[] trajectory = new float[N + 1];
        trajectory[0] = 0f;

        // sort the afv in descending order and keep everything else synchronized
        // Shell sort
        for (int increment = alarmFunctionValues.length / 2; increment > 0; increment = (increment == 2 ? 1 : (int) Math.round(increment / 2.2))) {
            for (int i = increment; i < alarmFunctionValues.length; i++) {
                float afv_temp = alarmFunctionValues[i];
                float costMap_temp = costMapValues[i];
                short eqk_temp = eqkMap[i];
                for (int j = i; j >= increment && alarmFunctionValues[j - increment] < afv_temp; j -= increment) {
                    alarmFunctionValues[j] = alarmFunctionValues[j - increment];
                    alarmFunctionValues[j - increment] = afv_temp;

                    costMapValues[j] = costMapValues[j - increment];
                    costMapValues[j - increment] = costMap_temp;

                    eqkMap[j] = eqkMap[j - increment];
                    eqkMap[j - increment] = eqk_temp;
                }
            }
        }

        float tau = 0f;
        short hits = 0;
        for (int i = 0; i < eqkMap.length; i++) {
            tau += costMapValues[i];
            short hitsInThisCell = eqkMap[i];
            if (hitsInThisCell > 0) {
                float thresholdInThisCell = alarmFunctionValues[i];
                float thresholdInNextCell = alarmFunctionValues[Math.min(i + 1, alarmFunctionValues.length - 1)];
                while (thresholdInThisCell == thresholdInNextCell && i < alarmFunctionValues.length - 2) {
                    i++;
                    tau += costMapValues[i];
                    hitsInThisCell += eqkMap[i];
                    thresholdInNextCell = alarmFunctionValues[i + 1];
                }
                for (int j = 0; j < hitsInThisCell; j++) {
                    trajectory[hits + j + 1] = tau;
                }
                hits += hitsInThisCell;
                if (hits == N) {
//                    System.out.println(i);
                    break;
                }
            }
        }

        return trajectory;
    }

    /**
     * For the specified reference forecast and the forecasts w/ which to compare it, compute the Molchan trajectory and the desired confidence bounds for the specified target 
     * eqk catalog and save the results in XML format and (optionally) Matlab format
     *
     * @param referenceForecast forecast to use as the reference model in computing Molchan trajectory
     * @param forecastsToCompare array of forecast paths w/ which we will compare the reference forecast
     * @param targetEqkCatalog catalog of observed target eqks of interest
     * @param resultsML path to file to which we'll be writing the Molchan results
     * @param seed value w/ which to seed the random number generator
     * @param alpha significance value of interest; we'll compute two-sided confidence bounds (i.e., alpha and (1 - alpha))     
     * @param saveInMatlabFormat Do you also want to save the results in Matlab format?
     * @param useMaskBit Should we pay attention to the forecast masking bit
     */
    public static void molchanAndASS(String referenceForecastPath, String[] forecastsToCompare, Catalog targetEqkCatalog, String resultsMLFile,
            float alpha, long seed, boolean saveInMatlabFormat, boolean useMaskBit) {
        CSEPForecast referenceForecast = new CSEPForecast(referenceForecastPath, useMaskBit);
        float[] originalReferenceValues = referenceForecast.values();
        boolean[] forecastOverlapFilter = new boolean[originalReferenceValues.length];
        for (int i = 0; i < forecastOverlapFilter.length; i++) {
            forecastOverlapFilter[i] = true;
        }
        
        if (useMaskBit) { // If we want to honor the forecast masking bit, we will calculate the spatially overlapping section
            forecastOverlapFilter = Utility.determineForecastOverlapFilter(forecastOverlapFilter, referenceForecast.values());


            for (int i = 0; i < forecastsToCompare.length; i++) {
                CSEPForecast forecastToCompare = new CSEPForecast(forecastsToCompare[i], useMaskBit);
                forecastOverlapFilter = Utility.determineForecastOverlapFilter(forecastOverlapFilter, forecastToCompare.values());
            }
        }

        referenceForecast = new CSEPForecast(referenceForecastPath, forecastOverlapFilter);
        // Bin the target eqk catalog into the forecast grid
        short[] targetEqksMap = referenceForecast.binnedCatalog(targetEqkCatalog);
        short N = ArrayUtil.sum(targetEqksMap); // total number of target eqks
       // System.out.println("Now we're dealing w/ " + N + " eqks in the overlapping region");

        String[] forecastsToCompareNames = new String[forecastsToCompare.length];

        // We'll need room for a trajectory for every forecast in forecastsToCompare, as well as one for the reference forecast relative to itself, and each trajectory will contain N + 1 elements
        float[][] molchanTrajectories = new float[forecastsToCompare.length + 1][N + 1];
        float[][] assTrajectories = new float[forecastsToCompare.length + 1][100];

        // Compute the self-trajectory
        float[] referenceValues = referenceForecast.values();
        molchanTrajectories[molchanTrajectories.length - 1] = MolchanTools.molchanTrajectory(referenceValues, referenceValues, targetEqksMap);
        assTrajectories[assTrajectories.length - 1] = MolchanTools.assTrajectory(molchanTrajectories[molchanTrajectories.length - 1]);

        // Compute the trajectories for all other forecasts relative to the reference trajectory
        for (int i = 0; i < forecastsToCompare.length; i++) {
            CSEPForecast forecastToCompare = new CSEPForecast(forecastsToCompare[i], forecastOverlapFilter);
            forecastsToCompareNames[i] = forecastToCompare.modelName();
            molchanTrajectories[i] = MolchanTools.molchanTrajectory(forecastToCompare.values(), referenceValues, targetEqksMap);
            assTrajectories[i] = MolchanTools.assTrajectory(molchanTrajectories[i]);
        }

        // Compute the confidence bounds for the specified target eqk distribution and reference model gridding
        float[][] bounds = MolchanTools.confidenceBoundsMolchanAndASS(targetEqksMap, referenceValues, alpha, seed);

        Calendar startTime = Calendar.getInstance();
        SimpleDateFormat df = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss'Z'");
        String creationTime = df.format(startTime.getTime());

        // Write out the trajectories to the results file
        try {
            FileOutputStream oOutFIS = new FileOutputStream(resultsMLFile);
            BufferedOutputStream oOutBIS = new BufferedOutputStream(oOutFIS);
            BufferedWriter oWriter = new BufferedWriter(new OutputStreamWriter(oOutBIS));

            oWriter.write("<?xml version=\"1.0\" encoding=\"utf-8\"?>\n");
            oWriter.write("<CSEPResult xmlns=\"http://www.scec.org/xml-ns/csep/0.1\">\n");
            oWriter.write("   <resultData publicID=\"smi://org.scec/csep/results/1\">\n");
            oWriter.write("      <MolchanASSTest publicID=\"smi://org.scec/csep/tests/molchanasstest/1\">\n");
            oWriter.write("         <creationInfo creationTime=\"" + creationTime + "\" />\n");
            oWriter.write("         <seed value=\"" + seed + "\" />\n");

            oWriter.write("         <name>Molchan-ASS-Test_" + referenceForecast.modelName() + "</name>\n");

            oWriter.write("         <molchanNu>");
            for (int i = N; i >= 0; i--) {
                oWriter.write((float) i / (float) N + " ");
            }
            oWriter.write("         </molchanNu>\n");

            oWriter.write("         <molchanTrajectories>");
            // Write out the trajectories for each forecast we're comparing
            for (int i = 0; i < forecastsToCompare.length; i++) {
                oWriter.write("            <molchanTrajectory forecast=\"" + forecastsToCompareNames[i] + "\">");
                for (int j = 0; j < molchanTrajectories[i].length; j++) {
                    oWriter.write(molchanTrajectories[i][j] + " ");
                }
                oWriter.write("            </molchanTrajectory>\n");
            }
            // Write the self-trajectory
            oWriter.write("            <molchanTrajectory forecast=\"" + referenceForecast.modelName() + "\">");
            for (int i = 0; i < molchanTrajectories[molchanTrajectories.length - 1].length; i++) {
                oWriter.write(molchanTrajectories[molchanTrajectories.length - 1][i] + " ");
            }
            oWriter.write("            </molchanTrajectory>\n");
            oWriter.write("         </molchanTrajectories>\n");

            oWriter.write("         <molchanLowerConfidence>");
            for (int i = 0; i <= N; i++) {
                oWriter.write(bounds[i][0] + " ");
            }
            oWriter.write("         </molchanLowerConfidence>\n");

            oWriter.write("         <molchanUpperConfidence>");
            for (int i = 0; i <= N; i++) {
                oWriter.write(bounds[i][1] + " ");
            }
            oWriter.write("         </molchanUpperConfidence>\n");

            oWriter.write("         <assTau>");
            for (int i = 1; i < 101; i++) {
                oWriter.write((float) i * 0.01f + " ");
            }
            oWriter.write("         </assTau>\n");

            oWriter.write("         <assTrajectories>");
            // Write out the trajectories for each forecast we're comparing
            for (int i = 0; i < forecastsToCompare.length; i++) {
                oWriter.write("            <assTrajectory forecast=\"" + forecastsToCompareNames[i] + "\">");
                for (int j = 0; j < assTrajectories[i].length; j++) {
                    oWriter.write(assTrajectories[i][j] + " ");
                }
                oWriter.write("            </assTrajectory>\n");
            }
            // Write the self-trajectory
            oWriter.write("            <assTrajectory forecast=\"" + referenceForecast.modelName() + "\">");
            for (int i = 0; i < assTrajectories[assTrajectories.length - 1].length; i++) {
                oWriter.write(assTrajectories[assTrajectories.length - 1][i] + " ");
            }
            oWriter.write("            </assTrajectory>\n");
            oWriter.write("         </assTrajectories>\n");

            oWriter.write("         <assLowerConfidence>");
            for (int i = N + 1; i < N + 101; i++) {
                oWriter.write(bounds[i][0] + " ");
            }
            oWriter.write("         </assLowerConfidence>\n");

            oWriter.write("         <assUpperConfidence>");
            for (int i = N + 1; i < N + 101; i++) {
                oWriter.write(bounds[i][1] + " ");
            }
            oWriter.write("         </assUpperConfidence>\n");

            oWriter.write("      </MolchanASSTest>\n");
            oWriter.write("   </resultData>\n");
            oWriter.write("</CSEPResult>\n");

            oWriter.close();
            oOutBIS.close();
            oOutFIS.close();

            if (saveInMatlabFormat) {
                String matlabFile = resultsMLFile.replace(".xml", ".m");
                oOutFIS = new FileOutputStream(matlabFile);
                oOutBIS = new BufferedOutputStream(oOutFIS);
                oWriter = new BufferedWriter(new OutputStreamWriter(oOutBIS));

                // Generate Matlab script to plot Molchan trajectories and confidence bounds
                oWriter.write("close all;\n");
                oWriter.write("nu = [1:-1/" + N + ":0];\n");
                oWriter.write("figure;\n");
                oWriter.write("hold on;\n");
                oWriter.write("axis([0 1 0 1]);\n");
                oWriter.write("daspect([1 1 1]);\n");
                oWriter.write("xlabel('Fraction of space-time occupied by alarm (\\tau)');\n");
                oWriter.write("ylabel('Miss rate (\\nu)');\n");

                for (int i = 0; i < molchanTrajectories.length; i++) {
                    oWriter.write("tau = [");
                    for (int j = 0; j < molchanTrajectories[i].length; j++) {
                        oWriter.write(molchanTrajectories[i][j] + ";");
                    }
                    oWriter.write("];\n");
                    oWriter.write("plot(tau, nu, 'ok', 'MarkerFaceColor', 'black');\n");
                }

                oWriter.write("legend(");
                for (int i = 0; i < forecastsToCompare.length; i++) {
                    oWriter.write("'" + forecastsToCompareNames[i].replace("_", ".") + "', ");
                }
                oWriter.write("'" + referenceForecast.modelName().replace("_", ".") + "');\n");

                oWriter.write("lowerMolchanBounds = [");
                for (int i = 0; i <= N; i++) {
                    oWriter.write(bounds[i][0] + ";");
                }
                oWriter.write("];\n");

                oWriter.write("upperMolchanBounds = [");
                for (int i = 0; i <= N; i++) {
                    oWriter.write(bounds[i][1] + ";");
                }
                oWriter.write("];\n");

                oWriter.write("hold on;\n");
                oWriter.write("plot(lowerMolchanBounds, nu, '--k');\n");
                oWriter.write("plot(upperMolchanBounds, nu, '--k');\n");

                // Generate Matlab script to plot ASS trajectories and confidence bounds
                oWriter.write("figure;\n");
                oWriter.write("hold on;\n");
                oWriter.write("tau = [0.01:0.01:1];\n");
                oWriter.write("axis([0 1 0 1]);\n");
                oWriter.write("daspect([1 2 1]);\n");
                oWriter.write("xlabel('Fraction of space-time occupied by alarm (\\tau)');\n");
                oWriter.write("ylabel('Area skill score');\n");

                for (int i = 0; i < assTrajectories.length; i++) {
                    oWriter.write("ass = [");
                    for (int j = 0; j < assTrajectories[i].length; j++) {
                        oWriter.write(assTrajectories[i][j] + ";");
                    }
                    oWriter.write("];\n");
                    oWriter.write("plot(tau, ass, '.');\n");
                }

                oWriter.write("legend(");
                for (int i = 0; i < forecastsToCompare.length; i++) {
                    oWriter.write("'" + forecastsToCompareNames[i].replace("_", ".") + "', ");
                }
                oWriter.write("'" + referenceForecast.modelName().replace("_", ".") + "');\n");

                oWriter.write("lowerASSBounds = [");
                for (int i = N + 1; i < N + 101; i++) {
                    oWriter.write(bounds[i][0] + ";");
                }
                oWriter.write("];\n");

                oWriter.write("upperASSBounds = [");
                for (int i = N + 1; i < N + 101; i++) {
                    oWriter.write(bounds[i][1] + ";");
                }
                oWriter.write("];\n");

                oWriter.write("hold on;\n");
                oWriter.write("plot(tau, lowerASSBounds, '--k');\n");
                oWriter.write("plot(tau, upperASSBounds, '--k');\n");

                oWriter.close();
                oOutBIS.close();
                oOutFIS.close();
            }
        } catch (Exception ex) {
            System.out.println("Error in MolchanTools.molchan()");
            ex.printStackTrace();
            System.exit(-1);
        }
    }
    /**
     * For the specified reference forecast and the forecasts w/ which to compare it, compute the Molchan trajectory and the desired confidence bounds for the specified target 
     * eqk catalog and save the results in Matlab format
     *
     * @param referenceForecast forecast to use as the reference model in computing Molchan trajectory
     * @param forecastsToCompare array of forecasts w/ which we will compare the reference forecast
     * @param targetEqkCatalog catalog of observed target eqks of interest
     * @param resultsML path to file to which we'll be writing the Molchan results
     * @param seed value w/ which to seed the random number generator
     * @param alpha significance value of interest; we'll compute two-sided confidence bounds (i.e., alpha and (1 - alpha))     
     */
    /*
    public static void molchanAndASSMatlab(CSEPForecast referenceForecast, CSEPForecast[] forecastsToCompare, Catalog targetEqkCatalog, String matlabFile,
    float alpha, long seed) {
    // Bin the target eqk catalog into the forecast grid
    short[] targetEqksMap = referenceForecast.binnedCatalog(targetEqkCatalog);
    short N = ArrayUtil.sum(targetEqksMap); // total number of target eqks
    
    // We'll need room for a trajectory for every forecast in forecastsToCompare, as well as one for the reference forecast relative to itself, and each trajectory will contain N + 1 elements
    float[][] molchanTrajectories = new float[forecastsToCompare.length + 1][N + 1];
    float[][] assTrajectories = new float[forecastsToCompare.length + 1][100];
    
    // Compute the self-trajectory
    float[] referenceValues = referenceForecast.values();
    molchanTrajectories[molchanTrajectories.length - 1] = MolchanTools.molchanTrajectory(referenceValues, referenceValues, targetEqksMap);
    assTrajectories[assTrajectories.length - 1] = MolchanTools.assTrajectory(molchanTrajectories[molchanTrajectories.length - 1]);
    
    // Compute the trajectories for all other forecasts relative to the reference trajectory
    for (int i = 0; i < forecastsToCompare.length; i++) {
    molchanTrajectories[i] = MolchanTools.molchanTrajectory(forecastsToCompare[i].values(), referenceValues, targetEqksMap);
    assTrajectories[i] = MolchanTools.assTrajectory(molchanTrajectories[i]);
    }
    
    // Compute the confidence bounds for the specified target eqk distribution and reference model gridding
    float[][] bounds = MolchanTools.confidenceBoundsMolchanAndASS(targetEqksMap, referenceValues, alpha, seed);
    
    
    // Write out the trajectories to the results file
    try {
    FileOutputStream oOutFIS = new FileOutputStream(matlabFile);
    BufferedOutputStream oOutBIS = new BufferedOutputStream(oOutFIS);
    BufferedWriter oWriter = new BufferedWriter(new OutputStreamWriter(oOutBIS));
    
    // Generate Matlab script to plot Molchan trajectories and confidence bounds
    oWriter.write("close all;\n");
    oWriter.write("nu = [1:-1/" + N + ":0];\n");
    oWriter.write("figure;\n");
    oWriter.write("hold on;\n");
    oWriter.write("axis([0 1 0 1]);\n");
    
    for (int i = 0; i < molchanTrajectories.length; i++) {
    oWriter.write("tau = [");
    for (int j = 0; j < molchanTrajectories[i].length; j++) {
    oWriter.write(molchanTrajectories[i][j] + ";");
    }
    oWriter.write("];\n");
    oWriter.write("plot(tau, nu, 'o');\n");
    }
    
    oWriter.write("legend(");
    for (int i = 0; i < forecastsToCompare.length; i++) {
    oWriter.write("'" + forecastsToCompare[i].modelName() + "', ");
    }
    oWriter.write("'" + referenceForecast.modelName() + "');\n");
    
    oWriter.write("lowerMolchanBounds = [");
    for (int i = 0; i <= N; i++) {
    oWriter.write(bounds[i][0] + ";");
    }
    oWriter.write("];\n");
    
    oWriter.write("upperMolchanBounds = [");
    for (int i = 0; i <= N; i++) {
    oWriter.write(bounds[i][1] + ";");
    }
    oWriter.write("];\n");
    
    oWriter.write("hold on;\n");
    oWriter.write("plot(lowerMolchanBounds, nu, '--k');\n");
    oWriter.write("plot(upperMolchanBounds, nu, '--k');\n");
    
    // Generate Matlab script to plot ASS trajectories and confidence bounds
    oWriter.write("figure;\n");
    oWriter.write("hold on;\n");
    oWriter.write("tau = [0.01:0.01:1];\n");
    oWriter.write("axis([0 1 0 1]);\n");
    
    for (int i = 0; i < assTrajectories.length; i++) {
    oWriter.write("ass = [");
    for (int j = 0; j < assTrajectories[i].length; j++) {
    oWriter.write(assTrajectories[i][j] + ";");
    }
    oWriter.write("];\n");
    oWriter.write("plot(tau, ass, '.');\n");
    }
    
    oWriter.write("legend(");
    for (int i = 0; i < forecastsToCompare.length; i++) {
    oWriter.write("'" + forecastsToCompare[i].modelName() + "', ");
    }
    oWriter.write("'" + referenceForecast.modelName() + "');\n");
    
    oWriter.write("lowerASSBounds = [");
    for (int i = N + 1; i < N + 101; i++) {
    oWriter.write(bounds[i][0] + ";");
    }
    oWriter.write("];\n");
    
    oWriter.write("upperASSBounds = [");
    for (int i = N + 1; i < N + 101; i++) {
    oWriter.write(bounds[i][1] + ";");
    }
    oWriter.write("];\n");
    
    oWriter.write("hold on;\n");
    oWriter.write("plot(tau, lowerASSBounds, '--k');\n");
    oWriter.write("plot(tau, upperASSBounds, '--k');\n");
    
    oWriter.close();
    oOutBIS.close();
    oOutFIS.close();
    } catch (Exception ex) {
    System.out.println("Error in ASSTools.molchanAndASSMatlab()");
    ex.printStackTrace();
    System.exit(-1);
    }
    }
     */
}