import java.io.*;
import java.util.Date;

public class Catalog {
    protected float[] lats;
    protected float[] lons;
    protected float[] depths;
    protected float[] mbs;
    protected float[] mss;
    protected float[] mc1s;
    protected float[] mc2s;
    protected String[] times;

    public int numberOfEqks(){
        return this.times.length;
    }

    public Catalog() {
    }

    /**
     * In the simplest case, we can create a catalog from arrays of data
     * 
     * @param times array of strings of the form yyyy/MM/dd HH:mm:ss
     * @param lats array of epicentral latitude points
     * @param lons array of epicentral longitude points
     * @param depths array of hypocentral depths
     * @param mbs array of mb magnitudes
     * @param mss array of ms magnitudes
     * @param mc1s array of mc1 magnitudes
     * @param mc2s array of mc2 magnitudes
     */
    public Catalog(String[] times, float[] lats, float[] lons, float[] depths,
            float[] mbs, float[] mss, float[] mc1s, float[] mc2s) {
        this.times = new String[times.length];
        this.lats = new float[lats.length];
        this.lons = new float[lons.length];
        this.depths = new float[depths.length];
        this.mbs = new float[mbs.length];
        this.mss = new float[mss.length];
        this.mc1s = new float[mc1s.length];
        this.mc2s = new float[mc2s.length];

        System.arraycopy(times, 0, this.times, 0, times.length);
        System.arraycopy(lats, 0, this.lats, 0, lats.length);
        System.arraycopy(lons, 0, this.lons, 0, lons.length);
        for (int i = 0; i < this.lons.length; i++) {
            if (this.lons[i] < 0){
                this.lons[i] += 360;
            }

        }
        System.arraycopy(depths, 0, this.depths, 0, depths.length);
        System.arraycopy(mbs, 0, this.mbs, 0, mbs.length);
        System.arraycopy(mss, 0, this.mss, 0, mss.length);
        System.arraycopy(mc1s, 0, this.mc1s, 0, mc1s.length);
        System.arraycopy(mc2s, 0, this.mc2s, 0, mc2s.length);
    }

    /**
     * Parse a catalog file in PDE EHDF format to instantiate a catalog object
     *
     * @param catalogFile file containing earthquake events
     */
    public Catalog(String catalogFile) {
        try {
            int numberOfEvents = numberOfEqksInFile(catalogFile);
            this.times = new String[numberOfEvents];
            this.lats = new float[numberOfEvents];
            this.lons = new float[numberOfEvents];
            this.depths = new float[numberOfEvents];
            this.mbs = new float[numberOfEvents];
            this.mss = new float[numberOfEvents];
            this.mc1s = new float[numberOfEvents];
            this.mc2s = new float[numberOfEvents];

            String sRecord = null;

            // Get a handle to the input catalog file
            FileInputStream oFIS = new FileInputStream(catalogFile);
            BufferedInputStream oBIS = new BufferedInputStream(oFIS);
            BufferedReader oReader = new BufferedReader(new InputStreamReader(oBIS));

            int eventNumber = 0;

            // Parse each line of the catalog file
            while ((sRecord = oReader.readLine()) != null) {
                // parse the details
                String[] eqkDetails = eqkParametersFromRecord(sRecord);

                String time = eqkDetails[0];
                String sLatitude = eqkDetails[1];
                String sLongitude = eqkDetails[2];
                String sDepth = eqkDetails[3];
                String sMb = eqkDetails[4];
                String sMs = eqkDetails[5];
                String sMc1 = eqkDetails[6];
                String sMc2 = eqkDetails[7];
                float lat = 0.0f;
                float lon = 0.0f;
                float depth = 0.0f;
                float mb = 0.0f;
                float ms = 0.0f;
                float mc1 = 0.0f;
                float mc2 = 0.0f;

                if (sLatitude.trim().length() > 0) {
                    lat = Float.parseFloat(sLatitude);
                }
                if (sLongitude.trim().length() > 0) {
                    lon = Float.parseFloat(sLongitude);
                    if (lon < 0){
                        lon += 360;
                    }
                }
                if (sDepth.trim().length() > 0) {
                    depth = Float.parseFloat(sDepth);
                }
                if (sMb.trim().length() > 1) {
                    mb = Float.parseFloat(sMb);
                }
                if (sMs.trim().length() > 1) {
                    ms = Float.parseFloat(sMs);
                }
                if (sMc1.trim().length() > 1) {
                    mc1 = Float.parseFloat(sMc1);
                }
                if (sMc2.trim().length() > 1) {
                    mc2 = Float.parseFloat(sMc2);
                }

                this.times[eventNumber] = time;
                this.lats[eventNumber] = lat;
                this.lons[eventNumber] = lon;
                this.depths[eventNumber] = depth;
                this.mbs[eventNumber] = mb;
                this.mss[eventNumber] = ms;
                this.mc1s[eventNumber] = mc1;
                this.mc2s[eventNumber] = mc2;

                eventNumber++;
            }
        } catch (Exception e) {
            System.out.println("error in Catalog(" + catalogFile + ")");
            e.printStackTrace();
            System.exit(-1);
        }
    }

    /**
     * Create a subcatalog containing only events falling within the given
     * magnitude range
     * 
     * @param minMag minimum magnitude of event we want in the subcatalog
     * @param maxMag maximum magnitude of event we want in the subcatalog
     */
    public Catalog subcatalogByMagnitude(float minMag, float maxMag) {
        int numberOfQualifyingEvents = 0;
        // be conservative: make room for the special case where every event in 
        // the catalog belongs to the subcatalog
        int[] indicesOfQualifyingEvents = new int[this.lats.length];

        for (int i = 0; i < this.times.length; i++) {
            float[] currentMags = {this.mbs[i], this.mss[i], this.mc1s[i],
                this.mc2s[i]};
            float currentMag = ArrayUtil.maximum(currentMags);
            if (currentMag >= minMag && currentMag <= maxMag) {
                indicesOfQualifyingEvents[numberOfQualifyingEvents] = i;
                numberOfQualifyingEvents++;
            }
        }
        return this.subcatalogByIndices(indicesOfQualifyingEvents,
                numberOfQualifyingEvents);
    }

    /**
     * Create a subcatalog with the specified number of eqks containing only the
     * eqks that are specified by the array of event indices and the number
     * 
     * @param indices indices of events to include in the subcatalog
     * @param numberOfEqks number of eqks to include in the subcatalog
     * @return subset of events corresponding to the specified eqk indices
     */
    private Catalog subcatalogByIndices(int[] indices, int numberOfEqks) {
        String[] timesLocal = new String[numberOfEqks];
        float[] latsLocal = new float[numberOfEqks];
        float[] lonsLocal = new float[numberOfEqks];
        float[] depthsLocal = new float[numberOfEqks];
        float[] mbsLocal = new float[numberOfEqks];
        float[] mssLocal = new float[numberOfEqks];
        float[] mc1sLocal = new float[numberOfEqks];
        float[] mc2sLocal = new float[numberOfEqks];

        for (int i = 0; i < numberOfEqks; i++) {
            int indexOfQualifyingEvent = indices[i];
            timesLocal[i] = this.times[indexOfQualifyingEvent];
            latsLocal[i] = this.lats[indexOfQualifyingEvent];
            lonsLocal[i] = this.lons[indexOfQualifyingEvent];
            depthsLocal[i] = this.depths[indexOfQualifyingEvent];
            mbsLocal[i] = this.mbs[indexOfQualifyingEvent];
            mssLocal[i] = this.mss[indexOfQualifyingEvent];
            mc1sLocal[i] = this.mc1s[indexOfQualifyingEvent];
            mc2sLocal[i] = this.mc2s[indexOfQualifyingEvent];
        }

        Catalog subCatalog = new Catalog(timesLocal, latsLocal, lonsLocal, 
                depthsLocal, mbsLocal, mssLocal, mc1sLocal, mc2sLocal);
        return subCatalog;
    }

    /**
     * Create a subcatalog containing only events falling within the given date range
     * 
     * @param minDate minimum origin time of event we want in the subcatalog
     * @param maxDate maximum origin time of event we want in the subcatalog
     */
    public Catalog subcatalogByTime(String minDate, String maxDate) {
        int numberOfQualifyingEvents = 0;
        // make room for the special case where every event in the catalog 
        // belongs to the subcatalog
        int[] indicesOfQualifyingEvents = new int[this.lats.length];

        Date eventTime = new Date();
        Date start = DateUtil.dateFromString(minDate);
        Date end = DateUtil.dateFromString(maxDate);

        for (int i = 0; i < this.times.length; i++) {
            eventTime = DateUtil.dateFromString(this.times[i]);
            //System.out.println("eventTime=" +eventTime.toString());
            if (eventTime.after(start) && eventTime.before(end)) {
                indicesOfQualifyingEvents[numberOfQualifyingEvents] = i;
                numberOfQualifyingEvents++;
            }
            // Assume the catalog is ordered by time; the first event that 
            // occurs after the end period of interest marks the line after
            // which no events will fall in the subcatalog
            if (eventTime.after(end)) {
                break;
            }
        }

        return this.subcatalogByIndices(indicesOfQualifyingEvents,
                numberOfQualifyingEvents);
    }

    /**
     * Parse earthquake parameters from a single line of a catalog file
     * 
     * @param sRecord chunk of text containing information on a single eqk
     * @param catalogType type of catalog
     * @return array of earthquake parameters in the following form:
     * [0]=origin time
     * [1]=latitude (in decimal degrees) of epicenter
     * [2]=longitude (in decimal degrees) of epicenter
     * [3]=magnitude
     * [4]=depth
     */
    protected String[] eqkParametersFromRecord(String sRecord) {
        String[] eqkParameters = new String[8];
        String sYear = sRecord.substring(4, 8);
        String sMonth = sRecord.substring(8, 10);
        String sDay = sRecord.substring(10, 12);
        String sHour = sRecord.substring(12, 14);
        String sMinute = sRecord.substring(14, 16);
        String sSecond = sRecord.substring(16, 18);
        String time = sYear + "/" + sMonth + "/" + sDay + " " + sHour + ":" + 
                sMinute + ":" + sSecond; // origin time
//        System.out.println(time);
        String sLatitude = sRecord.substring(20, 22) + "." +
                sRecord.substring(22, 25);
        if (sRecord.charAt(25) == 'S'){
            sLatitude = "-" + sLatitude.trim();
        }
        String sLongitude = sRecord.substring(26, 29) + "." +
                sRecord.substring(29, 32);
        if (sRecord.charAt(32) == 'W'){
            sLongitude = "-" + sLongitude.trim();
        }
        String sDepth = sRecord.substring(33, 36).trim() + "." +
                sRecord.substring(36, 37);
        String sMb = sRecord.substring(47, 48) + "." + sRecord.substring(48, 49);
        String sMs = sRecord.substring(51, 52) + "." + sRecord.substring(52, 53);
        String sMc1 = sRecord.substring(56, 57) + "." + sRecord.substring(57, 59);
        String sMc2 = sRecord.substring(66, 67) + "." + sRecord.substring(67, 69);

        eqkParameters[0] = time;
        eqkParameters[1] = sLatitude;
        eqkParameters[2] = sLongitude;
        eqkParameters[3] = sDepth;
        eqkParameters[4] = sMb;
        eqkParameters[5] = sMs;
        eqkParameters[6] = sMc1;
        eqkParameters[7] = sMc2;
        return eqkParameters;
    }

    /**
     * Create a subcatalog containing only events falling within the specified
     * distance from the specified reference point
     *
     * @param lat latitude of reference point
     * @param lon latitude of reference point
     * @param distance maximum distance from reference point within which an eqk
     *          will be placed in the subcatalog
     */
    public Catalog subcatalogByDistanceFromPoint(float lat, float lon,
            float distance) {
        int numberOfQualifyingEvents = 0;
        // make room for the special case where every event in the catalog
        // belongs to the subcatalog
        int[] indicesOfQualifyingEvents = new int[this.lats.length]; 
        if (lon < 0){
            lon += 360;
        }
        // Determine which events fall w/i the specified distance
        for (int i = 0; i < this.times.length; i++) {
            float currentLat = this.lats[i];
            float currentLon = this.lons[i];
//            float currentDistance = (float) GeoUtil.
//                  vincentyDistanceBetweenPoints(lat, lon, currentLat, currentLon);
            float currentDistance = (float) GeoUtil.
                    kossobokovDistanceBetweenPoints(lat, lon, currentLat,
                    currentLon);
            if (currentDistance <= distance) {
                indicesOfQualifyingEvents[numberOfQualifyingEvents] = i;
                numberOfQualifyingEvents++;
            }
        }
        return this.subcatalogByIndices(indicesOfQualifyingEvents,
                numberOfQualifyingEvents);
    }

    /**
     * Count the number of events in a catalog file (just the number of lines
     * in the file)
     *
     * @param catalogFile path to the catalog of interest
     * @return the number of events in the specified catalog file
     */
    private int numberOfEqksInFile(String catalogFile) {
        int numberOfEqks = 0;

        try {
            String sRecord = null;

            // Get a handle to the catalog file
            FileInputStream oFIS = new FileInputStream(catalogFile);
            BufferedInputStream oBIS = new BufferedInputStream(oFIS);
            BufferedReader oReader = new BufferedReader(new InputStreamReader(oBIS));

            // pass through the file once quickly to see how many events there are
            while ((sRecord = oReader.readLine()) != null) {
                numberOfEqks++;
            }

            oReader.close();
            oReader = null;
            oBIS.close();
            oBIS = null;
            oFIS.close();
            oFIS = null;
        } catch (Exception ex) {
            System.err.println("Trouble counting the number of events in " +
                    catalogFile);
            ex.printStackTrace();
            System.exit(-1);
        }

        return numberOfEqks;
    }

    /**
     * Print first 500 events
     */
    public void print() {
        System.out.println("time\tlon\tlat\tdepth\tmb\tms\tmc1\tmc2");

        int numberOfEvents = this.times.length;

        if (numberOfEvents > 50) {
            System.out.println("printing only the first 50 events");
        }
        for (int i = 0; i < numberOfEvents; i++) {
            if (i > 49) {
                break;
            }
            System.out.println(this.times[i] + "\t" + this.lons[i] + "\t" +
                    this.lats[i] + "\t" + this.depths[i] + "\t" + this.mbs[i] +
                    "\t" + this.mss[i] + "\t" + this.mc1s[i] + "\t" +
                    this.mc2s[i]);
        }
    }

}