import java.io.FileInputStream;
import java.util.Properties;
import java.util.Random;

public class GamblingTools {
    /** Creates a new instance of GamblingTools */
    public GamblingTools() {
    }

    /**
     * Calculate the gambling return for a single bet
     * @param probability reference probability, set by the house
     * @param bet Will there be one or more target eqks?
     * @param outcome Was there one or more target eqks?
     * @return the gambling return defined by Zhuang 2010
     */
    public static float winningsForOneBet(float probability, boolean bet,
            boolean outcome){
        if (bet & outcome){
            return ((1f - probability) / probability);
        }
        else if(!bet & !outcome)
        {
            return (probability / (1f - probability));
        }
        else{
            return -1;
        }
    }

    /**
     * Given an array of reference probabilities, bets, and outcomes, return the
     * total winnings
     * @param houseProbabilities array of probabilities
     * @param bets array of binary bets
     * @param outcomes array of binary outcomes
     * @return total winnings
     */
    public static float winnings(float[] houseProbabilities,
            boolean[] bets, boolean[] outcomes){
        float winnings = 0f;
        for (int i = 0; i < houseProbabilities.length;i++){
            winnings += winningsForOneBet(houseProbabilities[i], bets[i],
                    outcomes[i]);
        }
        return winnings;
    }

    /**
     * Based on the specified reference probabilities, simulate a naive bettor's
     * bets and compute his total winnings conditional on the specified binary
     * outcomes. Repeat this process for the specified number of bettors and
     * return each bettor's winnings
     */
    public static float[] winningsForSeveralRandomBettors(int numberOfBettors,
            float[] houseProbabilities, boolean[] outcomes, String seedFile){
        float[] winnings = new float[numberOfBettors];
        int numberOfBets = houseProbabilities.length;
        boolean[] bets = new boolean[numberOfBets];
        Random rndgen = new Random();
        long seed = 0l;
        Properties Props = new Properties();
        try {
            FileInputStream in = new FileInputStream(seedFile);
            Props.load(in);
            in.close();
            in = null;

            seed = Long.parseLong(Props.getProperty("seed"));
        } catch (Exception ex) {
            System.out.println("Error in GamblingTools."
                    + "winningsForSeveralRandomBettors(" + seedFile + ")");
            ex.printStackTrace();
            System.exit(-1);
        }

        rndgen.setSeed(seed);

        for (int i = 0; i < numberOfBettors;i++){
            for (int j = 0; j < numberOfBets; j++){
                bets[j] = false;
                float houseProbability = houseProbabilities[j];
                if (rndgen.nextFloat() < houseProbability){
                    bets[j] = true;
                }
            }
            winnings[i] = winnings(houseProbabilities, bets, outcomes);
        }
        return winnings;
    }
}
