public class M8Alarm {
    private String start;
    private String end;
    private float minMag;
    private float maxMag;
    private float centerLat;
    private float centerLon;
    private float radius;
    private boolean alarmFlag;
    private double referenceProbability = 0;
    private boolean refProbFromSmallerEvents = false;
    private boolean refProbFromZeroEvents = false;

    public boolean isAlarmFlag() {
        return alarmFlag;
    }

    public float getCenterLat() {
        return centerLat;
    }

    public float getCenterLon() {
        return centerLon;
    }

    public String getEnd() {
        return end;
    }

    public float getMaxMag() {
        return maxMag;
    }

    public float getMinMag() {
        return minMag;
    }

    public float getRadius() {
        return radius;
    }

    public boolean isRefProbFromSmallerEvents() {
        return refProbFromSmallerEvents;
    }

    public boolean isRefProbFromZeroEvents() {
        return refProbFromZeroEvents;
    }

    public double getReferenceProbability() {
        return referenceProbability;
    }

    public String getStart() {
        return start;
    }

    /*
     * Instantiate a circular M8 alarm with the specified start and end dates, 
     min/max magnitudes, center and radius, and binary prediction (true--1+ eqks
     * are expected, false--no eqks are expected)
     */
    public M8Alarm(String start, String end, float minMag, float maxMag,
            float centerLat, float centerLon, float radius, boolean alarmFlag){
        this.start = start;
        this.end = end;
        this.minMag = minMag;
        this.maxMag = maxMag;
        this.centerLat = centerLat;
        this.centerLon = centerLon;
        this.radius = radius;
        this.alarmFlag = alarmFlag;
    }

    /**
     * Calculate the reference probability of a target eqk for this alarm.  To do
     * this, we count the number of past target earthquakes occurring within the
     * alarm's spatial region.  We divide by the length of the global catalog to
     * obtain the annual rate of target eqks.  If there have been no target
     * earthquakes, then we infer the rate by computing the rate of smaller
     * events in the region and assume G-R w/ b-value = 1.  We then compute the
     * reference probability as p = 1 - e^-rt where r is the rate and t is the
     * duration of the alarm (again, in years)
     */
    public float referenceProbability(Catalog referenceCatalog,
            String referenceCatalogStartDate, String referenceCatalogEndDate) {
        float catalogDuration = Math.min(
                DateUtil.durationInDays(referenceCatalogStartDate, this.start),
                DateUtil.durationInDays(referenceCatalogStartDate,
                referenceCatalogEndDate)) / 365.25f;
//        System.out.println(referenceCatalogStartDate + "\t" + this.start +
//                "\t" + referenceCatalogEndDate);
        float alarmDuration = DateUtil.durationInDays(this.start,
                this.end) / 365.25f;
        float deltaM = 1f;

        int numberofPastTargetEqks =
                M8Tools.numberOfPastEqksWithinAlarmRegion(
                    referenceCatalog, this.centerLat, this.centerLon,
                    this.radius, this.minMag, this.maxMag, this.start);
//        System.out.println("Number of past eqks w/ m=" + this.minMag + " to " + this.maxMag + " for RTP Alarm " + this.jzName + " = " + numberofPastTargetEqks);
        boolean usedSmallerEarthquakes = false;
        if (numberofPastTargetEqks == 0) {
            usedSmallerEarthquakes = true;
            numberofPastTargetEqks =
                    M8Tools.numberOfPastEqksWithinAlarmRegion(
                    referenceCatalog, this.centerLat, this.centerLon,
                    this.radius, this.minMag - deltaM,
                    this.maxMag - deltaM, this.start);
//            System.out.println("Number of past eqks w/ m=" + (this.minMag - deltaM) + " to " + this.maxMag + " for RTP Alarm " + this.jzName + " = " + numberofPastTargetEqks);
        }

        float annualRateOfTargetEqks = (float) (numberofPastTargetEqks) /
                catalogDuration;

        if (usedSmallerEarthquakes) {
            annualRateOfTargetEqks /= (float) Math.pow(10, deltaM);
        }

        if (numberofPastTargetEqks == 0) {
            this.refProbFromZeroEvents = true;
            annualRateOfTargetEqks = (float) (0.5f) / catalogDuration;
        }

        float probability = (float) (1.0 - Math.exp(-1 *
                annualRateOfTargetEqks * alarmDuration));
        this.referenceProbability = probability;
        if (usedSmallerEarthquakes & ! this.refProbFromZeroEvents) {
            this.refProbFromSmallerEvents = true;
        }
//            if (arbitrarilySetRate) {
//                this.refProbabilityFromZeroEvents = true;
////                System.out.println(alarmDuration + "\t" + this.minMag + "\t0\t" + annualRateOfTargetEqks + "*\t" + probability);
//            } else {
//                this.refProbabilityFromSmallerEvents = true;
////                System.out.println(alarmDuration + "\t" + this.minMag + "\t0\t" + annualRateOfTargetEqks + "\t" + probability);
//            }
//        } //else {
//            System.out.println(alarmDuration + "\t" + this.minMag + "\t" + annualRateOfTargetEqks + "\tn/a\t" + probability);
//        }
        return probability;
    }
}