import java.io.OutputStreamWriter;
import java.io.BufferedOutputStream;
import java.io.FileOutputStream;
import java.io.BufferedWriter;
import java.io.InputStreamReader;
import java.io.BufferedInputStream;
import java.io.FileInputStream;
import java.io.BufferedReader;
import java.util.Arrays;
import java.util.StringTokenizer;

public class M8Tools {

    /** Creates a new instance of M8Tools */
    public M8Tools() {
    }

    /**
     * Determine the number of eqks w/ specified magnitude range occurring
     * within the specified alarm region before the specified date,
     * where the alarm region is specified by a center and a radius.
     *
     * @param superCatalog eqk catalog to use for
     *          within a given region of interest
     * @param lat latitude of alarm center
     * @param lon longitude of alarm center
     * @param radius distance around the cemter within which to count
     *          target eqks are interesting
     * @param minMag minimum target magnitude
     * @param maxMag maximum target magnitude
     * @param alarmStart start of M8 alarm, so we can count all events up to
     *          this point
     * @return number of eqks that happened before the specified alarmStart
     * date AND fall w/i the specified magnitude range AND within the specified
     * distance from the specified lat/lon point
     */
    public static int numberOfPastEqksWithinAlarmRegion(Catalog superCatalog,
            float lat, float lon, float radius, float minMag, float maxMag,
            String alarmStart) {
        // Filter the catalog so that it only contains events that fall w/i the
        // space-magnitude extent of the alarm and before the alarm started
        Catalog cat = superCatalog.subcatalogByMagnitude(minMag, maxMag);
        cat = cat.subcatalogByDistanceFromPoint(lat, lon, radius);
        cat = cat.subcatalogByTime("1000/01/01 00:00:00", alarmStart);
//        cat.print();

        int numberOfTargetsWithinAlarm = cat.numberOfEqks();
        return numberOfTargetsWithinAlarm;
    }
    
    /**
     * Determine the number of eqks w/ specified magnitude range occurring
     * within the specified alarm region within the specified date range,
     * where the alarm region is specified by a center and a radius.
     *
     * @param superCatalog eqk catalog to use for
     *          within a given region of interest
     * @param lat latitude of alarm center
     * @param lon longitude of alarm center
     * @param radius distance around the cemter within which to count
     *          target eqks are interesting
     * @param minMag minimum target magnitude
     * @param maxMag maximum target magnitude
     * @param alarmStart start of M8 alarm
     * @param alarmEnd end of M8 alarm
     * @return number of eqks that happened during the specified date range AND
     * fall w/i the specified magnitude range AND within the specified distance
     * from the specified lat/lon point
     */
    public static int numberOfTargetEqksWithinAlarmRegion(Catalog superCatalog,
            float lat, float lon, float radius, float minMag, float maxMag,
            String alarmStart, String alarmEnd) {
        // Filter the catalog so that it only contains events that fall w/i the
        // space-time-magnitude extent of the alarm
        Catalog cat = superCatalog.subcatalogByMagnitude(minMag, maxMag);
        cat = cat.subcatalogByDistanceFromPoint(lat, lon, radius);
        cat = cat.subcatalogByTime(alarmStart, alarmEnd);

        int numberOfTargetsWithinAlarm = cat.numberOfEqks();
//        if (numberOfTargetsWithinAlarm > 0){
//            cat.print();
//        }
        return numberOfTargetsWithinAlarm;
    }

    /**
     * Instantiate an array of M8 alarms with the centers specified in the given
     * file.  The file looks like this:
     *
 On 2013/01/01 current TIPs
    1.    -15.00   -175.00   0
    2.    -17.50   -174.00   1
...
 On 2013/01/01 current TIPs
    1.     -2.00    136.00   0
    2.     -2.25    138.50   0
     *
     * Each line is either a header or an alarm specification. For header lines,
     * the first non-whitespace character is a letter. Alarm specifications are
     * defined by a latitude, a longitude, and a Yes/No 0/1 flag.
     *
     * @param alarmsFile path to file containing alarm declarations
     * @param startDate start date for each alarm in the file
     * @param endDate end date for each alarm in the file
     * @param minMag minimum magnitude for each alarm in the file
     * @param maxMag maximum magnitude for each alarm in the file
     * @param radius radius (in km) of each alarm in the file
     */
    public M8Alarm[] AlarmsFromFile(String alarmsFile, String startDate,
            String endDate, float minMag, float maxMag, float radius) {
        M8Alarm[] alarms = null;
        try {
            int numberOfAlarms = numberOfAlarmsInFile(alarmsFile);
            alarms = new M8Alarm[numberOfAlarms];

            String sRecord = null;

            // Get a handle to the input catalog file
            FileInputStream oFIS = new FileInputStream(alarmsFile);
            BufferedInputStream oBIS = new BufferedInputStream(oFIS);
            BufferedReader oReader = new BufferedReader(new InputStreamReader(oBIS));

            int alarmNumber = 0;
            float centerLat = 0f;
            float centerLon = 0f;
            boolean alarmFlag = false;

            // Parse each line of the alarms file
            while ((sRecord = oReader.readLine()) != null) {
                sRecord = sRecord.replaceFirst("\\s+", "");
//                System.out.println(sRecord);
                // if the first character of this line is a digit, this is an alarm
                if (Character.isDigit(sRecord.charAt(0))) {
                    String[] splitString = (sRecord.split("\\s+"));
//                    System.out.println(Arrays.toString(splitString));
                    centerLat = Float.parseFloat(splitString[1]);
                    centerLon = Float.parseFloat(splitString[2]);
                    if (splitString[3].equalsIgnoreCase("1")){
                        alarmFlag = true;
                    }else{
                        alarmFlag = false;
                    }

                    alarms[alarmNumber] = new M8Alarm(startDate, endDate,
                            minMag, maxMag, centerLat, centerLon, radius,
                            alarmFlag);
                    alarmNumber++;
                }

            }
        } catch (Exception e) {
            System.out.println("error in AlarmsFromFile(" + alarmsFile + ")");
            e.printStackTrace();
            System.exit(-1);
        }
        return alarms;
    }

    /**
     * Count the number of M8 alarms in the specified file, neglecting comment
     * lines (marked by starting a line w/ a non-numeric character).
     *
     * @param alarmsFile path to the alarm file of interest
     * @return the number of alarms in the specified file
     */
    private static int numberOfAlarmsInFile(String alarmsFile) {
        int numberOfAlarms = 0;

        try {
            String sRecord = null;

            // Get a handle to the alarms file
            FileInputStream oFIS = new FileInputStream(alarmsFile);
            BufferedInputStream oBIS = new BufferedInputStream(oFIS);
            BufferedReader oReader = new BufferedReader(new InputStreamReader(oBIS));

            // pass through the file once quickly to see how many alarms there are
            while ((sRecord = oReader.readLine()) != null) {
                sRecord = sRecord.replace(" ", ""); // remove whitespace
//                System.out.println(sRecord);
                // if the first character of this line is a digit, this is an alarm
                if (Character.isDigit(sRecord.charAt(0))) {
                    numberOfAlarms++;
                }
            }

            oReader.close();
            oReader = null;
            oBIS.close();
            oBIS = null;
            oFIS.close();
            oFIS = null;
        } catch (Exception ex) {
            System.err.println("Trouble counting the number of alarms in " +
                    alarmsFile);
            ex.printStackTrace();
            System.exit(-1);
        }

        return numberOfAlarms;
    }

}
