"""
Module M8FOGTest
"""

__version__ = "$Revision$"
__revision__ = "$Id$"


import os, xml.dom.minidom
import matplotlib.pylab as plt

import CSEP, Environment, CSEPPropertyFile, CSEPLogging, \
       CSEPFile, EvaluationTest, ReproducibilityFiles, \
       CSEPInputParams
       
from DataSourceFactory import DataSourceFactory
from PDEDataSource import PDEDataSource
from cseprandom import CSEPRandom


#-------------------------------------------------------------------------------
#
# M8 Fixed Odds Gambling test (M8FOGTest).
#
# This class is designed to evaluate M8 external forecast within CSEP.
#
class M8FOGTest (EvaluationTest.EvaluationTest,
                 ReproducibilityFiles.ReproducibilityFiles):

    # Static data

    # Prefix for the test generated files
    __filePrefix = 'm8' + EvaluationTest.EvaluationTest.FilePrefix
    
    
    # Keyword identifying the class
    Type = "M8FOG"
    
    __type = "M8Test"
    
    # Common directory for alarm-based evaluation tests and related files
    __codePath = os.path.join(Environment.Environment.Variable[Environment.CENTER_CODE_ENV],
                              'src', 
                              'M8EvaluationTests')
    
    # Name of executable 
    __executableFile = 'Sandbox'
    
    class KML (object):
        
        Icons = {'style1': 'http://northridge.usc.edu/trac/csep/raw-attachment/wiki/M8Evaluation/FalseTrueIcon.png', 
                 'style2': 'http://northridge.usc.edu/trac/csep/raw-attachment/wiki/M8Evaluation/FalseFalseIcon.png',
                 'style3': 'http://northridge.usc.edu/trac/csep/raw-attachment/wiki/M8Evaluation/TrueTrueIcon.png', 
                 'style4': 'http://northridge.usc.edu/trac/csep/raw-attachment/wiki/M8Evaluation/TrueFalseIcon.png'}
        

        #-----------------------------------------------------------------------
        # Read data from result file and initialize alarm dictionary
        #-----------------------------------------------------------------------
        def __init__ (self,
                      result_file):

            # List of dictionaries that represent each alarm in the result table
            self.__results = []
            
            fhandle = CSEPFile.openFile(result_file)
            
            # Get the header line
            table_header = fhandle.readline().split()

            # Collect table information
            num_cols = 13

            for line in fhandle:
                # Split the line and capture data
                tmp_line = line.split()
                
                if len(tmp_line) == num_cols:
                    del tmp_line[4], tmp_line[5]
                    self.__results.append(dict(zip(table_header, tmp_line)))
                    
                    
        def createGMap(self,
                       kml_file):
            """Creates Google Maps KML document.
               Returns a KML file to be uploaded at maps.google.com.
               Navigate to 'My places' -> 'Create Map' -> 'Import' to
               upload the file and see the data."""
        
            # Create a new KML doc with our previously-defined
            # create_document() function
            kml_doc = self.createDocument("Reference probability and winnings map", 
                                          "Plots for reference probability and winnings map")
        
            # Get the specific DOM element that we created with create_document()
            # Returns a list, so call the first one
            document = kml_doc.documentElement.getElementsByTagName("Document")[0]
        
            # Iterate over our data to create KML document
            for placemark_info in self.__results:
                
                # create placemark to parse line of data into KML-format
                placemark = self.createPlacemark(placemark_info)
        
                # Adds the placemark we just created to the KML doc
                document.appendChild(placemark.documentElement)
        
            # Now that all data is parsed in KML-format, write to a file so we
            # can upload it to maps.google.com
            with CSEPFile.openFile(kml_file, CSEPFile.Mode.WRITE) as f:
                f.write(kml_doc.toprettyxml(indent=" ", 
                                            encoding='UTF-8'))
                
                
        @staticmethod
        def createDocument(name, 
                           description=''):
            """Create the overall KML document."""
        
        
            # Initialization of an XML doc
            main_doc = xml.dom.minidom.Document()
            
            # Define as a KML-type XML doc
            kml = main_doc.createElement('kml')
            kml.setAttribute('xmlns', 
                             'http://www.opengis.net/kml/2.2')
            main_doc.appendChild(kml)
            
            document = main_doc.createElement('Document')
            kml.appendChild(document)
            
            docName = main_doc.createElement('name')
            document.appendChild(docName)
            docName_text = main_doc.createTextNode(name)
            docName.appendChild(docName_text)
            
            docDescription = main_doc.createElement('description')
            document.appendChild(docDescription)
            docDescription_text = main_doc.createTextNode(description)
            docDescription.appendChild(docDescription_text)
            
            # Create four different picture elements
            for each_key in M8FOGTest.KML.Icons:  
                picture = main_doc.createElement('Style')
                picture.setAttribute('id', each_key)
                document.appendChild(picture)
                
                pictureIconStyle = main_doc.createElement('IconStyle')
                picture.appendChild(pictureIconStyle)
                
                pictureIcon = main_doc.createElement('Icon')
                pictureIconStyle.appendChild(pictureIcon)
                
                picturehref = main_doc.createElement('href')
                pictureIcon.appendChild(picturehref)
                picturehref_text = main_doc.createTextNode(M8FOGTest.KML.Icons[each_key])
                picturehref.appendChild(picturehref_text)
                
                picturescale = main_doc.createElement('scale')
                pictureIconStyle.appendChild(picturescale)
                picturescale_text = main_doc.createTextNode('1.100000')
                picturescale.appendChild(picturescale_text)
                
            return main_doc


        @staticmethod
        def createPlacemark(address):
            """Generate the KML Placemark for a given address."""
            
            
            # Create an initial XML document
            main_doc = xml.dom.minidom.Document()
            
            # Set up one placemark
            placemark_main = main_doc.createElement('Placemark')
            main_doc.appendChild(placemark_main)
            
            placemark_name = main_doc.createElement('name')
            placemark_main.appendChild(placemark_name)
            placemark_name_text = main_doc.createTextNode('Alarm start date: %(start)s, Reference probability: %(prob)s, Winnings: %(return)s' % address)
            placemark_name.appendChild(placemark_name_text)
        
            placemark_description = main_doc.createElement('description')
            placemark_main.appendChild(placemark_description)
            placemark_description_text = main_doc.createTextNode('Bet: %(bet)s, Outcome: %(outcome)s, Min magnitude: %(min_mag)s, Max magnitude: %(max_mag)s' % address)
            placemark_description.appendChild(placemark_description_text)
                
            placemark_styleUrl = main_doc.createElement('styleUrl')
            placemark_main.appendChild(placemark_styleUrl)
            if '%(bet)s' % address == 'false' and '%(outcome)s' % address == 'false':
                placemark_styleUrl_text = main_doc.createTextNode("#style2")
            if '%(bet)s' % address == 'false' and '%(outcome)s' % address == 'true':
                placemark_styleUrl_text = main_doc.createTextNode("#style1")
            if '%(bet)s' % address == 'true' and '%(outcome)s' % address == 'true':
                placemark_styleUrl_text = main_doc.createTextNode("#style3")
            if '%(bet)s' % address == 'true' and '%(outcome)s' % address == 'false':
                placemark_styleUrl_text = main_doc.createTextNode("#style4")
                
            placemark_styleUrl.appendChild(placemark_styleUrl_text)
            
            placemark_Point = main_doc.createElement('Point')
            placemark_main.appendChild(placemark_Point)
            
            placemark_Point_coordinates = main_doc.createElement('coordinates')
            placemark_Point.appendChild(placemark_Point_coordinates)
            placemark_Point_coordinates_text = main_doc.createTextNode('%(lon)s,%(lat)s' % address)
            placemark_Point_coordinates.appendChild(placemark_Point_coordinates_text)
        
            return main_doc                


    #===========================================================================
    # Nested class with matplotlib settings for the test     
    #===========================================================================
    class Matplotlib (EvaluationTest.EvaluationTest.Matplotlib):

        # Static data
        _plotConfidenceBounds = {'markersize' : None,
                                 'color': 'k',
                                 'facecolor' : '0.80', # 0.90
                                 'edgecolor' : '0.80', # 0.90
                                 'alpha': 0.5,
                                 'linestyle' : '--',
                                 'linewidth' : 1,
                                 'shadeLowerLimit': 0.0, # 0.00
                                 'shadeUpperLimit': 1.0 }    # 0.99
    
        _plotTrajectory = {'markersize' : 4.0,
                           'color': 'k'}
    
    
    __logger = None
    
    # Option to specify minimum magnitude for the forecast under evaluation:
    # 8.0 for M80@*.res files, 7.5 for M75@*.res files
    __minMagnitudeOption = "minMagnitude"

    # Option to alarm radius for the forecast under evaluation:
    # 668 for M80@*.res files, 427 for M75@*.res files
    __alarmRadiusOption = "alarmRadius"
    
    # Option to specify number of random bettors
    __numRandomBettorsOption = "numRandomBettors"
    
    # Option to draw random seed value by the system or to read it from specified
    # file
    __randomSeedFileOption = "randomSeedFile"
    
    # Default values for input parameters (for M8 forecast)
    __defaultArgs = {__randomSeedFileOption : None,
                     __minMagnitudeOption : '8.0',
                     __alarmRadiusOption : '668',
                     __numRandomBettorsOption : '1000'}
    
    
    #---------------------------------------------------------------------------
    #
    # Initialization.
    #
    # Input: 
    #        group - ForecastGroup object. This object identifies forecast
    #                models to be evaluated.
    #        args - Optional input arguments for the test. Default is None.
    # 
    def __init__ (self, group, args = None):
        """ Initialization for M8FOGTest class."""
        
        # Constructors for base classes
        EvaluationTest.EvaluationTest.__init__(self, group)
        ReproducibilityFiles.ReproducibilityFiles.__init__(self)
        
        if M8FOGTest.__logger is None:
           M8FOGTest.__logger = CSEPLogging.CSEPLogging.getLogger(M8FOGTest.__name__)
        
        # Input arguments for the model were provided:
        self.__args = CSEPInputParams.CSEPInputParams.parse(M8FOGTest.__defaultArgs,
                                                            args)
        
        
    #---------------------------------------------------------------------------
    #
    # Returns file prefix for test result file.
    #
    # Input: None
    #
    # Output: File prefix used by test results.
    #
    @classmethod
    def filePrefix (cls):
        """ Returns file prefix for test result file."""
        
        return M8FOGTest.__filePrefix
        
        
    #---------------------------------------------------------------------------
    #
    # Formats filename for the evaluation test summary of all models in the 
    # forecast group. This method overwrites base-class implementation of the
    # method since there are no summary files for all models for now.
    #
    # Input: None
    #
    # Output: Filename for all-models summary file
    #
    def allModelsSummaryFile (self):
        """ Formats filename for the evaluation test summary of all models in the 
            forecast group."""

        # Path to the all models summary file - None
        return None
        
        
    #-----------------------------------------------------------------------------
    #
    # Returns description word for the test. Implemented by derived classes.
    #
    # Input: None
    #
    # Output: Description of the test (such RELMTest, AlarmTest, etc.)
    #
    def typeDescriptor (self):
        """ Returns test type descriptor."""

        return M8FOGTest.__type
        

    #-----------------------------------------------------------------------------
    #
    # Returns unique identifier for the test.
    #
    # Input: None
    #
    # Output: Description of the test (such RELMTest, AlarmTest, etc.)
    #
    def type (self):
        """ Returns test identifier."""

        return M8FOGTest.Type


    #----------------------------------------------------------------------------
    #
    # Create input parameter file for the run.
    #
    # File format is per Jeremy's definition:
    #
    # alarmsPath: path to the .res file
    # alarmsStartDate: obvious, right?
    # alarmsEndDate: This should be 5 years after alarmsStartDate
    # alarmsMinMag: 8.0 for M80@*.res files, 7.5 for M75@*.res files
    # alarmsMaxMag: just something big, 12 is okay
    # alarmsRadius: This is 668 for M80@*.res files, 427 for M75@*.res files
    # catalogPath: path to the full PDE catalog
    # referenceCatalogStartDate: for now, it seems to be 1990/01/01
    #                            00:00:00, but this may be changed to 1973
    # numberOfRandomBettors: 1000 or 10K, we'll have to play w/ this to see
    #                        how fast it is
    # resultsPath=/Users/liukis/Documents/workspace/csep/fromJeremy/M8Assessment/results.txt
    #
    # Input: 
    #       forecast_name - Name of forecast model for the test
    #
    # Output: filename for parameter file
    #
    def createParameterFile (self, forecast_name):
        """ Create input parameter file for the run."""
        
        result_prefix = "%s_%s" %(self.filePrefixPattern(),
                                  CSEPFile.Name.extension(forecast_name))
        
        parameter_file = os.path.join(self.testDir,
                                      result_prefix + EvaluationTest.EvaluationTest._paramFilePostfix)
        
        fhandle = CSEPFile.openFile(parameter_file,
                                    CSEPFile.Mode.WRITE)
        
        fhandle.write("alarmsPath=%s\n" %os.path.join(self.forecasts.dir(),
                                                      forecast_name))
        
        # 2013/01/01 00:00:00
        fhandle.write("alarmsStartDate=%s\n" %self.forecasts.postProcess().start_date.strftime(CSEP.Time.DateTimeFormat))
        fhandle.write("alarmsEndDate=%s\n" %self.forecasts.postProcess().end_date.strftime(CSEP.Time.DateTimeFormat))

        fhandle.write("alarmsMinMag=%s\n" %self.__args[M8FOGTest.__minMagnitudeOption])
        fhandle.write("alarmsMaxMag=12.0\n")
        fhandle.write("alarmsRadius=%s\n" %self.__args[M8FOGTest.__alarmRadiusOption])

        fhandle.write("catalogPath=%s\n" %CSEPFile.Name.ascii(self.catalogFile.name))
        
        data_source = DataSourceFactory().object(PDEDataSource.Type,
                                                 isObjReference = True) 
        fhandle.write("referenceCatalogStartDate=%s\n" %data_source.StartDate.strftime(CSEP.Time.DateTimeFormat))

        fhandle.write("numberOfRandomBettors=%s\n" %self.__args[M8FOGTest.__numRandomBettorsOption])
        
        result_file = CSEPFile.Name.ascii(result_prefix + EvaluationTest.EvaluationTest._resultFilePostfix)
        fhandle.write("resultsPath=%s\n" %os.path.join(self.testDir,
                                                       result_file))

        # Flag if random seed value should be drawn by the system
        seed_file = os.path.join(self.testDir,
                                 '%s_%s' %(result_prefix,
                                           EvaluationTest.EvaluationTest._randomSeedFile))
        
        if self.__args[M8FOGTest.__randomSeedFileOption] is not None:
           seed_file = self.__args[M8FOGTest.__randomSeedFileOption]
        else:
           
           # Create seed value and store it in the file to be passed to the test
           seed = CSEPRandom.createSeed()
           
           seed_fhandle = CSEPFile.openFile(seed_file,
                                            CSEPFile.Mode.WRITE)
           seed_fhandle.write('seed=%s\n' %seed)
           seed_fhandle.close()
           
           # Register for reproducibility
           info_msg = "Seed value used by Java random number generator for %s \
evaluation test for forecast model '%s' in '%s' directory." %(self.type(),
                                                              forecast_name,
                                                              self.forecasts.dir())


           # Record parameter file with reproducibility registry
           ReproducibilityFiles.ReproducibilityFiles.add(self,
                                                         os.path.basename(seed_file),
                                                         info_msg,
                                                         CSEPFile.Format.ASCII)
           
              
        fhandle.write('pathToSeedFile=%s\n' %seed_file)
        
        fhandle.close()

        # Register input parameters file for reproducibility
        info_msg = "Input parameters file used by %s evaluation test for forecast \
'%s' in '%s' directory." %(self.type(),
                           forecast_name,
                           self.forecasts.dir())


        # Record parameter file with reproducibility registry
        ReproducibilityFiles.ReproducibilityFiles.add(self,
                                                      os.path.basename(parameter_file),
                                                      info_msg,
                                                      CSEPFile.Format.ASCII)
     
        return parameter_file


    #----------------------------------------------------------------------------
    #
    # Returns list of file patterns to be published.
    #
    # Input: None
    #
    # Output: Empty list of file patterns.
    #
    def publishPatterns (self):
        """ Returns list of file patterns to publish."""

        # PNG images and KML files are published only
        return ['*%s*' %CSEPFile.Extension.PNG]


    #----------------------------------------------------------------------------
    # Returns file extension for evaluation test result
    # 
    def fileExtension (self):
        """ Returns extension for evaluation test result file. Default is XML format file.
            This method should be overwritten by child classes if result format file
            is other than default one (XML)."""

        return CSEPFile.Extension.ASCII


    #----------------------------------------------------------------------------
    #
    # Invoke Java code for the test.
    #
    # Input: 
    #        forecast_name - Forecast model to test
    #
    def evaluate (self,
                  forecast_name):
        """ Invoke Java code for the evaluation test."""


        M8FOGTest.__logger.info("%s test for %s" %(self.type(),
                                                   forecast_name))

       
        # Create parameter file for the run
        parameter_file = self.createParameterFile(forecast_name)
       
        # invoke evaluation test:
        # -Xms128m -Xmx1024m - to increase heap size
        Environment.invokeCommand('java -Xms128m -Xmx1024m -classpath %s %s %s' 
                                  %(M8FOGTest.__codePath,
                                    M8FOGTest.__executableFile,
                                    parameter_file))
           
        # Handle files required by reproducibility: create unique copies and 
        # remove original data
        ReproducibilityFiles.ReproducibilityFiles.copyAndCleanup(self,
                                                                 M8FOGTest.Type,
                                                                 self.testDir)
        return []


    #----------------------------------------------------------------------------
    #
    # Plot test results. Child classes should overwrite the method.
    #
    # Input: 
    #        result_file - Path to the result file in XML format
    #
    # Output: 
    #        DOM tree object for result file
    #
    @classmethod
    def plot (cls, 
              result_file,
              output_dir = None):
        """ Plot test results in XML format."""

        total_winnings_line = None
        random_winnings_line = None
        
        results = CSEPFile.openFile(result_file)
        
        # Get both lines
        for line in results:
            if 'total winnings' in line:
                total_winnings_line = line
            if 'random bettor winnings' in line:
                random_winnings_line = line
                
        # Extract the values from the line
        win_index = total_winnings_line.index('=')
        total_winnings_line = float(total_winnings_line[win_index+1:])
        
        # Find the random bettor winnings winnings line
        # Extract the values from the winnings line
        win_index = random_winnings_line.index('[')
        random_winnings_line = random_winnings_line[win_index+1:-2]
        
        # Convert the values into float
        random_winnings_line = [float(item) for item in random_winnings_line.split(", ")]
        
        # Plot the Histogram
        num_bins = 50
        
        # The histogram of the data
        n, bins, patches = plt.hist(random_winnings_line, 
                                    num_bins, 
                                    facecolor='green', 
                                    alpha=0.5)
        ymin, ymax = plt.ylim()
        plt.vlines(total_winnings_line, 0, ymax, 
                   colors='red', 
                   linestyles='solid', 
                   label='Observed', 
                   linewidth=3,
                   alpha=0.5)
        
        plt.legend()
        plt.xlabel('Winnings')
        plt.ylabel('Number of simulated bettors')
        plt.title('Observed and simulated winnings')
        
        # Tweak spacing to prevent clipping of ylabel
        plt.subplots_adjust(left=0.15)           
        
        image_file = result_file
        if output_dir is not None:
            image_file = os.path.join(output_dir,
                                      os.path.basename(result_file))

        image_file = CSEPFile.Name.extension(image_file,
                                             CSEPFile.Extension.PNG)
          
        plt.savefig(image_file)
        plt.close()
         
        return [image_file, 
                cls.__resultToKML(result_file,
                                   output_dir)]
        

    #---------------------------------------------------------------------------
    # Create KML format file of evaluation outcome for display in Google maps
    #
    # Icons are stored on CSEP Trac (for example,
    # http://northridge.usc.edu/trac/csep/raw-attachment/wiki/M8Evaluation/FalseFalseIcon.png)
    #
    @classmethod
    def __resultToKML (cls, 
                       result_file,
                       output_dir):
        """ Create KML format file of evaluation outcome for display in Google maps."""
        
        
        
        
        kml_file = result_file
        if output_dir is not None:
            kml_file = os.path.join(output_dir,
                                    os.path.basename(result_file))

        kml_file = CSEPFile.Name.extension(kml_file,
                                           CSEPFile.Extension.KML)
        
        kml = M8FOGTest.KML(result_file)
        kml.createGMap(kml_file)
        return kml_file
          
        
    #-----------------------------------------------------------------------------
    #
    # Update cumulative test result data with daily result. This method can
    # be overwritten by derived classes. Alarm-based evaluation tests don't have
    # cumulative test results, therefore the class overwrites the method not to 
    # update the cumulative results.
    #
    # Input: 
    #        result_file - Daily result file to be used to update 
    #                      corresponding cumulative result file.
    #
    # Output: None.
    #
    def updateCumulativeResultData (self, result_file):
        """ Update cumulative test result data with daily result."""

        pass

