"""
Module PostProcess
"""

__version__ = "$Revision: 5062 $"
__revision__ = "$Id: PostProcess.py 5062 2015-03-10 19:56:22Z liukis $"

import os, datetime

import CSEPGeneric, CSEPFile, Environment, CSEPUtils, CSEP, MatlabLogical, \
       CSEPLogging
from ReproducibilityFiles import ReproducibilityFiles


#--------------------------------------------------------------------------------
#
# PostProcess.
#
# This class is designed for post-processing of the catalog data.
# It constructs a Matlab script that invokes actual post-processing for the
# catalog data.
#
class PostProcess (ReproducibilityFiles):

    ### Class to store catalog filter thresholds 
    class Threshold (object):
        
        def __init__ (self,
                      min_magnitude,
                      max_depth,
                      start_date = None):
            """ Initialize filter thresholds.
                
                Input arguments:
                min_magnitude - Minimum magnitude
                max_depth - Maximum depth
                start_date - Start date (default is None)
            """
            
            self.MinMagnitude = min_magnitude
            self.MaxDepth = max_depth
            self.DecimalStartDate = CSEPUtils.decimalYear(start_date)

    # If class is defined within 'class Files', can't reference it from Files.__init__--->
    # had to bring it outside of 'class Files'
    class Uncertainties (object):
        """Structure-like class to hold files and related directories for
           catalogs with applied uncertainties.
           
           Input arguments:
           catalog - Name of file with catalog uncertainties (default is 
                     CSEP.Catalog.Filename.Uncertainties)
           dir - Directory to store random number used to generate catalog
                 uncertainties (default is 'uncertainties')
           result_var - Matlab variable to store catalog uncertainties to
                        the file (default is MatlabLogical.CatalogVar.Uncertainties)
        """
        
        def __init__ (self,
                      catalog = CSEP.Catalog.Filename.Uncertainties,
                      dir = 'uncertainties',
                      result_var = MatlabLogical.CatalogVar.Uncertainties):
            
            self.catalog = catalog
            # Directory to store files with random numbers used to generate
            # catalog uncertainties
            self.dir = dir
            # Generated catalog uncertainties are still stored in Matlab 
            # cell array by scipy (TODO: legacy code from when CSEP was
            # using Matlab)
            self.matlabVar = result_var
    
    
    class Files (object):
        """Structure-like class to hold information for generated catalog files
           by post-processing filtering routines for the task (input catalog for 
           forecast generation, or observation catalog for evaluation test).
        """
        def __init__ (self,
                      catalog = CSEP.Catalog.Filename.Undeclustered,
                      catalog_uncertainties = CSEP.Catalog.Filename.Uncertainties,
                      cumulative_catalog = 'cumulative.'):
            """Initialize filenames for result catalogs and related files/directories
               to be generated by PostProcess.
               
               Input arguments:
               catalog - Filename for result catalog (default is CSEP.Catalog.Filename.Undeclustered)
               catalog_uncertainties - Filename for catalog with applied 
                                       uncertainties (default is CSEP.Catalog.Filename.Uncertainties)
               cumulative_catalog - Filename for cumulative catalog  (default is
                                    'cumulative.'). Data product is introduced in 
                                    CSEP V11.7.0 to support T and W evaluation tests.
                                    This catalog is constructed by the same 
                                    PostProcess thresholds, except for 
                                    the start date of the catalog - it's set to 
                                    the date when forecast group was introduced 
                                    to the Testing Center for evaluation. For example,
                                    3-month models were introduced on April 1, 2009, for
                                    testing. Start date for current 3-month forecasts is January 1, 2011.
                                    Current observation catalog for the testing period
                                    would include events [2011-01-01; now] while
                                    cumulative catalog would include events [2009-04-01; now]
            """
            
            self.catalog = catalog
            self.uncertainties = PostProcess.Uncertainties(catalog_uncertainties)
            self.cumulativeCatalog = cumulative_catalog
            
            if (self.catalog is not None) and (self.cumulativeCatalog is not None):
                self.cumulativeCatalog += self.catalog
            

    # Static data of the class
    # Full path to the source code top level path
    CenterCode = Environment.Environment.Variable[Environment.CENTER_CODE_ENV]
      
    # Name of the file that contains scaling factor for the forecasts that is
    # applied during evaluation test
    __scaleFactorFile = 'ForecastScaleFactor.dat'
    
    
    #--------------------------------------------------------------------
    #
    # Initialization.
    #
    def __init__ (self, 
                  magnitude,
                  depth,
                  duration = None,
                  catalogs = None,
                  xml_template = None,
                  test_intervals = {}):
        """ Initialize PostProcess class.
        
            Input arguments:
            magnitude - Minimum magnitude for catalog events
            depth - Maximum depth for catalog events
            duration - Time duration for the testing period (default is None) 
            catalogs - PostProcess.Files object that represents filenames of
                       results catalogs generated by PostProcess (default is 
                       None which implies PostProcess.Files() - could not use
                       default value of PostProcess.Files() due to namespace
                       issues by Python).
                       Derived child post-processing class can overwrite 
                       filename for the result data.
            xml_template - XML master template for the forecasts models
                           post-processing is applied to (default is None)
        """
        
        # Call base class constructor
        ReproducibilityFiles.__init__(self)
        
        # XML master template to be populated by data from already existing
        # forecast files
        self.__xmlTemplate = xml_template
        
        ### TODO: ?Define units for duration period?
        self.duration = duration
          
        # Start date of the testing period - it will vary depending on the 
        # forecast group (group is defined by the entry date into the testing center)
        self.__startDate = None
        
        # End date for the testing period
        self.__expirationDate = None
        
        # Start date for cumulative testing period - this date is defined by
        # forecast group entry date into the Testing Center
        self.__cumulativeStartDate = None
        
        self.threshold = PostProcess.Threshold(magnitude,
                                               depth)

        self.files = catalogs
        if catalogs is None:
            self.files = PostProcess.Files()
        elif isinstance(catalogs, str):
            self.files = PostProcess.Files(catalogs)

        # Register files for reproducibility 
        # NOTE: more reproducibility files might be added by derived from 
        #       PostProccess classes during catalog generation
        if self.files.catalog is not None:
           
           file_format = CSEPFile.Extension.toFormat(self.files.catalog)
           info_msg = "Result catalog '%s' in %s format generated by '%s' \
post-processing." %(self.files.catalog, 
                    file_format, 
                    self.type())

           ReproducibilityFiles.add(self,
                                    self.files.catalog, 
                                    info_msg, 
                                    file_format)
           
           if self.files.cumulativeCatalog is not None:
               file_format = CSEPFile.Extension.toFormat(self.files.cumulativeCatalog)
               
               info_msg = "Cumulative catalog '%s' in %s format generated by '%s' \
post-processing." %(self.files.cumulativeCatalog, 
                    file_format, 
                    self.type())
    
               ReproducibilityFiles.add(self,
                                        self.files.cumulativeCatalog, 
                                        info_msg, 
                                        file_format,
                                        is_required = False)
               
               
        # Add optional for reproducibility catalog with applied uncertainties
        # Should be done at object construction time to guarantee staging of 
        # the file if it exists
        if self.files.uncertainties.catalog is not None:
            
            file_format = CSEPFile.Extension.toFormat(self.files.uncertainties.catalog)
            info = "'%s' catalogs with applied uncertainties file used by '%s' post-processing." \
                   %(self.files.uncertainties.catalog, 
                     self.type())

            # It's optional data product - some data sources (CMT) don't apply
            # provided errors
            ReproducibilityFiles.add(self,
                                     self.files.uncertainties.catalog,
                                     info,
                                     file_format,
                                     is_required = False)

        # Support reset of forecast's start time in the middle of testing period:
        # dictionary of current forecast's start time and end time (instead of 
        # fixed forecast class )            
        self.testIntervals = test_intervals      


    #--------------------------------------------------------------------
    #
    # Get the type of post-processing.
    # This method is implemented by derived children classes.
    #
    # Input: None.
    # 
    # Output:
    #         Post-processing type.
    # 
    def type (self):
        """ Get type of the post-processing."""
        
        pass
     

    #---------------------------------------------------------------------------
    #
    # Register directory with random seed values that were used to generate
    # catalog uncertainties.
    #
    # Input:
    #        dir_path - Path to the directory that stores random seed values.
    # 
    # Output: None
    # 
    def registerUncertaintiesDir (self, dir_path):
        """ Register directory with random seed values that were used to
             generate catalog uncertainties."""

        if dir_path is not None:
           
           info = "'%s' directory with modified catalogs and used random seed \
files for '%s' post-processing." %(dir_path, 
                                   self.type())

           ReproducibilityFiles.add(self,
                                    dir_path,
                                    info,
                                    CSEPFile.Format.DIR)        


    #--------------------------------------------------------------------
    #
    # Set start date for the testing period used by the post-processing.
    #
    # Input: 
    #        test_date - Testing period start date
    # 
    # Output:
    #         None.
    # 
    def startDate (self, test_date):
        """ Set start date for the testing period used by the post-processing."""

        # Start date is not set yes
        if self.__startDate is None:
            self.__startDate = test_date
            self.threshold.DecimalStartDate = CSEPUtils.decimalYear(test_date)
        
            # Set cumulative start date if it's not set yet
            self.cumulativeStartDate(test_date)
        return
     

    #--------------------------------------------------------------------
    #
    # Set end date for the testing period used by the post-processing.
    #
    # Input: 
    #        test_date - Testing period end date
    # 
    # Output:
    #         None.
    # 
    def endDate (self, test_date):
        """ Set end date for the testing period used by the post-processing."""

        self.__expirationDate = test_date
        return


    #--------------------------------------------------------------------
    #
    # Set start date for the testing period used by the post-processing.
    #
    # Input: 
    #        test_date - Testing period start date
    # 
    # Output:
    #         None.
    # 
    def cumulativeStartDate (self, test_date):
        """ Set start date for the testing period used by the post-processing."""

        if (test_date is not None) and (self.__cumulativeStartDate is None):
            self.__cumulativeStartDate = test_date
        
        return


    def testDateDuration(self,
                         test_date):
        """ Since evaluation is invoked for the test date, 24-hours of the whole
            testing date should be included into calculation of the scale factor
            that represents the test date. It returns datetime.timedelta()
            representation of number of hours."""
            
        test_interval = datetime.timedelta(hours=24)
        
        if test_date in self.testIntervals:
            # Calculate duration of the test interval
            end_date = self.testIntervals[test_date]
            
            test_interval = end_date - test_date
            
        return test_interval 
    

    #---------------------------------------------------------------------------
    #
    # Write scale factor (since it's computed by Matlab code for now) that 
    # corresponds to the test date to the file
    #
    # Input: 
    #        test_date - Test date
    #        test_dir - Directory to write scale factor file to (!!!ONLY WHILE 
    #                   MATLAB computes it!!!) 
    #        compute_factor - Flag to indicate if scale factor should be 
    #                         computed and applied to the forecast during 
    #                         evaluation. Default is to use scale factor of 1.0.
    # 
    # Output:
    #        Scale factor
    # 
    def scaleFactor (self,
                     test_date,
                     test_dir,
                     compute_factor=True):
        """ Compute scale factor that represents test date and write it to the
            file (for now since it's computed by Matlab codes)."""

        
        # Create file with forecast scale factor no matter what - used
        # by map generation and evaluation test
        scale_file_path = os.path.join(test_dir,
                                       PostProcess.__scaleFactorFile)

        if not os.path.exists(scale_file_path):
    
            # Write scale factor to the file
            scale_factor = 1.0
            
            # Test date is the same as start date of the forecast period for
            # one-day models, use scale factor of 1.0, otherwise compute 
            # scale factor relative to provided duration in decimal years
#            print 'TestDate=', test_date, type(test_date), \
#                  'StartDate=', self.start_date, type(self.start_date)
            if test_date != self.start_date and \
               MatlabLogical.Boolean[compute_factor] == MatlabLogical.Boolean[True]: 
      
                 dec_test_date = CSEPUtils.decimalYear(test_date + self.testDateDuration(test_date))
                 dec_start_date = CSEPUtils.decimalYear(self.start_date)
                 
                 scale_factor = (dec_test_date - dec_start_date)/self.duration
                 
            CSEPLogging.CSEPLogging.getLogger(PostProcess.__name__).info("scaleFactor=%s start=%s test=%s duration=%s" %(scale_factor,
                                                                                                                         self.start_date,
                                                                                                                         test_date,
                                                                                                                         self.duration))
            
            fhandle = CSEPFile.openFile(scale_file_path,
                                        CSEPFile.Mode.WRITE)
            fhandle.write('%s\n' %scale_factor)
            fhandle.close()

        # Read value from the file
        scale_factor = [float(x) for x in open(scale_file_path)][0]
        
        return scale_factor

     
    #--------------------------------------------------------------------
    #
    # Get start date for the testing period used by the post-processing.
    #
    # Input: None.
    # 
    # Output:
    #         datetime object.
    # 
    def __getStartDate (self):
        """ Get start date for the testing period used by the post-processing."""

        return self.__startDate

    start_date = property(__getStartDate, startDate, 
                          doc = "Start date for the testing period.")    


    #--------------------------------------------------------------------
    #
    # Get end date for the testing period used by the post-processing.
    #
    # Input: None.
    # 
    # Output:
    #         datetime object.
    # 
    def __getEndDate (self):
        """ Get end date for the testing period used by the post-processing."""

        return self.__expirationDate

    end_date = property(__getEndDate, endDate, 
                        doc = "End date for the testing period.")    


    #--------------------------------------------------------------------
    #
    # Get end date for the testing period used by the post-processing.
    #
    # Input: None.
    # 
    # Output:
    #         datetime object.
    # 
    def __getCumulativeStartDate (self):
        """ Get start date for cumulative testing period used by the post-processing
            (defined by entry date of the forecast model into Testing Center for
             evaluation) 
        """

        return self.__cumulativeStartDate

    cumulative_start_date = property(__getCumulativeStartDate, 
                                     cumulativeStartDate, 
                                     doc = "Start date for cumulative testing period.")    


    #----------------------------------------------------------------------------
    #
    # Get XML master template for the models post-processing is applied to.
    #
    # Input: None.
    # 
    # Output:
    #         Filename for master XML template.
    # 
    def __getTemplate (self):
        """ Get master XML template used by the post-processing."""

        return self.__xmlTemplate

    template = property(__getTemplate, 
                          doc = "Master XML template for forecast models.")    


    #--------------------------------------------------------------------
    #
    # Post-process catalog data for the test.
    # This method is to be implemented by derived children classes.
    #
    # Input: 
    #        test_date - datetime object that represents the test date.    
    #        raw_file - Raw file to apply post-processing to.
    #
    def apply (self, test_date, raw_file):
        """ Post-process downloaded data.
            This method invokes post-processing that is specific to the
            forecast model. It should be implemented by all child classes.
            It constructs and invokes series of Matlab scripts that construct the post-processing.""" 

        pass
     

    #--------------------------------------------------------------------
    #
    # Does forecast expire on a given date?
    #
    # Input: 
    #        test_date - datetime object that represents the test date.    
    #
    # Output:
    #         True if forecast expires on a given date, False otherwise.
    #
    def expires (self, test_date):
        """ Does forecast expire on a given date?""" 

        if self.__expirationDate is None:
           return False
        else:
           return self.__expirationDate == test_date
