"""
Module ReproducibilityFiles
"""

__version__ = "$Revision: 4218 $"
__revision__ = "$Id: ReproducibilityFiles.py 4218 2013-03-04 21:01:48Z liukis $"

import os
import CSEPPropertyFile, CSEPFile, CSEPLogging


#--------------------------------------------------------------------------------
#
# ReproducibilityFiles.
#
# This class is designed for capture files required for reproducibility of 
# a given test result.
#
class ReproducibilityFiles (object):

    # Static data of the class
      
    # Logger object for the class  
    __logger = None 
    
    # Structure-like class to store information about file or directory used
    # by the testing framework for reproducibility
    class EntryInfo (object):
        
        def __init__ (self, 
                      info,
                      format,
                      preserve):
            
            # String representing information (comment) about the file
            self.comment = info
            # String representing internal format for the file 
            self.format = format
            # Flag if original file/directory should be preserved within
            # testing framework (see Trac ticket #306: Don't preserve 
            # OneDayModelInputCatalog data products within testing framework)
            self.preserve = preserve
            
        
        # Overwritten string representation of the class data
        def __repr__ (self):
            return "%s %s %s" %(self.comment,
                                self.format,
                                self.preserve)
      
    #----------------------------------------------------------------------------
    #
    # Initialization.
    #
    # Input: None
    # 
    def __init__ (self):
        """ Initialization for ReproducibilityFiles class"""
        
        # Names of the files generated by the object that would need to be preserved
        # for results reproducibility.
        # This dictionary stores a tuple of information and file format per each
        # entry.
        self.__files = {}

        # Names of optional files generated by the object that would need to be preserved
        # for results reproducibility. For example the same type of catalog 
        # post-processing (OneDayModel) might generate catalog uncertainties for
        # California testing region, and not generate them for Western Pacific
        # testing region.
        # This dictionary stores a tuple of information and file format per each
        # entry.
        self.__optionalFiles = {}
        
        if ReproducibilityFiles.__logger is None:
           ReproducibilityFiles.__logger = CSEPLogging.CSEPLogging.getLogger(ReproducibilityFiles.__name__)
        

    #----------------------------------------------------------------------------
    #
    # Add file required for reproducibility of results.
    #
    # Input:
    #        filename - Filename.
    #        info - Information about the file.
    #        format - Format for the file.
    #        is_required - Flag if file is required for reproducibility. Default
    #                      is True meaning that file is required, set to False 
    #                      if file is optional for the reproducibility.
    # 
    # Output:
    #         Tuple of two dictionaries: required and optional files.
    # 
    def add (self, filename, info, format, is_required = True, preserve = True):
        """ Add file required for reproducibility of results."""

        registry = self.__files
        if is_required is False:
            registry = self.__optionalFiles
            
        registry[filename] = ReproducibilityFiles.EntryInfo(info, 
                                                            format, 
                                                            preserve)


    #----------------------------------------------------------------------------
    #
    # Get dictionary of files required for reproducibility of results.
    #
    # Input: None.
    # 
    # Output:
    #         dictionary of registered files and metadata about them
    # 
    def __get (self):
        """ Get dictionary of files required for reproducibility of results."""

        return (self.__files, self.__optionalFiles)

    reproducibility = property(__get,  
                               doc = "Tuple of files dictionaries required for reproducibility of results.")    


    #----------------------------------------------------------------------------
    #
    # Remove registered entry
    #
    # Input:
    #        entry - Registered entry to remove
    # 
    # Output:
    #         dictionary of registered files and metadata about them
    # 
    def _delete (self, entry, is_required = True):
        """ Remove registered entry."""

        if is_required is True:
            del self.__files[entry]
        else:
            del self.__optionalFiles[entry]


    #----------------------------------------------------------------------------
    #
    # Create unique copies of registered files and remove original files.
    #
    # Input:
    #        entry_type - Keyword that identifies files registered for the 
    #                     reproducibility
    #        dir_path - Optional directory for registered files. Default is None.
    #        entry_description - Optional description for registered files to be
    #                            used as part of new filenames. Default is None.
    #                            If description is not provided, original filename
    #                            is used as a description.
    #        cleanup - Flag if original entries should be removed after copies
    #                  with unique filenames were generated.
    # 
    # Output: List of removed files 
    # 
    def copyAndCleanup (self, 
                        entry_type, 
                        dir_path = None, 
                        entry_description = None,
                        cleanup = True):
       """Create unique copies of registered files and remove original files."""

       # List of entries that were renamed and removed (some registered entries
       # are optional)
       existing_entries = []
       
       # Generate unique copies of files required for forecast reproducibility.
       # There are two sets of registered files: required and optional.
       # Raise an exception is required file is missing, generate a warning if
       # optional file is missing
       required_flag = True
       for each_registry, is_required in ((self.__files, required_flag), 
                                          (self.__optionalFiles, not required_flag)):

          # 'value' is of ReproducibilityFiles.EntryInfo type
          for entry, value in each_registry.items():
             
             entry_path = entry

             # Path for the entry was specified
             if (dir_path is not None) and (os.path.isabs(dir_path) is True):
                entry_path = os.path.join(dir_path, entry)
                
             if os.path.exists(entry_path):
                 
                existing_entries.append(entry)
             
                description = entry
                if entry_description is not None:
                   description = entry_description
                   
                filenames = CSEPPropertyFile.CSEPPropertyFile.filenamePair(entry_type, 
                                                                           description)
      
                # Unpack the sequence
                datafile, metafile = filenames
      
                datafile_path = datafile
                if (dir_path is not None) and (os.path.isabs(dir_path) is True):
                   datafile_path = os.path.join(dir_path,
                                                datafile)

                if value.preserve:
                    # Create unique copy only if flag for the data product is
                    # set to preserve it
                    CSEPFile.copy(entry_path, 
                                  datafile_path)
                
                # Check if corresponding data file is a link - only if directory
                # is provided
                is_datafile_link = None
                if dir_path is not None:
                   is_datafile_link = entry_path
                
                metafile_path = metafile
                if dir_path is not None:
                   metafile_path = os.path.join(dir_path,
                                                metafile)   

                # Create metadata file
                CSEPPropertyFile.CSEPPropertyFile.createMetafile(metafile_path, 
                                                                 entry,
                                                                 value.format,
                                                                 value.comment,
                                                                 is_datafile_link,
                                                                 preserve_data = value.preserve)

                # Remove original file
                if cleanup is True:
                   ReproducibilityFiles.__logger.info("copyAndCleanup(): \
Removing file %s for '%s' identifier." %(entry, entry_type))
                   os.remove(entry_path)

             else:

                error_msg = "copyAndCleanup(): \
File entry %s does not exist for '%s' identifier." %(entry_path, entry_type)

                if is_required is False:

                   # Optional entry does not exist, generate a warning
                   ReproducibilityFiles.__logger.warning(error_msg)
                   
                else:
                   
                   # Required entry does not exist, generate an error
                   ReproducibilityFiles.__logger.error(error_msg)
                   raise RuntimeError, error_msg
         
                      
       # Re-set registry if original files have been already cleaned up
       if cleanup is True:
          self.__files = {}
          self.__optionalFiles = {}
          
          
       return existing_entries                    
       
             