"""
Module M8EvaluationTests
"""

__version__ = "$Revision$"
__revision__ = "$Id$"


import sys, os, unittest, shutil, datetime, filecmp

import CSEPFile, CSEPInitFile, CSEP

from CSEPTestCase import CSEPTestCase
from EvaluationTest import EvaluationTest
from M8FOGTest import M8FOGTest
from RELMAftershockPostProcess import RELMAftershockPostProcess


from RELMCatalog import RELMCatalog
from PostProcessFactory import PostProcessFactory
from ForecastGroup import ForecastGroup
from CSEPLogging import CSEPLogging
from ForecastHandlerFactory import ForecastHandlerFactory
from PolygonForecastHandler import PolygonForecastHandler


# Logger object for the module
__logger = None


#-------------------------------------------------------------------------------
# Function to access logger object for the module.
#-------------------------------------------------------------------------------
def _moduleLogger():
    """ Get logger object for the module, initialize one if it does not exist"""
    
    global __logger
    if __logger is None:
        __logger = CSEPLogging.getLogger(__name__)
    
    return __logger


 #------------------------------------------------------------------------------
 #
 # Test evaluation tests tailored for M8 forecasts assessment.
 #
class M8EvaluationTests (CSEPTestCase):

    # Directory with reference data for the tests
    __referenceDir = os.path.join(CSEPTestCase.ReferenceDataDir, 
                                  'evaluationTests',
                                  'm8')
    

    #---------------------------------------------------------------------------
    #
    # Invoke M8FOG evaluation test for M8.0 forecast by Volodya Kossobokov, 
    # and validate the results.
    #
    def testM8FOGm80Test(self):
        """ Run M8FOG evaluation test for M8.0 forecast evaluation and succeed."""

        # Setup test name
        CSEPTestCase.setTestName(self, 
                                 self.id())
        
        test_list = M8FOGTest.Type

        # Directory with forecast file
        forecast_dir = os.path.join(CSEPTestCase.TestDirPath,
                                    'forecasts')
        os.makedirs(forecast_dir)
        
        forecast_file = 'M80@08b.dat'
        shutil.copyfile(os.path.join(M8EvaluationTests.__referenceDir, 
                                     forecast_file),
                        os.path.join(forecast_dir, 
                                     forecast_file))    

        # Reference file with test results
        result_filename = "m8Test_M8FOG-Test_M80@08b_result.dat"        

        args = [datetime.datetime(2008, 7, 1), 
                datetime.datetime(2013, 7, 1)]
        post_process = PostProcessFactory().object(RELMAftershockPostProcess.Type,
                                                   args)
        catalog_file = post_process.files.catalog
        
        # Observation catalog for evaluation
        reference_catalog = 'ehdf1990_to_2013w23_inclusive.dat' 
        
        _moduleLogger().info('Copying reference catalog %s to %s' %(os.path.join(M8EvaluationTests.__referenceDir, 
                                                                                 reference_catalog),
                                                                   os.path.join(CSEPTestCase.TestDirPath, 
                                                                                catalog_file)))
        shutil.copyfile(os.path.join(M8EvaluationTests.__referenceDir, 
                                     reference_catalog),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     catalog_file))        
        
        m8_inputs = "randomSeedFile=%s" %os.path.join(M8EvaluationTests.__referenceDir,
                                                      'seed.txt')
        self.__evaluationTest(test_list,
                              post_process,
                              forecast_dir,
                              result_filename,
                              m8_inputs)


    #---------------------------------------------------------------------------
    #
    # Invoke M8FOG evaluation test for M7.5 forecast by Volodya Kossobokov, 
    # and validate the results.
    #
    def testM8FOGm75Test(self):
        """ Run M8FOG evaluation test for M7.5 forecast evaluation and succeed."""

        # Setup test name
        CSEPTestCase.setTestName(self, 
                                 self.id())
        
        test_list = M8FOGTest.Type

        # Directory with forecast file
        forecast_dir = os.path.join(CSEPTestCase.TestDirPath,
                                    'forecasts')
        os.makedirs(forecast_dir)
        
        forecast_file = 'M75@08b.dat'
        shutil.copyfile(os.path.join(M8EvaluationTests.__referenceDir, 
                                     forecast_file),
                        os.path.join(forecast_dir, 
                                     forecast_file))    

        # Reference file with test results
        result_filename = "m8Test_M8FOG-Test_M75@08b_result.dat"        

        args = [datetime.datetime(2008, 7, 1), 
                datetime.datetime(2013, 7, 1)]
        post_process = PostProcessFactory().object(RELMAftershockPostProcess.Type,
                                                   args)
        catalog_file = post_process.files.catalog
        
        # Observation catalog for evaluation
        reference_catalog = 'ehdf1990_to_2013w23_inclusive.dat' 
        
        _moduleLogger().info('Copying reference catalog %s to %s' %(os.path.join(M8EvaluationTests.__referenceDir, 
                                                                                 reference_catalog),
                                                                   os.path.join(CSEPTestCase.TestDirPath, 
                                                                                catalog_file)))
        shutil.copyfile(os.path.join(M8EvaluationTests.__referenceDir, 
                                     reference_catalog),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     catalog_file))        
        
        m75_inputs = "minMagnitude=7.5, alarmRadius=427, randomSeedFile=%s" %os.path.join(M8EvaluationTests.__referenceDir,
                                                                                          'seed.txt')
        self.__evaluationTest(test_list,
                              post_process,
                              forecast_dir,
                              result_filename,
                              m75_inputs)
          

    #--------------------------------------------------------------------
    #
    # Run evaluation test for the forecast, and validate the results.
    #
    # Inputs:
    #            test_list - List of evaluation tests to invoke.
    #            post_process_type - Keyword identifying PostProcessing that
    #                                         has been applied to the catalog data.
    #            random_file - Name of the file with random numbers used by the 
    #                                 evaluation test.
    #            forecast_dir - Directory that stores forecast files for evaluation.
    #            result_file - Name of the result file in ascii format used for
    #                             test validation.
    #            
    #
    def __evaluationTest(self, 
                         test_list,
                         post_process, 
                         forecast_dir,
                         ref_result_file,
                         tests_inputs=None):
        """ Run evaluation test for the forecast and succeed."""


        ### Generate test directory
        RELMCatalog(CSEPTestCase.TestDirPath)

        # ForecastGroup object that represents forecast models for the test
        forecast_group = ForecastGroup(forecast_dir,
                                       post_process,
                                       test_list,
                                       tests_inputs)

        result_file = os.path.join(CSEPTestCase.TestDirPath, 
                                   ref_result_file)
        
        # Use the same directory for catalog data and test results
        plot_file = None
        for each_set in forecast_group.tests:
            for each_test in each_set:
               each_test.run(datetime.datetime(2013, 8, 1), 
                             CSEPTestCase.TestDirPath,
                             CSEPTestCase.TestDirPath)
               
               # Check for existence of histogram plot
               plot_file = each_test.plot(result_file)[0]

               self.failIf(os.path.exists(plot_file) is False, 
                    "Plot result file %s does not exist." %plot_file)
               
               # Check for existence of histogram plot
               kml_file = CSEPFile.Name.extension(result_file,
                                                  CSEPFile.Extension.KML)

               self.failIf(os.path.exists(kml_file) is False, 
                    "KML result file %s does not exist." %kml_file)
               

        ### Check existence of test results
        self.failIf(os.path.exists(result_file) is False, 
                    "Test result file %s does not exist." %result_file)
        
        # Compare to provided reference result file
        reference_file = os.path.join(M8EvaluationTests.__referenceDir,
                                      ref_result_file)

        self.failIf(filecmp.cmp(reference_file, result_file) is False, 
                    "Failed to compare evaluation test results: %s vs. %s" %(reference_file,
                                                                             result_file))
        

# Invoke the module
if __name__ == '__main__':

   import logging
   from PDEDataSource import PDEDataSource
   from DataSourceFactory import DataSourceFactory
   from PostProcessFactory import PostProcessFactory


   DataSourceFactory.composite().clear()
   DataSourceFactory().object(PDEDataSource.Type,
                              input_variables = CSEPTestCase.DataSourceArgs)
   
   # Invoke all tests
   unittest.main()

   # Shutdown logging
   logging.shutdown()
        
# end of main