"""
Module StatisticalEvaluationTests
"""

__version__ = "$Revision$"
__revision__ = "$Id$"


import sys, os, unittest, shutil, datetime, glob
from xml.etree.cElementTree import tostring
import numpy as np

import CSEPFile

from CSEPTestCase import CSEPTestCase
from EvaluationTest import EvaluationTest
from RELMCatalog import RELMCatalog
from OneDayModelPostProcess import OneDayModelPostProcess
from ForecastGroup import ForecastGroup
from CSEPLogging import CSEPLogging
from TStatisticalTest import TStatisticalTest
from WStatisticalTest import WStatisticalTest
from RELMAftershockPostProcess import RELMAftershockPostProcess
from CSEPInitFile import CSEPInitFile
from ThreeMonthsModelPostProcess import ThreeMonthsModelPostProcess
from CSEPSchedule import CSEPSchedule
from ForecastHandlerFactory import ForecastHandlerFactory
from TXStatisticalTest import TXStatisticalTest
from WXStatisticalTest import WXStatisticalTest
from ForecastGroupInitFile import ForecastGroupInitFile

from BogusForecastModel1 import BogusForecastModel1
from BogusForecastModel2 import BogusForecastModel2
from BogusForecastModel3 import BogusForecastModel3
from BogusThirtyMinutesModel1 import BogusThirtyMinutesModel1
from BogusThirtyMinutesModel2 import BogusThirtyMinutesModel2
from ThirtyMinutesModelPostProcess import ThirtyMinutesModelPostProcess
from OneYearModelPostProcess import OneYearModelPostProcess
from OneDayModelPostProcess import OneDayModelPostProcess


# Logger object for the module
__logger = None


#-------------------------------------------------------------------------------
# Function to access logger object for the module.
#-------------------------------------------------------------------------------
def _moduleLogger():
    """ Get logger object for the module, initialize one if it does not exist"""
    
    global __logger
    if __logger is None:
        __logger = CSEPLogging.getLogger(__name__)
    
    return __logger


 #------------------------------------------------------------------------------
 #
 # Test statistical evaluation tests for the forecasts models. This module tests
 # reproducibility of tests results.
 #
class StatisticalEvaluationTests (CSEPTestCase):

    # Directory with reference data for the tests
    __referenceDir = os.path.join(CSEPTestCase.ReferenceDataDir, 
                                  'statistical_tests')
    
    # Test date 
    __testDate = datetime.datetime(2011, 3, 8)

    __TVars = ['meanInformationGain',
               'lowerConfidenceLimits',
               'upperConfidenceLimits',
               'numberEvents']
    # XML elements to validate for each test
    __XMLElements = {TStatisticalTest.Type: __TVars,
                     TXStatisticalTest.Type: __TVars,
                     WStatisticalTest.Type: ['WilcoxonSignificance'],
                     WXStatisticalTest.Type: ['WilcoxonSignificance']}

    __referenceData = {TStatisticalTest.Type: 'sTest_T-Test.xml',
                       WStatisticalTest.Type: 'sTest_W-Test.xml',
                       TXStatisticalTest.Type: 'sTest_TX-Test.xml',
                       WXStatisticalTest.Type: 'sTest_WX-Test.xml'}


    #---------------------------------------------------------------------------
    #
    # Run T and W evaluation tests for already prepared forecasts rates (Canterbury
    # aftershock sequence is used) and validate the results.
    #
    def testPreparedDataForTWTests(self):
        """ Run T and W evaluation tests for the already prepared forecast \
data and succeed. This test is using already collected forecasts rates for \
observed events, and verifies that Python impelementation of the test is \
working properly."""

        # Setup test name
        CSEPTestCase.setTestName(self, 
                                 self.id())
        
        test_list = '%s %s' %(TStatisticalTest.Type, 
                              WStatisticalTest.Type)

        # Directory with forecasts files
        forecast_dir = 'forecasts-rates'
        cumulative_start_date = '2010-09-04'
        forecast_group = ForecastGroup(os.path.join(StatisticalEvaluationTests.__referenceDir,
                                                    forecast_dir),
                                       OneDayModelPostProcess.Type,
                                       test_list,
                                       post_process_inputs = cumulative_start_date)
        
        # Create matrix of total forecasts rates (as it was used by original R code)
        np_rates = np.array([[66.611278,  66.611278,  4.608372,  4.608372],
                             [288.523330, 288.523330, 19.960993, 19.960993],
                             [4.648330,   4.648330,   4.648330,  4.648330],
                             [2.091676,   2.091676,   2.091676,  2.091676]])
        
        test_obj = forecast_group.tests[0][0]
        test_rates = test_obj.rates(forecast_group)
        
        test_rates.np_sum_rates = np_rates
        
        all_models = ['PPEETAS-rates-fromXML.dat',
                      'ETAS-rates-fromXML.dat',
                      'PPEEEPAS-0F-rates-fromXML.dat',
                      'ppe5yrnzdec-fromXML.dat.RATES-fromXML.dat']
        
        test_rates.all_models = all_models
        
        np_event_rates = np.zeros((len(all_models),),
                                  dtype = np.object)
        
        for index, model in enumerate(all_models):
             np_event_rates[index] = ForecastHandlerFactory().CurrentHandler.load(os.path.join(StatisticalEvaluationTests.__referenceDir,
                                                                                             forecast_dir,
                                                                                             model))
        test_rates.np_event_rates = np_event_rates
        
        shutil.copyfile(os.path.join(StatisticalEvaluationTests.__referenceDir, 
                                     OneDayModelPostProcess().files.catalog),
                        os.path.join(CSEPTestCase.TestDirPath,
                                     OneDayModelPostProcess().files.catalog))
        
        catalog = RELMCatalog.load(os.path.join(CSEPTestCase.TestDirPath,
                                                OneDayModelPostProcess().files.catalog))
        
        for each_test in forecast_group.tests[0]:
            each_test.cumulativeCatalogFile.npObject = catalog
            each_test.testDir = CSEPTestCase.TestDirPath
            each_test.testDate = StatisticalEvaluationTests.__testDate
       
        for each_test in forecast_group.tests[0]:
            for each_forecast in all_models:
                each_test.evaluate(each_forecast)

            # Evaluate result data
            self.__evaluateResults(each_test)


    #---------------------------------------------------------------------------
    #
    # Run T and W evaluation tests for file-based forecasts and validate 
    # the results.
    #
    def testFileBasedForecastsTests(self):
        """ Run T and W evaluation tests for file-based forecasts \
and succeed. This test collects forecasts rates for the whole testing period \
and per each observed event, invokes T and W evaluation tests and verifies \
results."""

        # Setup test name
        CSEPTestCase.setTestName(self, 
                                 self.id())
        
        catalog_file = '2006_01_01.catalog.nodecl.dat'

        # Copy over catalog file to the test directory
        _moduleLogger().info('Copying reference catalog %s to %s' %(os.path.join(StatisticalEvaluationTests.__referenceDir, 
                                                                                 catalog_file),
                                                                    os.path.join(CSEPTestCase.TestDirPath, 
                                                                                 RELMAftershockPostProcess().files.catalog)))
        shutil.copyfile(os.path.join(StatisticalEvaluationTests.__referenceDir, 
                                     catalog_file),
                        os.path.join(CSEPTestCase.TestDirPath,
                                     RELMAftershockPostProcess().files.catalog))    
                
        # Tests to invoke
        test_list = '%s %s' %(TStatisticalTest.Type, 
                              WStatisticalTest.Type)

        # Directory with forecasts files
        forecast_dir = 'file_forecasts'
        test_date = datetime.datetime(2011, 5, 2)

        forecast_group = ForecastGroup(os.path.join(StatisticalEvaluationTests.__referenceDir,
                                                    forecast_dir),
                                       RELMAftershockPostProcess.Type,
                                       test_list)
        
        
        for each_test in forecast_group.tests[0]:
            each_test.run(test_date,
                          CSEPTestCase.TestDirPath,
                          CSEPTestCase.TestDirPath)

            # Evaluate result data
            self.__evaluateResults(each_test,
                                   os.path.join(StatisticalEvaluationTests.__referenceDir,
                                                'file_forecasts_results'))


    #---------------------------------------------------------------------------
    #
    # Run T and W evaluation tests for one-day forecasts and validate 
    # the results.
    #
    def testOneDayForecasts(self):
        """ Run T and W evaluation tests for one-day forecasts \
and succeed. This test collects forecasts rates for the whole testing period \
and per each observed event, invokes T and W evaluation tests and verifies \
results."""

        # Setup test name
        CSEPTestCase.setTestName(self, 
                                 self.id())
        
        # Observation catalogs
        catalog_ref_dir = os.path.join(StatisticalEvaluationTests.__referenceDir, 
                                       'one_day_catalogs')

        # Observation catalog for test date        
        catalog_file = 'catalog.nodecl.dat'

        # Copy over catalog file to the test directory
        _moduleLogger().info('Copying reference catalog %s to %s' %(os.path.join(catalog_ref_dir, 
                                                                                 catalog_file),
                                                                    os.path.join(CSEPTestCase.TestDirPath, 
                                                                                 OneDayModelPostProcess().files.catalog)))
        shutil.copyfile(os.path.join(catalog_ref_dir, 
                                     catalog_file),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     OneDayModelPostProcess().files.catalog))    

        # Observation catalog for test date        
        catalog_file = 'cumulative.catalog.nodecl.dat'

        # Copy over catalog file to the test directory
        _moduleLogger().info('Copying reference catalog %s to %s' %(os.path.join(catalog_ref_dir, 
                                                                                 catalog_file),
                                                                    os.path.join(CSEPTestCase.TestDirPath, 
                                                                                 OneDayModelPostProcess().files.cumulativeCatalog)))
        shutil.copyfile(os.path.join(catalog_ref_dir, 
                                     catalog_file),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     OneDayModelPostProcess().files.cumulativeCatalog))    
        
        # One-day models for the test
        models = '%s %s' %(BogusForecastModel1.Type,
                           BogusForecastModel2.Type)
                
        # Tests to invoke
        test_list = '%s %s' %(TStatisticalTest.Type, 
                              WStatisticalTest.Type)

        # Directory with forecasts files
        group_dir = 'one_day_forecasts'
        
        shutil.copytree(os.path.join(StatisticalEvaluationTests.__referenceDir, 
                                     group_dir),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     group_dir))
        
        test_date = datetime.datetime(2011, 4, 3)

        forecast_group = ForecastGroup(os.path.join(CSEPTestCase.TestDirPath, 
                                                    group_dir),
                                       OneDayModelPostProcess.Type,
                                       test_list,
                                       model_list = models,
                                       post_process_inputs = [datetime.datetime(2011, 3, 28)])
        
        
        for each_test in forecast_group.tests[0]:
            each_test.run(test_date,
                          CSEPTestCase.TestDirPath,
                          CSEPTestCase.TestDirPath)

            # Evaluate result data
            self.__evaluateResults(each_test,
                                   os.path.join(StatisticalEvaluationTests.__referenceDir,
                                                'one_day_forecasts_results'))


    #---------------------------------------------------------------------------
    #
    # Run T and W evaluation tests for three_month forecasts and validate 
    # the results.
    #
    def testThreeMonthForecasts(self):
        """ Run T and W evaluation tests for three-month forecasts \
and succeed. This test collects forecasts rates for the whole testing period \
and per each observed event, invokes T and W evaluation tests and verifies \
results."""

        # Setup test name
        CSEPTestCase.setTestName(self, 
                                 self.id())
        
        # One-day models for the test
        models = '%s %s %s' %(BogusForecastModel1.Type,
                              BogusForecastModel2.Type,
                              BogusForecastModel3.Type)
                
        # Tests to invoke
        test_list = '%s %s' %(TStatisticalTest.Type, 
                              WStatisticalTest.Type)

        # Directory with forecasts files
        group_dir = 'three_month_forecasts'
        
        shutil.copytree(os.path.join(StatisticalEvaluationTests.__referenceDir, 
                                     group_dir),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     group_dir))
        
        test_date = datetime.datetime(2011, 5, 15)

        forecast_group = ForecastGroup(os.path.join(CSEPTestCase.TestDirPath, 
                                                    group_dir),
                                       ThreeMonthsModelPostProcess.Type,
                                       test_list,
                                       model_list = models,
                                       post_process_inputs = [datetime.datetime(2011, 4, 1),
                                                              datetime.datetime(2011, 7, 1),
                                                              datetime.datetime(2010, 10, 1)])
        
        # Observation catalogs
        catalog_ref_dir = os.path.join(StatisticalEvaluationTests.__referenceDir, 
                                       'three_month_catalogs')

        # Copy over observation catalog file to the test directory
        _moduleLogger().info('Copying reference catalog %s to %s' %(os.path.join(catalog_ref_dir, 
                                                                                 forecast_group.postProcess().files.catalog),
                                                                    os.path.join(CSEPTestCase.TestDirPath, 
                                                                                 forecast_group.postProcess().files.catalog)))
        shutil.copyfile(os.path.join(catalog_ref_dir, 
                                     forecast_group.postProcess().files.catalog),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     forecast_group.postProcess().files.catalog))    

        # Copy over cumulative catalog file to the test directory
        _moduleLogger().info('Copying reference catalog %s to %s' %(os.path.join(catalog_ref_dir, 
                                                                                 forecast_group.postProcess().files.cumulativeCatalog),
                                                                    os.path.join(CSEPTestCase.TestDirPath, 
                                                                                 forecast_group.postProcess().files.cumulativeCatalog)))
        shutil.copyfile(os.path.join(catalog_ref_dir, 
                                     forecast_group.postProcess().files.cumulativeCatalog),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     forecast_group.postProcess().files.cumulativeCatalog))    
        
        
        models_schedule = CSEPSchedule()
        models_schedule.add('*',
                            '1 4 7 10',
                            '1')
        forecast_group.models.schedule = models_schedule
        
        for each_test in forecast_group.tests[0]:
            each_test.run(test_date,
                          CSEPTestCase.TestDirPath,
                          CSEPTestCase.TestDirPath)

            # Evaluate result data
            self.__evaluateResults(each_test,
                                   os.path.join(StatisticalEvaluationTests.__referenceDir,
                                                'three_month_results'))


    #---------------------------------------------------------------------------
    #
    # Run T and W evaluation tests for one-day forecasts with observation
    # catalog of 1 event to make sure that statistical tests are not invoked.
    #
    def testLessThanTwoEventsCatalogs(self):
        """ Run T and W evaluation tests for less than 2 observed events \
to make sure that tests are not invoked."""

        # Setup test name
        CSEPTestCase.setTestName(self, 
                                 self.id())
        
        # Observation catalogs
        catalog_ref_dir = os.path.join(StatisticalEvaluationTests.__referenceDir, 
                                       'one_event_catalog')

        # Observation catalog for test date        
        catalog_file = 'catalog.nodecl.dat'

        # Copy over catalog file to the test directory
        _moduleLogger().info('Copying reference catalog %s to %s' %(os.path.join(catalog_ref_dir, 
                                                                                 catalog_file),
                                                                    os.path.join(CSEPTestCase.TestDirPath, 
                                                                                 OneDayModelPostProcess().files.catalog)))
        shutil.copyfile(os.path.join(catalog_ref_dir, 
                                     catalog_file),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     OneDayModelPostProcess().files.catalog))    

        # Observation catalog for test date        
        catalog_file = 'cumulative.catalog.nodecl.dat'

        # Copy over catalog file to the test directory
        _moduleLogger().info('Copying reference catalog %s to %s' %(os.path.join(catalog_ref_dir, 
                                                                                 catalog_file),
                                                                    os.path.join(CSEPTestCase.TestDirPath, 
                                                                                 OneDayModelPostProcess().files.cumulativeCatalog)))
        shutil.copyfile(os.path.join(catalog_ref_dir, 
                                     catalog_file),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     OneDayModelPostProcess().files.cumulativeCatalog))    
        
        # One-day models for the test
        models = '%s %s' %(BogusForecastModel1.Type,
                           BogusForecastModel2.Type)
                
        # Tests to invoke
        test_list = '%s %s' %(TStatisticalTest.Type, 
                              WStatisticalTest.Type)

        # Directory with forecasts files
        group_dir = 'one_day_forecasts'
        
        shutil.copytree(os.path.join(StatisticalEvaluationTests.__referenceDir, 
                                     group_dir),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     group_dir))
        
        test_date = datetime.datetime(2011, 4, 3)

        forecast_group = ForecastGroup(os.path.join(CSEPTestCase.TestDirPath, 
                                                    group_dir),
                                       OneDayModelPostProcess.Type,
                                       test_list,
                                       model_list = models,
                                       post_process_inputs = [datetime.datetime(2011, 3, 28)])
        
        
        for each_test in forecast_group.tests[0]:
            each_test.run(test_date,
                          CSEPTestCase.TestDirPath,
                          CSEPTestCase.TestDirPath)

            # Evaluate result data
            test_file = os.path.join(CSEPTestCase.TestDirPath,
                                     StatisticalEvaluationTests.__referenceData[each_test.type()])

            self.failIf(os.path.exists(test_file) is True,
                        "Test result file %s is generated for observation catalog of less than 32events")


    #---------------------------------------------------------------------------
    #
    # Run T and W evaluation tests for thirty-minutes forecasts and validate 
    # the results.
    #
    def testThirtyMinutesForecasts(self):
        """ Run T and W evaluation tests for thirty-minutes forecasts \
and succeed. This test collects forecasts rates for the whole testing period \
and per each observed event, invokes T and W evaluation tests and verifies \
results."""

        # Setup test name
        CSEPTestCase.setTestName(self, 
                                 self.id())
        
        # Observation catalogs
        catalog_ref_dir = os.path.join(StatisticalEvaluationTests.__referenceDir, 
                                       'thirty_minutes_catalogs')

        # Observation catalog for test date        
        catalog_file = 'catalog.nodecl.dat'

        # Copy over catalog file to the test directory
        _moduleLogger().info('Copying reference catalog %s to %s' %(os.path.join(catalog_ref_dir, 
                                                                                 catalog_file),
                                                                    os.path.join(CSEPTestCase.TestDirPath, 
                                                                                 OneDayModelPostProcess().files.catalog)))
        shutil.copyfile(os.path.join(catalog_ref_dir, 
                                     catalog_file),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     OneDayModelPostProcess().files.catalog))    

        # Observation catalog for test date        
        catalog_file = 'cumulative.catalog.nodecl.dat'

        # Copy over catalog file to the test directory
        _moduleLogger().info('Copying reference catalog %s to %s' %(os.path.join(catalog_ref_dir, 
                                                                                 catalog_file),
                                                                    os.path.join(CSEPTestCase.TestDirPath, 
                                                                                 OneDayModelPostProcess().files.cumulativeCatalog)))
        shutil.copyfile(os.path.join(catalog_ref_dir, 
                                     catalog_file),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     OneDayModelPostProcess().files.cumulativeCatalog))    
        
        # Thirty-minute model for the test
        models = '%s %s' %(BogusThirtyMinutesModel1.Type,
                           BogusThirtyMinutesModel2.Type)
                
        # Tests to invoke
        test_list = '%s %s' %(TStatisticalTest.Type, 
                              WStatisticalTest.Type)

        # Directory with forecasts files
        group_dir = 'thirty_minutes_forecasts'
        
        shutil.copytree(os.path.join(StatisticalEvaluationTests.__referenceDir, 
                                     group_dir),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     group_dir))
        
        test_date = datetime.datetime(2013, 8, 1, 23, 30, 0)

        forecast_group = ForecastGroup(os.path.join(CSEPTestCase.TestDirPath, 
                                                    group_dir),
                                       ThirtyMinutesModelPostProcess.Type,
                                       test_list,
                                       model_list = models,
                                       post_process_inputs = [datetime.datetime(2013, 8, 1, 23, 30, 0),
                                                              datetime.datetime(2013, 8, 2, 0, 0, 0),
                                                              datetime.datetime(2013, 8, 1)])
        forecast_group.postProcess().endDate(datetime.datetime(2013, 8, 2, 0, 0, 0))
        
        forecast_group.models.schedule.add('*', '*', '*', '0:23', '0:30:30', '0')
        
        
        for each_test in forecast_group.tests[0]:
            each_test.run(test_date,
                          CSEPTestCase.TestDirPath,
                          CSEPTestCase.TestDirPath)

            # Evaluate result data
            self.__evaluateResults(each_test,
                                   os.path.join(StatisticalEvaluationTests.__referenceDir,
                                                'thirty_minutes_forecasts_results'))


    #---------------------------------------------------------------------------
    #
    # Validate a fix for Trac ticket #318: T and W tests for 30-minutes class 
    # won't include event observed during the last 30 minutes of the test date
    #
    def testThirtyMinutesForecastsTrac318(self):
        """ Run T and W evaluation tests for thirty-minutes forecasts \
with observation event during last 30-minutes of the test date and succeed."""

        # Setup test name
        CSEPTestCase.setTestName(self, 
                                 self.id())
        
        # Observation catalogs
        catalog_ref_dir = os.path.join(StatisticalEvaluationTests.__referenceDir, 
                                       'thirty_minutes_catalogs_Trac318')

        # Observation catalog for test date        
        catalog_file = 'catalog.nodecl.dat'

        # Copy over catalog file to the test directory
        _moduleLogger().info('Copying reference catalog %s to %s' %(os.path.join(catalog_ref_dir, 
                                                                                 catalog_file),
                                                                    os.path.join(CSEPTestCase.TestDirPath, 
                                                                                 OneDayModelPostProcess().files.catalog)))
        shutil.copyfile(os.path.join(catalog_ref_dir, 
                                     catalog_file),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     OneDayModelPostProcess().files.catalog))    

        # Observation catalog for test date        
        catalog_file = 'cumulative.catalog.nodecl.dat'

        # Copy over catalog file to the test directory
        _moduleLogger().info('Copying reference catalog %s to %s' %(os.path.join(catalog_ref_dir, 
                                                                                 catalog_file),
                                                                    os.path.join(CSEPTestCase.TestDirPath, 
                                                                                 OneDayModelPostProcess().files.cumulativeCatalog)))
        shutil.copyfile(os.path.join(catalog_ref_dir, 
                                     catalog_file),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     OneDayModelPostProcess().files.cumulativeCatalog))    
        
        # Thirty-minute model for the test
        models = '%s %s' %(BogusThirtyMinutesModel1.Type,
                           BogusThirtyMinutesModel2.Type)
                
        # Tests to invoke
        test_list = '%s %s' %(TStatisticalTest.Type, 
                              WStatisticalTest.Type)

        # Directory with forecasts files
        group_dir = 'thirty_minutes_forecasts_Trac318'
        
        shutil.copytree(os.path.join(StatisticalEvaluationTests.__referenceDir, 
                                     group_dir),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     group_dir))
        
        test_date = datetime.datetime(2012, 8, 26, 23, 30, 0)

        forecast_group = ForecastGroup(os.path.join(CSEPTestCase.TestDirPath, 
                                                    group_dir),
                                       ThirtyMinutesModelPostProcess.Type,
                                       test_list,
                                       model_list = models,
                                       post_process_inputs = [datetime.datetime(2012, 8, 26, 23, 30, 0),
                                                              datetime.datetime(2012, 8, 27, 0, 0, 0),
                                                              datetime.datetime(2012, 8, 26, 23, 30, 0)])
        forecast_group.postProcess().endDate(datetime.datetime(2012, 8, 27, 0, 0, 0))
        
        forecast_group.models.schedule.add('*', '*', '*', '0:23', '0:30:30', '0')
        
        
        for each_test in forecast_group.tests[0]:
            each_test.run(test_date,
                          CSEPTestCase.TestDirPath,
                          CSEPTestCase.TestDirPath)

            # Evaluate result data
            self.__evaluateResults(each_test,
                                   os.path.join(StatisticalEvaluationTests.__referenceDir,
                                                'thirty_minutes_results_Trac318'))


    #---------------------------------------------------------------------------
    #
    # Run T and W evaluation tests for one-day forecasts with other than
    # 00:00:00 start time and validate the results.
    #
    def testOneDayNonZeroStartTimeForecasts(self):
        """ Run T and W evaluation tests for one-day forecasts \
and succeed. This test collects forecasts rates for the whole testing period \
and per each observed event, invokes T and W evaluation tests and verifies \
results."""

        # Setup test name
        CSEPTestCase.setTestName(self, 
                                 self.id())
        
        # Directory with forecasts files
        group_dir = 'one_day_forecasts_nonZeroTime'
        
        shutil.copytree(os.path.join(StatisticalEvaluationTests.__referenceDir, 
                                     group_dir),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     group_dir))
        
        test_date = datetime.datetime(2014, 4, 3, 2, 35, 45)

        forecast_group = ForecastGroup(os.path.join(CSEPTestCase.TestDirPath, 
                                                    group_dir))
        
        
        for each_test in forecast_group.tests[0]:
            each_test.run(test_date,
                          os.path.join(CSEPTestCase.TestDirPath, 
                                       forecast_group.catalogDir()),
                          CSEPTestCase.TestDirPath)
#                           os.path.join(CSEPTestCase.TestDirPath, 
#                                        forecast_group.resultDir()))

            # Evaluate result data
            self.__evaluateResults(each_test,
                                   os.path.join(StatisticalEvaluationTests.__referenceDir,
                                                'one_day_forecasts_nonZeroTime_results'))


    #---------------------------------------------------------------------------
    #
    # Run T and W evaluation tests for one-year forecasts which reset
    # start time within testing period and validate the results.
    #
    def testOneYearTimeResetForecasts(self):
        """ Run T and W evaluation tests for one-year forecasts that reset start \
time during testing period and succeed. This test collects forecasts rates for \
the whole testing period and per each observed event, invokes T and W evaluation \
tests and verifies results."""

        # Setup test name
        CSEPTestCase.setTestName(self, 
                                 self.id())
        
        # One-day models for the test
        models = '%s %s' %(BogusForecastModel1.Type,
                           BogusForecastModel2.Type)
                
        # Tests to invoke
        test_list = '%s %s' %(TStatisticalTest.Type, 
                              WStatisticalTest.Type)

        # Directory with forecasts files
        group_dir = 'one_year_resetTime_forecasts'
        
        shutil.copytree(os.path.join(StatisticalEvaluationTests.__referenceDir, 
                                     group_dir),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     group_dir))
        
        test_date = datetime.datetime(2012, 2, 29, 2, 18, 4)
        test_input = "endDates=2010-09-03T16:35:42/2011-09-03T16:35:42 \
                               2011-02-21T23:51:43/2012-02-21T23:51:43 \
                               2011-06-13T02:20:50/2012-06-13T02:20:50 \
                               2011-12-23T02:18:04/2012-12-23T02:18:04|"

        forecast_group = ForecastGroup(os.path.join(CSEPTestCase.TestDirPath, 
                                                    group_dir),
                                       OneYearModelPostProcess.Type,
                                       test_list,
                                       test_input,
                                       model_list = models,
                                       post_process_inputs = [datetime.datetime(2011, 12, 23, 2, 18, 4),
                                                              datetime.datetime(2012, 12, 23, 2, 18, 4),
                                                              datetime.datetime(2010, 9, 3, 16, 35, 42)])
        
        # Observation catalogs
        catalog_ref_dir = os.path.join(StatisticalEvaluationTests.__referenceDir, 
                                       'one_year_resetTime_catalogs')

        # Copy over observation catalog file to the test directory
        ref_catalog = os.path.join(catalog_ref_dir, 
                                   forecast_group.postProcess().files.catalog)
        test_catalog = os.path.join(CSEPTestCase.TestDirPath, 
                                    forecast_group.postProcess().files.catalog)
        _moduleLogger().info('Copying reference catalog %s to %s' %(ref_catalog,
                                                                    test_catalog))
        shutil.copyfile(ref_catalog,
                        test_catalog)    

        # Copy over cumulative catalog file to the test directory
        ref_catalog = os.path.join(catalog_ref_dir, 
                                   forecast_group.postProcess().files.cumulativeCatalog)
        test_catalog = os.path.join(CSEPTestCase.TestDirPath, 
                                    forecast_group.postProcess().files.cumulativeCatalog)

        _moduleLogger().info('Copying reference catalog %s to %s' %(ref_catalog,
                                                                    test_catalog))
        shutil.copyfile(ref_catalog,
                        test_catalog)    
        
        
        models_schedule = CSEPSchedule()
        models_schedule.add('2010','9','3', '16', '35', '42')
        models_schedule.add('2011','2','21', '23', '51', '43')
        models_schedule.add('2011','6','13', '2', '20', '50')
        models_schedule.add('2011','12','23', '2', '18', '4')
        models_schedule.add('2012','12','23', '2', '18', '4')
        
        forecast_group.models.schedule = models_schedule
        
        for each_test in forecast_group.tests[0]:
            each_test.run(test_date,
                          CSEPTestCase.TestDirPath,
                          CSEPTestCase.TestDirPath)

            # Evaluate result data
            self.__evaluateResults(each_test,
                                   os.path.join(StatisticalEvaluationTests.__referenceDir,
                                                'one_year_resetTime_results'))


    #---------------------------------------------------------------------------
    #
    # Run T and W evaluation tests for one-day forecasts which reset
    # start time within testing period and validate the results.
    #
    def testOneDayTimeResetForecasts(self):
        """ Run T and W evaluation tests for one-day forecasts that reset start \
time during testing period and succeed. This test collects forecasts rates for \
the whole testing period and per each observed event, invokes T and W evaluation \
tests and verifies results."""

        # Setup test name
        CSEPTestCase.setTestName(self, 
                                 self.id())
        
        # One-day models for the test
        models = '%s %s' %(BogusForecastModel1.Type,
                           BogusForecastModel2.Type)
                
        # Tests to invoke
        test_list = '%s %s' %(TStatisticalTest.Type, 
                              WStatisticalTest.Type)

        # Directory with forecasts files
        group_dir = 'one_day_resetTime_forecasts'
        
        shutil.copytree(os.path.join(StatisticalEvaluationTests.__referenceDir, 
                                     group_dir),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     group_dir))
        
        test_date = datetime.datetime(2011, 2, 23, 23, 51, 43)
        test_input = "endDates=2011-02-21T16:35:42/2011-02-22T16:35:42|"

        forecast_group = ForecastGroup(os.path.join(CSEPTestCase.TestDirPath, 
                                                    group_dir),
                                       OneDayModelPostProcess.Type,
                                       test_list,
                                       test_input,
                                       model_list = models,
                                       post_process_inputs = [datetime.datetime(2011, 2, 20, 16, 35, 42)])
        
        # Observation catalogs
        catalog_ref_dir = os.path.join(StatisticalEvaluationTests.__referenceDir, 
                                       'one_day_resetTime_catalogs')

        # Copy over observation catalog file to the test directory
        ref_catalog = os.path.join(catalog_ref_dir, 
                                   forecast_group.postProcess().files.catalog)
        test_catalog = os.path.join(CSEPTestCase.TestDirPath, 
                                    forecast_group.postProcess().files.catalog)
        _moduleLogger().info('Copying reference catalog %s to %s' %(ref_catalog,
                                                                    test_catalog))
        shutil.copyfile(ref_catalog,
                        test_catalog)    

        # Copy over cumulative catalog file to the test directory
        ref_catalog = os.path.join(catalog_ref_dir, 
                                   forecast_group.postProcess().files.cumulativeCatalog)
        test_catalog = os.path.join(CSEPTestCase.TestDirPath, 
                                    forecast_group.postProcess().files.cumulativeCatalog)

        _moduleLogger().info('Copying reference catalog %s to %s' %(ref_catalog,
                                                                    test_catalog))
        shutil.copyfile(ref_catalog,
                        test_catalog)    
        
        
        models_schedule = CSEPSchedule()
        models_schedule.add('2011','2','20', '16', '35', '42')
        models_schedule.add('2011','2','21', '16', '35', '42')
        models_schedule.add('2011','2','21:25', '23', '51', '43')
        
        forecast_group.models.schedule = models_schedule
        
        for each_test in forecast_group.tests[0]:
            each_test.run(test_date,
                          CSEPTestCase.TestDirPath,
                          CSEPTestCase.TestDirPath)

            # Evaluate result data
            self.__evaluateResults(each_test,
                                   os.path.join(StatisticalEvaluationTests.__referenceDir,
                                                'one_day_resetTime_results'))


    #---------------------------------------------------------------------------
    #
    # Run TX and WX evaluation tests for one-year and one-month forecasts which
    # reset start time within testing period and validate the results.
    #
    def testOneYearOneMonthTimeResetForecasts(self):
        """ Run TX and WX evaluation tests for one-year and one-month forecasts that reset start \
time during testing period and succeed. This test collects forecasts rates for \
the whole testing period and per each observed event, invokes T and W evaluation \
tests and verifies results."""

        # Setup test name
        CSEPTestCase.setTestName(self, 
                                 self.id())

        # Forecast group to compare to
        group_dir = 'one_year'
        shutil.copytree(os.path.join(StatisticalEvaluationTests.__referenceDir, 
                                     group_dir),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     group_dir))

        # Directory with forecasts files
        group_dir = 'one_month'
        shutil.copytree(os.path.join(StatisticalEvaluationTests.__referenceDir, 
                                     group_dir),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     group_dir))

        # Put absolute path for the referred forecast group
        xmldoc = ForecastGroupInitFile(os.path.join(CSEPTestCase.TestDirPath,
                                                    group_dir))
        group = xmldoc.elements(TXStatisticalTest.XML.GroupElement)[0]
        group.attrib[TXStatisticalTest.XML.DirAttribute] = os.path.join(CSEPTestCase.TestDirPath, 
                                                                        group.attrib[TXStatisticalTest.XML.DirAttribute])
        
        # Write modified file to the test directory
        init_file = os.path.join(CSEPTestCase.TestDirPath,
                                 group_dir,
                                 "forecast.init.xml")
      
        fhandle = CSEPFile.openFile(init_file, 
                                    CSEPFile.Mode.WRITE)
        xmldoc.write(fhandle)
        fhandle.close()

        
        test_date = datetime.datetime(2011, 2, 25, 23, 51, 43)

        forecast_group = ForecastGroup(os.path.join(CSEPTestCase.TestDirPath, 
                                                    group_dir))
        
        for each_test in forecast_group.tests[0]:
            each_test.run(test_date,
                          os.path.join(CSEPTestCase.TestDirPath, 
                                       forecast_group.catalogDir()),
                          CSEPTestCase.TestDirPath)
            # Verify that plotting is working
            plots = each_test.plot(os.path.join(CSEPTestCase.TestDirPath,
                                                'sTest_%s-Test.xml' %each_test.type()))

            self.failIf(len(plots) == 0,
                    "Plot files are expected for %s test" %each_test.type())

            # Evaluate result data
            self.__evaluateResults(each_test,
                                   os.path.join(StatisticalEvaluationTests.__referenceDir,
                                                'one_year_one_month_results'))


    #---------------------------------------------------------------------------
    #
    # Run TX and WX evaluation tests for one-day and static five-year forecasts
    # with provided start time and validate the results.
    #
    def testOneDayFiveYearForecasts(self):
        """ Run TX and WX evaluation tests for one-day and five-year forecasts.\
            This test provides minimum magnitude threshold and other than group
            entry date for the evaluation. The test collects forecasts rates for \
            the whole testing period and per each observed event across forecasts
            groups of different classes, and invokes TX and WX evaluation \
            tests and verifies results."""   

        # Setup test name
        CSEPTestCase.setTestName(self, 
                                 self.id())

        # Forecast group to compare to
        group_dir = 'five_year'
        shutil.copytree(os.path.join(StatisticalEvaluationTests.__referenceDir, 
                                     group_dir),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     group_dir))
        # Copy static forecasts for the group
        files_dir = os.path.join(StatisticalEvaluationTests.__referenceDir,
                                 'file_forecasts')
        for each_file in os.listdir(files_dir):
            if not os.path.isdir(os.path.join(files_dir,
                                              each_file)):
                shutil.copyfile(os.path.join(files_dir, 
                                             each_file),
                                os.path.join(CSEPTestCase.TestDirPath,
                                             group_dir,
                                             each_file))
            

        # Directory with daily forecasts configuration
        group_dir = 'one_day'
        shutil.copytree(os.path.join(StatisticalEvaluationTests.__referenceDir, 
                                     group_dir),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     group_dir))
        # Copy daily forecasts to the group
        shutil.copytree(os.path.join(StatisticalEvaluationTests.__referenceDir, 
                                     'one_day_forecasts'),
                        os.path.join(CSEPTestCase.TestDirPath,
                                     group_dir,
                                     'forecasts'))
         
        # Put absolute path for the referred forecast group
        xmldoc = ForecastGroupInitFile(os.path.join(CSEPTestCase.TestDirPath,
                                                    group_dir))
        group = xmldoc.elements(TXStatisticalTest.XML.GroupElement)[0]
        group.attrib[TXStatisticalTest.XML.DirAttribute] = os.path.join(CSEPTestCase.TestDirPath, 
                                                                        group.attrib[TXStatisticalTest.XML.DirAttribute])
        
        # Write modified file to the test directory
        init_file = os.path.join(CSEPTestCase.TestDirPath,
                                 group_dir,
                                 "forecast.init.xml")
      
        fhandle = CSEPFile.openFile(init_file, 
                                    CSEPFile.Mode.WRITE)
        xmldoc.write(fhandle)
        fhandle.close()

        
        test_date = datetime.datetime(2011, 4, 3)

        forecast_group = ForecastGroup(os.path.join(CSEPTestCase.TestDirPath, 
                                                    group_dir))
        
        for each_test in forecast_group.tests[0]:
            each_test.run(test_date,
                          os.path.join(CSEPTestCase.TestDirPath, 
                                       forecast_group.catalogDir()),
                          CSEPTestCase.TestDirPath)
            # Verify that plotting is working
            plots = each_test.plot(os.path.join(CSEPTestCase.TestDirPath,
                                                'sTest_%s-Test.xml' %each_test.type()))
 
            self.failIf(len(plots) == 0,
                    "Plot files are expected for %s test" %each_test.type())

            # Evaluate result data
            self.__evaluateResults(each_test,
                                   os.path.join(StatisticalEvaluationTests.__referenceDir,
                                                'one_day_five_year_results'))



    #----------------------------------------------------------------------------
    #
    # Validate results for provided EvaluationTest object
    #
    def __evaluateResults(self, 
                          test_obj,
                          results_dir = None):                               
        """ Validate evaluation test results."""


        ### Evaluate test results
        reference_dir = StatisticalEvaluationTests.__referenceDir
        if results_dir is not None:
            reference_dir = results_dir
            
        reference_file = os.path.join(reference_dir,
                                      StatisticalEvaluationTests.__referenceData[test_obj.type()])

        test_file = os.path.join(CSEPTestCase.TestDirPath,
                                 StatisticalEvaluationTests.__referenceData[test_obj.type()])
        
        CSEPLogging.getLogger(StatisticalEvaluationTests.__name__).info("Comparing reference evaluation \
test file %s with generated evaluation test file %s..." %(reference_file, 
                                                          test_file)) 
                          
        # Extract values of specified XML elements from both files and compare
        reference_xml = CSEPInitFile(reference_file)
        test_xml = CSEPInitFile(test_file)
        
        # Use percent difference validating results        
        percent_diff = 0.01
                
        for each_tag in StatisticalEvaluationTests.__XMLElements[test_obj.type()]:
           ref_elements = reference_xml.elements(each_tag)
           test_elements = test_xml.elements(each_tag)           
           
           for each_ref_elem, each_test_elem in zip(ref_elements,
                                                    test_elements):

              _moduleLogger().info("Comparing %s \
XML element: %s to %s" %(each_tag, tostring(each_ref_elem), tostring(each_test_elem))) 
              
              self.failIf(CSEPFile.compareLines(each_ref_elem.text,
                                                each_test_elem.text,
                                                percent_diff,
                                                True) is False,
                          "Failed to compare %s XML elements: expected '%s', received '%s'"
                          %(each_tag, tostring(each_ref_elem), tostring(each_test_elem)))
              
        
        # Make sure plots are generated for each test result 
        plot_files = test_obj.plot(test_file)
       
        for each_plot in plot_files:
            self.failIf(os.path.exists(each_plot) is False,
                        "Failed to generate plot file '%s'")
    
        # Fix for Trac ticket #150: Make sure result file is renamed to 
        # unique filename (malformed filename won't be renamed) 
        test_obj.resultData()
        
        # Make sure that all results in XML files are renamed:
        xml_files = glob.glob('%s/*%s' %(CSEPTestCase.TestDirPath,
                                         CSEPFile.Extension.XML))
        self.failIf(len(xml_files) != 0,
                    "Copies of XML files with unique filenames were not generated for: %s" \
                    %xml_files)

        # Make sure that all plots in SVG files are renamed:
        svg_files = glob.glob('%s/*%s' %(CSEPTestCase.TestDirPath,
                                         CSEPFile.Extension.SVG))
        self.failIf(len(svg_files) != 0,
                    "Copies of SVG plot files with unique filenames were not generated for: %s" \
                   %svg_files)

          
# Invoke the module
if __name__ == '__main__':
   
   # Invoke all tests
   unittest.main()
        
# end of main
