"""
Module CSEPScheduleTest
"""

__version__ = "$Revision: 4906 $"
__revision__ = "$Id: CSEPScheduleTest.py 4906 2014-09-29 23:30:02Z liukis $"


import sys, os, unittest, shutil, datetime

from CSEPTestCase import CSEPTestCase
from ForecastGroupInitFile import ForecastGroupInitFile
from BatchInitFile import BatchInitFile
from ForecastGroup import ForecastGroup


#--------------------------------------------------------------------
#
# Validate that ForecastGroup class properly parses 
# and initializes CSEPSchedule objects from the initialization file.
#
class CSEPScheduleTest (CSEPTestCase):

   # Static data of the class
   
   # Unit tests use sub-directory of global reference data directory
   __referenceDataDir = os.path.join(CSEPTestCase.ReferenceDataDir, 
                                     'unitTest', 'schedule')

   
   #-----------------------------------------------------------------------------
   #
   # This test verifies that CSEPSchedule class is creating dates according to
   # schedule as specified in configuration file.
   #
   def test(self):
      """ Confirm that CSEPSchedule class properly specifies dates according \
to the configuration file."""

      # Setup test name
      CSEPTestCase.setTestName(self, 
                               self.id())

      # Copy reference init file to the test directory
      init_file = "forecast.init.xml"
      shutil.copyfile(os.path.join(CSEPScheduleTest.__referenceDataDir, init_file),
                      os.path.join(CSEPTestCase.TestDirPath, init_file))    

      # Use test directory as forecast group directory
      init_file = ForecastGroupInitFile(CSEPTestCase.TestDirPath)

      ### Validate results
      
      ### Models schedule
      element = ForecastGroupInitFile.ModelElement
      models_schedule = init_file.schedule(element)
      
      error = "Expected valid schedule element for '%s'." %(element)
      self.failIf(models_schedule is None, error)
      
      ### Test for specific dates
      # Year is specified as any
      test_date = datetime.datetime(2006, 1, 15)
      error = "Expected '%s' date to be found for '%s' schedule." \
              %(test_date.date(), element)
      self.assertTrue(models_schedule.has(test_date), error)


      # Year is specified as any, month is provided by second 'month' element
      test_date = datetime.datetime(2005, 11, 10)
      error = "Expected '%s' date to be found for '%s' schedule." \
              %(test_date.date(), element)
      self.assertTrue(models_schedule.has(test_date), error)


      # Year is explicit, month is any
      test_date = datetime.datetime(2007, 7, 21)
      error = "Expected '%s' date to be found for '%s' schedule." \
              %(test_date.date(), element)
      self.assertTrue(models_schedule.has(test_date), error)


      # Year is explicit, month is explicit, day is any
      test_date = datetime.datetime(2007, 9, 28)
      error = "Expected '%s' date to be found for '%s' schedule." \
              %(test_date.date(), element)
      self.assertTrue(models_schedule.has(test_date), error)


      # Date is explicit
      test_date = datetime.datetime(2006, 5, 1)
      error = "Expected '%s' date to be found for '%s' schedule." \
              %(test_date.date(), element)
      self.assertTrue(models_schedule.has(test_date), error)

      # Date is not in the schedule
      test_date = datetime.datetime(2005, 8, 1)
      error = "Unexpected '%s' date is found for '%s' schedule." \
              %(test_date.date(), element)
      self.assertFalse(models_schedule.has(test_date), error)

      # Test ranges
      for each_year in xrange(2009, 2012):
          for each_day in xrange(1, 16):
              test_date = datetime.datetime(each_year, 3, each_day)
              
              error = "Expected to find '%s' date for '%s' schedule." \
                      %(test_date.date(), element)
              self.assertTrue(models_schedule.has(test_date), error)

      # Test ranges with increments
      for each_year in xrange(2000, 2007, 2):
          for each_month in xrange(4, 13, 4):
              for each_day in xrange(22, 29, 2):
                  test_date = datetime.datetime(each_year, each_month, each_day)
                  
                  error = "Expected to find '%s' date for '%s' schedule." \
                          %(test_date.date(), element)
                  self.assertTrue(models_schedule.has(test_date), error)


      ### evaluationTests schedule
      element = ForecastGroupInitFile.EvaluationTestElement
      tests_schedule = init_file.schedule(element)
      error = "Expected valid schedule element for '%s'." %(element)
      self.failIf(tests_schedule is None, error)


      # Date is implicit: should match any date
      test_date = datetime.datetime(2007, 6, 19)
      error = "Expected '%s' date to be found for '%s' schedule." \
              %(test_date.date(), element)
      self.assertTrue(tests_schedule.has(test_date), error)
      
      
   #-----------------------------------------------------------------------------
   #
   # This test verifies that CSEPSchedule class creates a complete set of dates
   # that represent the schedule.
   #
   def testBatchDates(self):
      """ Confirm that CSEPSchedule class creates complete list of dates that \
represent specified schedule."""

      # Setup test name
      CSEPTestCase.setTestName(self, 
                               self.id())

      try:
         
         # Configuration file for the test directory
         config_file = "forecast.init.xml"
   
         # Use test directory as forecast group directory
         init_file = ForecastGroupInitFile(CSEPScheduleTest.__referenceDataDir,
                                           config_file)
         
         ### Models schedule
         element = ForecastGroupInitFile.ModelElement
         models_schedule = init_file.schedule(element)         
         
         for d in models_schedule.dates():
            pass

         self.fail("Failed to raise an exception for an ambiguous start date")

      except RuntimeError, error:

         self.failIf(error.args[0].find("dates(): Start date is ambiguous") < 0, 
            "Failed to raise exception of expected content.")
      
         
      # Configuration file for the test directory
      config_file = "forecast_dates.init.xml"

      # Use test directory as forecast group directory
      init_file = ForecastGroupInitFile(CSEPScheduleTest.__referenceDataDir,
                                        config_file)

      element = ForecastGroupInitFile.EvaluationTestElement
      tests_schedule = init_file.schedule(element)

      ### Validate results
      all_dates = []
      
      for each_date in tests_schedule.dates():
         all_dates.append(each_date)
         
      # Expected results
      reference_value = [datetime.datetime(2007, 11, 5, 0, 0),
                         datetime.datetime(2007, 11, 6, 0, 0),
                         datetime.datetime(2007, 11, 7, 0, 0),
                         datetime.datetime(2007, 12, 1, 0, 0),
                         datetime.datetime(2007, 12, 2, 0, 0),
                         datetime.datetime(2007, 12, 3, 0, 0),
                         datetime.datetime(2007, 12, 5, 0, 0),
                         datetime.datetime(2007, 12, 6, 0, 0),
                         datetime.datetime(2007, 12, 7, 0, 0),
                         datetime.datetime(2007, 12, 31, 0, 0),
                         datetime.datetime(2008, 1, 1, 0, 0), 
                         datetime.datetime(2008, 1, 15, 0, 0), 
                         datetime.datetime(2008, 1, 30, 0, 0), 
                         datetime.datetime(2008, 2, 1, 0, 0), 
                         datetime.datetime(2008, 2, 2, 0, 0), 
                         datetime.datetime(2008, 2, 3, 0, 0), 
                         datetime.datetime(2008, 2, 4, 0, 0), 
                         datetime.datetime(2008, 2, 5, 0, 0), 
                         datetime.datetime(2008, 2, 6, 0, 0), 
                         datetime.datetime(2008, 2, 7, 0, 0), 
                         datetime.datetime(2008, 2, 8, 0, 0), 
                         datetime.datetime(2008, 2, 9, 0, 0), 
                         datetime.datetime(2008, 2, 10, 0, 0), 
                         datetime.datetime(2008, 2, 11, 0, 0), 
                         datetime.datetime(2008, 2, 12, 0, 0), 
                         datetime.datetime(2008, 2, 13, 0, 0), 
                         datetime.datetime(2008, 2, 14, 0, 0), 
                         datetime.datetime(2008, 2, 15, 0, 0), 
                         datetime.datetime(2008, 2, 16, 0, 0), 
                         datetime.datetime(2008, 2, 17, 0, 0), 
                         datetime.datetime(2008, 2, 18, 0, 0), 
                         datetime.datetime(2008, 2, 19, 0, 0), 
                         datetime.datetime(2008, 2, 20, 0, 0), 
                         datetime.datetime(2008, 2, 21, 0, 0), 
                         datetime.datetime(2008, 2, 22, 0, 0), 
                         datetime.datetime(2008, 2, 23, 0, 0), 
                         datetime.datetime(2008, 2, 24, 0, 0), 
                         datetime.datetime(2008, 2, 25, 0, 0), 
                         datetime.datetime(2008, 2, 26, 0, 0), 
                         datetime.datetime(2008, 2, 27, 0, 0), 
                         datetime.datetime(2008, 2, 28, 0, 0), 
                         datetime.datetime(2008, 2, 29, 0, 0),
                         datetime.datetime(2008, 3, 1, 0, 0), 
                         datetime.datetime(2008, 3, 2, 0, 0), 
                         datetime.datetime(2008, 3, 3, 0, 0)]

      self.failIf(reference_value != all_dates,
                  "Failed to get all schedule dates: expected %s, got %s"
                  %(reference_value,
                    all_dates))
      

   #----------------------------------------------------------------------------
   #
   # This test verifies that CSEPSchedule class is creating datetime objects
   #  according to the schedule as specified in configuration file.
   #
   def testTime(self):
      """ Confirm that CSEPSchedule class properly specifies datetime objects \
according to the configuration file."""

      # Setup test name
      CSEPTestCase.setTestName(self, 
                               self.id())

      # Copy reference init file to the test directory
      init_file = "forecast_times.init.xml"
      shutil.copyfile(os.path.join(CSEPScheduleTest.__referenceDataDir, init_file),
                      os.path.join(CSEPTestCase.TestDirPath, init_file))    

      # Use test directory as forecast group directory
      init_file = ForecastGroupInitFile(CSEPTestCase.TestDirPath,
                                        init_file)

      ### Validate results
      
      ### Models schedule
      element = ForecastGroupInitFile.EvaluationTestElement
      tests_schedule = init_file.schedule(element)
      
      error = "Expected valid schedule element for '%s'." %(element)
      self.failIf(tests_schedule is None, error)
      
      ### Test for specific dates and times
      # Date and time are explicit in the schedule
      test_date = datetime.datetime(2008, 3, 1, 4, 0, 0)
      error = "Expected '%s' date to be found for '%s' schedule." \
              %(test_date, element)
      self.assertTrue(tests_schedule.has(test_date), error)

      test_date = datetime.datetime(2008, 3, 1, 4, 30, 0)
      error = "Expected '%s' date to be found for '%s' schedule." \
              %(test_date, element)
      self.assertTrue(tests_schedule.has(test_date), error)

      test_date = datetime.datetime(2008, 3, 1, 4, 29, 0)
      error = "Unexpected '%s' date is specified by the '%s' schedule." \
              %(test_date, element)
      self.assertFalse(tests_schedule.has(test_date), error)

      test_date = datetime.datetime(2008, 3, 1, 13, 0, 0)
      error = "Unexpected '%s' date is specified by the '%s' schedule." \
              %(test_date, element)
      self.assertFalse(tests_schedule.has(test_date), error)

      # Only date is specified by the schedule, not time
      test_date = datetime.datetime(2008, 1, 15, 0, 0, 0)
      error = "Expected '%s' date to be found for '%s' schedule." \
              %(test_date, element)
      self.assertTrue(tests_schedule.has(test_date), error)

      # Only date is specified by the schedule, use not-existent time
      test_date = datetime.datetime(2008, 1, 15, 1, 0, 0)
      error = "Unexpected '%s' date is specified by the '%s' schedule." \
              %(test_date, element)
      self.assertFalse(tests_schedule.has(test_date), error)

      # Any day within a month is specified by the schedule
      test_date = datetime.datetime(2008, 2, 10, 0, 0, 0)
      error = "Expected '%s' date to be found for '%s' schedule." \
              %(test_date, element)
      self.assertTrue(tests_schedule.has(test_date), error)

      # Only date is specified by the schedule, use not-existent time
      test_date = datetime.datetime(2008, 2, 11, 11, 30, 0)
      error = "Unexpected '%s' date is specified by the '%s' schedule." \
              %(test_date, element)
      self.assertFalse(tests_schedule.has(test_date), error)


      # Test ranges
      for each_year in xrange(2009, 2011):
          for each_month in xrange(6, 7):
              for each_hour in xrange(1, 24):
                  test_date = datetime.datetime(each_year, each_month, 9,
                                                each_hour, 0, 0)
                  
                  error = "Expected to find '%s' date for '%s' schedule." \
                          %(test_date, element)
                  self.assertTrue(tests_schedule.has(test_date), error)


                  test_date = datetime.datetime(each_year, each_month, 9,
                                                each_hour, 30, 0)
                  
                  error = "Expected to find '%s' date for '%s' schedule." \
                          %(test_date, element)
                  self.assertTrue(tests_schedule.has(test_date), error)

                  test_date = datetime.datetime(each_year, each_month, 15,
                                                2, 45, 20)
                  
                  error = "Expected to find '%s' date for '%s' schedule." \
                          %(test_date, element)
                  self.assertTrue(tests_schedule.has(test_date), error)


   #----------------------------------------------------------------------------
   #
   # This test verifies that CSEPSchedule class is creating datetime objects
   # according to the schedule as specified in BatchProcessing
   # configuration file.
   #
   def testBatchInitFile(self):
      """ Confirm that CSEPSchedule class properly specifies datetime objects \
according to the configuration file."""

      # Setup test name
      CSEPTestCase.setTestName(self, 
                               self.id())

      # Copy reference init file to the test directory
      init_file = "batch_config.init.xml"

      config_file = BatchInitFile(os.path.join(CSEPScheduleTest.__referenceDataDir,
                                               init_file))
      
      reference_dates = [datetime.datetime(2010, 9, 3, 16, 35, 42),
                         datetime.datetime(2011, 2, 20, 23, 51, 43),
                         datetime.datetime(2011, 2, 21, 23, 51, 43),
                         datetime.datetime(2011, 6, 12, 2, 20, 50),
                         datetime.datetime(2011, 6, 13, 2, 20, 50),
                         datetime.datetime(2011, 7, 1, 2, 20, 50),
                         datetime.datetime(2011, 7, 2, 2, 20, 50),
                         datetime.datetime(2011, 7, 3, 2, 20, 50),
                         datetime.datetime(2011, 7, 4, 2, 20, 50),
                         datetime.datetime(2011, 7, 5, 2, 20, 50),
                         datetime.datetime(2011, 7, 6, 2, 20, 50),
                         datetime.datetime(2011, 7, 7, 2, 20, 50),
                         datetime.datetime(2011, 7, 8, 2, 20, 50),
                         datetime.datetime(2011, 7, 9, 2, 20, 50),
                         datetime.datetime(2011, 7, 10, 2, 20, 50),
                         datetime.datetime(2011, 7, 11, 2, 20, 50),
                         datetime.datetime(2011, 7, 12, 2, 20, 50),
                         datetime.datetime(2011, 7, 13, 2, 20, 50),
                         datetime.datetime(2011, 7, 14, 2, 20, 50),
                         datetime.datetime(2011, 7, 15, 2, 20, 50),
                         datetime.datetime(2011, 7, 16, 2, 20, 50),
                         datetime.datetime(2011, 7, 17, 2, 20, 50),
                         datetime.datetime(2011, 7, 18, 2, 20, 50),
                         datetime.datetime(2011, 7, 19, 2, 20, 50),
                         datetime.datetime(2011, 7, 20, 2, 20, 50),
                         datetime.datetime(2011, 7, 21, 2, 20, 50),
                         datetime.datetime(2011, 7, 22, 2, 20, 50),
                         datetime.datetime(2011, 7, 23, 2, 20, 50),
                         datetime.datetime(2011, 7, 24, 2, 20, 50),
                         datetime.datetime(2011, 7, 25, 2, 20, 50),
                         datetime.datetime(2011, 7, 26, 2, 20, 50),
                         datetime.datetime(2011, 7, 27, 2, 20, 50),
                         datetime.datetime(2011, 7, 28, 2, 20, 50),
                         datetime.datetime(2011, 7, 29, 2, 20, 50),
                         datetime.datetime(2011, 7, 30, 2, 20, 50),
                         datetime.datetime(2011, 7, 31, 2, 20, 50),
                         datetime.datetime(2011, 12, 22, 2, 18, 4),
                         datetime.datetime(2011, 12, 23, 2, 18, 4),
                         datetime.datetime(2012, 2, 29, 16, 35, 42)]

      ### Validate results
      result_dates = []
      for each_config in config_file.processInfo:
          for each_date in each_config.schedule.dates():
              result_dates.append(each_date)
      
      self.failIf(reference_dates != result_dates,
                  "Failed to get all schedule dates: expected %s, got %s"
                  %(reference_dates,
                    result_dates))


   #----------------------------------------------------------------------------
   #
   # This test verifies that CSEPSchedule class is creating datetime objects
   # according to the schedule as specified in ForecastGroup
   # configuration file with two sets of evaluation tests.
   #
   def testTwoEvaluationSchedules(self):
      """ Confirm that CSEPSchedule class properly specifies datetime objects \
according to the configuration file with two sets of evaluaiton tests."""

      # Setup test name
      CSEPTestCase.setTestName(self, 
                               self.id())

      # Copy reference init file to the test directory
      init_file = "forecast_two_schedules.init.xml"

      shutil.copyfile(os.path.join(self.__referenceDataDir, 
                                   init_file),
                      os.path.join(CSEPTestCase.TestDirPath, 
                                   "forecast.init.xml"))    

      group = ForecastGroup(os.path.join(CSEPTestCase.TestDirPath))
      
      reference_dates = [datetime.datetime(2010, 8, 5),
                         datetime.datetime(2010, 9, 15),
                         datetime.datetime(2010, 10, 25),
                         datetime.datetime(2010, 11, 30),
                         datetime.datetime(2010, 12, 31),
                         datetime.datetime(2011, 1, 5),
                         datetime.datetime(2012, 2, 2),
                         datetime.datetime(2013, 3, 3),
                         datetime.datetime(2014, 4, 4)]

      ### Validate results
      result_dates = []
      tw_set = group.tests[1]
      for each_day in reference_dates:

        # If date is within set's schedule or last date of the testing period
        self.assertFalse(group.hasTests(each_day,
                                        test_set = tw_set),
                    "Unexpected %s date is found in T/W evaluation schedule"
                    %each_day)

      for each_month in xrange(1, 13):
          self.assertTrue(group.hasTests(datetime.datetime(2014, each_month, 1),
                                         test_set = tw_set),
                    "%s date should be found in T/W evaluation schedule"
                    %each_day)


# Invoke the module
if __name__ == '__main__':
   
   # Invoke all tests
   unittest.main()
        
# end of main
