"""
Module ForecastGroupTest
"""

__version__ = "$Revision: 4768 $"
__revision__ = "$Id: ForecastGroupTest.py 4768 2014-08-11 17:53:28Z liukis $"


import sys, os, unittest, shutil, datetime

import CSEP, GeographicalRegions, CSEPFile

from Environment import *
from CSEPTestCase import CSEPTestCase
from ForecastGroup import ForecastGroup
from RELMAftershockPostProcess import RELMAftershockPostProcess
from OneDayModelPostProcess import OneDayModelPostProcess
from OneMonthModelPostProcess import OneMonthModelPostProcess
from OneYearModelPostProcess import OneYearModelPostProcess
from BogusForecastModel1 import BogusForecastModel1
from BogusForecastModel2 import BogusForecastModel2
from HybridForecast import HybridForecast
from RELMNumberTest import RELMNumberTest
from cseprandom import CSEPRandom
from PostProcessFactory import PostProcessFactory
from ForecastHandlerFactory import ForecastHandlerFactory


#-------------------------------------------------------------------------------
#
# Validate that ForecastGroup class is working properly.
#
class ForecastGroupTest (CSEPTestCase):

   # Static data of the class
   
   # Unit tests use sub-directory of global reference data directory
   __referenceDataDir = os.path.join(CSEPTestCase.ReferenceDataDir, 
                                     'unitTest', 
                                     'forecastGroup')

   
   #----------------------------------------------------------------------------
   #
   # This test verifies that ForecastGroup class identifies existing 
   # forecast files properly.
   #
   def testExistingFiles(self):
      """ Confirm that ForecastGroup identifies existing \
forecast files properly."""

      # Setup test name
      CSEPTestCase.setTestName(self, 
                               self.id())

      # Use csep/data/forecasts
      forecast_dir = os.path.join(Environment.Variable[CENTER_CODE_ENV],
                                  "data", 
                                  "forecasts")
      
      group = ForecastGroup(forecast_dir, 
                            RELMAftershockPostProcess.Type)

      ### Validate results
      # 13 forecast files are expected in specified directory
      expected_num_files = 13
      error_message = "ForecastGroupTest: failed to get expected %s \
forecast files." %expected_num_files

      self.assertEqual(len(group.files()),
                       expected_num_files, 
                       error_message)
        

   #----------------------------------------------------------------------------
   #
   # This test verifies that ForecastGroup class properly parses 
   # initialization file.
   #
   def testInitializationFile(self):
      """ Confirm that ForecastGroup properly parses initialization file."""

      # Setup test name
      CSEPTestCase.setTestName(self, 
                               self.id())

      # Copy reference init file to the test directory
      init_file = "forecast.init.xml"
      shutil.copyfile(os.path.join(self.__referenceDataDir, 
                                   init_file),
                      os.path.join(CSEPTestCase.TestDirPath, 
                                   init_file))    

      # Use test directory as forecast group directory
      group = ForecastGroup(CSEPTestCase.TestDirPath)

      ### Validate results
      
      # Forecast directory
      forecast_dir = os.path.join(CSEPTestCase.TestDirPath, 
                                  "forecasts")
      self.assertEqual(group.dir(),
                       forecast_dir, 
                       "Expected '%s' forecast directory, got %s." 
                       %(forecast_dir,
                         group.dir()))

      # PostProcess object
      post_processing = group.postProcess()
      self.assertIsNotNone(post_processing,
                           "Expected valid PostProcessing object.")
      
      type = RELMAftershockPostProcess.Type
      self.assertEqual(post_processing.type(),
                       type, 
                       "Expected '%s' type of postProcessing object, got %s" 
                       %(type,
                         post_processing.type()))
    
      # Check for start date of post-processing 
      expected_date = datetime.datetime(2006, 6, 1)
      self.assertEqual(post_processing.start_date, 
                       expected_date, 
                       "Expected startDate='%s', got '%s'" 
                       %(expected_date, 
                         post_processing.start_date))
      
      # Check for expiration date of post-processing
      expire_date = datetime.datetime(2011, 6, 1)
      self.assertTrue(post_processing.expires(expire_date),
                      "Expected to expire on '%s'" %expire_date)      


      # Models 
      self.assertTrue(group.hasModels(), 
                      "Expected forecast models in initialization file")

      # Existing forecast files
      self.assertEqual(len(group.files()), 0,
                       "There should not be any forecast files, found %s"
                       %group.files())
      
      # Catalog directory
      catalog_dir = os.path.join(CSEPTestCase.TestDirPath, 
                                 "observations")
      self.assertEqual(group.catalogDir(),
                       catalog_dir,
                       "Expected '%s' catalog directory, got %s"
                       %(catalog_dir,
                         group.catalogDir()))
                  
      # Evaluation tests
      tests = "N L ROC MASS"
      
      file_tests = []
      for each_set in group.tests:
          for each_test in each_set:
              file_tests.append(each_test.type())
      file_tests_str = ' '.join(file_tests)
      
      self.assertEqual(file_tests_str,
                       tests, 
                       "Expected '%s' evaluation tests, got '%s'"
                       %(tests, 
                         file_tests_str))


   #----------------------------------------------------------------------------
   #
   # This test verifies that ForecastGroup class properly parses 
   # initialization file.
   #
   def testInitializationFileWithMultipleSchedules(self):
      """ Confirm that ForecastGroup properly parses initialization file with \
multiple schedules for evaluation tests."""

      # Setup test name
      CSEPTestCase.setTestName(self, 
                               self.id())

      # Copy reference init file to the test directory
      ref_init_file = "multiple_schedule_forecast.init.xml"
      init_file = "forecast.init.xml"
      shutil.copyfile(os.path.join(self.__referenceDataDir, 
                                   ref_init_file),
                      os.path.join(CSEPTestCase.TestDirPath, 
                                   init_file))    

      # Use test directory as forecast group directory
      group = ForecastGroup(CSEPTestCase.TestDirPath)

      ### Validate results
      
      # Forecast directory
      forecast_dir = os.path.join(CSEPTestCase.TestDirPath, 
                                  "forecasts")
      self.assertEqual(group.dir(),
                       forecast_dir, 
                       "Expected '%s' forecast directory, got %s."
                       %(forecast_dir,
                         group.dir()))

      # PostProcess object
      post_processing = group.postProcess()
      self.assertIsNotNone(post_processing,
                           "Expected valid PostProcessing object.")
      
      # Models 
      self.assertTrue(group.hasModels(), 
                      "Expected forecast models in initialization file")

      # Evaluation tests: two sets with it's own schedule
      all_tests = group.tests
      
      # 1st set of tests
      expected_tests = "N L MASS"
      file_tests = []
      tests = all_tests[0]
      
      for each_test in tests:
          file_tests.append(each_test.type())

      file_tests_str = ' '.join(file_tests)
      
      self.assertEqual(file_tests_str,
                       expected_tests,
                       "Expected '%s' evaluation tests, got '%s'"
                       %(expected_tests, 
                         file_tests_str))
      
      # Test date in whole group, and first test set schedule
      test_date = datetime.datetime(2013, 1, 1) 
      
      # Test date for the end of the testing period
      end_test_date = datetime.datetime(2010, 12, 31)
      
      self.assertTrue(group.hasTests(test_date),
                      "Expected test date within the group: %s" %test_date)

      self.assertTrue(group.hasTests(end_test_date),
                      "Expected testing period end date within the group: %s"
                      %end_test_date)
      
      self.assertTrue(group.hasTests(test_date,
                                     test_set = tests),
                      "Expected test date within first set of tests: %s"
                      %test_date)

      self.assertTrue(group.hasTests(end_test_date,
                                     test_set = tests),
                      "Expected testing period end date within first set \
of tests: %s" %end_test_date)

      # 2nd set of tests
      expected_tests = "T W"
      file_tests = []
      tests = all_tests[1]
      
      for each_test in tests:
          file_tests.append(each_test.type())

      file_tests_str = ' '.join(file_tests)
      
      self.assertEqual(file_tests_str,
                       expected_tests, 
                       "Expected '%s' evaluation tests, got '%s'"
                       %(expected_tests, 
                         file_tests_str))
      
      self.assertFalse(group.hasTests(test_date,
                                      test_set = tests),
                      "Second set of tests should not include \
test date: %s" %test_date)
      
      

      test_date = datetime.datetime(2013, 1, 15) 
      self.assertTrue(group.hasTests(test_date,
                                     test_set = tests),
                      "Expected test date within second test set: %s"
                      %test_date)
      
      self.assertFalse(group.hasTests(end_test_date,
                                      test_set = tests),
                       "Expected testing period end date within second \
set of tests: %s" %end_test_date)


   #----------------------------------------------------------------------------
   #
   # This test verifies that ForecastGroup class properly parses 
   # initialization file that contains HybridForecast configuration.
   #
   def testHybridModel(self):
      """ Confirm that ForecastGroup properly parses initialization file which \
contains HybridModel configuration."""

      # Setup test name
      CSEPTestCase.setTestName(self, 
                               self.id())

      # Copy reference init file to the test directory
      init_file = "forecast_with_hybrid_model.init.xml"
      shutil.copyfile(os.path.join(self.__referenceDataDir, 
                                   init_file),
                      os.path.join(CSEPTestCase.TestDirPath, 
                                   "forecast.init.xml"))    

      # Use test directory as forecast group directory
      group = ForecastGroup(CSEPTestCase.TestDirPath)

      ### Validate results
      
      # Models within the group should include HybridForecast
      self.assertEqual(len(group.models), 3,
                       "Expected 3 forecasts models within the group, \
got %s models" %group.models)
      
      # One of the models should be HybridModels
      self.assertTrue(any(isinstance(model, HybridForecast) for model in group.models),
                  "One of group's forecasts must by of HybridForecast type, \
got %s objects" %group.models)


   #----------------------------------------------------------------------------
   #
   # This test verifies that ForecastGroup class is properly initialized
   # by configuration file with missing evaluation tests.
   #
   def testInitializationFileNoTests(self):
      """ Confirm that ForecastGroup properly initialized by configuration file."""

      # Setup test name
      CSEPTestCase.setTestName(self, 
                               self.id())

      # Copy reference init file to the test directory
      init_file = "forecast_no_tests.init.xml"
      shutil.copyfile(os.path.join(self.__referenceDataDir,
                                   init_file),
                      os.path.join(CSEPTestCase.TestDirPath,
                                   'forecast.init.xml'))    

      # Use test directory as forecast group directory
      group = ForecastGroup(CSEPTestCase.TestDirPath)

      ### Validate results
      
      # Tests are not specified for the date
      self.assertFalse(group.hasTests(CSEPTestCase.Date),
                      "Group should not have any evaluation tests specified")


   #----------------------------------------------------------------------------
   #
   # Fix for Trac ticket #94
   # This test verifies that ForecastGroup class stages archived XML format of 
   # the forecast file to create Matlab format file used by evaluation tests.
   #
   def testXMLForecastStaging(self):
      """ Confirm that ForecastGroup stages archived XML format of the forecast \
file to generate final Matlab format file used by evaluation tests."""

      # Setup test name
      CSEPTestCase.setTestName(self, 
                               self.id())

      # Copy forecast group directory to the runtime test directory
      group_dir = "forecasts"
      shutil.copytree(os.path.join(self.__referenceDataDir, 
                                   group_dir),
                      os.path.join(CSEPTestCase.TestDirPath, 
                                   group_dir))

      # Copy input catalog to the test directory
      input_catalog = 'OneDayModelInputCatalog.mat'
      shutil.copyfile(os.path.join(self.__referenceDataDir, 
                                   input_catalog),
                      os.path.join(CSEPTestCase.TestDirPath, 
                                   input_catalog))    
      
      CSEP.Forecast.UseXMLMasterTemplate = True
      
      try:
         
         group = ForecastGroup(os.path.join(CSEPTestCase.TestDirPath,
                                            group_dir),
                               post_process = OneDayModelPostProcess.Type,
                               model_list = '%s %s' %(BogusForecastModel1.Type,
                                                      BogusForecastModel2.Type))
         
         group.create(datetime.datetime(2008, 6, 1),
                      CSEPTestCase.TestDirPath)
         
      finally:
         
         CSEP.Forecast.UseXMLMasterTemplate = False
         
      # Validate that XML format files in top-level forecast directory are links
      # to archived files
      xml_files = ['BogusForecastModel1_6_1_2008.xml', 
                   'BogusForecastModel2_6_1_2008.xml']
      
      for each_file in xml_files:
         self.assertTrue(os.path.islink(os.path.join(CSEPTestCase.TestDirPath, 
                                                     group_dir,
                                                     each_file)),
                         "Expected '%s' link for archived forecast file."
                         %each_file)


   #----------------------------------------------------------------------------
   #
   # This test verifies that ForecastGroup class stages archived DAT.GZIP format 
   # of the forecast file, and extracts HDF5 format file used by evaluation 
   # tests.
   #
   def testASCIItoGZIPForecastStaging(self):
      """ Confirm that ForecastGroup stages archived ASCII format of the \
forecast file to generate final dat.targz archive format file used by \
evaluation tests."""

      # Setup test name
      CSEPTestCase.setTestName(self, 
                               self.id())

      # Copy forecast group directory to the runtime test directory
      group_dir = "forecasts"
      shutil.copytree(os.path.join(self.__referenceDataDir, 
                                   'datforecasts'),
                      os.path.join(CSEPTestCase.TestDirPath, 
                                   group_dir))

      # Copy input catalog to the test directory
      input_catalog = 'OneDayModelInputCatalog.mat'
      shutil.copyfile(os.path.join(self.__referenceDataDir, 
                                   input_catalog),
                      os.path.join(CSEPTestCase.TestDirPath, 
                                   input_catalog))    
      
      #CSEP.Forecast.UseBinaryFormat = True
      CSEP.Forecast.UseCompression = True
      #CSEP.Forecast.UseXMLMasterTemplate = True
      
      try:
         
         group = ForecastGroup(os.path.join(CSEPTestCase.TestDirPath,
                                            group_dir),
                               post_process = OneDayModelPostProcess.Type,
                               model_list = '%s %s' %(BogusForecastModel1.Type,
                                                      BogusForecastModel2.Type))
         
         files = group.create(datetime.datetime(2014, 2, 15),
                              os.path.join(CSEPTestCase.TestDirPath,
                                           group_dir))
         
      finally:
         
         CSEP.Forecast.UseCompression = False
         
      # Validate that HDF5 format files in top-level forecast directory are 
      # generated
      result_files = ['BogusForecastModel1_2_15_2014.dat.targz', 
                      'BogusForecastModel2_2_15_2014.dat.targz']
      
      for each_file in result_files:
         self.assertTrue(os.path.exists(os.path.join(CSEPTestCase.TestDirPath, 
                                                     group_dir,
                                                     each_file)),
                         "Expected '%s' file to exist." %each_file)


   #----------------------------------------------------------------------------
   #
   # This test verifies that ForecastGroup class creates archived DAT.GZIP 
   # format of the forecast file, and extracts HDF5 format file used by 
   # evaluation tests.
   #
   def testCreateASCIIGZIPForecast(self):
      """ Confirm that ForecastGroup creates and archives ASCII format of the \
forecast file to generate final dat.targz archive format file used by \
evaluation tests."""

      # Setup test name
      CSEPTestCase.setTestName(self, 
                               self.id())

      # Copy forecast group directory to the runtime test directory
      group_dir = "forecasts"

      # Copy input catalog to the test directory
      input_catalog = 'OneDayModelInputCatalog.mat'
      shutil.copyfile(os.path.join(self.__referenceDataDir, 
                                   input_catalog),
                      os.path.join(CSEPTestCase.TestDirPath, 
                                   input_catalog))    
      
      #CSEP.Forecast.UseBinaryFormat = True
      CSEP.Forecast.UseCompression = True
      BogusForecastModel1.RequireInputCatalog = False
      BogusForecastModel2.RequireInputCatalog = False
            
      try:
          
          # Set forecast data for the model
          forecast_file = os.path.join(self.__referenceDataDir, 
                                       'datforecasts/archive/2014_2/BogusForecastModel1_2_15_2014.dat')
         
          np_forecast = ForecastHandlerFactory().CurrentHandler.load(forecast_file)
        
          tests = RELMNumberTest.Type
          group = ForecastGroup(os.path.join(CSEPTestCase.TestDirPath,
                                             group_dir),
                                OneDayModelPostProcess.Type,
                                tests,
                                model_list = '%s %s' %(BogusForecastModel1.Type,
                                                       BogusForecastModel2.Type))
         
          for each_model in group.models:   
              each_model.data(np_forecast)
         
          files = group.create(datetime.datetime(2014, 2, 15),
                               os.path.join(CSEPTestCase.TestDirPath,
                                            group_dir))

          # Invoke N-test to verify that evaluation can be invoked on forecasts
          # archives
          catalog_file = PostProcessFactory().object(OneDayModelPostProcess.Type).files.catalog
          reference_catalog = "%s.%s" %(OneDayModelPostProcess.Type, 
                                        catalog_file)
          shutil.copyfile(os.path.join(CSEPTestCase.ReferenceDataDir, 
                                       reference_catalog),
                          os.path.join(CSEPTestCase.TestDirPath, 
                                       catalog_file))
      
          for each_test in group.tests[0]:
              # Use the same directory for catalog data and test results
              each_test.run(datetime.datetime(2014, 2, 15), 
                            CSEPTestCase.TestDirPath,
                            CSEPTestCase.TestDirPath)
         
      finally:
         
          CSEP.Forecast.UseCompression = False
          BogusForecastModel1.RequireInputCatalog = True
          BogusForecastModel2.RequireInputCatalog = True
         
         
      # Validate that HDF5 format files in top-level forecast directory are 
      # generated
      result_files = ['BogusForecastModel1_2_15_2014.dat.targz', 
                      'BogusForecastModel2_2_15_2014.dat.targz']
      
      for each_file in result_files:
         self.assertTrue(os.path.exists(os.path.join(CSEPTestCase.TestDirPath, 
                                                     group_dir,
                                                     each_file)),
                         "Expected '%s' file to exist." %each_file)


   #----------------------------------------------------------------------------
   #
   # This test verifies that ForecastGroup class stages archived HDF5.GZIP format 
   # of the forecast file, and extracts HDF5 format file used by evaluation tests.
   #
   def testASCIIGZIPForecastStaging(self):
      """ Confirm that ForecastGroup stages archived dat.targz format of the \
forecast file, and evaluation test can be invoked for archived version of the \
forecast."""

      # Setup test name
      CSEPTestCase.setTestName(self, 
                               self.id())

      # Copy forecast group directory to the runtime test directory
      group_dir = "forecasts"
      shutil.copytree(os.path.join(self.__referenceDataDir, 
                                   'datgzforecasts'),
                      os.path.join(CSEPTestCase.TestDirPath, 
                                   group_dir))

      # Copy input catalog to the test directory
      input_catalog = 'OneDayModelInputCatalog.mat'
      shutil.copyfile(os.path.join(self.__referenceDataDir, 
                                   input_catalog),
                      os.path.join(CSEPTestCase.TestDirPath, 
                                   input_catalog))    
      
      CSEP.Forecast.UseCompression = True
      CSEPRandom.ReadSeedFromFile = False
      
      try:
         
         tests = RELMNumberTest.Type
         group = ForecastGroup(os.path.join(CSEPTestCase.TestDirPath,
                                            group_dir),
                               OneDayModelPostProcess.Type,
                               tests,
                               model_list = '%s %s' %(BogusForecastModel1.Type,
                                                      BogusForecastModel2.Type))
         
         group.create(datetime.datetime(2014, 2, 15),
                      os.path.join(CSEPTestCase.TestDirPath,
                                   group_dir))
         
         # Invoke N-test to verify that evaluation can be invoked on forecasts
         # archives
         catalog_file = PostProcessFactory().object(OneDayModelPostProcess.Type).files.catalog
         reference_catalog = "%s.%s" %(OneDayModelPostProcess.Type, 
                                       catalog_file)
         shutil.copyfile(os.path.join(CSEPTestCase.ReferenceDataDir, 
                                      reference_catalog),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     catalog_file))
      
         for each_test in group.tests[0]:
           # Use the same directory for catalog data and test results
           each_test.run(datetime.datetime(2014, 2, 15), 
                         CSEPTestCase.TestDirPath,
                         CSEPTestCase.TestDirPath)
         
      finally:
         
         CSEP.Forecast.UseCompression = False
         CSEPRandom.ReadSeedFromFile = True
         
      # Validate that HDF5 format files in top-level forecast directory are 
      # generated
      result_files = ['BogusForecastModel1_2_15_2014.dat.targz', 
                      'BogusForecastModel2_2_15_2014.dat.targz']
      
      for each_file in result_files:
          self.assertTrue(os.path.islink(os.path.join(CSEPTestCase.TestDirPath, 
                                                     group_dir,
                                                     each_file)),
                         "Expected '%s' link for archived forecast file." %each_file)
      

   #----------------------------------------------------------------------------
   #
   # This test verifies that ForecastGroup class creates GZIP archive of HDF5
   # format forecast file based on original XML format forecast, and extracts
   # HDF5 format file for the evaluation tests.
   #
   def testXMLtoHDF5GZIPForecastStaging(self):
      """ Confirm that ForecastGroup converts existing XML format of forecast
          to archive h5.targz format of the forecast file, and extracts HDF5
          format forecast for evaluation test."""

      # Introduce forecast class for testing of high-resolution partial Global
      # region
      class BogusForecastModel1XML (BogusForecastModel1):
          
          Type = 'BogusXML1'
          
          def __init__ (self, dir_path):
              BogusForecastModel1.__init__(self, dir_path)
              
          def filename (self):
              return CSEPFile.Name.xml(BogusForecastModel1.filename(self))
          
          def type (self):
              return BogusForecastModel1XML.Type
          
      from ForecastFactory import ForecastFactory
      
      # Update ForecastFactory with new model
      ForecastFactory(os.path.join(CSEPTestCase.ReferenceDataDir, 
                                   CSEPTestCase.ForecastFactoryConfigFile),
                      {BogusForecastModel1XML.Type: BogusForecastModel1XML})
      
      # Setup test name
      CSEPTestCase.setTestName(self, 
                               self.id())

      # Copy forecast group directory to the runtime test directory
      group_dir = "forecasts"
      shutil.copytree(os.path.join(self.__referenceDataDir, 
                                   'xmlforecasts'),
                      os.path.join(CSEPTestCase.TestDirPath, 
                                   group_dir))

      # Copy input catalog to the test directory
      input_catalog = 'OneDayModelInputCatalog.mat'
      shutil.copyfile(os.path.join(self.__referenceDataDir, 
                                   input_catalog),
                      os.path.join(CSEPTestCase.TestDirPath, 
                                   input_catalog))    
      
      CSEP.Forecast.UseCompression = True
      CSEP.Forecast.UseBinaryFormat = True
      CSEP.Forecast.UseXMLMasterTemplate = True
      CSEP.Forecast.ApplyXMLMasterValidation = True
      
      GeographicalRegions.Region().set(GeographicalRegions.Global)

      # Set post-processing specific data 
      min_magn = OneDayModelPostProcess.MinMagnitude
      max_depth = OneDayModelPostProcess.MaxDepth
      template = OneDayModelPostProcess.ForecastTemplate
    
      OneDayModelPostProcess.MinMagnitude = 5.95
      OneDayModelPostProcess.MaxDepth = 30.0
      OneDayModelPostProcess.ForecastTemplate = os.path.join(self.__referenceDataDir,
                                                             'ShortGlobalTemplateM6.xml')
      
      try:
         tests = RELMNumberTest.Type
         group = ForecastGroup(os.path.join(CSEPTestCase.TestDirPath,
                                            group_dir),
                               OneDayModelPostProcess.Type,
                               tests,
                               model_list = BogusForecastModel1XML.Type)
         
         group.create(datetime.datetime(2013, 11, 18),
                      os.path.join(CSEPTestCase.TestDirPath,
                                   group_dir))
         
         # Invoke N-test to verify that evaluation can be invoked on forecasts
         # archives
         catalog_file = PostProcessFactory().object(OneDayModelPostProcess.Type).files.catalog
         reference_catalog = "%s.%s" %(OneDayModelPostProcess.Type, 
                                       catalog_file)
         shutil.copyfile(os.path.join(CSEPTestCase.ReferenceDataDir, 
                                      reference_catalog),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     catalog_file))
      
         for each_test in group.tests[0]:
           # Use the same directory for catalog data and test results
           each_test.run(datetime.datetime(2013, 11, 18),
                         CSEPTestCase.TestDirPath,
                         CSEPTestCase.TestDirPath)
         
      finally:
         
         CSEP.Forecast.UseCompression = False
         CSEP.Forecast.UseBinaryFormat = False
         CSEP.Forecast.UseXMLMasterTemplate = False
         CSEP.Forecast.ApplyXMLMasterValidation = False
         
         GeographicalRegions.Region().set(GeographicalRegions.California)

         OneDayModelPostProcess.MinMagnitude = min_magn
         OneDayModelPostProcess.MaxDepth = max_depth
         OneDayModelPostProcess.ForecastTemplate = template
         
      # Validate that HDF5 format files in top-level forecast directory are 
      # generated
      result_files = ['BogusXML1_11_18_2013-fromXML.h5.targz'] 
      
      for each_file in result_files:
          self.assertTrue(os.path.exists(os.path.join(CSEPTestCase.TestDirPath, 
                                                      group_dir,
                                                      each_file)),
                         "Expected '%s' archive of HDF5 forecast file."
                         %each_file)


   #----------------------------------------------------------------------------
   #
   # This test verifies that ForecastGroup class creates GZIP archive of HDF5
   # format forecast file based on validated by the CSEP XML format forecast, 
   # and extracts HDF5 format file for the evaluation tests.
   #
   def testValidateXMLtoHDF5GZIPForecastStaging(self):
      """ Confirm that ForecastGroup validates existing XML format of forecast,
          stores into archive h5.targz format of the forecast file, and extracts 
          HDF5 format forecast for the evaluation test."""

      # Introduce forecast class for testing of high-resolution partial Global
      # region
      class BogusForecastModel1XML (BogusForecastModel1):
          
          Type = 'BogusXML1'
          
          def __init__ (self, dir_path):
              BogusForecastModel1.__init__(self, dir_path)
              
          def filename (self):
              return CSEPFile.Name.xml(BogusForecastModel1.filename(self))
          
          def type (self):
              return BogusForecastModel1XML.Type
          
      from ForecastFactory import ForecastFactory
      
      # Update ForecastFactory with new model
      ForecastFactory(os.path.join(CSEPTestCase.ReferenceDataDir, 
                                   CSEPTestCase.ForecastFactoryConfigFile),
                      {BogusForecastModel1XML.Type: BogusForecastModel1XML})

      
      # Setup test name
      CSEPTestCase.setTestName(self, 
                               self.id())

      # Copy forecast group directory to the runtime test directory
      group_dir = "forecasts"
      shutil.copytree(os.path.join(self.__referenceDataDir, 
                                   'validatexmlforecasts'),
                      os.path.join(CSEPTestCase.TestDirPath, 
                                   group_dir))

      # Copy input catalog to the test directory
      input_catalog = 'OneDayModelInputCatalog.mat'
      shutil.copyfile(os.path.join(self.__referenceDataDir, 
                                   input_catalog),
                      os.path.join(CSEPTestCase.TestDirPath, 
                                   input_catalog))    
      
      CSEP.Forecast.UseCompression = True
      CSEP.Forecast.UseBinaryFormat = True
      CSEP.Forecast.UseXMLMasterTemplate = True
      CSEP.Forecast.ApplyXMLMasterValidation = True
      
      GeographicalRegions.Region().set(GeographicalRegions.Global)

      # Set post-processing specific data 
      min_magn = OneDayModelPostProcess.MinMagnitude
      max_depth = OneDayModelPostProcess.MaxDepth
      template = OneDayModelPostProcess.ForecastTemplate
    
      OneDayModelPostProcess.MinMagnitude = 5.95
      OneDayModelPostProcess.MaxDepth = 30.0
      OneDayModelPostProcess.ForecastTemplate = os.path.join(self.__referenceDataDir,
                                                             'ShortGlobalTemplateM6.xml')
      
      try:
         tests = RELMNumberTest.Type
         group = ForecastGroup(os.path.join(CSEPTestCase.TestDirPath,
                                            group_dir),
                               OneDayModelPostProcess.Type,
                               tests,
                               model_list = BogusForecastModel1XML.Type)
         
         group.create(datetime.datetime(2013, 11, 18),
                      os.path.join(CSEPTestCase.TestDirPath,
                                   group_dir))
         
         # Invoke N-test to verify that evaluation can be invoked on forecasts
         # archives
         catalog_file = PostProcessFactory().object(OneDayModelPostProcess.Type).files.catalog
         reference_catalog = "%s.%s" %(OneDayModelPostProcess.Type, 
                                       catalog_file)
         shutil.copyfile(os.path.join(CSEPTestCase.ReferenceDataDir, 
                                      reference_catalog),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     catalog_file))
      
         for each_test in group.tests[0]:
           # Use the same directory for catalog data and test results
           each_test.run(datetime.datetime(2013, 11, 18),
                         CSEPTestCase.TestDirPath,
                         CSEPTestCase.TestDirPath)
         
      finally:
         
         CSEP.Forecast.UseCompression = False
         CSEP.Forecast.UseBinaryFormat = False
         CSEP.Forecast.UseXMLMasterTemplate = False
         CSEP.Forecast.ApplyXMLMasterValidation = False
         
         GeographicalRegions.Region().set(GeographicalRegions.California)

         OneDayModelPostProcess.MinMagnitude = min_magn
         OneDayModelPostProcess.MaxDepth = max_depth
         OneDayModelPostProcess.ForecastTemplate = template
         
      # Validate that HDF5 format files in top-level forecast directory are 
      # generated
      result_files = ['BogusXML1_11_18_2013-fromXML.h5.targz'] 
      
      for each_file in result_files:
          self.assertTrue(os.path.exists(os.path.join(CSEPTestCase.TestDirPath, 
                                                      group_dir,
                                                      each_file)),
                         "Expected '%s' archive of HDF5 forecast file."
                         %each_file)


   def testOneDayForecastWithNonZeroStartTime(self):
      """ Confirm that start time other than 00:00:00 can be provided for \
OneDayForecast class, and verify that forecast with proper name is generated."""

      # Setup test name
      CSEPTestCase.setTestName(self, 
                               self.id())

      # Copy reference init file to the test directory
      init_file = "forecastTimeStamp.init.xml"
      shutil.copyfile(os.path.join(self.__referenceDataDir, 
                                   init_file),
                      os.path.join(CSEPTestCase.TestDirPath, 
                                   "forecast.init.xml"))    

      # Copy input catalog to the test directory
      input_catalog = 'OneDayModelInputCatalog.mat'
      shutil.copyfile(os.path.join(self.__referenceDataDir, 
                                   input_catalog),
                      os.path.join(CSEPTestCase.TestDirPath, 
                                   "OneDayModelInputCatalog.dat"))    
      
      group = ForecastGroup(CSEPTestCase.TestDirPath)
      
      # Set forecast data for the model
      forecast_file = os.path.join(self.__referenceDataDir, 
                                   'datforecasts/archive/2014_2/BogusForecastModel1_2_15_2014.dat')
     
      np_forecast = ForecastHandlerFactory().CurrentHandler.load(forecast_file)
    
      for each_model in group.models:   
          each_model.data(np_forecast)
      
      files = group.create(datetime.datetime(2014, 7, 15, 16, 34, 45),
                           CSEPTestCase.TestDirPath)

      # Invoke N-test to verify that evaluation can be invoked on forecasts
      # archives
      catalog_file = PostProcessFactory().object(OneDayModelPostProcess.Type).files.catalog
      reference_catalog = "%s.%s" %(OneDayModelPostProcess.Type, 
                                    catalog_file)
      shutil.copyfile(os.path.join(CSEPTestCase.ReferenceDataDir, 
                                   reference_catalog),
                     os.path.join(CSEPTestCase.TestDirPath, 
                                  catalog_file))
  
      for each_test in group.tests[0]:
          # Use the same directory for catalog data and test results
          each_test.run(datetime.datetime(2014, 7, 15, 16, 34, 45),
                        CSEPTestCase.TestDirPath,
                        CSEPTestCase.TestDirPath)


   def testOneMonthForecastWithNonZeroStartTime(self):
      """ Confirm that start time other than 00:00:00 can be provided for \
OneMonthForecast class, and verify that forecast with proper name is generated."""

      # Setup test name
      CSEPTestCase.setTestName(self, 
                               self.id())

      # Copy reference init file to the test directory
      init_file = "forecastTimeStampOneMonth.init.xml"
      shutil.copyfile(os.path.join(self.__referenceDataDir, 
                                   init_file),
                      os.path.join(CSEPTestCase.TestDirPath, 
                                   "forecast.init.xml"))    

      # Copy input catalog to the test directory
      input_catalog = 'OneDayModelInputCatalog.mat'
      shutil.copyfile(os.path.join(self.__referenceDataDir, 
                                   input_catalog),
                      os.path.join(CSEPTestCase.TestDirPath, 
                                   "OneDayModelInputCatalog.dat"))    
      
      group = ForecastGroup(CSEPTestCase.TestDirPath)
      # Set forecast data for the model
      forecast_file = os.path.join(self.__referenceDataDir, 
                                   'datforecasts/archive/2014_2/BogusForecastModel1_2_15_2014.dat')
     
      np_forecast = ForecastHandlerFactory().CurrentHandler.load(forecast_file)
    
      for each_model in group.models:   
          each_model.data(np_forecast)
      
      # Create forecasts
      files = group.create(datetime.datetime(2014, 7, 1, 16, 34, 45),
                           CSEPTestCase.TestDirPath)

      # Remove lock directory manually since deleting forecast group object
      # just decrements ref. counter for it
      shutil.rmtree(os.path.join(group.dir(),
                                 'lock'))
      # Reset forecast
      group = ForecastGroup(CSEPTestCase.TestDirPath)

      # Invoke N-test to verify that evaluation can be invoked on forecasts
      # archives
      catalog_file = PostProcessFactory().object(OneMonthModelPostProcess.Type,
                                                 [datetime.datetime(2014, 7, 1, 16, 34, 45),
                                                  datetime.datetime(2014, 8, 1, 16, 34, 45)]).files.catalog
      reference_catalog = "%s.%s" %(OneDayModelPostProcess.Type, 
                                    catalog_file)
      shutil.copyfile(os.path.join(CSEPTestCase.ReferenceDataDir, 
                                   reference_catalog),
                     os.path.join(CSEPTestCase.TestDirPath, 
                                  catalog_file))
  
      # ATTN: To simulate real-time CSEP processing, would need to re-generate 
      # forecast group to force reading of correct start and end dates for 
      # the current testing period as set by model generation 
      # OR set already correct dates in the forecast.init.xml configuration file
      for each_test in group.tests[0]:
          # Use the same directory for catalog data and test results
          each_test.run(datetime.datetime(2014, 7, 15, 16, 34, 45),
                        CSEPTestCase.TestDirPath,
                        CSEPTestCase.TestDirPath)

          self.failIf(not os.path.exists(os.path.join(CSEPTestCase.TestDirPath,
                                                      "rTest_N-Test_BogusOneMonthModel1_7_1_2014.16_34_45.xml")),
                      "Failed to generate expected test result for BogusOneMonthModel1_7_1_2014.16_34_45 forecast")
          
          # Delete scale factor file since it should be re-generated for 
          # another test date
          os.remove(os.path.join(CSEPTestCase.TestDirPath,
                                 "ForecastScaleFactor.dat"))
          os.remove(os.path.join(CSEPTestCase.TestDirPath,
                                 "rTest_N-Test_BogusOneMonthModel1_7_1_2014.16_34_45.xml"))
          

      shutil.rmtree(os.path.join(group.dir(),
                                 'lock'))

      # Reset forecast
      group = ForecastGroup(CSEPTestCase.TestDirPath)
      for each_test in group.tests[0]:
          each_test.run(datetime.datetime(2014, 7, 31, 16, 34, 45),
                        CSEPTestCase.TestDirPath,
                        CSEPTestCase.TestDirPath)

          self.failIf(not os.path.exists(os.path.join(CSEPTestCase.TestDirPath,
                                                      "rTest_N-Test_BogusOneMonthModel1_7_1_2014.16_34_45.xml")),
                      "Failed to generate expected test result for BogusOneMonthModel1_7_1_2014.16_34_45 forecast")


   def testOneYearForecastWithNonZeroStartTime(self):
      """ Confirm that start time other than 00:00:00 can be provided for \
OneYearForecast class, and verify that forecast with proper name is generated."""

      # Setup test name
      CSEPTestCase.setTestName(self, 
                               self.id())

      # Copy reference init file to the test directory
      init_file = "forecastTimeStampOneYear.init.xml"
      shutil.copyfile(os.path.join(self.__referenceDataDir, 
                                   init_file),
                      os.path.join(CSEPTestCase.TestDirPath, 
                                   "forecast.init.xml"))    

      # Copy input catalog to the test directory
      input_catalog = 'OneDayModelInputCatalog.mat'
      shutil.copyfile(os.path.join(self.__referenceDataDir, 
                                   input_catalog),
                      os.path.join(CSEPTestCase.TestDirPath, 
                                   "OneDayModelInputCatalog.dat"))    
      
      group = ForecastGroup(CSEPTestCase.TestDirPath)
      # Set forecast data for the model
      forecast_file = os.path.join(self.__referenceDataDir, 
                                   'datforecasts/archive/2014_2/BogusForecastModel1_2_15_2014.dat')
     
      np_forecast = ForecastHandlerFactory().CurrentHandler.load(forecast_file)
    
      for each_model in group.models:   
          each_model.data(np_forecast)
      
      # Create forecasts
      files = group.create(datetime.datetime(2013, 9, 1, 16, 34, 45),
                           CSEPTestCase.TestDirPath)

      # Remove lock directory manually since deleting forecast group object
      # just decrements ref. counter for it
      shutil.rmtree(os.path.join(group.dir(),
                                 'lock'))
      # Reset forecast
      group = ForecastGroup(CSEPTestCase.TestDirPath)

      # Invoke N-test to verify that evaluation can be invoked on forecasts
      # archives
      catalog_file = PostProcessFactory().object(OneYearModelPostProcess.Type,
                                                 [datetime.datetime(2013, 9, 1, 16, 34, 45),
                                                  datetime.datetime(2014, 9, 1, 16, 34, 45)]).files.catalog
      reference_catalog = "%s.%s" %(OneDayModelPostProcess.Type, 
                                    catalog_file)
      shutil.copyfile(os.path.join(CSEPTestCase.ReferenceDataDir, 
                                   reference_catalog),
                     os.path.join(CSEPTestCase.TestDirPath, 
                                  catalog_file))
  
      # ATTN: To simulate real-time CSEP processing, would need to re-generate 
      # forecast group to force reading of correct start and end dates for 
      # the current testing period as set by model generation 
      # OR set already correct dates in the forecast.init.xml configuration file
      for each_test in group.tests[0]:
          # Use the same directory for catalog data and test results
          each_test.run(datetime.datetime(2014, 7, 15, 16, 34, 45),
                        CSEPTestCase.TestDirPath,
                        CSEPTestCase.TestDirPath)

          self.failIf(not os.path.exists(os.path.join(CSEPTestCase.TestDirPath,
                                                      "rTest_N-Test_BogusOneYearModel1_9_1_2013.16_34_45.xml")),
                      "Failed to generate expected test result for BogusOneMonthModel1_9_1_2013.16_34_45 forecast")
          
          # Delete scale factor file since it should be re-generated for 
          # another test date
          os.remove(os.path.join(CSEPTestCase.TestDirPath,
                                 "ForecastScaleFactor.dat"))
          os.remove(os.path.join(CSEPTestCase.TestDirPath,
                                 "rTest_N-Test_BogusOneYearModel1_9_1_2013.16_34_45.xml"))

      shutil.rmtree(os.path.join(group.dir(),
                                 'lock'))

      # Reset forecast
      group = ForecastGroup(CSEPTestCase.TestDirPath)
      for each_test in group.tests[0]:
          each_test.run(datetime.datetime(2014, 8, 31, 16, 34, 45),
                        CSEPTestCase.TestDirPath,
                        CSEPTestCase.TestDirPath)

          self.failIf(not os.path.exists(os.path.join(CSEPTestCase.TestDirPath,
                                                      "rTest_N-Test_BogusOneYearModel1_9_1_2013.16_34_45.xml")),
                      "Failed to generate expected test result for BogusOneMonthModel1_9_1_2013.16_34_45 forecast")


# Invoke the module
if __name__ == '__main__':
   
   # Invoke all tests
   unittest.main()

# end of main
