"""
Module to assist with CSEP one-day and thirty-minutes ETAS_DR model evaluation results for USGS report

1. Utility to scale down one-day forecast to 30-minute forecast and to save it as 30-min forecasts
for the required testing period.
"""

__version__ = "$Revision$"
__revision__ = "$Id$"

import os, datetime, glob, sys, time, re
import numpy as np
import CSEPFile, CSEP, CSEPLogging, CSEPGeneric, Environment, \
       CSEPXML
from CSEPSchedule import CSEPSchedule
from CSEPPropertyFile import CSEPPropertyFile
from RELMCatalog import RELMCatalog
from RateForecastHandler import RateForecastHandler
from Forecast import Forecast


#--------------------------------------------------------------------------------
#
# ForecastConversion
#
# This module is designed to scale-down one-day forecast to 30-minute forecast,
# and to save it as target forecasts within required testing time interval. 
#
class ForecastConversion (object):

    # Static data members

    # Logger for the class
    __logger = None
    
    
    #===============================================================================
    # Constructor for ObservationCatalog class
    #
    # Inputs:
    #         dirpath - Directory path to model files.
    #
    #===============================================================================
    def __init__ (self):
   
        if ForecastConversion.__logger is None:
           ForecastConversion.__logger = CSEPLogging.CSEPLogging.getLogger(ForecastConversion.__name__)
        

    #===========================================================================
    # create
    # 
    # Inputs: 
    #         model_key - Keyword identifying forecast model
    #         masking_bit - Masking bit common to all forecasts participating in
    #                       evaluation test
    #         static_forecast - Path to the static model. Default is None meaning
    #                           fore
    #===========================================================================
    def create (self,
                forecast_dir, 
                thirty_min_dir="thirty_mins_forecasts"):
        """Create 30-min forecasts."""

        forecasts = os.listdir(forecast_dir)
        
        if not os.path.exists(thirty_min_dir):
            os.makedirs(thirty_min_dir)
        
        scale_factor = 48.0 # 48 30-min forecasts within one-day
        test_day = datetime.date(2012, 8, 26)
        
        # Step through one-day forecasts and save scaled down version to thirty-minutes ones:
        # one of <ETAS_DROneDayPPEMd3_8_26_2012-fromXML.dat, ETAS_DROneDayMd3_8_26_2012-fromXML.dat>
        for each_forecast in forecasts:
            forecast = RateForecastHandler().load(os.path.join(forecast_dir,
                                                               each_forecast))
            
            print "Converting", forecast
            forecast[:, CSEPGeneric.Forecast.Format.Rate] /= scale_factor
            
            # Search for date and optional time formatted strings in the name
            name_info = re.search(r"([\w\d_]+)(?=_\d+_\d+_\d+-fromXML.dat)",
                                  each_forecast)
            
            
            name = name_info.group()
            print "name:", name
            
            # Save as 30-min forecasts
            for each_hour in range(19, 24):
                for each_minute in [0, 30]:
                     
                     test_time = datetime.time(each_hour,
                                               each_minute,
                                               0)
                     
                     
                     forecast_file = '%s_%s_%s_%s.%s.dat' %(name,
                                                            test_day.month,
                                                            test_day.day,
                                                            test_day.year,
                                                            test_time.strftime('%H_%M_%S'))
                     
                     forecast_file = os.path.join(thirty_min_dir, 
                                                  forecast_file)
                     print "Creating ", forecast_file  
                     np.savetxt(forecast_file, 
                                forecast)       


# Invoke the module
if __name__ == '__main__':


    import optparse
    
    
    command_options = optparse.OptionParser()

    command_options.add_option('--convertPlots',
                               action="store_true", 
                               dest='convert_plots',
                               default=False,
                               help="Convert PS plots as generated by the test to PNG")
    
    command_options.add_option('--from',
                               dest='one_day_dir',
                               default=None,
                               help="Full path to directory with one-day forecasts")

    command_options.add_option('--to',
                               dest='thirty_min_dir',
                               default=None,
                               help="Full path to directory with thirty-min forecasts")

    (values, args) = command_options.parse_args()


    observations = ForecastConversion()
    observations.create(values.one_day_dir,
                        values.thirty_min_dir)
    