"""
Module EEPASModel
"""

__version__ = "$Revision$"
__revision__ = "$Id$"


import os, datetime

import Environment, CSEPFile, CSEPLogging, CSEP
from Forecast import Forecast
from OneYearForecast import OneYearForecast
from ReproducibilityFiles import ReproducibilityFiles
from CatalogDataSource import CatalogDataSource
from CSEPStorage import CSEPStorage


#--------------------------------------------------------------------------------
#
# EEPAS-0F forecast model.
#
# This class is designed to invoke three-months PPE and EEPAS forecast model. 
# It prepares input catalog data, formats control file with model parameters,
# and invokes the model. It places forecast file under user specified directory.
#
class EEPASOneYearModel (OneYearForecast):

    # Static data of the class
    
    # Keyword identifying the type of the forecast
    Type = "EEPAS" + OneYearForecast.Type

    TimeVariantPostfix = 'TV'
    
    # 30-day lag in input catalog - specific to the EEPAS models
    __catalogLagDays = 0
    
    # Top-level path for the model executable and configuration files
    __NZRootPath = os.environ["NZHOME"]
    __execPath = os.path.join(__NZRootPath,
                              'EEPAS')
    __executableFile = "eepasof22ng77"
    __catalogCoversionExecutable = "reformatscec"
    
    CombineForecastsExecutable = os.path.join(__execPath, 
                                              'combineforecasts')
    
    
    #----------------------------------------------------------------------------
    #
    # Initialization.
    #
    # Input: 
    #        dir_path - Directory to store forecast file to.
    #        args - Optional arguments for the model. EEPAS models don't 
    #               require any configurable inputs by the CSEP system.
    #        param_file - Parameter file for a specific model configuration.
    #        use_weights - Flag if model shouuld use weights. Default is 0 (False).
    # 
    def __init__ (self, dir_path, args, param_file, use_weights = 0):
        """ Initialization for EEPASModel class"""
        
        OneYearForecast.__init__(self, dir_path)
        
        # Make sure that user is not providing any inputs to the model
        if args is not None:
            error_msg = "%s forecast model does not support any input arguments. \
'%s' input string is provided." %(EEPASOneYearModel.Type, args)

            CSEPLogging.CSEPLogging.getLogger(EEPASOneYearModel.__name__).error(error_msg)
            raise RuntimeError, error_msg

        # Flag for model weights 
        self.__weights = use_weights
        
        # Filename for input catalog in a model expected format
        self.__modelInputCatalog = CSEPFile.Name.extension(self.inputCatalogFilename(),
                                                           'ref')
        
        # Parameter file for configuration
        self.__paramFile = param_file


    #----------------------------------------------------------------------------
    #
    # Return keyword identifying the model.
    #
    # Input: None.
    #
    # Output:
    #           String identifying the type
    #
    def type (self):
        """ Returns keyword identifying the forecast model type."""
        
        return self.Type


    #----------------------------------------------------------------------------
    #
    # Return filename for the input catalog data. It overwrites parent's
    # method because EEPAS is using ASCII format catalog file.
    #
    # Input: None.
    #
    # Output:
    #        String identifying the filename.
    #
    def inputCatalogFilename (self):
        """ Returns filename of input catalog data."""
        
        return CSEPFile.Name.ascii(Forecast.inputCatalogFilename(self))

        
    #----------------------------------------------------------------------------
    #
    # Write input parameter file for the model.
    #
    # Input: None
    #        
    def writeParameterFile (self):
        """ Format input parameter file for the model.
            Created file will be used by Fortran executable that invokes the
            model."""

        fhandle = Forecast.writeParameterFile(self)
        
        # Write generic part of input parameters at the top of the file:
        fhandle.write("FILES\n")
        
        fhandle.write("licence '%s'\n" \
                      %os.path.join(EEPASOneYearModel.__execPath, 'licencekey.dat'))
        
        fhandle.write("grdm '%s'\n" \
                      %os.path.join(EEPASOneYearModel.__execPath, 'relmmagsnz.dat'))
        
        fhandle.write("grdl '%s'\n" \
                      %os.path.join(EEPASOneYearModel.__execPath, 'relmblocksnz.dat'))

        fhandle.write("catalog '%s'\n" \
                      %os.path.join(self.catalogDir, self.__modelInputCatalog))
        
        fhandle.write("polygon '%s'\n" %os.path.join(EEPASOneYearModel.__execPath,
                                                     self.polygonFile()))
        fhandle.write("end files\n")
        
        fhandle.write("POLY/GON  reads in target polygon from file\n")
         
        fhandle.write("CATA/LOGUE selects input data from catalogue file\n")
        
        fhandle.write("all /latmin,latmax,longmin,longmx\n")

        fhandle.write("1951, 01, 01, 0, 0, 0.0\n")
        
        catalog_end_date = self.start_date - \
                           datetime.timedelta(days=EEPASOneYearModel.__catalogLagDays)
        fhandle.write("%s, %s, %s, %s, %s, %s,\n"
                       %(catalog_end_date.year,
                         catalog_end_date.month,
                         catalog_end_date.day,
                         catalog_end_date.hour,
                         catalog_end_date.minute,
                         catalog_end_date.second))
        fhandle.write("2.95 ,0.0, 40.0, 5.0 / min mag,depmin, depmax, depmar (depth range in km)\n")
        fhandle.write("PARAMETERS\n")
        fhandle.write("'%s' /\n" %os.path.join(EEPASOneYearModel.__execPath,
                                               self.__paramFile))

    	fhandle.write("PARAMETERS\n")
        self.writeDeclusterFlag(fhandle)
        
        fhandle.write("nfile %s\n" %self.nFile())
        fhandle.write("endp/\n")
        fhandle.write("FILES\n")
        fhandle.write("array '%s'\n" %self.filename())
        fhandle.write("end files\n")
        
        # Save time-variant component of the forecast
        if self.timeVariantForecastFile() is not None:
            
            fhandle.write("PARAMETERS\n")
            fhandle.write("nfile 26\n")
            fhandle.write("endp/\n")
            fhandle.write("FILES\n")
            fhandle.write("array '%s'\n" %self.timeVariantForecastFile())
            fhandle.write("end files\n")
        
        
        self.writePPEForecastFile(fhandle)
        self.writeWeightingLikelihood(fhandle)
	
        fhandle.write("PARAMETERS\n")
        fhandle.write("dely %s\n" %EEPASOneYearModel.__catalogLagDays)
        fhandle.write("nfile 26\n")
        
        # Write forecast period
        fhandle.write("gyr1 %s\n" %self.start_date.year)
        fhandle.write("gmo1 %s\n" %self.start_date.month)
        fhandle.write("gdy1 %s\n" %self.start_date.day)
        fhandle.write("ghr1 %s\n" %self.start_date.hour)
        fhandle.write("gmn1 %s\n" %self.start_date.minute)
        fhandle.write("gsc1 %s\n" %self.start_date.second)

        fhandle.write("gyr2 %s\n" %self.end_date.year)
        fhandle.write("gmo2 %s\n" %self.end_date.month)
        fhandle.write("gdy2 %s\n" %self.end_date.day)
        fhandle.write("ghr2 %s\n" %self.end_date.hour)
        fhandle.write("gmn2 %s\n" %self.end_date.minute)
        fhandle.write("gsc2 %s\n" %self.end_date.second)
        fhandle.write("endp/\n")

        self.writeGrids(fhandle)
        fhandle.write("STOP\n")

        # Close the file
        fhandle.close()


    def nFile (self):
        """ Return file identifier for whole forecast file"""
        
        return '28'
    
    
    def writeDeclusterFlag (self, fhandle):
        """ Writes decluster flag to parameter file if there are any."""

        return fhandle
    

    #----------------------------------------------------------------------------
    #
    # Prepare R execution script and invoke the model.
    #
    # Input: None
    #        
    def run (self):
        """ Run EEPAS forecast."""

        # If modelers output some debug/progress info to stderr, ignore it - 
        # don't trigger an exception 
        ignore_model_errors = True


        # Create link for CSEP input catalog and model formatted catalog files if
        # one doesn't exist yet
        if os.path.exists(os.path.join(self.catalogDir, 
                                       self.__modelInputCatalog)) is False:
           
           command = "ln -s %s fort.10" %os.path.join(self.catalogDir,
                                                      self.inputCatalogFilename())
           Environment.invokeCommand(command)
           
           command = "ln -s %s fort.11" %os.path.join(self.catalogDir, 
                                                      self.__modelInputCatalog)
           Environment.invokeCommand(command)
           
           
           # Reformat input catalog into model expected one
           Environment.invokeCommand(os.path.join(EEPASOneYearModel.__execPath, 
                                                  EEPASOneYearModel.__catalogCoversionExecutable),
                                     ignore_model_errors) 
   
           # Clean up temporary links to the catalog files
           Environment.invokeCommand("rm fort.*")
        
        
        # Invoke the model
        command = "%s %s" %(os.path.join(EEPASOneYearModel.__execPath,
                                         EEPASOneYearModel.__executableFile),
                            self.parameterFile)

        Environment.invokeCommand(command, 
                                  ignore_model_errors)

        # Clean up temporary files generated by the run
        command = "rm -rf *.[0-9][0-9] *.ou* fort.*"
        Environment.invokeCommand(command)


    #---------------------------------------------------------------------------
    #
    # Create forecast.
    # This method is overwritten to create metadata file per each forecast file
    # as generated by the model.
    #
    # Input: 
    #       test_date - datetime object that represents testing date.
    #       catalog_dir - Directory with catalog data
    #       archive_dir - Directory to store intermediate model results if any.
    #                     Default is None.    
    #        
    # Output:
    #        Name of created forecast file.
    #
    def create (self, 
                test_date, 
                catalog_dir, 
                archive_dir=None, 
                data_source = CatalogDataSource()):
        """ Generate forecast."""
        
        # Invoke the models
        result_files = Forecast.create(self, 
                                       test_date, 
                                       catalog_dir, 
                                       archive_dir)

        # Check if PPE forecast file wasn't generated (EEPAS-0F forecast was staged) 
        # and PPE file is archived ---> stage the file
        file_is_staged = False
        if self.ppeForecastFile() is not None and \
           os.path.exists(os.path.join(self.dir,
                                       self.ppeForecastFile())) is False and \
           archive_dir is not None:
            
           file_is_staged = CSEPStorage.stage(self, 
                                              [self.ppeForecastFile()],
                                              os.path.join(archive_dir,
                                                           Forecast.archiveDir(self.ppeForecastFile())))


        # PPE model is invoked implicitly by the EEPAS configuration for ETAS,
        # create metadata file for it.
        if file_is_staged is False and self.ppeForecastFile() is not None:
           
           comment = "PPE forecast file with start date '%s' and end date '%s'" \
                     %(self.start_date, self.end_date)
           
           Forecast.metadata(os.path.join(self.dir, 
                                          self.ppeForecastFile()),
                             comment,
                             archive_dir)
        
        if self.ppeForecastFile() is not None:
            result_files.append(os.path.basename(self.ppeForecastFile()))

        # Stage time-variant component if it exists
        file_is_staged = False
        if self.timeVariantForecastFile() is not None and \
           os.path.exists(os.path.join(self.dir,
                                       self.timeVariantForecastFile())) is False and \
           archive_dir is not None:
            
           file_is_staged = CSEPStorage.stage(self, 
                                              [self.timeVariantForecastFile()],
                                              os.path.join(archive_dir,
                                                           Forecast.archiveDir(self.timeVariantForecastFile())))

        # Time-variant component of the model is saved by each EEPAS,
        # create metadata file for it.
        if file_is_staged is False and self.timeVariantForecastFile() is not None:
           
           comment = "Time-variant component for %s with start date '%s' and end date '%s'" \
                     %(self.filename(),
                       self.start_date, 
                       self.end_date)
           
           Forecast.metadata(os.path.join(self.dir, 
                                          self.timeVariantForecastFile()),
                             comment,
                             archive_dir)
        
        if self.timeVariantForecastFile() is not None:
            result_files.append(os.path.basename(self.timeVariantForecastFile()))
        
        return result_files


    #----------------------------------------------------------------------------
    #
    # Writes forecasts output files to the file handle that are specific to the 
    # model configuration - some configurations generate two forecasts files.
    #
    # Input:
    #        fhandle - Handle to the open control file used to invoke the model.
    #
    # Output:
    #        Handle to the control file.
    #        
    def writePPEForecastFile (self, fhandle):
        """ Writes PPE forecast output file(s) to the file that are specific to the
            model configuration."""

        if self.ppeForecastFile() is not None:
            
            fhandle.write("PARAMETERS\n")
            
            fhandle.write("nfile 27\n")
            fhandle.write("endp/\n")
            fhandle.write("FILES\n")
            fhandle.write("array '%s'\n" %self.ppeForecastFile())
            fhandle.write("end files\n")
        
        return fhandle


    #---------------------------------------------------------------------------
    #
    # Write weight and likelihood directives if any are defined for the model
    #
    def writeWeightingLikelihood(self, fhandle):
    
        fhandle.write("PERIOD\n")
        fhandle.write("1987, 01, 01, 2006, 03,31/ y1,m1,d1,y2,m2,d2\n")
        fhandle.write("WEIGHTS\n")
        fhandle.write("%s\n" %self.__weights)
        fhandle.write("COMPUTE WEIGHTS\n")
        fhandle.write("LIKELIHOODS\n")


    #---------------------------------------------------------------------------
    #
    # Creates filename for PPE foreacast if it does not exist yet.
    #
    # Input:
    #        start_date - Start date of the forecast. Default is None.
    #
    # Output:
    #        Handle to the control file.
    def ppeForecastFile (self):
        """ Creates filename for PPE foreacast."""
        
        # There is no PPE forecast generated by generic EEPAS configuration
        return None
    
    
    def timeVariantForecastFile (self):
        
        if self.filename() is not None:
            return self.filename().replace(self.type(), 
                                           self.type() + EEPASOneYearModel.TimeVariantPostfix)
        return None
    

    #----------------------------------------------------------------------------
    #
    # Writes grid information that are specific to the model configuration.
    #
    # Input:
    #        fhandle - Handle to the open control file used to invoke the model.
    #
    # Output:
    #        Handle to the control file.
    #        
    def writeGrids (self, fhandle):
        """ Writes grid information that are specific to the model configuration."""

        fhandle.write("EEPGRID\n")
        
        return fhandle


    #----------------------------------------------------------------------------
    #
    # Returns name of the testing area polygon file that is specific to the 
    # model configuration.
    #
    # Input: None
    #
    # Output:
    #        Name of the file.
    #        
    def polygonFile (self):
        """ Returns name of the testing area polygon file that is specific to the 
            model configuration."""

        return 'nztestpoly.dat'


    #---------------------------------------------------------------------------
    #
    # Return commands that should be used to capture version of external
    # software packages the model is dependent on. 
    #
    # Input: None.
    #
    # Output:
    #           String identifying the type
    #
    @staticmethod
    def externalSoftwareVersions ():
        """ Returns dictionary of command to determine external software 
            version and flag if output of that command is redirected to the
            stderr (True) or not (False) (java -version, for example)."""
        

        return {} 

