"""
Module ETAS_HWOneMonthModel
"""

__version__ = "$Revision$"
__revision__ = "$Id$"


import os

import Environment, CSEPFile
from Forecast import Forecast
from OneMonthForecast import OneMonthForecast
from HelmstetterWernerModel import HelmstetterWernerModel


#--------------------------------------------------------------------------------
#
# ETAS_HW one-month forecast model.
#
# This class is designed to invoke one-month ETAS forecast model by Agnes 
# Helmstetter and Max Werner. It prepares input catalog data, formats control 
# file with model parameters, and invokes the model. It places forecast file 
# under user specified directory.
#
class ETAS_HWOneMonthModel (OneMonthForecast, HelmstetterWernerModel):

    # Static data of the class
    
    # Keyword identifying type of the class
    Type = "ETAS_HW" + OneMonthForecast.Type
    
    __executableFile = "csepETAS_HW"
    
    __parameterFile = "E_m3M4_T30_pl.par"
    
    
    #--------------------------------------------------------------------
    #
    # Initialization.
    #
    # Input: 
    #        dir_path - Directory to store forecast file to.
    #        args - Optional arguments for the model. Default is None.
    # 
    def __init__ (self, dir_path, args = None):
        """ Initialization for ETAS_HWOneMonthModel class"""
        
        OneMonthForecast.__init__(self, dir_path)
        HelmstetterWernerModel.__init__(self, args)
        

    #---------------------------------------------------------------------------
    #
    # Return keyword identifying the model.
    #
    # Input: None.
    #
    # Output:
    #           String identifying the type
    #
    def type (self):
        """ Returns keyword identifying the forecast model type."""
        
        return ETAS_HWOneMonthModel.Type


    #----------------------------------------------------------------------------
    #
    # Return full path for the result forecast file that is based on XML
    # master template.
    #
    # Input: None.
    #
    # Output:
    #           String identifying the filename.
    #
    def filename (self):
        """ Returns filename of generated forecast in XML format."""
        
        return CSEPFile.Name.xml(Forecast.filename(self))
    

    #--------------------------------------------------------------------
    #
    # Write input parameter file for the model in following format:
    # forecastStartDate=1992-07-01T00:00:00Z
    # forecastEndDate=1992-07-02T00:00:00Z
    # InputCatalogFile=INPUT/OneDayModelInputCatalog-2009-07-01-M2.95-1932.dat
    # OutputForecastFile=OUT_ETAS_Md3_Mmin4_T1_tmp/csep-test.xml
    # OutputScratch=OUT_SCRATCH/    
    #
    # Input: None
    #        
    def writeParameterFile (self, 
                            filename = None):
        """ Format input parameter file for the model.
            Path to created input parameter file will be passed to the 
            model's executable."""

        fhandle = Forecast.writeParameterFile(self,
                                              filename)
        
        # To avoid inheritance from two classes that are derived from the same
        # base "Forecast" class, use this "pass all values" approach
        HelmstetterWernerModel.writeFile(self,
                                         fhandle,
                                         self.start_date,
                                         self.end_date,
                                         os.path.join(self.catalogDir,
                                                      self.inputCatalogFilename()),
                                                      self.filename(),
                                         ETAS_HWOneMonthModel.__parameterFile)

        # Close the file
        fhandle.close()


    #--------------------------------------------------------------------
    #
    # Invoke the model to generate forecast.
    #
    # Input: None
    #        
    def run (self):
        """ Invoke ETAS_HW model."""

        # If modelers output some debug/progress info to stderr, ignore it - 
        # don't trigger an exception 
        ignore_model_errors = True
        
        command = "%s %s" %(os.path.join(HelmstetterWernerModel._path,
                                         ETAS_HWOneMonthModel.__executableFile),
                            self.parameterFile)
        Environment.invokeCommand(command,
                                  ignore_model_errors)
            
