"""
Module HainzlModel
"""

__version__ = "$Revision: 4699 $"
__revision__ = "$Id: HainzlModel.py 4699 2014-07-21 23:15:31Z liukis $"

import os, shutil

import Environment, CSEPFile, CSEP, CSEPLogging
from Forecast import Forecast
from DataSourceFactory import DataSourceFactory
from CSEPInputParams import CSEPInputParams
from SlipModels import SlipModels


#--------------------------------------------------------------------------------
#
# HainzlModel forecast model for New Zealand.
#
# This class is a helper to invoke any of Sebastian Hainzl forecasts models.
#
class HainzlModel (object):

    # Static data of the class
    Type = 'HAINZL'

    # Center code path for the model
    Path = os.path.join(Forecast.CodePath, 
                        'NewZealand',
                        'src',
                        'HainzlModels')

    # Option to specify which model configuration to invoke
    ModelOption = 'model'
    
    ModelSet = ['ETAS', 'RETAS']
    
    # useMainshockprobmaps option for the model
    MainshockProbMapsOption = 'config'
    
    # Allowed model configurations
    MainshockProbMapsSet = ['0', '1', '2']
    
    # Coulumb file list option
    CoulombFileListOption = 'coulombList'
    
    # Coulumb file list option
    SlipModelFileListOption = 'slipModelList'

    # Default options for the model
    __defaultArgs = {ModelOption: None,
                     MainshockProbMapsOption: None,
                     CoulombFileListOption: os.path.join(Path,
                                                         'darefield_farfalleheaderfile.dat'),
                     SlipModelFileListOption: None}

    
    #--------------------------------------------------------------------
    #
    # Initialization.
    #
    # Input: 
    #        args - Optional arguments for the model. Default is None.
    # 
    def __init__ (self, args = None):
        """ Initialization for HainzlModel class"""
        
        # Input arguments for the model were provided:
        self.__args = CSEPInputParams.parse(HainzlModel.__defaultArgs, 
                                            args)
        if self.__args[HainzlModel.ModelOption] not in HainzlModel.ModelSet:
            error_msg = "One of %s models options is allowed, '%s' is provided" %(HainzlModel.ModelSet,
                                                                                  self.__args[HainzlModel.ModelOption])
             
            CSEPLogging.CSEPLogging.getLogger(HainzlModel.__name__).error(error_msg)
            raise RuntimeError, error_msg
        
        if self.__args[HainzlModel.MainshockProbMapsOption] not in HainzlModel.MainshockProbMapsSet:
            error_msg = "One of %s models options is allowed, '%s' is provided" %(HainzlModel.MainshockProbMapsSet,
                                                                                  self.__args[HainzlModel.MainshockProbMapsOption])
             
            CSEPLogging.CSEPLogging.getLogger(HainzlModel.__name__).error(error_msg)
            raise RuntimeError, error_msg


    #----------------------------------------------------------------------------
    #
    # Return sub-type keyword identifying the model: based on testing region.
    #
    # Input: None.
    #
    # Output:
    #           String identifying the sub-type
    #
    def subtype (self):
        """ Returns keyword identifying the forecast model sub-type."""

        # Specify selected type of forecast as the sub-type for the model        
        return '%s%s' %(self.__args[HainzlModel.ModelOption],
                        self.__args[HainzlModel.MainshockProbMapsOption])


    #----------------------------------------------------------------------------
    #
    # Return filename for model's executable
    #
    # Input: None.
    #
    # Output:
    #         Name of the executable file for the model.
    #
    def executableScript (self):
        """ Returns name of the Perl script to use to invoke the model."""

        # Specify selected type of forecast as the sub-type for the model        
        return 'run%s.pl' %self.subtype()
    

    #--------------------------------------------------------------------
    #
    # Write input parameter file for the model in following format:
    #    [csep@csep-devel-13-1 ~/fromSebastian]$ cat testinput/exampleinputfile.txt 
    #    ForecastStartDate=2010-09-03T16:35:42Z
    #    ForecastEndDate=2010-09-04T16:35:42Z
    #    InputCatalogFile=testinput/examplecatalog.dat
    #    OutputForecastFile=result.test
    #    ForecastTemplate=testinput/darf_temp.txt
    #    InputCoulombFile=testinput/darefield_farfalleheaderfile.dat
    #    InputListSlipModels=testinput/list_of_slip_models.dat    
    #        
    def writeFile (self, 
                   fhandle,
                   start_date,
                   end_date,
                   input_catalog,
                   result_forecast,
                   template_file):
        """ Format input parameter file for the model.
            Path to created input parameter file will be passed to the 
            model's executable."""

        # Input parameters include:
        line = "ForecastStartDate=%sZ\n" \
               %start_date.strftime(CSEP.Time.ISO8601Format)
        fhandle.write(line)

        line = "ForecastEndDate=%sZ\n" \
               %end_date.strftime(CSEP.Time.ISO8601Format)
        fhandle.write(line)
        
        line = "InputCatalogFile=%s\n" %input_catalog
        fhandle.write(line)

        line = "OutputForecastFile=%s\n" %result_forecast
        fhandle.write(line)
        
        line = "ForecastTemplate=%s\n" %CSEPFile.Name.ascii(template_file)
        fhandle.write(line)
        
        # Use existing file list if it's provided, otherwise identify available 
        # Coulomb files within file system
        coulomb_file_list = self.__args[HainzlModel.CoulombFileListOption]
        line = "InputCoulombFile=%s\n" %coulomb_file_list
        fhandle.write(line)
        
        slip_models_list = self.__args[HainzlModel.SlipModelFileListOption]
        if slip_models_list is None:
            # Identify slip model files for the run
            local_path, local_file = os.path.split(input_catalog)
            slip_models_list = SlipModels().files(local_path,
                                                  start_date)
        
        line = "InputListSlipModels=%s\n" %slip_models_list
        fhandle.write(line)

        return fhandle


    #--------------------------------------------------------------------
    #
    # Invoke the model to generate forecast.
    #
    # Input: None
    #        
    def runScript (self, parameter_file):
        """ Invoke model."""

        # If modelers output some debug/progress info to stderr, ignore it - 
        # don't trigger an exception 
        ignore_model_errors = True
        
        # Create temporary directory for model's run
        output_dir = os.path.join(os.getcwd(),
                                  'output')
        os.makedirs(output_dir)
        
        # ( echo testinput/exampleinputfile.txt ) | ./runRETAS1.pl        
        command = "(echo \"%s\"; echo \"%s\") | %s" %(parameter_file,
                                                      HainzlModel.Path,
                                                      os.path.join(HainzlModel.Path,
                                                                   self.executableScript()))
        Environment.invokeCommand(command,
                                  ignore_model_errors)
        
        # Remove model's runtime directory
        shutil.rmtree(output_dir,
                      ignore_errors = True,
                      onerror = CSEP.OnError.shutil)


    #---------------------------------------------------------------------------
    #
    # Return commands that should be used to capture version of external
    # software packages the model is dependent on. 
    #
    # Input: None.
    #
    # Output:
    #           String identifying the type
    #
    @staticmethod
    def externalSoftwareVersions ():
        """ Returns dictionary of command to determine external software 
            version and flag if output of that command is redirected to the
            stderr (True) or not (False) (java -version, for example)."""
        
        return {"gcc --version" : False}

