"""
Module HainzlOneYearModel
"""

__version__ = "$Revision: 4555 $"
__revision__ = "$Id: HainzlOneYearModel.py 4555 2014-02-11 22:16:09Z liukis $"


import os

from Forecast import Forecast
from OneYearForecast import OneYearForecast
from HainzlModel import HainzlModel


#--------------------------------------------------------------------------------
#
# HAINZL one-year forecast model.
#
# This class is designed to invoke one-year HAINZL forecast model by Sebastian 
# Hainzl. It prepares input catalog data, formats control 
# file with model parameters, identifies list of available coulomb and slip 
# models and provides it as an input to the model. It places forecast file 
# under user specified directory.
#
class HainzlOneYearModel (OneYearForecast, HainzlModel):

    # Static data of the class
    
    # Keyword identifying type of the class
    Type = HainzlModel.Type + OneYearForecast.Type
    
    
    #--------------------------------------------------------------------
    #
    # Initialization.
    #
    # Input: 
    #        dir_path - Directory to store forecast file to.
    #        args - Optional arguments for the model. Default is None.
    # 
    def __init__ (self, dir_path, args = None):
        """ Initialization for HainzlOneYearModel class"""
        
        OneYearForecast.__init__(self, dir_path)
        HainzlModel.__init__(self, args)
        

    #---------------------------------------------------------------------------
    #
    # Return keyword identifying the model.
    #
    # Input: None.
    #
    # Output:
    #           String identifying the type
    #
    def type (self):
        """ Returns keyword identifying the forecast model type."""
        
        return HainzlOneYearModel.Type


    #----------------------------------------------------------------------------
    #
    # Return sub-type keyword identifying the model: based on testing region.
    #
    # Input: None.
    #
    # Output:
    #           String identifying the sub-type
    #
    def subtype (self):
        """ Returns keyword identifying the forecast model sub-type."""
        
        return HainzlModel.subtype(self)


    #--------------------------------------------------------------------
    #
    # Write input parameter file for the model in following format:
    # forecastStartDate=1992-07-01T00:00:00Z
    # forecastEndDate=1992-07-02T00:00:00Z
    # InputCatalogFile=INPUT/OneDayModelInputCatalog-2009-07-01-M2.95-1932.dat
    # OutputForecastFile=OUT_ETAS_Md3_Mmin4_T1_tmp/csep-test.xml
    # OutputScratch=OUT_SCRATCH/    
    #
    # Input: None
    #        
    def writeParameterFile (self, 
                            filename = None):
        """ Format input parameter file for the model.
            Path to created input parameter file will be passed to the 
            model's executable."""

        fhandle = Forecast.writeParameterFile(self,
                                              filename)
        
        # To avoid inheritance from two classes that are derived from the same
        # base "Forecast" class, use this "pass all values" approach
        HainzlModel.writeFile(self,
                              fhandle,
                              self.start_date,
                              self.end_date,
                              os.path.join(self.catalogDir,
                                           self.inputCatalogFilename()),
                              self.filename(),
                              self.templateFile)

        # Close the file
        fhandle.close()


    #--------------------------------------------------------------------
    #
    # Invoke the model to generate forecast.
    #
    # Input: None
    #        
    def run (self):
        """ Invoke model."""

        HainzlModel.runScript(self,
                              self.parameterFile)
        
