"""
Module STEPCoulombModel
"""

__version__ = "$Revision:$"
__revision__ = "$Id:$"

import os, datetime

import Environment, CSEPFile, CSEP, CSEPLogging
from Forecast import Forecast
from CSEPInputParams import CSEPInputParams
from cseprandom import CSEPRandom
from GeoNetNZDataSource import GeoNetNZDataSource, FOCAL_MECHANISM_PATH_ENV
from DataSourceFactory import DataSourceFactory
from SlipModels import SlipModels


#-------------------------------------------------------------------------------
#
# STEPCoulombModel forecast model for New Zealand.
#
# This class is a helper to invoke any of Sandy Steacy/Matt Gerstenberger
# forecasts models.
#
class STEPCoulombModel(object):

    # Static data of the class
    Type = 'STEPCOULOMB'

    # Center code path for the model
    Path = os.path.join(Forecast.CodePath,
                        'NewZealand',
                        'src',
                        'STEPCoulombModel')

    __JavaPath = os.path.join(Path,
                              'OpenSHA')

    # Coulumb file list option
    CoulombFileOption = 'CoulombFile'

    # Coulumb file list option
    SlipModelFileListOption = 'slipModelList'

    # Parameter file for the Coulomb model: to support multiple 
    # geographical regions
    ParamFileOption = 'masterParameter'
    __backgroundFileOption = "backgroundFile"

    # Default options for the model
    __defaultArgs = {ParamFileOption: os.path.join(Path,
                                                   "master_file_GNS.txt"),
                     SlipModelFileListOption: None,
                     __backgroundFileOption: "NZZeroRate05.dat"}

    __CoulombExecutableFile = 'CSEPCoulomb'

    __ifortVersionFile = 'ifort.version'
    

    #--------------------------------------------------------------------
    #
    # Initialization.
    #
    # Input:
    #        args - Optional arguments for the model. Default is None.
    #
    def __init__ (self, args=None):
        """ Initialization for STEPCoulombModel class"""
        
        # Input arguments for the model were provided:
        self.__args = CSEPInputParams.parse(STEPCoulombModel.__defaultArgs, 
                                            args)

        self.CoulombOutput = None
        

    #---------------------------------------------------------------------------
    #
    # Write input parameter file for the model in following format:
    # 
    # Coulomb part of the model:
    # InputParameterFile master_file_GNS_2.txt
    # PathToThePrograms /home/ajimenez/programs
    # ForecastStartingDate 20120520050600
    # Slipmodelfile slip_models.txt
    # MomentTensorFile GeoNet_CMT_solutions.csv
    #        
    def writeCoulombFile (self, 
                          filename,
                          start_date,
                          local_dir):
        """ Format input parameter file for the model.
            Path to created input parameter file will be passed to the 
            model's executable."""

        fhandle = file(filename, 
                       CSEPFile.Mode.WRITE)

        line = "InputParameterFile %s\n" %self.__args[STEPCoulombModel.ParamFileOption]
        fhandle.write(line)
        
        line = "PathToThePrograms %s/\n" %self.Path
        fhandle.write(line)
        
        line = "ForecastStartingDate %s\n" %start_date.strftime("%Y%m%d%H%M%S")
        fhandle.write(line)

        slip_models_list = self.__args[STEPCoulombModel.SlipModelFileListOption]
        if slip_models_list is None:
            # Identify slip model files for the run
            slip_models_list = SlipModels().files(local_dir,
                                                  start_date,
                                                  include_eventID=True)
        
        line = "Slipmodelfile %s\n" %slip_models_list
        fhandle.write(line)

        data_path, data_file = os.path.split(DataSourceFactory().object(GeoNetNZDataSource.Type, 
                                                                        isObjReference=True).RawFile)
        
        if FOCAL_MECHANISM_PATH_ENV in os.environ:
            data_path = os.environ[FOCAL_MECHANISM_PATH_ENV]
        
        line = "MomentTensorFile %s\n" %os.path.join(data_path,
                                                     GeoNetNZDataSource.ProcessedFocalMechanismFile)
        fhandle.write(line)

        line = "Outputfile %s\n" %self.CoulombOutput
        fhandle.write(line)


    #---------------------------------------------------------------------------
    #
    # Write input parameter file for the model in following format:
    # 
    # Coulomb part of the model:
    # InputParameterFile master_file_GNS_2.txt
    # PathToThePrograms /home/ajimenez/programs
    # ForecastStartingDate 20120520050600
    # Slipmodelfile slip_models.txt
    # MomentTensorFile GeoNet_CMT_solutions.csv
    #        
    def writeSTEPFile (self, 
                       fhandle,
                       start_date,
                       num_days,
                       catalog_file,
                       result_forecast):
        """ Format input parameter file for the STEP part of the model.
            Path to created input parameter file will be passed to the 
            model's executable."""


        result_path, result_file = os.path.split(result_forecast)
        self.CoulombOutput = os.path.join(result_path,
                                          "CoulombPart_" + result_file)
            
        # Start time of the data - to save on computations use later than
        # input catalog's start date
        line = "1 1 2010 0 0 0\n"
        fhandle.write(line)

        # Test date and time 
        line = "%s %s %s %s %s %s\n" \
               %(start_date.day,
                 start_date.month,
                 start_date.year,
                 start_date.hour,
                 start_date.minute,
                 start_date.second)
        fhandle.write(line)

        # Duration in days
        line = "%s\n" %int(num_days)
        fhandle.write(line)

        # Path to the input catalog file
        fhandle.write(catalog_file + '\n')

        # Path to the output forecast file
        fhandle.write(result_forecast + "\n")

        # Path to the CSEP_background file
        fhandle.write(os.path.join(self.__JavaPath,
                                   self.__args[STEPCoulombModel.__backgroundFileOption]) + "\n")
        
        fhandle.write(self.CoulombOutput + '\n')


    #--------------------------------------------------------------------
    #
    # Invoke the model to generate forecast.
    #
    # Input: None
    #        
    def invokeModel (self, 
                     parameter_file, 
                     step_parameter_file):
        """ Invoke model."""

                
        command = "%s %s" %(os.path.join(STEPCoulombModel.Path,
                                         STEPCoulombModel.__CoulombExecutableFile),
                            parameter_file)
        Environment.invokeCommand(command)
        
        cwd = os.getcwd()
        os.chdir(os.path.join(STEPCoulombModel.__JavaPath,
                              'build'))

        try:
            # Model requests to change to the model installation directory
            # since it's using relative paths to locate other files
            __command = "java -Xms2048m -Xmx6144m -jar lib/step-aftershock.jar 0 -f %s" %os.path.join(cwd,
                                                                                                      step_parameter_file)

            Environment.invokeCommand(__command)

        finally:
            # Go back to the directory
            os.chdir(cwd)
            
        # Return path to the Coulomb part of the model that needs to be 
        # archived to avoid it's evaluation
        return self.CoulombOutput
        

    #---------------------------------------------------------------------------
    #
    # Return commands that should be used to capture version of external
    # software packages the model is dependent on. 
    #
    # Input: None.
    #
    # Output:
    #           String identifying the type
    #
    @staticmethod
    def externalSoftwareVersions ():
        """ Returns dictionary of command to determine external software 
            version and flag if output of that command is redirected to the
            stderr (True) or not (False) (java -version, for example)."""
        
        version_file = CSEPFile.openFile(os.path.join(STEPCoulombModel.Path,
                                                      STEPCoulombModel.__ifortVersionFile))

        return {"ifort --version" : version_file.read(),
                "java -version"   : True} # Java outputs info to stderr

