"""
Module STEPJAVAOneYearModel
"""

__version__ = "$Revision$"
__revision__ = "$Id$"

import os, datetime

import CSEPFile, CSEPLogging, Environment
from Forecast import Forecast
from OneYearForecast import OneYearForecast
from CSEPInputParams import CSEPInputParams


#-------------------------------------------------------------------------------------
#
# STEPJAVAOneYear forecast model.
#
# This class is designed to invoke a one-year STEP forecast model. It prepares 
# input catalog data, formats input file with model parameters, and invokes
# the STEP model. It places forecast file under user specified directory.
#
class STEPJAVAOneYearModel (OneYearForecast):

    # Static data of the class

    # Keyword identifying type of the class
    Type = "STEPJAVA" + OneYearForecast.Type
    
    __backgroundFileOption = "backgroundFile"

    # Center code path for the model
    __NZRootPath = os.environ["NZHOME"]
    __modelPath = os.path.join(__NZRootPath,
                               'OpenSHA')
    
    # This data is static for the class - safe because we generate
    # only one forecast per model for any CSEP run.
    __defaultArgs = {__backgroundFileOption : "NZyearlyRates.txt"}


    #--------------------------------------------------------------------
    #
    # Initialization.
    #
    # Input: 
    #        dir_path - Directory to store forecast file to.
    #        args - Optional arguments for the model. Default is None.
    # 
    def __init__ (self, dir_path, args=None):
        """ Initialization for STEPJAVAOneYearModel class"""

        # Use later than actual download start date for the input catalog
        # to make STEP code happy
        OneYearForecast.__init__(self, dir_path)

        # Input arguments for the model were provided:
        self.__args = CSEPInputParams.parse(STEPJAVAOneYearModel.__defaultArgs, 
                                            args)
        


    #--------------------------------------------------------------------
    #
    # Return keyword identifying the model.
    #
    # Input: None.
    #
    # Output:
    #           String identifying the type
    #
    def type (self):
        """ Returns keyword identifying the forecast model type."""

        return self.Type


    #---------------------------------------------------------------------------
    #
    # Return sub-type keyword identifying the model: based on background file 
    # used by the model
    #
    # Input: None.
    #
    # Output:
    #           String identifying the sub-type
    #
    def subtype (self):
        """ Returns keyword identifying the forecast model sub-type."""
        
        return CSEPFile.Name.extension(self.__args[STEPJAVAOneYearModel.__backgroundFileOption])


    #--------------------------------------------------------------------
    #
    # Write input parameter file for the model.
    #
    # Input: None.
    #        
    def writeParameterFile (self):
        """ Format input parameter file for the model."""


        fhandle = Forecast.writeParameterFile(self)

        # Start time of the data - to save on computations use later than
        # input catalog's start date
        line = "1 1 1980 0 0 0\n"
        fhandle.write(line)


        # Test date - append start "hour minute second" as all zero's 
        line = "%s %s %s 0 0 0\n" \
               %(self.start_date.day,
                 self.start_date.month,
                 self.start_date.year)
        fhandle.write(line)

        # Duration in days
        line = "%s\n" %int(self.numDays(self.start_date,
                                        self.end_date))
        fhandle.write(line)

        # Path to the input catalog file
        fhandle.write(os.path.join(self.catalogDir,
                                   CSEPFile.Name.extension(self.inputCatalogFilename(),
                                                           'dat')+ "\n"))

        # Path to the output forecast file
        fhandle.write(self.filename() + "\n")

        # Path to the CSEP_background file
        fhandle.write(os.path.join(self.__modelPath,
                                   self.__args[STEPJAVAOneYearModel.__backgroundFileOption]) + "\n")

        # Close the file
        fhandle.close()


    #---------------------------------------------------------------------------
    #
    # Invoke the model.
    #
    # Input: None
    #        
    def run (self):
        """ Run STEPJAVAOneYear forecast."""

        cwd = os.getcwd()
        os.chdir(os.path.join(STEPJAVAOneYearModel.__modelPath,
                              'build'))

        try:
            # Model requests to change to the model installation directory
            # since it's using relative paths to locate other files
            __command = "java -Xms2048m -Xmx6144m -jar lib/step-aftershock.jar 0 -f %s" %os.path.join(cwd,
                                                                                  self.parameterFile)

            Environment.invokeCommand(__command)

        finally:
            # Go back to the directory
            os.chdir(cwd)


    #---------------------------------------------------------------------------
    #
    # Return commands that should be used to capture version of external
    # software packages the model is dependent on. 
    #
    # Input: None.
    #
    # Output:
    #           String identifying the type
    #
    @staticmethod
    def externalSoftwareVersions ():
        """ Returns dictionary of command to determine external software 
            version and flag if output of that command is redirected to the
            stderr (True) or not (False) (java -version, for example)."""


        return {"java -version" : True} # Java output info to stderr


