"""
Module SUPCanterburyModel
"""

__version__ = "$Revision$"
__revision__ = "$Id$"


import os, datetime

import Environment, CSEPFile, CSEPXML
from Forecast import Forecast


#--------------------------------------------------------------------------------
#
# SUPCanterbury reference forecast model.
#
# This class is designed to invoke three-months PPE and EEPAS forecast model. 
# It prepares input catalog data, formats control file with model parameters,
# and invokes the model. It places forecast file under user specified directory.
#
class SUPCanterburyModel (object):

    # Static data of the class
    
    # Keyword identifying the type of the forecast
    Type = "SUPCanterbury"

    # 30-day lag in input catalog - specific to the EEPAS models
    __catalogLagDays = 0
    
    # Top-level path for the model executable and configuration files
    Path = os.path.join(Forecast.CodePath,
                        'NewZealand',
                        'src',
                        'EEPAS')
                        
    __executableFile = "eepasof22ng77"
    __catalogCoversionExecutable = "reformatscec"

    __paramFile = "sup_cant.par"
    
    __weights = 0
    
    
    #----------------------------------------------------------------------------
    #
    # Initialization.
    #
    # Input:
    #        input_catalog - Input catalog filename 
    # 
    def __init__ (self, 
                  input_catalog):
        """ Initialization for EEPASModel class"""
        

        # Filename for input catalog in a model expected format
        self.__modelInputCatalog = CSEPFile.Name.extension(input_catalog,
                                                           'ref')
        
        # Darfield experiment is defined for 0.05 degree grid, which results
        # in cell center coordinates to have 3 digits of precision
        CSEPXML.FloatFormatStr = '%.3f'
        

    #----------------------------------------------------------------------------
    #
    # Write input parameter file for the model.
    #        
    def writeFile (self, 
                   fhandle,
                   catalog_dir,
                   start_date,
                   end_date,
                   forecast_file):
        """ Format input parameter file for the model.
            Created file will be used by Fortran executable that invokes the
            model."""
        
        
        # Write generic part of input parameters at the top of the file:
        fhandle.write("FILES\n")
        
        fhandle.write("licence '%s'\n" \
                      %os.path.join(SUPCanterburyModel.Path, 
                                    'licencekey.dat'))
        
        fhandle.write("grdm '%s'\n" \
                      %os.path.join(SUPCanterburyModel.Path, 
                                    'relmmagscant.dat'))
        
        fhandle.write("grdl '%s'\n" \
                      %os.path.join(SUPCanterburyModel.Path,
                                    'relmblockscant.dat'))

        fhandle.write("catalog '%s'\n" \
                      %os.path.join(catalog_dir, 
                                    self.__modelInputCatalog))
        
        fhandle.write("polygon '%s'\n" %os.path.join(SUPCanterburyModel.Path,
                                                     "canttestpoly.dat"))
        fhandle.write("end files\n")
        
        fhandle.write("POLY/GON  reads in target polygon from file\n")
         
        fhandle.write("CATA/LOGUE selects input data from catalogue file\n")
        
        fhandle.write("all /latmin,latmax,longmin,longmx\n")

        fhandle.write("1964, 01, 01, 0, 0, 0.0\n")
        
        catalog_end_date = start_date - \
                           datetime.timedelta(days=SUPCanterburyModel.__catalogLagDays)
        fhandle.write("%s, %s, %s, %s, %s, %s,\n"
                       %(catalog_end_date.year,
                         catalog_end_date.month,
                         catalog_end_date.day,
                         catalog_end_date.hour,
                         catalog_end_date.minute,
                         catalog_end_date.second))
        fhandle.write("3.95 ,0.0, 40.0, 5.0 / min mag,depmin, depmax, depmar (depth range in km)\n")
        fhandle.write("PARAMETERS\n")
        fhandle.write("'%s' /\n" %os.path.join(SUPCanterburyModel.Path,
                                               self.__paramFile))

    	fhandle.write("PARAMETERS\n")
        
        fhandle.write("nfile %s\n" %self.nFile())
        fhandle.write("endp/\n")
        fhandle.write("FILES\n")
        fhandle.write("array '%s'\n" %forecast_file)
        fhandle.write("end files\n")
        
        catalog_end_date = start_date
        
        fhandle.write("PERIOD\n")
        fhandle.write("1964, 01, 01, %s, %s, %s/ y1,m1,d1,y2,m2,d2\n" %(catalog_end_date.year,
                                                                        catalog_end_date.month,
                                                                        catalog_end_date.day))
        fhandle.write("WEIGHTS\n")
        fhandle.write("%s\n" %SUPCanterburyModel.__weights)
        fhandle.write("COMPUTE WEIGHTS\n")
        fhandle.write("LIKELIHOODS\n")
	
        fhandle.write("PARAMETERS\n")
        fhandle.write("dely %s\n" %SUPCanterburyModel.__catalogLagDays)
        fhandle.write("nfile 26\n")
        
        # Write forecast period
        fhandle.write("gyr1 %s\n" %start_date.year)
        fhandle.write("gmo1 %s\n" %start_date.month)
        fhandle.write("gdy1 %s\n" %start_date.day)
        fhandle.write("ghr1 %s\n" %start_date.hour)
        fhandle.write("gmn1 %s\n" %start_date.minute)
        fhandle.write("gsc1 %s\n" %start_date.second)

        fhandle.write("gyr2 %s\n" %end_date.year)
        fhandle.write("gmo2 %s\n" %end_date.month)
        fhandle.write("gdy2 %s\n" %end_date.day)
        fhandle.write("ghr2 %s\n" %end_date.hour)
        fhandle.write("gmn2 %s\n" %end_date.minute)
        fhandle.write("gsc2 %s\n" %end_date.second)
        fhandle.write("endp/\n")

        fhandle.write("SUPGRID\n")
        fhandle.write("STOP\n")

        # Close the file
        fhandle.close()


    def nFile (self):
        """ Return file identifier for whole forecast file"""
        
        return '26'
    

    #--------------------------------------------------------------------
    #
    # Invoke the model to generate forecast.
    #
    # Input: None
    #        
    def invokeModel (self,
                     catalog_dir,
                     catalog_file,
                     parameter_file):
        """ Invoke model."""

        # If modelers output some debug/progress info to stderr, ignore it - 
        # don't trigger an exception 
        ignore_model_errors = True

        # Create link for CSEP input catalog and model formatted catalog files if
        # one doesn't exist yet
        if os.path.exists(os.path.join(catalog_dir, 
                                       self.__modelInputCatalog)) is False:
           
           command = "ln -s %s fort.10" %os.path.join(catalog_dir,
                                                      catalog_file)
           Environment.invokeCommand(command)
           
           command = "ln -s %s fort.11" %os.path.join(catalog_dir, 
                                                      self.__modelInputCatalog)
           Environment.invokeCommand(command)
           
           
           # Reformat input catalog into model expected one
           Environment.invokeCommand(os.path.join(SUPCanterburyModel.Path, 
                                                  SUPCanterburyModel.__catalogCoversionExecutable),
                                     ignore_model_errors) 
   
           # Clean up temporary links to the catalog files
           Environment.invokeCommand("rm fort.*")
        
        
        # Invoke the model
        command = "%s %s" %(os.path.join(SUPCanterburyModel.Path,
                                         SUPCanterburyModel.__executableFile),
                            parameter_file)

        Environment.invokeCommand(command, 
                                  ignore_model_errors)

        # Clean up temporary files generated by the run
        command = "rm -rf *.[0-9][0-9] *.ou* fort.*"
        Environment.invokeCommand(command)
        
    
    #---------------------------------------------------------------------------
    #
    # Return commands that should be used to capture version of external
    # software packages the model is dependent on. 
    #
    # Input: None.
    #
    # Output:
    #           String identifying the type
    #
    @staticmethod
    def externalSoftwareVersions ():
        """ Returns dictionary of command to determine external software 
            version and flag if output of that command is redirected to the
            stderr (True) or not (False) (java -version, for example)."""
        

        return {} 

