"""
Module SlipModels
"""

__version__ = "$Revision$"
__revision__ = "$Id$"

import os, re, glob, datetime

import CSEPLogging, CSEPFile, CSEP

# Environment variable to provide location of the slip models files
SLIP_MODELS_ENV = "SLIP_MODELS_DIR"
SLIP_MODELS_LAG_ENV ='SLIP_MODELS_DAYS_LAG'


#-------------------------------------------------------------------------------
#
# SlipModels
#
# This class provides an interface to identify available slip models that 
# correspond to the forecast start date.
#
class SlipModels(object):
    """ Class to provide an interface to identify available slip models that 
        correspond to the forecast start date.
    """
    
    Filename = "slipmodels.dat"
    
    __logger = None
    
    
    #---------------------------------------------------------------------------
    #
    # Initialization.
    # 
    def __init__ (self):
        """ Initialization for SlipModels class."""
    
        if SlipModels.__logger is None:
           SlipModels.__logger = CSEPLogging.CSEPLogging.getLogger(SlipModels.__name__)
           
        # Slip model related env. variables must be set
        if SLIP_MODELS_ENV not in os.environ:
            msg = "Environment variable '%s' is required to provide directory \
location of available slip models with *.dat extension." %SLIP_MODELS_ENV
            
            SlipModels.__logger.error(msg)
            raise RuntimeError, msg

        # Directory with slip models must exist
        self.__dir = os.environ[SLIP_MODELS_ENV]
        if not os.path.exists(self.__dir):
            msg = "Specified '%s' directory with slip models does not exist" \
                   %self.__dir
            
            SlipModels.__logger.error(msg)
            raise RuntimeError, msg

        if SLIP_MODELS_LAG_ENV not in os.environ:
            msg = "Environment variable '%s' is required to provide number of \
lag days to slip models." %SLIP_MODELS_LAG_ENV
            
            SlipModels.__logger.error(msg)
            raise RuntimeError, msg
        
        self.__lagDays = int(os.environ[SLIP_MODELS_LAG_ENV])

        
    #---------------------------------------------------------------------------
    #
    # Return file format of pre-processed catalog data.
    #
    # Input: None.
    #
    # Output: String representing the file format of pre-processed catalog data.
    #
    def files (self,
               dir_path,
               test_date,
               include_eventID=False):
        """ Create file that lists identified slip models.
            Inputs: 
            dir_path - Local directory path where to store file with models
            test_date - Test date for which to identify slip models
            include_eventID - Flag if eventID should be parsed out from the slip
                              model filename and provided as separate metadata.
                              Default is False."""

        local_file = os.path.join(dir_path,
                                  "%s.%s" %(test_date.strftime(CSEP.Time.ISO8601Format),
                                                               SlipModels.Filename))
        if include_eventID:
            local_file = os.path.join(dir_path,
                                  "%s.withEventId.%s" %(test_date.strftime(CSEP.Time.ISO8601Format),
                                                               SlipModels.Filename))

        if os.path.exists(local_file):
            # File already exists, use it for the run
            return local_file

                
        entries = glob.glob("%s/*.dat" %self.__dir)
        entries.sort()
        
        __dateStr = "date"
        __magStr = "Mg"
        __eventID = "ID"

        # Go 10-days back from the test date
        adj_test_date = test_date - datetime.timedelta(days=self.__lagDays)

        
        # List of strings that will become lines in the slip model file
        models = []
        
        for each_file in entries:
            
            # Each slip model filename is expected to have the following format:
            # /path/to/dir/2010-09-03T16:35:41Z_Mg7.1_ANYTHING.dat
            file_info = re.search(r"(?P<%s>\d+-\d+-\d+T\d+:\d+:\d+)Z_Mg(?P<%s>\d+(.\d+)?)_(ID(?P<%s>\d+)_)?" \
                                  %(__dateStr,
                                    __magStr,
                                    __eventID),
                                  os.path.basename(each_file))
     
            # Found match
            if file_info:
                
                # Slip model should be 10 days priory to the test date provided
                model_date = datetime.datetime.strptime(file_info.group(__dateStr),
                                                        CSEP.Time.ISO8601Format)
                
                # use slip model if it corresponds to any event priory to
                # 10-days before the test date
                if model_date <= adj_test_date:
                    if not include_eventID:
                        models.append('\t'.join([file_info.group(__dateStr) + 'Z',
                                                 file_info.group(__magStr),
                                                 each_file]))
                    else:
                        models.append('\t'.join([file_info.group(__dateStr) + 'Z',
                                                 file_info.group(__magStr),
                                                 file_info.group(__eventID),
                                                 each_file]))
                        
                    

        # Expected format of the file listing slip models for the 
        # forecast model:
        #             # no. of slip models (lines to read)
        #             # time    magnitude    path_to_file 
        #             5
        #             2010-09-03T16:35:41Z    7.1     darfield_slip.dat
        #             2011-02-21T23:51:42Z    6.31    chch_feb_2011_slip.dat
        #             2011-06-13T02:20:49Z    6.41    chch_june_2011_slip.dat
        #             2011-12-23T00:58:38Z    5.84    dec_m5.8_2011-slip.dat
        #             2011-12-23T02:18:03Z    6.16    chch_dec_2011_slip.dat
        fhandle = CSEPFile.openFile(local_file,
                                    CSEPFile.Mode.WRITE)
        fhandle.write("# no. of slip models (lines to read)\n")
        fhandle.write("# time    magnitude    path_to_file\n")
        fhandle.write('%s\n' %len(models))
        
        fhandle.write("\n".join(models))
        fhandle.write("\n")
        fhandle.close()
            
        return local_file
