"""
Module BestAvailableCatalogToZMAP

command example:
python ~/source/src/SCECModels/NewZealand/generic/utils/NZCatalog.py --csvCatalog=earthquakesto20140703.ref --historicalCatalog=import_processed.dat --ZMAPCatalog=inputcatalog.dat
"""

__version__ = "$Revision$"
__revision__ = "$Id$"

import datetime
import numpy as np

import CSEPFile, CSEPUtils, CSEPGeneric


class CSVFormat (object):
    YEAR   = 0 
    MONTH  = 1
    DAY    = 2
    HOUR   = 3
    MINUTE = 4
    SECOND = 5
    LAT    = 6
    LONG   = 7
    DEPTH  = 8 
    MAGN   = 9
    ERLAT  = 10
    ERZ    = 11
    ERMAG  = 12 


#-------------------------------------------------------------------------------
#
# Utility to convert "best available catalog", as provided by GNS for 
# Canterbury experiment of 2014, into ZMAP format. 
#
class NZCatalog (object):


    
    def __init__ (self):
        """ Initialization"""
    
        
    @staticmethod
    def toZMAP (csv_file,
                historical_file,
                result_file):
                
        """ This method converts CSV format of catalog into ZMAP format, 
            and saves newly formatted catalog to 'result_file'."""


        result_fhandle = CSEPFile.openFile(result_file,
                                           CSEPFile.Mode.WRITE)

        if historical_file:
            # Use historical events before 1950/1/1 for result catalog
            historical_np = CSEPFile.read(historical_file)
            
            year = historical_np[:, CSEPGeneric.Catalog.ZMAPFormat.DecimalYear].astype(np.int)
            
            selection = year < 1950
            
            np.savetxt(result_fhandle,
                       historical_np[selection, :])

        
        # Use "best available" catalog for recent events
        csv_np = CSEPFile.read(csv_file)
        nrows, ncols = csv_np.shape
        
        csv_np = np.append(csv_np, 
                           np.zeros((nrows, 3)),
                                     axis = 1)
        
        # Since errors are not provided, put NaN values for error fields
        # which will be replaced by default error values
        csv_np[:, CSVFormat.ERLAT] = 'NaN'
        csv_np[:, CSVFormat.ERZ] = 'NaN'
        csv_np[:, CSVFormat.ERMAG] = 'NaN'
        
        # There are seconds that are outside of [0..59] range, fix those
        seconds_selection = csv_np[:, CSVFormat.SECOND] >= 60.0
        csv_np[seconds_selection, CSVFormat.MINUTE] += 1.0
        csv_np[seconds_selection, CSVFormat.SECOND] -= 60.0

        # There are minutes that are outside of [0..59] range, fix those
        mins_selection = csv_np[:, CSVFormat.MINUTE] >= 60.0
        csv_np[mins_selection, CSVFormat.HOUR] += 1.0
        csv_np[mins_selection, CSVFormat.MINUTE] -= 60.0

        # Compute decimal year based on date and time of all catalog events
        catalog_date = np.array([datetime.datetime.combine(datetime.datetime.strptime('/'.join([str(int(event[CSVFormat.YEAR])),
                                                                                                str(int(event[CSVFormat.MONTH])),
                                                                                                str(int(event[CSVFormat.DAY]))]),
                                                                                      '%Y/%m/%d'),
                                                           datetime.time(int(event[CSVFormat.HOUR]),
                                                                         int(event[CSVFormat.MINUTE]),
                                                                         int(str(event[(CSVFormat.SECOND)]).split('.')[0]),
                                                                         int(str(event[CSVFormat.SECOND]).split('.')[1].ljust(6, '0'))))  for event in csv_np])        

        # Replace year entry in catalog with decimal year
        csv_np[:, CSVFormat.YEAR] = np.array([CSEPUtils.decimalYear(each_date) for each_date in catalog_date])

        # Save catalog in ZMAP format to the file
        np.savetxt(result_fhandle,
                   csv_np[:, (CSVFormat.LONG,
                              CSVFormat.LAT,
                              CSVFormat.YEAR,
                              CSVFormat.MONTH,
                              CSVFormat.DAY,
                              CSVFormat.MAGN,
                              CSVFormat.DEPTH,
                              CSVFormat.HOUR,
                              CSVFormat.MINUTE,
                              CSVFormat.SECOND,
                              CSVFormat.ERLAT,
                              CSVFormat.ERZ,
                              CSVFormat.ERMAG)])


# Invoke the module
if __name__ == '__main__':

     from CSEPOptions import CommandLineOptions
     from CSEPOptionParser import CSEPOptionParser

     parser = CSEPOptionParser()

     parser.add_option('--csvCatalog',
                       type='string',
                       dest='csv_catalog',
                       default=None,
                       help='CSV catalog to convert to ZMAP format')
     
     parser.add_option('--ZMAPCatalog',
                       type='string',
                       dest='zmap_catalog',
                       default=None,
                       help='Result ZMAP catalog')
     
     parser.add_option('--historicalCatalog',
                       type='string',
                       dest='historical_catalog',
                       default=None,
                       help='Historical catalog to append pre-1950/1/1 events from to the ZMAP catalog')
     
     # List of requred options for the test
     required_options = ['--csvCatalog', 
                         '--ZMAPCatalog']

     options = parser.options(required_options)
     
     catalog = NZCatalog()
     
     catalog.toZMAP(options.csv_catalog,
                    options.historical_catalog,
                    options.zmap_catalog)
     