/* Okada
 *
 * - to calculate Coulomb stress change based probability map
 *
 *
 * INPUT: pscmpfile
 *        depth_min [km]
 *        depth_max [km]
 *        ddepth [km]
 *        no_mechanisms
 *        uncertainty [°]
 *
 * OUTPUT: lat/ lon/ probability density  --> outputfile name taken from pscmpfile
 *
 *
 * compilation flags: math library (-lm)
 *                    openmp (-lopenmp)   --> to parallelize the code, if not used, code will run in one thread only
 *
 */


#include <stdio.h>
#include <math.h>
#include <string.h>

#include "defines.h"
#include "pscokada.h"
#include "nrutil.h"
#include "read_pscmpfile.h"
#include "coord_trafos.h"
#include "c_recipes.h"




double dist(double clat1, double clon1, double clat2, double clon2){
  double pig, alat1, along1, rp1, x1, y1, z1, alat2, along2, rp2, x2, y2, z2;
  double valo, dummy, cang, reff, dist;

  pig   = asin(1.0) * 2.0;
  alat1 = clat1 * pig/180.0;
  along1= clon1 * pig/180.0;
  rp1 = 6378.14 * (1.0 - (1.0/298.25) * sq(sin(alat1)));
  x1 = rp1 * cos(alat1) * cos(along1);
  y1 = rp1 * cos(alat1) * sin(along1);
  z1 = rp1 * sin(alat1);
  alat2  = clat2 * pig/180.0;
  along2 = clon2 * pig/180.0;
  rp2 = 6378.14 * (1.0 - (1.0/298.25) * sq(sin(alat2)));
  x2 = rp2 * cos(alat2) * cos(along2);
  y2 = rp2 * cos(alat2) * sin(along2);
  z2 = rp2 * sin(alat2);
/* c */
/* c calcola l'angolo tra le 2 semirette passanti per il centro */
/* c */
  valo = (x1*x2 + y1*y2 + z1*z2) / sqrt((sq(x1) + sq(y1) + sq(z1)) * (sq(x2) + sq(y2) + sq(z2)));
  dummy=fabs(valo)-1.0;
  //if (dummy>0){ printf("*** Problemi nella subroutine DIST ***\n valo=%lf\n",valo); exit(1);}
     
  cang = acos(valo);
  reff = sqrt(rp1*rp2);
  dist = reff*cang;
  return dist;
}



int main(){
  FILE *fout, *fin;
  char pscmpfile[CMAX], outfolder[CMAX], outfile[CMAX], probabilityFile[2*CMAX], gridfile[160], resfile[160], sdum[60];
  
  double **segdata, **patchdata, **observation, *probability, sumprob=0;
  double **receivermechanisms, *gridlat, *gridlon;
  
  double obs_lat_min, obs_lat_max, obs_lon_min, obs_lon_max, ref_lat, ref_lon;
  double sigma1, sigma2, sigma3;
  double dlat, dlon, segment_ref_north, segment_ref_east, patch_length, patch_width;
  double patch_east, patch_north, patch_depth, obs_depth, DCFS, alpha, lambda, mu, rho, vp, vs;
  double friction, skempton, sig, strike_pscmp, dip_pscmp, rake_pscmp, uncertainty;
  double porepressure, sxx, syy, szz, sxy, syz, szx, sum_sxx, sum_syy, sum_szz, sum_sxy, sum_syz,  sum_szx;
  double dum, dum_segment_strike, dum_segment_dip, dum_slip_strikedirection, dum_slip_dipdirection;
  double obs_north_min, obs_north_max, obs_east_min, obs_east_max;
  double depth_min, depth_max, ddepth, L, W;
  int i, j, k, l, m, ii, count, obs_count, no_segs, *no_patches, Ngrid;
  int no_patches_total = 0, no_mechanisms;
  int warnung, idum;
  long seed;
  
  double DCFSmax = 5000000; // upper cutoff for CFS change considered for probability maps (50^6Pa = 50 bar)
  

  
  // negative integer number to initialize random generator
  seed = -1;
  dum = gasdev(&seed);
  
  /**************************************************************************/
  /**************************************************************************/
  
  /********** INPUT **********/
  //printf("\n\nSpecify PSCMP input file:");
  warnung = scanf("%s", pscmpfile);
  if (warnung==0 || warnung==EOF)
    fprintf(stderr, "\nERROR: scanf Input\n");

   //printf("\n\nSpecify Input Grid-file:");
  warnung = scanf("%s", gridfile);
  if (warnung==0 || warnung==EOF)
    fprintf(stderr, "\nERROR: scanf Input\n"); 

   //printf("\n\nSpecify OUTPUT file:");
  warnung = scanf("%s", resfile);
  if (warnung==0 || warnung==EOF)
    fprintf(stderr, "\nERROR: scanf Input\n"); 

  //printf("\n\nObservation depth minimum [km]:");
  warnung = scanf("%lf", &depth_min);
  if (warnung==0 || warnung==EOF)
    fprintf(stderr, "\nERROR: scanf Input\n");
  
  //printf("\n\nObservation depth maximum [km]:");
  warnung = scanf("%lf", &depth_max);
  if (warnung==0 || warnung==EOF)
    fprintf(stderr, "\nERROR: scanf Input\n");
  
  //printf("\n\nObservation depth step [km]:");
  warnung = scanf("%lf", &ddepth);
  if (warnung==0 || warnung==EOF)
    fprintf(stderr, "\nERROR: scanf Input\n");
  
  //	printf("\nMean receiver fault mechanism taken from pscmp file!\n"
  //		   "Use how many receiver fault mechanisms (>= 1, if >1, strike, dip, rake with Gaussian distribution)?");
  warnung = scanf("%d", &no_mechanisms);
  if (warnung==0 || warnung==EOF)
    fprintf(stderr, "\nERROR: scanf Input\n");
  
  //printf("\n\nStandard deviation of strike/dip/rake values [°]:");
  warnung = scanf("%lf", &uncertainty);
  if (warnung==0 || warnung==EOF)
    fprintf(stderr, "\nERROR: scanf Input\n");
  
  
  /**************************************************************************/
  
  
  // read pscmp file
  // segment header, patch data, number segments and patches, observation geometry
  //
  // Segment Spezifikationen
  // lat   lon    depth length width strike dip   np_st np_di start_time
  // [deg] [deg]  [km]  [km]   [km]  [deg]  [deg] [-]   [-]   [day]
  //
  // Patch Spezifikationen
  // pos_s   pos_d    slp_stk slp_dip open
  // [km]    [km]     [m]     [m]     [m]
  read_pscmpfile(pscmpfile, &segdata, &patchdata, &no_segs, &no_patches,
		 &friction, &skempton, &strike_pscmp, &dip_pscmp, &rake_pscmp,
		 &sigma1, &sigma2, &sigma3, &lambda, &mu);
    
  
  // crustal properties
  porepressure = 0.0;
  alpha = (lambda + mu) / (lambda + 2 * mu);

  printf("\n\t Parameters read: friction=%.2lf  lambda=%.1e  mu=%.1e  alpha=%.2lf  skempton=%.1lf strike_pscmp=%.1lf dip_pscmp=%.1lf rake_pscmp=%.1lf\n\n",friction,lambda,mu,alpha,skempton,strike_pscmp,dip_pscmp,rake_pscmp);

  // receiver fault mechanisms (strike,dip,rake)
  receivermechanisms = dmatrix(1,no_mechanisms,1,3);
  
  // create receiver fault mechanisms
  // <=0 --> ERROR
  // ==1 --> calculate for one specific mechanism
  // >1  --> use standard deviation for mechanisms
  if (no_mechanisms<=0) {
    fprintf(stderr, "\nERROR: number of receiver fault mechanisms <=0 (%d)!!!\n\n", no_mechanisms);
  }
  else {
    for(i=1; i<=no_mechanisms; i++) {
      receivermechanisms[i][1] = strike_pscmp + gasdev(&seed)*uncertainty;
      receivermechanisms[i][2] = dip_pscmp    + gasdev(&seed)*uncertainty;
      receivermechanisms[i][3] = rake_pscmp   + gasdev(&seed)*uncertainty;
    }
  }
  
  
  
  /**************************************************************************/
  /**************************************************************************/
  // create observation array using observation borders
  // from pscmp file and number of observation points
  
  // READ GRID-FILE
  // read  Ngrid, lat[i], lon[i], dlat, dlon
  fin=fopen(gridfile,"r");
  fscanf(fin,"%s",&sdum);   fscanf(fin,"%d", &Ngrid);   
  gridlat = dvector(1,Ngrid);
  gridlon = dvector(1,Ngrid);
  
  ref_lat = 0.0;        // reference lat/lon (necessary for transformation to local coordinate system)
  ref_lon = 0.0;

  for(i=1;i<=Ngrid;i++){ 
    fscanf(fin,"%d",&idum);  
    fscanf(fin,"%lf",&gridlon[i]); ref_lon += gridlon[i];
    fscanf(fin,"%lf",&gridlat[i]); ref_lat += gridlat[i];
    fscanf(fin,"%lf",&dum); 
  }
  ref_lat /= 1.0*Ngrid;
  ref_lon /= 1.0*Ngrid;

  printf("--> ref-lat=%.4lf  ref-lon=%.4lf\n",ref_lat,ref_lon);
 
  fclose(fin);
  // *********************************************************************************************
   
  // total number of slip patches
  for (i=1;i<=no_segs;i++) no_patches_total += no_patches[i];
  
  
  /**************************************************************************/
  /**************************************************************************/
  
  // observation coordinates
  // lat, lon, north, east
  observation = dmatrix(1,4+no_mechanisms,1,Ngrid);
  probability = dvector(1,Ngrid);
  
  count = 0;
  
  // create observation coordinate points
  for (i=1;i<=Ngrid;i++) {
    observation[1][i] = gridlat[i];
    observation[2][i] = gridlon[i];
    latlon2localcartesian(observation[1][i], observation[2][i], ref_lat, ref_lon, &observation[3][i], &observation[4][i]);
    observation[5][i] = 0.0;
  }
  
  
  /**************************************************************************/
  /**************************************************************************/
  
#pragma omp parallel for						\
    private(j,obs_count,obs_depth,sum_sxx,sum_syy,sum_szz,sum_sxy,sum_syz,sum_szx,\
            k,segment_ref_north,segment_ref_east,patch_length,patch_width,\
            l,count,ii,patch_east,patch_north,patch_depth,sxx,syy,szz,sxy,syz,szx,\
            dum_segment_strike,dum_segment_dip,dum_slip_strikedirection,dum_slip_dipdirection,\
            m,DCFS,sig) \
    reduction(+:sumprob)
    // loop over observation points
    for (i=1;i<=Ngrid;i++) {

      obs_count = i;

      for (obs_depth=depth_min; obs_depth<=depth_max; obs_depth+=ddepth) {

	// set total stress change to zero at the beginning of the calculation for every grid point
	sum_sxx = 0.0;
	sum_syy = 0.0;
	sum_szz = 0.0;
	sum_sxy = 0.0;
	sum_syz = 0.0;
	sum_szx = 0.0;
	
	// loop over segments
	for (k=1;k<=no_segs;k++) {
	  
	  // segment reference point to local cartesian coordinate system
	  latlon2localcartesian(segdata[k][1], segdata[k][2], ref_lat, ref_lon, &segment_ref_north, &segment_ref_east);
	  
	  // length of one patch
	  // = segment length divided by number of patches in strike direction
	  patch_length = segdata[k][4]/segdata[k][8];
	  // width of one patch
	  // = segment width divided by number of patches in dip direction
	  patch_width  = segdata[k][5]/segdata[k][9];
	  
	  
	  for (l=1;l<=no_patches[k];l++){
	    count = l;
	    if(k>1) for (ii=1;ii<k;ii++) count += no_patches[ii];
	    
	    // calculate absolute patch coordinates
	    patch_north = segment_ref_north + cos(segdata[k][6]*DEG2RAD)*patchdata[count][1] + cos((segdata[k][6]+90)*DEG2RAD)*cos(segdata[k][7]*DEG2RAD)*patchdata[count][2];
	    patch_east  = segment_ref_east  + sin(segdata[k][6]*DEG2RAD)*patchdata[count][1] + sin((segdata[k][6]+90)*DEG2RAD)*cos(segdata[k][7]*DEG2RAD)*patchdata[count][2];
	    patch_depth = segdata[k][3] + sin(segdata[k][7]*DEG2RAD)*patchdata[count][2];
	    
	    sxx=syy=szz=sxy=syz=szx=0.0;
	    dum_segment_strike = segdata[k][6];
	    dum_segment_dip    = segdata[k][7];
	    dum_slip_strikedirection = patchdata[count][3];
	    dum_slip_dipdirection = patchdata[count][4];
	    
	    // calculate stress tensor at observation point
	    // INPUT:
	    // patch north, east, depth, strike, dip, rake,
	    // patch length, width, slip in strike direction, slip in dip direction
	    // observation north, east, depth
	    // stress tensor, alpha, lambda, mu, friction
	    pscokada(patch_north, patch_east, patch_depth, dum_segment_strike, dum_segment_dip,
		     patch_length, patch_width, dum_slip_strikedirection, dum_slip_dipdirection,
		     observation[3][obs_count], observation[4][obs_count], obs_depth,
		     &sxx, &syy, &szz, &sxy, &syz, &szx, alpha, lambda, mu, friction); //patch_rake is not used, slip in strike and dip-direction contains the same information
	    
	    // sum up stress tensors at the same grid point for the contribution of all slip patches
	    sum_sxx += sxx;
	    sum_syy += syy;
	    sum_szz += szz;
	    sum_sxy += sxy;
	    sum_syz += syz;
	    sum_szx += szx;
	    
	  } // end loop over patches
	  
	} // end loop over segments
	
	// calculate coulomb stress change for every receiver mechanism
	for (m=1; m<=no_mechanisms; m++){
	  DCFS=sig=0.0;
	  
	  cmbfix(sum_sxx, sum_syy, sum_szz, sum_sxy, sum_syz, sum_szx, porepressure, friction,
		 &DCFS, &sig, receivermechanisms[m][1], receivermechanisms[m][2], receivermechanisms[m][3]);
	  observation[5 + (m-1)][obs_count] = DCFS;
	  
	  /*******************/
	  /* probability map */
	  /*******************/
	  // values <= 0 are not considered
	  if (DCFS>0) {
	    // set upper limit because of strong heterogeneities close to the fault
	    if(DCFS>DCFSmax) {
	      probability[obs_count] += DCFSmax;
	      sumprob                += DCFSmax;
	    }
	    else {
	      probability[obs_count] += DCFS;
	      sumprob                += DCFS;
	    }
	  }
	  /*******************/
	  /*******************/
	} // end loop over receiver mechanisms
	
      } // end loop over depth layers
      
    } // end loop over gridpoints


    
    //dum = sumprob*area_per_point;
    dum = sumprob;
    for(i=1;i<=Ngrid;i++) probability[i] /= dum;

    
    
    /**************************************************************************/
    /******************************** OUTPUT **********************************/
    /**************************************************************************/
    
    // save probability data
    fout = fopen(resfile, "w");
    fprintf(fout, "lat               lon                probability density\n");
    for(i=1;i<=Ngrid;i++) fprintf(fout, "%-16.9lf   %-16.9lf   %-20.12lf   \n", observation[1][i], observation[2][i], probability[i]); 
    fclose(fout);
     
    
    free_dmatrix(observation,1,4+no_mechanisms,1,Ngrid);
    free_dmatrix(patchdata,1,no_patches_total,1,5);
    free_dmatrix(segdata,1,no_segs,1,10);
    free_ivector(no_patches,1,no_segs);
    free_dmatrix(receivermechanisms, 1, no_mechanisms, 1, 3);
    free_dvector(probability, 1, Ngrid);
    
    
    
    
    return(0);

}
