/*
 * read_pscmpfile.c
 *
 *  Created on: Nov 2, 2010
 *      Author: bach
 *
 *      reads pscmp file
 *
 *      variables: filename, segment data matrix (Nx10), patch data matrix (Nx5),
 *                 number segments, number patches per segment (N)
 *
 *      Achtung: Pointer zu nicht initialisierten Matrizen/Vektoren muessen uebergeben werden,
 *               da diese dann innerhalb der Funktion angelegt werden.
 *               Diese Matrizen müssen am Ende des aufrufenden Programms wieder freigegeben werden.
 *
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "nrutil.h"

int read_pscmpfile(char *pscmpfile, double ***segment_data, double ***patch_data, int *no_segments, int **no_patches,
		double *friction, double *skempton, double *strike, double *dip, double *rake,
		   double *sigma1, double *sigma2, double *sigma3, double *lambda, double *mu) {

  FILE    *fin;
  char    cdum[256], *cerror, *cdum2, *cdum3;
  int     obs_type, no_outputs, segment, patches = 0, sum_no_patches = 0, i, j;
  fpos_t  pos;


  // open input file
  fin = fopen(pscmpfile, "r");
  if (fin == NULL) {
    fprintf(stderr, "\nError opening pscmp file %s\n\n", pscmpfile);
    exit(1);
  }

  
  //	printf("\nBeginn read_pscmpfile\n");
  cerror = fgets(cdum, 256, fin);	
  while(cdum[0] == '#') cerror = fgets(cdum, 256, fin);
  sscanf(cdum, "%lf %lf",lambda,mu);
    
  // Kommentarzeilen ueberspringen
  cerror = fgets(cdum, 256, fin);
  while(cdum[0] == '#') cerror = fgets(cdum, 256, fin);
  
  if (cerror == NULL) fprintf(stderr, "Error reading from pscmpfile!\n");
  
  sscanf(cdum, "%*d %lf %lf %lf %lf %lf %lG %lG %lG", friction, skempton, strike, dip, rake, sigma1, sigma2, sigma3);

  
  // Kommentarzeilen ueberspringen
  cerror = fgets(cdum, 256, fin);
  while(cdum[0] == '#') cerror = fgets(cdum, 256, fin);
  
  
  // number of slip segments
  sscanf(cdum, "%d", no_segments);
  
  cerror = fgets(cdum, 256, fin);
  while (cdum[0] == '#') {
    fgetpos(fin, &pos);
    cerror = fgets(cdum, 256, fin);
  }
  // Position zurueck setzen auf Zeilenanfang
  fsetpos(fin, &pos);
  
  // Vektor fuer Segmentdaten, pro Zeile ein Segment
  *segment_data = dmatrix(1, *no_segments, 1, 10);
  *no_patches = ivector(1, *no_segments);
  
  
  
  /************************************************************************/
  /*                                                                      */
  /*                  Read Segment and Patch data                         */
  /*                                                                      */
  /************************************************************************/
  
  // read Segment data and determine amount of patches
  for (i = 1; i <= *no_segments; i++) {
    // Segment Spezifikationen
    // lat   lon    depth length width strike dip   np_st np_di start_time
    // [deg] [deg]  [km]  [km]   [km]  [deg]  [deg] [-]   [-]   [day]
    cerror = fgets(cdum, 256, fin);
    sscanf(cdum, "%d %lf %lf %lf %lf %lf %lf %lf %lf %lf %lf",
	   &segment, &(*segment_data)[i][1], &(*segment_data)[i][2], &(*segment_data)[i][3],
	   &(*segment_data)[i][4], &(*segment_data)[i][5], &(*segment_data)[i][6],
	   &(*segment_data)[i][7], &(*segment_data)[i][8], &(*segment_data)[i][9], &(*segment_data)[i][10]);
    (*no_patches)[i] = (int) ((*segment_data)[i][8] * (*segment_data)[i][9]);
    sum_no_patches += (*no_patches)[i];
    
    for (j = 1; j <= (*no_patches)[i]; j++) {
      cerror = fgets(cdum, 256, fin);
    }
  }
  
  // Matrix fuer Patch data erzeugen
  *patch_data = dmatrix(1, sum_no_patches, 1, 5);
  // File Pointer zuruecksetzen auf Beginn der Daten
  fsetpos(fin, &pos);
  patches = 0;
  
  for (i = 1; i <= *no_segments; i++) {
    // Segment Daten ueberspringen, da bereits eingelesen
    cerror = fgets(cdum, 256, fin);
    
    // Patch Spezifikationen
    // pos_s   pos_d    slp_stk slp_dip open
    // [km]    [km]     [m]     [m]     [m]
    for (j = 1; j <= (*no_patches)[i]; j++) {
      patches++;
      cerror = fgets(cdum, 256, fin);
      sscanf(cdum, "%lf %lf %lf %lf %lf", &(*patch_data)[patches][1], &(*patch_data)[patches][2], &(*patch_data)[patches][3], &(*patch_data)[patches][4], &(*patch_data)[patches][5]);
    }
  }
  
  if (sum_no_patches != patches) {
    printf("\nAnzahl Patches nicht konsistent!\n\n");
    exit(2);
  }
  
  printf("\n******************************************************************************************\n");
  printf("* Reading pscmp file successful:                                                         *\n");
  printf("* %-87s*\n", pscmpfile);
  printf("*                                                                                        *\n");
  printf("* number of segments = %-10d number of patches  = %-10d                        *\n", *no_segments, sum_no_patches);
  printf("******************************************************************************************\n");
  
  fclose(fin);
  return (0);
  
}






