#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include "recipes/nrutil.c"
#include "recipes/ran1.c"
#include "recipes/files.h" 
#include "recipes/einlese.c"
#include "recipes/sortd.c"
#include "recipes/indexxd.c"
#include "recipes/julday.c"

#define log10(A) (log(A)/log(10))
#define sq(A) ((A)*(A))
#define sign(A) ((int) (A/fabs(A)))
#define eps0 1.0e-3

static int     setRETAS;                       //  if=1  RETAS-model; else ETAS-model
static int     useMainshockprobmaps;           //  if=1  probability-gridmaps are use for mainshocks, if available 
static int     spacebackground;                //  if=1  space-dependent background probability map is used if available; else constant rate is assumed
static int     setdt;                          //  if=1  incomplete times after mainshocks are not used (California-relation) 
static int     setprint;                       //  if=1  intermediate LL-values will be printed in LL-maximization routine (for testing)

static double  **tc;                           //  NK time periods of completeness [ tc[i][1] -  tc[i][2] ]
static int     NK;
static double  **probmain;                     //  probability-maps for direct aftershocks for NSLIP mainshocks based on Coulomb-Stress calculations
static double  **probmaindist;                 //  probability-maps for direct aftershocks for NSLIP mainshocks based on nearest fault-distance relation
static int     *indmain;                       //  index of the earthquake (tall, mall, latall, lonall) corresponding to probability-map-index 
static int     NSLIP;
static double  *probback;                      //  probability-map  for background events
static double  *gridlat, *gridlon, *dvol;      //  Ngrid latitude, longitude and cell-area [km2] of grid points for parameter estimation and forecast
static int     Ngrid;
static double  dlat, dlon;                     // corresponding (mean) gridspacing
static double  *t, *m, *lat, *lon;             //  i=1,...Z: Earthquakes for parameter estimation 
static int     Z;
static int     *indgrid;                       //  corresponding gridpoint of these events
static double  *tall, *latall, *lonall, *mall; //  i=1,...Zall: All earthquakes used for earthquake rate calculation
static int     Zall;
static int     *indgridall;                    //  corresponding gridpoint of these events
static int     *insidegrid;                    //  if=1, then event is inside LL-space                     
static double  *xweightD, *xweightDL;          //  spatial weight of aftershock triggering inside LL-space related to event i=1,...Zall: D with constand d and DL with d=d0*L(M)
static double  **spaceweightall;               //  [j][i] defines the fraction of aftershocks expected to be triggered by event j in the LL-spatial-grid at time of event i

static int     set_latlon;                     //  if =1 lat, lon  (latall, lonall) are given in [degrees]; else in [km]
static int     *setmain;                       //  index of the probability-map of the corresponding tall-mall-latall-lonall-earthquake (=0 if no slip-map exists)
static int     *Nepi;                          //  i=1,... Zall: Number of identified direct aftershocks
static int     **Indexepi;                     //  i=1,..., Zall  j=1,...Nepi:  Indexes of these direct aftershocks k=Indexepi[i][j]  

static double  T1, T2, T;                      // definition of time intervals for parameter estimation [T1 T2] and forecasting [T2  T2+T]
static int     Nlargest;                       //  the Nlargest-magnitude values in [Tlargestmin  T2] are optimized  
static double  Mcut, Mcutpredict;              // minimum magnitude cutoff for parameter estimation (Mcut) and forecast (Mcutpredict)
static double  Tafter;                         // maximum duration of direct aftershock triggering --> p<1 values allowed!
static double  Mmin, Mstar, C, Cstar, alpha0;  // help-parameters for the RETAS model
static double  locerror;                       // location error = minimum value for the dL(M)-value 
static double  maxdepth;                       // [km] maximum considered depth of the earthquakes in the catalog


// internal parameters:
static double  vol, bvalue;                    // total-area and b-value
static double  MUvalue, Cvalue, Pvalue, M0value, N0value, ALPHAvalue, Qvalue, Dvalue, D0value;
static int     *set;                           //  if=1 then the corresponding parameter is estimated
static char    catalogfile[160], resfile[160];

#define dlonmax     0.051                       // maximum longitude gridspacing   
#define dlatmax     0.051                       // maximum latitude  gridspacing   
#define NDLmin     10                           // spatial distribution calculated with fixed d-value if N>NDLmin spatial-mainshock-sources; otherwise with d=d0*L(M)
#define dTmin     365.0                         // minimum length of time interval [T1 T2] for parameter estimation
#define Nmax      500                           // maximum number of events in the LL-space (T1 will be adopted accordingly)
#define Zmin1      50                           // minimum number of events in LLspace to start parameter estimation (b-value & trigger parameter)
#define Zmin2     100                           // minimum number of events in LLspace to start full parameter estimation; else only bvalue, mu & N0/M0 estimated 
#define Nsmoothmin 20                           // minimum number too smooth declustered events for estimation space-dependent background; else use uniform background

#include "calculatejulday.c"
#include "distance.c"
#include "calculate_timeintegral.c"
#include "loglike_nospace.c"
#include "loglike_mainshockdistribution.c"
#include "loglike_nospace_magoptimization.c"
#include "loglike_nospace_directinput.c"
#include "calculatetransformedtime_dtcomplete.c"
#include "mmaxvalue.c"
#include "davidn_nospace.c"
#include "davidn.c"
#include "readinput+catalog.c"
#include "calculatespaceweights.c"
#include "estimation_parameter.c" 
#include "decluster.c"
#include "simulationforecast.c"


int readinputfile(char *input_fname, char *farfalleheaderfile, char *slipfilelistorg, char *templatefile){
  
  /* input: file name input_fname
   *
   * output:
   * 		outname: output file
   * 		T2, T: start and length of forecast period in days
   * 		farfalleheaderfile: input file for farfalle code (general file with info about the crust etc).
   * 		templatefile: forecast template
   * 		catalogfile: catalog
   * 		slipmodellist: file containing a list of slip models 
   *
   * NB: all pointers will be ignored if NULL is passed; otherwise, char* should already be initialized.
   */
  
  FILE    *fin;
  int     Nchar=200;
  char    line[Nchar], *p;
  char    *key, *value;
  int     NP=7, i, n, yy, mm, dd, hh, min, sec;
  int     value_found[NP];
  double  tstart, tend;

  char *keys[]={
    /*0*/	"ForecastStartDate",	\
    /*1*/	"ForecastEndDate",	\
    /*2*/	"OutputForecastFile",	\
    /*3*/	"InputCatalogFile",		\
    /*4*/	"ForecastTemplate",		\
    /*5*/	"InputCoulombFile",		\
    /*6*/	"InputListSlipModels",		\
  };
  
  for (n=0;n<NP;n++) value_found[n]=0;



  if((fin = fopen(input_fname, "r"))==NULL) {
    fprintf(stderr, "Error read_input: unable to open input file %s.\n", input_fname);
    return 1;
  }

  while(!feof(fin)) {
    fgets(line,Nchar,fin);
    if (ferror(fin)) {
      printf("Error reading input data using fgets!\n");
      return 1;
    }

    p=strpbrk(line,"=");
    if(p == NULL) goto end;
   

      key=strtok(line,"=");
    value=strtok(NULL,"=");

    i=0;
    while (i<NP && strcmp(key,keys[i])) i++;
    if (i>=NP){
      fprintf(stderr, "Warning read_input: parameter \" %s\" in file \"%s\" not recognized.\n", value, input_fname);
      continue;
    }
    value_found[i]=1;

    switch(i){
    case 0:
      sscanf(value, "%d-%d-%dT%d:%d:%dZ", &yy, &mm, &dd, &hh, &min, &sec);
      calculatejulday(yy,mm,dd,hh,min,1.0*sec,&tstart);
      T2=tstart;
      break;
    case 1:
      sscanf(value, "%d-%d-%dT%d:%d:%dZ", &yy, &mm, &dd, &hh, &min, &sec);
      calculatejulday(yy,mm,dd,hh,min,1.0*sec,&tend);
      T=tend-tstart;
      break;
    case 2:
      if (resfile) sscanf(value,"%s",resfile);
      break;
    case 3:
      if (catalogfile) sscanf(value,"%s",catalogfile);
      break;
    case 4:
      if (templatefile) sscanf(value,"%s",templatefile);
      break;
    case 5:
      if (farfalleheaderfile) sscanf(value,"%s",farfalleheaderfile);
      break;
    case 6:
      if (slipfilelistorg) sscanf(value,"%s",slipfilelistorg);
      break;
    }
  }
 end:
  fclose(fin);

  for (n=0;n<NP;n++) {
    if (!value_found[n]) {
      switch (n){
	// key 6 are optional; all the others are compulsory, and error is returned if missing.
	
      case 6:
	printf("Warning: parameter %s not given in %s.\n\n", keys[n], input_fname);
	if (slipfilelistorg) strcpy(slipfilelistorg,""); 
	useMainshockprobmaps=0;
	break;
      default:
	printf("Error: parameter %s not given in %s.\n", keys[n], input_fname);
	return 1;
	break;
      }
    }
  }
  
  if((fin = fopen(slipfilelistorg, "r"))==NULL) useMainshockprobmaps=0;
  return 0;
  
}


int readwrite_slipmodellist(char *inputlist, char *outputlist){
  FILE   *fin, *fout;
  int    Nchar=200;
  char   line[Nchar], sdum[100], file[200];
  double teps, mageps, time, mag;
  int    i, ii, n, yy, mm, dd, hh, min, sec, sett, NSLIP1;

  teps   = 30.0;  // [min]  allowed maximum time difference to be identified with catalog event 
  mageps = 0.5;   // maximum allowed magnitude difference to be  " " "
 
  //printf("inputlist=%s\n outputlist=%s\n\n",inputlist,outputlist);

  fin=fopen(inputlist,"r");
  if (fin==NULL) return (1);
  
  line[0]='#';
  while (line[0]=='#') fgets(line,Nchar,fin); // title lines
  sscanf(line,"%d",&NSLIP1);
  
  NSLIP=0;
  for(i=1;i<=NSLIP1;i++){
    fscanf(fin,"%s",&sdum);
    sscanf(sdum, "%d-%d-%dT%d:%d:%dZ", &yy, &mm, &dd, &hh, &min, &sec);
    calculatejulday(yy,mm,dd,hh,min,1.0*sec,&time);
    fscanf(fin,"%lf",&mag);  
    fscanf(fin,"%s",&file);  
    if(time>T2)  printf("\n\n Slipmodel %s   (t=%lf  mag=%.2lf) not used because event occurred after Forecast start-time T2=%lf!!\n\n",file,time,mag,T2);
    else NSLIP++;
  }
  fclose(fin);

  fout=fopen(outputlist,"w");
  fprintf(fout,"%d\n",NSLIP);
  indmain = ivector(1,NSLIP);  // index of the earthquake in the catalog which corresponds to probability-map

  fin=fopen(inputlist,"r");
  line[0]='#';
  while (line[0]=='#') fgets(line,Nchar,fin); // title lines
  sscanf(line,"%d",&NSLIP1);

  ii=0;
  for(i=1;i<=NSLIP1;i++){
    fscanf(fin,"%s",&sdum);
    sscanf(sdum, "%d-%d-%dT%d:%d:%dZ", &yy, &mm, &dd, &hh, &min, &sec);
    calculatejulday(yy,mm,dd,hh,min,1.0*sec,&time);
    fscanf(fin,"%lf",&mag);
    fscanf(fin,"%s",&file);
    if(time<=T2){
      ii++;
      sett=0;
      for(n=1;n<=Zall;n++) if(fabs(tall[n]-time)<teps && fabs(mall[n]-mag)<mageps){ indmain[ii]=n; setmain[n]=ii; sett=1; }
      if(sett==0){  printf("\n\n\t ATTENTION: Corresponding %d.slip-event  (t=%lf  mag=%lf) not found in the catalog!!\n\n",ii,time,mag); exit(1); }
      fprintf(fout,"%d\t%d\t%s\n",ii,indmain[ii],file);
    }
  }
  fclose(fin);
  fclose(fout);
}


void readtemplate(char *inputfile, char *outname)
{
  FILE    *fin, *fout; 
  double  **res, *lathelp, *lonhelp, *volhelp;
  double  lon1, lon2, lat0, lat1, lat2, lon10, lon20, lat10, lat20, mag1, mag2, dep1, dep2, dum, ddlon, ddlat, L, W;
  int     *indlat, *indlon, i, ii, k, kk, j, count, idum, ans, lonfac, latfac;
  long    Zmax;

  set_latlon=1; /*  1= Koordinates are in Lat Lon */

  maxdepth   = 0.0;    // [km] maximum considered depth of the earthquakes in the catalog
  Mcutpredict=10.0;    // minimum Magnitude for the forecasts

  fin=fopen(inputfile,"r");
  ans=1;

  Zmax = 1000000; 
  res  = dmatrix(1,3,1,Zmax);

  Ngrid = 0;
  count = 0;
  vol   = 0.0;
  dlon  = 0.0;
  dlat  = 0.0; 
  while(ans!=EOF){
    ans=fscanf(fin,"%lf",&lon1);  fscanf(fin,"%lf",&lon2); 
    ans=fscanf(fin,"%lf",&lat1);  fscanf(fin,"%lf",&lat2);

    if(lon1!=lon10 || lon2!=lon20 || lat1!=lat10 || lat2!=lat20){
  
      lon10=lon1; lon20=lon2; lat10=lat1; lat20=lat2;

      ddlon = lon2-lon1;
      ddlat = lat2-lat1;
      //printf("ddlon=%.3lf  ddlat=%.3lf ",ddlon,ddlat);

      lonfac = (int) (1+ddlon/dlonmax);
      latfac = (int) (1+ddlat/dlatmax);

      ddlon /= 1.0*lonfac;
      ddlat /= 1.0*latfac;
      //printf("--> ddlon=%.3lf  ddlat=%.3lf  lonfac=%d  latfac=%d\n",ddlon,ddlat,lonfac,latfac);
     
      count++;
      dlon += ddlon;
      dlat += ddlat;
      
      for(i=1;i<=lonfac;i++) for(j=1;j<=latfac;j++){ 
	Ngrid++;
	res[1][Ngrid] = lon1 + (i-0.5)*ddlon;
	res[2][Ngrid] = lat1 + (j-0.5)*ddlat;

	L = distance(res[2][Ngrid]-0.5*ddlat,res[1][Ngrid],res[2][Ngrid]+0.5*ddlat,res[1][Ngrid]);
	W = distance(res[2][Ngrid],res[1][Ngrid]-0.5*ddlon,res[2][Ngrid],res[1][Ngrid]+0.5*ddlon);
	res[3][Ngrid] = L*W;
	vol += L*W;
      }
    }
 
    ans=fscanf(fin,"%lf",&dep1);  fscanf(fin,"%lf",&dep2); 
    ans=fscanf(fin,"%lf",&mag1);  fscanf(fin,"%lf",&mag2); 
    ans=fscanf(fin,"%lf",&dum);   fscanf(fin,"%d",&idum); 
  
    if(dep2>maxdepth)    maxdepth=dep2;
    if(mag1<Mcutpredict) Mcutpredict=mag1;
  }
  fclose(fin);

  dlon /= 1.0*count;
  dlat /= 1.0*count;

  printf("\n Template %s read  --> maxdepth=%.1lf  Mcutpredict=%.2lf  Ngrid=%d  vol=%.2lf [km2]  average: dlon=%.2lf  dlat=%.2lf\n\n",inputfile,maxdepth,Mcutpredict,Ngrid,vol,dlon,dlat);

  gridlat  = dvector(1,Ngrid);
  gridlon  = dvector(1,Ngrid);
  dvol     = dvector(1,Ngrid);
  probback = dvector(1,Ngrid);
  lathelp  = dvector(1,Ngrid);  
  lonhelp  = dvector(1,Ngrid);  
  volhelp  = dvector(1,Ngrid);
  indlat   = ivector(1,Ngrid);
  indlon   = ivector(1,Ngrid);

  for(i=1;i<=Ngrid;i++) lathelp[i]=res[2][i];
  indexxd(Ngrid,lathelp,indlat);

  count=0; j=0;
  for(i=1;i<=Ngrid;i++){
    ii=indlat[i];   
    lat1=res[2][ii];
    if(i==1) lat0=lat1;

    if(lat1!=lat0){
      indexxd(count,lonhelp,indlon);
      for(k=1;k<=count;k++){
	kk=indlon[k];
	j++;
	gridlon[j]=lonhelp[kk];
	gridlat[j]=lat0;
	dvol[j]   =volhelp[kk];
      }
      lat0=lat1;
      count=1;
      lonhelp[count]=res[1][ii];
      volhelp[count]=res[3][ii];
    }
    else{
      count++;
      lonhelp[count]=res[1][ii];
      volhelp[count]=res[3][ii];
    }
  }
  indexxd(count,lonhelp,indlon);
  for(k=1;k<=count;k++){
    kk=indlon[k];
    j++;
    gridlon[j]=lonhelp[kk];
    gridlat[j]=lat0;
    dvol[j]   =volhelp[kk];
  }
  if(j!=Ngrid){ printf("\n\n\t ATTENTION j=%d !=! Ngrid=%d  ... exit\n\n",j,Ngrid); exit(1); }

  sprintf(outname,"output/gridpoints.dat");
  fout=fopen(outname,"w");
  fprintf(fout,"#_Ngrid= %d\n",Ngrid);
  for(i=1;i<=Ngrid;i++) fprintf(fout,"%d\t%lf\t%lf\t%lf\n",i,gridlon[i],gridlat[i],dvol[i]);
  fclose(fout);

  printf("\n\t %s written\n",outname);
 
  free_dvector(lathelp,1,Ngrid);  
  free_dvector(lonhelp,1,Ngrid);  
  free_dvector(volhelp,1,Ngrid);
  free_ivector(indlat,1,Ngrid);
  free_ivector(indlon,1,Ngrid);

  free_dmatrix(res,1,3,1,Zmax);
}


void read_probmainshock()
{
  FILE   *fin, *fout;
  char   sdum[100], probfile[160], slipfile[160], outname[120];
  double dum, dumlat, dumlon, prob, rdist, time, mag, res, norm, lat0, normtest1, normtest2;
  int    i, n, idum, sett;

  probmain     = dmatrix(1,NSLIP,1,Ngrid);   // probability-map for direct aftershocks based on Coulomb-Stress
  probmaindist = dmatrix(1,NSLIP,1,Ngrid);   // probability-map for direct aftershocks based on nearest distance to fault ... will be specified for each given q & d 

  for(i=1;i<=NSLIP;i++){

    sprintf(probfile,"output/slipmodel%d_probr",i);

    fin=fopen(probfile,"r");
    fscanf(fin,"%s",&sdum);
    fscanf(fin,"%d",&idum);

    norm=0.0;
    for(n=1;n<=Ngrid;n++){
      fscanf(fin,"%d", &idum);
      fscanf(fin,"%lf",&dumlat);
      fscanf(fin,"%lf",&dumlon);
      fscanf(fin,"%lf",&prob);
      fscanf(fin,"%lf",&rdist);
      probmain[i][n] = prob;

      res = pow(sq(rdist)+sq(Dvalue),-(1+Qvalue));
      probmaindist[i][n] = res;
      norm += res;
    }
    for(n=1;n<=Ngrid;n++) probmaindist[i][n] /= norm;

    normtest1=0.0;
    normtest2=0.0;
    for(n=1;n<=Ngrid;n++){
      normtest1+=probmain[i][n];
      normtest2+=probmaindist[i][n];
    }

    fclose(fin);
    printf("\t mainshock-probmap %s (norm_cfs=%lf norm_dist=%lf) read\n",probfile,normtest1,normtest2);

 

    sprintf(outname,"output/probmap%d-logscale.out",i);
    fout=fopen(outname,"w"); 
    lat0=gridlat[1];
    for(n=1;n<=Ngrid;n++){
      if(gridlat[n]!=lat0){ lat0=gridlat[n];  fprintf(fout,"\n"); }
      fprintf(fout,"%lf\t",log10(probmain[i][n]));
    }  
    fclose(fout);

    sprintf(outname,"output/distmap%d-logscale.out",i);
    fout=fopen(outname,"w"); 
    lat0=gridlat[1];  

 
    for(n=1;n<=Ngrid;n++){
      if(gridlat[n]!=lat0){ lat0=gridlat[n];  fprintf(fout,"\n"); }
      fprintf(fout,"%lf\t",log10(probmaindist[i][n]));
    }  
    fclose(fout);

  }
  printf("\t output-File %s written\n",outname);

  printf("\n");
}

void setRETASparameters()
{
  // parameter of the RETAS model:
  Mmin   = -2.0;   // absolute minimum magnitude for M0 = meanM0 calculation
  Mstar  =  5.0;
  alpha0 = 1.07;   
  C      = 2*pow(10.0,bvalue*(Mmin-Mcut));    //  = <M0>/10^9.1  [Nm]
  Cstar  = pow(10.0,(1.5-alpha0)*Mstar);     
}

double calculatebvalue(){
  double  dM, dM1, dM2, dum, mean, b, db;
  int     i, k, sett1, sett2;

  if(Z<Zmin1){ 
    b=1.0;
    printf("\n\t set b-value:  b=%.1lf\n\n",b);
  }
  else{
    
    // calculate res = b-value:  
    dM1=1000.0;   dM2=1000.0;  sett1=0; sett2=0;
    for(i=1;i<=Z;i++) { 
      dum=Mcut-m[i]; if(dum>0.0  && dum<dM1){ dM1=dum; sett1++; }
      dum=m[i]-Mcut; if(dum>=0.0 && dum<dM2){ dM2=dum; sett2++; }
    }
    if(sett1==0) dM1=0.0;
    dM = dM1-dM2; 
    
    mean=0.0; 
    for(i=1;i<=Z;i++) mean += m[i]/(1.0*Z);
    
    b  = log10(exp(1.0))/(mean-(Mcut-0.5*dM));
    db = 1.96*b/sqrt(1.0*Z);
    
    printf("\n\t calculated b-value (dM=%lf):  b=%lf (+-%lf)",dM,b,db);
    if(b>1.0 && b-db<1.0) b=1.0;
    if(b<1.0 && b+db>1.0) b=1.0;
    if(b==1.0) printf("\t ---> set to b=%.1lf\n\n",b);
    else       printf("\n\n");
  }
  return b;
}


int main()
{
  FILE     *fout;
  char     inputfile[200], farfalleheaderfile[200], slipfilelistorg[200], templatefile[200], outname[160], slipmodellist[160], gridfile[160], befehl[160], suffx[120];
  char     executabledirectory[200];
  double   *mallorg;
  int      i, k, idum, Nsur;

  // setRETAS:             use RETAS_if=1 OR ETAS_if=0
  // useMainshockprobmaps: use NO_mainshockmaps_if=0  OR use_probmainmaps_if=1  OR use_distance-to-fault_if=2 
  scanf("%d",&setRETAS);       
  scanf("%d",&useMainshockprobmaps); 

 // spacebackground:  constant-background_if=0  ELSE space-dependent background calculated:
  spacebackground=1;

  if(setRETAS==1)                  printf("\n\t RETAS model used\n");                
  else                             printf("\n\t ETAS model used\n");
  if(useMainshockprobmaps==1)      printf("\t Mainshock-related Coulomb-space distribution is used\n");   
  else if(useMainshockprobmaps==2) printf("\t Mainshock-related distance-to-fault distribution is used\n");
  else                             printf("\t NO Mainshock-related spatial distributions is used\n");
  if(spacebackground==1)           printf("\t space-dependent background used\n\n"); 
  else                             printf("\t uniform background used\n\n");

  // read general inputfile:
  scanf("%s",&inputfile);

  // contains: catalogfile, T2 & T, slipmodellist & farfalleheaderfile
  idum=readinputfile(inputfile,farfalleheaderfile,slipfilelistorg,templatefile);
  if(idum==1){ printf("\n\n\t ERROR in INPUT-FILE %s READING!!\n\n",inputfile); exit(1); }
  
  printf("%s read:\n",inputfile);
  printf("\t --> T2=%.2lf [days]  T=%.2lf [days]\n",T2,T);
  printf("\t --> Outputfile     = %s\n",resfile);
  printf("\t --> Catalog        = %s\n",catalogfile);
  printf("\t --> Templatefile   = %s\n",templatefile);
  printf("\t --> Farfalleheader = %s\n",farfalleheaderfile);
  printf("\t --> Sipmodellist   = %s\n",slipfilelistorg);
  printf("\t --> useMainshockprobmaps=%d\n\n",useMainshockprobmaps);

  // read directory of local executable files:
  scanf("%s",&executabledirectory);

  readtemplate(templatefile,gridfile);

  // *********** internal parameter setting *******************************************************************************:
  
  setprint=0;  // if =1 then intermediate LL-values will be printed in davidn.c

  // aftershock duration:
  Tafter  = 10.0;   // [years]
  Tafter *= 365.25; // [days]

  // spatial decay parameter:
  Dvalue  = 1.0;
  Qvalue  = 0.5;              // corresponds to a 1/r^3 decay 

  D0value = 0.5;              // for DL = DO * L(M)  accounting for radius of half-rupture length
 
  // optimize the magnitude of the Nlargest:
  Nlargest    = 10;
  // ************************************************************************************************************************

  // READ CATALOG-DATA
  setdt=1;        // if=1  incomplete times after mainshocks are not used (California-relation) 
  readcatalog();
  
  // reset Dvalue to location error (minimum value 0.5km);
  Dvalue  = locerror;

  fout=fopen("output/test.kat","w");    for(i=1;i<=Z;i++)    fprintf(fout,"%d\t%lf\t%lf\t%lf\t%lf\n",i,t[i],lat[i],lon[i],m[i]);             fclose(fout);
  fout=fopen("output/test.allkat","w"); for(i=1;i<=Zall;i++) fprintf(fout,"%d\t%lf\t%lf\t%lf\t%lf\n",i,tall[i],latall[i],lonall[i],mall[i]); fclose(fout); 
  //exit(1);

  // calculate b-value: 
  bvalue=calculatebvalue(); 
  
  // set some parameters of RETAS:
  setRETASparameters();

  // ******** CALCULATE STRESS-RELATED CALCULATIONS **************
  if(useMainshockprobmaps!=0){

    sprintf(slipmodellist,"output/slipmodellist.txt");
    readwrite_slipmodellist(slipfilelistorg,slipmodellist);

    sprintf(outname,"output/run_stresscalculations.sh");
    fout=fopen(outname,"w");
    fprintf(fout,"(echo \"%s\"; echo \"%s\"; echo \"%s\"; echo \"%s\";) | %s/slipmodel_conversion\n",farfalleheaderfile,slipmodellist,gridfile,executabledirectory,executabledirectory);
    fclose(fout);  
    sprintf(befehl,"chmod u+x %s",outname); system(befehl); 
    sprintf(befehl,"./%s",outname);         system(befehl); 
    read_probmainshock(); 
    if(NSLIP==0) useMainshockprobmaps=0;
  }
  // *********************************************************************

 // calculate spatial weights firstly for aftershock-triggering within LL-space for epicenter- or mainshock-sources:
  calculatespaceweights();

  if(spacebackground==1){
    mallorg=dvector(1,Zall);
    for(i=1;i<=Zall;i++) mallorg[i]=mall[i];
    
    // estimate parameters for time interval [T1 T2]:
    if(setRETAS==1) sprintf(suffx,"RETASfirstestimate");
      else            sprintf(suffx,"ETASfirstestimate");
    spacebackground=0;
    estimation_parameter(suffx);   // estimate parameters
    decluster();                   // decluster + calculate probback[igrid]
    spacebackground=1;
    
    // reset values:
    for(i=1;i<=Zall;i++){
      mall[i]=mallorg[i];
      for(k=1;k<=Z;k++) spaceweightall[i][k]=0.0;  
      Nepi[i]=1; 
      Indexepi[i][1]=i; 
    }
    calculatespaceweights();

    free_dvector(mallorg,1,Zall);
  }
  
  if(setRETAS==1) sprintf(suffx,"RETAS-mainshockprobmaps%d-Mcut%.2lf",useMainshockprobmaps,Mcutpredict);  
  else            sprintf(suffx,"ETAS-mainshockprobmaps%d-Mcut%.2lf",useMainshockprobmaps,Mcutpredict); 

  // estimate parameters for time interval [T1 T2]:
  estimation_parameter(suffx);

  // forecast activity for [T2 T2+T]:
  Nsur=100000;  
  simulationforecast(suffx,Nsur,Mcutpredict,T,templatefile,resfile);

 
  printf("\n\t CODE-END\n\n");  
}

