! ------ MAGNITUDE DISTRIBUTION
!-------------------------------------------
	   subroutine get_pm(im,ix,iy,pm)	   
!---	   compute magnitude distribution in cell ix,iy, for magnitude bin im, 
!---	   for spatially variable b-value b(x,y)
!---	   using a tapered GR law, with a magnitude cutoff Mcorner
!---	   Including special case for the geysers
       implicit none
       integer LMAX,ix,iy,im
       parameter (LMAX=500)       ! max number of cells =LX*LY
	   real Mmin,Mmax,Mcorner,dm,Md,b0,M,pm
	   real b(LMAX,LMAX)
	   real lon1,lon2,lat1,lat2,b1,b2,mcg
	   double precision TFmin,TFmax,TF
	   integer gflag(LMAX,LMAX)
	   common /par/TFmin,TFmax,TF,Mmin,Mmax,Mcorner,dm,Md,b0 
	   common /bmap/b          
	   common /geysers/lon1,lon2,lat1,lat2,b1,b2,mcg,gflag		
	   ! ---  spatially variable b-value
	   M=Mmin+(im-1)*dm
	   if (gflag(ix,iy).eq.0) then
			call get_pm_taperedGR(pm,Mmin,Mmax,M,M+dm,Mcorner,b(ix,iy))
	   else  ! inside Geysers area
		    call get_pm_GRb1b2(pm,Mmin,Mmax,M,M+dm,mcg,b1,b2)
	   endif
	   if (pm<0. .or. pm>1.) then 
			write(*,*) '!gpm:   im,ix,iy,M,pm,Mmin,Mmax,Mcorner,dm,Md,b0'
			write(*,*) '      ',im,ix,iy,M,pm,Mmin,Mmax,Mcorner,dm,Md,b0
	   endif
	   end
!----------------------------------------------------
		subroutine get_pm_taperedGR(pm,mmin,mmax,m1,m2,mc,b)
! computes the probability pm for m1<m<m2 
! for a GR law with exponent b defined between mmin and mmax
! and tapered by an exponential cutoff (in seismic moment) at a corner magnitude mc
! ref: Helmstetter et al SRL 2007 (eq10); Helmstetter and Werner, BSSA 2012
		real pm,mmin,mmax,m1,m2,mc,b,p1,p2,s,f
		f=10.**(1.5*(mmin-mc))
		p1=10.**(-b*(m1-mmin)) *exp(f -10.**(1.5*(m1-mc)))        ! proba of m>m1	
		p2=10.**(-b*(m2-mmin)) *exp(f -10.**(1.5*(m2-mc)))        ! proba of m>m2	
		s=1.-10.**(-b*(mmax-mmin)) *exp(f -10.**(1.5*(mmax-mc)))  ! proba of mmin<m<mmax
		pm=(p1-p2)/s  ! proba of m1<m<m2
		end
!----------------------------------------------------
		subroutine get_pm_GRb1b2(pm,mmin,mmax,m1,m2,mc,b1,b2)
! computes the probability pm for m1<m<m2 
! for a piece-wise GR pdf with exponent b1 for m<mc and b2 for m>mc 
		real pm,mmin,mmax,m1,m2,mc,b1,b2,p1,p2,s,f
		if (mmin.ge.mc) then ! single GR law with b=b2
			p1=10.**(-b2*(m1-mmin))         ! proba of m>m1	
		    p2=10.**(-b2*(m2-mmin))         ! proba of m>m2	
			s=1.-10.**(-b2*(mmax-mmin))     ! proba of mmin<m<mmax
		    pm=(p1-p2)/s  ! proba of m1<m<m2
			return
		endif
		f=10.**(-b1*(mc-mmin))
		s=log(10.) /(1./b1 -f *(1./b1-1./b2))	
		if (m1.le.mc) then
			p1=s*( 10.**(-b1*(m1-mmin))-f) /log(10.)/b1 + s*f/log(10.)/b2
		else
			p1=s*f * 10.**(-b2*(m1-mc)) /log(10.)/b2
		endif
		if (m2.le.mc) then
			p2=s*( 10.**(-b1*(m2-mmin))-f) /log(10.)/b1 + s*f/log(10.)/b2
		else
			p2=s*f* 10.**(-b2*(m2-mc)) /log(10.)/b2
		endif
		pm=p1-p2     ! proba of m1<m<m2
		end

!------------------------------------------------------------
      subroutine Mcomp(tau,MM,Mc)
!     completeness magnitude at time tau after event an event of mag. MM
!     See Kagan, Short term prpoerties..., submitted 2003
	  double precision tau
      real MM,Mc
      Mc=2.5-0.76*log10(tau) + MM-7.
	  Mc=min(Mc,MM)
      end  
	  
!------------------------------------------------------------
      subroutine mag_cdf(mag,N,m,pcm)
!    cumulative magnitude distribution
	  integer nlmax,i,N
	  parameter (nlmax=200000)	
	  real mag(nlmax),m(nlmax),pcm(nlmax)
	  ! initialisation
	  do i=1,N
		m(i)=mag(i)
		pcm(i)=(N-i+1.)/N    ! cumulative distribution P(M>=m)
	  enddo		
	  call piksrt(N,m)	! sort m(1:N) by increasing values	  
      end  
	  
!------------------------------------------------------------
      real function mag_weight(mc,md,m,pcm,N)
!     mag_weight=pm(Mmin)/Pm(mag), where pm is the cumulative magnitude distribution
	  integer nlmax,i,j,N
	  parameter (nlmax=200000)	
	  real mc,md,m(nlmax),pcm(nlmax)
	  if (md .ge.mc) then	! above completeness magnitude
		mag_weight=1.
		return
	  else
		i=1
		do while (md.gt.m(i) .and. i.lt.N) 
			i=i+1
		enddo
		j=i
		do while (mc.gt.m(j)  .and. j.lt.N)
			j=j+1
		enddo
		mag_weight=pcm(i)/pcm(j) ! = pcf(m>=md) / pcf(m>=mc)
		! write(*,*) 'mag_weight: mc=',mc,' Md',Md,',i=',i,'j=',j,'w=',mag_weight
		! write(*,*) 'mag_weight: ',m(i),m(j),pcm(i),pcm(j)
	  endif
	  return
	  end
! -------------------------------------------------------
!	SPATIAL DISTRIBUTION
!--------------------------------------------------------
		real function distance(lon1,lat1,lon2,lat2)
		implicit none
		real lon1,lat1,lon2,lat2,radian,R
		real lon1r,lat1r,lon2r,lat2r,dlon,dlat,s1,s2,sa,a
		! great circle distance in km
		! http://www.movable-type.co.uk/scripts/gis-faq-5.1.html
		radian=0.017453    ! pi/180 conversion degrees -> degrees
		R=6378.            ! earth radius
		lat2r=lat2*radian
		lon2r=lon2*radian
		lat1r=lat1*radian 
		lon1r=lon1*radian
		dlon = lon2r - lon1r
		dlat = lat2r - lat1r
		s1=sin(dlat/2.) 
		s2=sin(dlon/2.)
		a = s1*s1 + cos(lat1r) * cos(lat2r) * s2*s2
		sa=min(1.,sqrt(a))
		distance = R * 2. * asin(sa)
		return
		end	
		
!-------------------------------------------------------		
	 real function feval(loni,lati,di,lonc,latc,l,kr) 
!!-------------------------------------------------------		
!	compute the integral f of a 2D Gaussian of width di located at (loni,lati)
!	over a cell of width l located at lonc,latc
	 implicit none
	 real distance,loni,lati,di,lonc,latc,dlon,l,lkm,r,dmax
	 real x1,y1,x2,y2,y,q,dx,dy,z,c,f,fevalgs2D,fevalpl
	 character *2 kr	! kernel function : 'gs' (gaussian) or 'pl' (power-law)
	 external distance,fevalgs2D,fevalpl
	 if (di.le.0.) then
		write(*,*) '!! feval: di=',di,loni,lati,lonc,latc,l,kr
		feval=0.
	    return
	 endif
	 f=5.
	 lkm=111.
	 dy=l*lkm			! cell size in km, x direction
	 dmax=f*di+dy	    ! maximum integration distance
	 ! distance between point loni,lati and center of cell
	 r=lkm*(abs(latc-lati) + abs (lonc-loni))  ! approx, only to check if > dmax
	 if (r.gt.2*dmax .and. kr.eq.'gs') then	   
		feval=0.	    ! too far from EQ, gaussian is almost 0
		return
	 endif
 	 c=cos(latc*0.017453)
	 dx=dy*c		    ! cell size in km, y direction
	 dlon=lonc-loni
	 dlon=min(dlon,360.-dlon)
	 x1=dlon*lkm*c	    -dx/2.	
	 y1=(latc-lati)*lkm	-dy/2.	
	 if ( (di>f*dy) .or. (r>f*dy) .or. (r>dmax)) then	
		! ----- cell size << smoothing distance
		!    or cell size << EQ-cell distance
		!    or EQ-cell distance >> ( smoothing distance + cell size )
		r=distance(loni,lati,lonc,latc)
		! --- take value of a 2D Gaussian kernel at the center of cell
		if(kr.eq.'gs') then	! 2D Gaussian kernel
			z=r/di
			feval=dx*dy*exp(-z*z/2.)/3.1415926/2./di/di
			return
		else				! power-law kernel
			! ---  at large distances relative to di and dx, 
			! take the value in the center of cell -> ... bad approx
			! fap=dx*dy /(r*r+di*di)**1.5 *di/3.1415926/2.
			! --- Better approximation: analytical integration for x
			! then take the value at (y1+y2)/2	-> er  < 2%	
			x2=x1+dx;  y2=y1+dy
			y=(y2+y1)/2.
			q=y*y+di*di
			feval=( x2/(x2*x2+q)**0.5 -x1/(x1*x1+q)**0.5 )/q  *dy*di/3.1415926/2.
			feval=max(feval,0.)
			!write(*,*) 'fevala:',loni,',',lati,',',di,',',lonc,',',latc,',',l,kr,feval,r
			!write(*,*) x1,x2,y1,y2,y,dx,dy,q,feval
			return
		endif
	 endif	
	 ! --- otherwise compute integral
	 if (kr.eq.'gs') f=fevalgs2D(x1,y1,di,dx,dy)
	 if (kr.eq.'pl') f=fevalpl(x1,y1,di,dx,dy)
	 feval=max(f,0.)
	 return
	 end	
!---------------------------------------------------------------
      function  fevalgs2D(x1,y1,di,dx,dy)       
!---------   integral of phi(r) for x1<x<x1+dx and y1<y<y1+dy with phi(r) gaussian
	  implicit none
	  real di,x1,y1,x2,y2,dx,dy,disr2,fevalgs2D
	  double precision a1,a2,A3,a4
      y2=y1+dy
      x2=x1+dx
	  disr2=di*sqrt(2.)
	  a1=x1/disr2; 	  a2=x2/disr2
	  a3=y1/disr2; 	  a4=Y2/disr2
	  fevalgs2D = (derf(a1)-derf(a2)) * ( derf(a3)-derf(a4) ) / 4.
	  return
	  end
!---------------------------------------------------------------
!     intergral of phi(r) for x1<x<x1+dx and y1<y<y1+dy with phi(r) ~1/(r^2+d^2)^1.5
      function fevalpl(x1,y1,di,dx,dy)       
      implicit none
      real di, x1,x1f,x2,x12,x22,y1,y2,dx,dy,di2,f,fevalpl
      common /fc/x1f,x2,di2,x12,x22    ! parameters of func function, integral over x      
      external func        
	  x2=x1+dx;  y2=y1+dy
	  x12=x1*x1; x22=x2*x2; 
	  x1f=x1                   
	  di2=di*di
	  call qromb(func,y1,y2,f) ! integral of phi(r) in the cell	  
	  f=f*di/3.1415926/2.
	  fevalpl=max(f,0.)
      end
!-------------------------------------------------------
	    function  func(y)
!       f(y)=integral of K(x,y)~1/(x^2+y^2+d^2)^1.5 for x1<x<x2
        real func
		real d2,y,x1,x2,x12,x22,y2,q 
        common /fc/x1,x2,d2,x12,x22
        q=y*y+d2       ! y^2+d^2
        func=(x2/(x22+q)**0.5-x1/(x12+q)**0.5)/q
		func=max(0.,func)
		!! if dx <<x et q << x, numerical errors, x2/(x22+q)**0.5 approx x1/(x12+q)**0.5 and func <0
		return
        end
!---------------------------------------------------------------	  
      SUBROUTINE qromb(func,a,b,ss) 
!-------------------------- numerical recipies. Integration
      INTEGER JMAX,JMAXP,K,KM
      REAL a,b,func,ss,EPS
      EXTERNAL func
      PARAMETER (EPS=1.e-5, JMAX=20, JMAXP=JMAX+1, K=5, KM=K-1)
CU    USES polint,trapzd
      INTEGER j
      REAL dss,h(JMAXP),s(JMAXP),small
	  small=1.e-10		! modif Agnes. otherwise no convergence for small aboslute values of errors
      h(1)=1.
      do j=1,JMAX
        call trapzd(func,a,b,s(j),j)
        if (j.ge.K) then
          call polint(h(j-KM),s(j-KM),K,0.,ss,dss)
          if (abs(dss).le.EPS*abs(ss) .or. abs(dss).le.small ) return ! modif
        endif
        s(j+1)=s(j)
        h(j+1)=0.25*h(j)
	  enddo
	  write(*,*) '!! too many steps in qromb'
	  write(*,*) '! qromb: j=',j,' JMAX=',JMAX,' a=',a,' b=',b,' s=',s(j),' dss=',dss,' ss=',ss
	  write(*,*) 's:',s
	  
	  ss=-10.		! modif 2013/3/11
	  
	  return 
      END
C-----------------------------------------------------------------------------
      SUBROUTINE polint(xa,ya,n,x,y,dy) ! numerical recipies
      INTEGER n,NMAX
      REAL dy,x,y,xa(n),ya(n)
      PARAMETER (NMAX=10)
      INTEGER i,m,ns
      REAL den,dif,dift,ho,hp,w,c(NMAX),d(NMAX)
      ns=1
      dif=abs(x-xa(1))
      do 11 i=1,n
        dift=abs(x-xa(i))
        if (dift.lt.dif) then
          ns=i
          dif=dift
        endif
        c(i)=ya(i)
        d(i)=ya(i)
11    continue
      y=ya(ns)
      ns=ns-1
      do 13 m=1,n-1
        do 12 i=1,n-m
          ho=xa(i)-x
          hp=xa(i+m)-x
          w=c(i+1)-d(i)
          den=ho-hp
          ! if(den.eq.0.) pause 'failure in polint'
          den=w/den
          d(i)=hp*den
          c(i)=ho*den
12      continue
        if (2*ns.lt.n-m)then
          dy=c(ns+1)
        else
          dy=d(ns)
          ns=ns-1
        endif
        y=y+dy
13    continue
      return
      END

!-----------------------------------------------------------------------------      
      SUBROUTINE trapzd(func,a,b,s,n)
!------------------ numerical recipies, computes integral of function func between a and b
      INTEGER n
      REAL a,b,s,func
      EXTERNAL func
      INTEGER it,j
      REAL del,sum,tnm,x
      if (n.eq.1) then
        s=0.5*(b-a)*(func(a)+func(b))
      else
        it=2**(n-2)
        tnm=it
        del=(b-a)/tnm
        x=a+0.5*del
        sum=0.
        do 11 j=1,it
          sum=sum+func(x)
          x=x+del
11      continue
        s=0.5*(s+(b-a)*sum/tnm)
      endif
      return
      END
!-----------------------------------------------------------------------------
	function  gammln(xx)
!------------------ numerical recipies, calculate log(gamma(xx)), used in LogLik
	real gammln,xx
	integer j
	real ser,stp,tmp,x,y,cof(6)
	SAVE cof,stp
	DATA cof,stp/76.18009172947146d0,-86.50532032941677d0,
     * 24.01409824083091d0,-1.231739572450155d0,.1208650973866179d-2,
     *    -.5395239384953d-5,2.5066282746310005d0/
	x=xx
	y=x
	tmp=x+5.5d0
	tmp=(x+0.5d0)*log(tmp)-tmp
	ser=1.000000000190015d0
	do j=1,6
	   y=y+1.d0
	   ser=ser+cof(j)/y
	enddo
	gammln=tmp+log(stp*ser/x)
	return
	END
c----------------------------------------------------
      SUBROUTINE CDAY (DAY)
C  program written by Y. Kagan (?)
C  PROGRAM TO COMPUTE time in days since 1/1/1900
      INTEGER*4 MD(13)  
	  INTEGER IL, IYE, IMO, IDA, IH, IMI, IDAY, NDAY, IT
	  DATA MD  / 0,31,59,90,120,151,181,212,243,273,304,334,365 /
      double precision DIDAY, PDAY, YL, H, AM, DAY
	  REAL SEC
      LOGICAL LEAP
      PARAMETER (YL = 365.25D0)
      COMMON /DATE/ SEC, IYE, IMO, IDA, IH, IMI
      IL = 0
      LEAP = MOD (IYE,4) .EQ. 0
      IDAY = DFLOAT (IYE - 1905)*YL + 1096.0D0
      IF(LEAP .AND. IMO. GT. 2 )  IL = 1
      IDAY = IDAY + MD(IMO) + IL + IDA
      DAY=DFLOAT(IDAY)+(DFLOAT(IH)+(DFLOAT(IMI)+SEC/60.D0)/60.D0)/24.D0
	  DAY=DAY+729.D0
      RETURN
      END
!----------------------------------------------------
      SUBROUTINE CDAYinv (DAY0)
C  program written by Y. Kagan (?)
C  COMPUTE YEAR,MONTH,DAY,HOUR,MINUTE AND SECONDS from day number since 1900/1/1
      INTEGER*4 MD(13)
	  INTEGER IL, IYE, IMO, IDA, IH, IMI, IDAY, NDAY, IT
	  DATA MD  / 0,31,59,90,120,151,181,212,243,273,304,334,365 /
	  double precision DIDAY, PDAY, YL, H, AM, day,day0
	  real SEC
	  LOGICAL LEAP
      PARAMETER (YL = 365.25D0)
      COMMON /DATE/ SEC, IYE, IMO, IDA, IH, IMI
	  DAY=DAY0-729.0D0	  
	  IL = 0
      NDAY = DAY
      DIDAY = NDAY
      IYE = (DIDAY - 731.0D0)/YL + 1904.0D0
      IDAY = DFLOAT (IYE - 1905)*YL + 1096.0D0
      IDAY = NDAY - IDAY
      LEAP = MOD (IYE,4) .EQ. 0
      IF(IYE .EQ. 1900)  LEAP = .FALSE.
      IF(DAY .LE. 0.0D0 ) THEN
        IDAY = IDAY - 1
        DIDAY = DIDAY - 1.0D0
	  END IF
C
      DO 10 IMO = 1, 13
      IF(LEAP .AND. IMO.GT.2)  IL = 1
      IT = IDAY - MD(IMO) - IL
      IF(IT ) 15, 15, 10
   10 IDA = IT
   15 IMO = IMO - 1
      PDAY = DAY - DIDAY
      IF (IYE .EQ. 1897)  IDA = IDA - 1
      H = PDAY*24.0D0
      IH = H
      AM = (H - FLOAT(IH))*60.0D0
      IMI = AM
      SEC = (AM - FLOAT(IMI))*60.0
      IF (SEC .GT. 59.996) THEN
         SEC = SEC - 60.00
         IMI = IMI + 1
         END IF
      RETURN
      END
!--------------------------------------------------------------------------------
!	OPTIMIZATINO (amoeba, amotry) numerical recipies
!--------------------------------------------------------------------------------
       SUBROUTINE amoeba(p,y,mp,np,ndim,ftol,funk,iter,ITMAX)
!--------------------------------------------------------------------------------
      INTEGER iter,mp,ndim,np,NMAX,ITMAX
      REAL ftol,p(mp,np),y(mp),funk
      PARAMETER (NMAX=10)
      EXTERNAL funk
CU    USES amotry,funk
      INTEGER i,ihi,ilo,inhi,j,m,n
      REAL rtol,sum,swap,ysave,ytry,psum(NMAX),amotry
      iter=0
1     do 12 n=1,ndim
        sum=0.
        do 11 m=1,ndim+1
          sum=sum+p(m,n) !sum over starting points of each parameter
11      continue
        psum(n)=sum !call the sum psum
12    continue
2     ilo=1
      if (y(1).gt.y(2)) then !if likelihood of set 1 > set 2
        ihi=1 !then call index high set 1
        inhi=2 ! and index is next highest set 2
      else
        ihi=2 !otherwise reverse the order 
        inhi=1
      endif
      do 13 i=1,ndim+1 !cycle through starting value sets
        if(y(i).le.y(ilo)) ilo=i !if ith less than previous ilow, set ilo to current index
        if(y(i).gt.y(ihi)) then !if ith larger than previous ihi, then 
          inhi=ihi ! set previous ihi to is next highest index
          ihi=i ! and current i to new ihi
        else if(y(i).gt.y(inhi)) then !if current LL is less than high LL but greater than previous is next highest index
          if(i.ne.ihi) inhi=i !and if current i is not equal to ihi, set is next highest to i
        endif
13    continue
      rtol=2.*abs(y(ihi)-y(ilo))/(abs(y(ihi))+abs(y(ilo))) !calculate normalized difference between high and low LL scores
      if (rtol.lt.ftol) then !if running tolerance is less than input tolerance
        swap=y(1) !call the first set swap
        y(1)=y(ilo) !assign y(1) the value of the lowest LL value
        y(ilo)=swap ! and assign y(ilo) the value of the previous y(1)
        do 14 n=1,ndim !cycle thru parameters
          swap=p(1,n) !call a parameter in the first starting set swap
          p(1,n)=p(ilo,n) !assign the first starting set parameter the value of the parameter that gives the lowest LL
          p(ilo,n)=swap !and assign p(ilo) the value from the first starting point
14      continue
		write(*,*) 'Tolerance level achieved.'
        return
      endif
      if (iter.ge.ITMAX) then !too many iterations - looks like tolerance requirement will never be obtained. 
		write(6,*) 'ITMAX exceeded in amoeba'
		write(*,*) 'ITMAX exceeded in amoeba'
		return
      endif	
      iter=iter+2 !why would you add 2 to the number of iterations
      ytry=amotry(p,y,psum,mp,np,ndim,funk,ihi,-1.0)
      if (ytry.le.y(ilo)) then
        ytry=amotry(p,y,psum,mp,np,ndim,funk,ihi,2.0)
      else if (ytry.ge.y(inhi)) then
        ysave=y(ihi)
        ytry=amotry(p,y,psum,mp,np,ndim,funk,ihi,0.5)
        if (ytry.ge.ysave) then
          do 16 i=1,ndim+1
            if(i.ne.ilo)then
              do 15 j=1,ndim
                psum(j)=0.5*(p(i,j)+p(ilo,j))
                p(i,j)=psum(j)
15            continue
              y(i)=funk(psum)
            endif
16        continue
          iter=iter+ndim
          goto 1
        endif
      else
        iter=iter-1
      endif
      goto 2
      END
!--------------------------------------------------------------
      FUNCTION amotry(p,y,psum,mp,np,ndim,funk,ihi,fac)
!--------------------------------------------------------------
      INTEGER ihi,mp,ndim,np,NMAX
      REAL amotry,fac,p(mp,np),psum(np),y(mp),funk
      PARAMETER (NMAX=20)
      EXTERNAL funk
CU    USES funk
      INTEGER j
      REAL fac1,fac2,ytry,ptry(NMAX)
      fac1=(1.-fac)/ndim !
      fac2=fac1-fac
      do 11 j=1,ndim
        ptry(j)=psum(j)*fac1-p(ihi,j)*fac2
		if (ptry(j).lt.0.) then					! This was added to stop parameters from becoming negative
			! write(*,*) 'Setting ptry(',j,')=',ptry(j),' to absolute value thereof.'
			ptry(j)=abs(ptry(j))		!!
		endif
11    continue
      ytry=funk(ptry)
      if (ytry.lt.y(ihi)) then
        y(ihi)=ytry
        do 12 j=1,ndim
          psum(j)=psum(j)-p(ihi,j)+ptry(j)
          p(ihi,j)=ptry(j)
12      continue
      endif
      amotry=ytry
      return
      END
!-----------------------------------------------------------------------
	subroutine interp2(x1,x2,y)
!	 compute interpolated value y at position x1,x2	(=value of nearest cell)
!-----------------------------------------------------------------------	   
	implicit none
	integer LMAX,i1,i2,Nxmc,Nymc,n,j1,j2
	parameter (LMAX=500)		! max grid size
	real lonmc(LMAX),latmc(LMAX),mcxy(LMAX,LMAX),x1,x2,y,dx1,dx2
	common /mcmap/lonmc,latmc,mcxy,Nxmc,Nymc
	if (Nxmc.eq.0) then				! uniform value of mc for all cells
		y=mcxy(1,1)
		return
	endif!	find cell in which point x1,x2 lies
	dx1=lonmc(2)-lonmc(1)			! find longitude difference between cells
	dx2=latmc(2)-latmc(1)			! find latitude  difference btw cells
	i1=1+(x1-lonmc(1)+0.0001)/dx1	! find longitude index of cell that contains location x1
	i2=1+(x2-latmc(1)+0.0001)/dx2	! find latitude  index of cell that contains location x2	
	i1=max(1,i1)					! make sure index is at least equal to 1
	i2=max(1,i2)					! 
	i1=min(Nxmc,i1)				    ! make sure index is no larger than number of cells
	i2=min(Nymc,i2)					
	y=mcxy(i1,i2)
	if ( mcxy(i1,i2).eq.0.) then	! where flag=0
		n=1
		y=0.
		do while (y.eq.0.)
			! look at adjacent cells
			do j1=i1-n,i1+n
				if (j1.gt.0 .and. j1 .le. Nxmc)  then
					do j2=i2-n,i2+n
						if (j2.gt.0 .and. j2.le. Nymc) 	y=mcxy(j1,j2)
						if (y.gt.0.) exit
					enddo
					if (y.gt.0.) exit
				endif
			enddo
			n=n+1
		enddo
	endif	
	if (y.gt.10. .or. y.lt.0.) then
		write(*,*) 'I2: ?? ',x1,x2,i1,i2,j1,j2,n,y
	endif
	return
	end

C--------------------------------------------------------------------------- 
	real function median(v,l) ! shell method, num. rec. p323
c   sort the catalog by increasing values v(1:l)
c	and return median value
c	! in output v is modified (sorted by increasing values)
	implicit none
	integer i,j,inc,l,LT
	parameter (LT=100000)
	real  v(LT),vv
	inc=1
 1	inc=3*inc+1
	if(inc.le.l) goto 1
 2	continue
	inc=inc/3
	do i=inc+1,l
		vv=v(i)
		j=i
 3	   if(v(j-inc).gt.vv) then
	      v(j)=v(j-inc)
	      j=j-inc
	      if(j.le.inc) goto 4
	      goto 3
	   endif
 4	   v(j)=vv
	enddo
	if(inc.gt.1) goto 2
	i=l/2
	median=v(i)
	return
	END
!----------------------------------	
	subroutine piksrt(n,arr)
	! num. rec, sort vctors arr(1:n) by increasing order, shell method
      INTEGER n
      REAL arr(n)
      INTEGER i,j
      REAL a
      do j=2,n
        a=arr(j)
        do i=j-1,1,-1
          if(arr(i).le.a)goto 10
          arr(i+1)=arr(i)
		enddo
        i=0
10      arr(i+1)=a
      enddo
      return
      END
