! written by Agnes Helmstetter and Max Werner
! last modified June 2012
!
! Compilation (g95,g77 or gfortran) 
!	g95  F/csepETAS.f F/common.f -o csepETAS -lm  -ffixed-line-length-132 -fimplicit-none
! Execution:
!	 csepETAS INPUT/csep-input-file-ETAS
!------------------------------------------
      program csepETAS
!-------------------------------------------
        implicit none
		character *200 inputfile
		call getarg(1, inputfile)
		
!------ read input parameters in file csepETAS.par. Also read EQ catalog and background rate
		call readpar(inputfile)          	
	   
!------ compute the expected number of aftershocks of each eq. inside testing area using ETAS
		call get_rho					 

!------ write output catalog
		call write_output_cat

!------ compute the spatial distribution of aftershocks of M>MBIG mainshock
     	call evalbigaft 

!------ compute the predicted rate per cell
		call mapnpred

! ----- write output xml file
		call csep_template(inputfile)    
        end
!-----------------------------------------------------  
		subroutine get_rho
!-----------------------------------------------------	 
!	Compute the number of aftershocks rho of each EQ inside the testing area,
!   and the triggering distance d
!		implicit none
		integer j,nlmax,N1,LX,LY,LMAX,ix,iy
  		parameter (nlmax=200000) ! nlmax = maximum number of events in the catalog     
		parameter (LMAX=500)         ! max number of cells =LMAX*LMAX
		real mub,alpha,K,p,c,br,fd
		real Mmin,Mmax,Mcorner,dm,Md,b0
        double precision TFmin,TFmax,TF,t(nlmax)
        real M(nlmax),lon(nlmax),lat(nlmax),rho(nlmax),d(nlmax) ! seismicity catalog
		real b(LMAX,LMAX),bj,Pm,lon1,lon2,lat1,lat2,b1,b2,mcg
		real latmin,latmax,lonmin,lonmax,zmax,l
 		integer gflag(LMAX,LMAX)
      	common /cat/t,lon,lat,rho,d,M,N1
		common /ETAS/mub,alpha,K,p,c,br,fd			! ETAS param. 
		common /par/TFmin,TFmax,TF,Mmin,Mmax,Mcorner,dm,Md,b0
		common /grid/latmin,latmax,lonmin,lonmax,zmax,l,LX,LY 
		common /bmap/b
		common /geysers/lon1,lon2,lat1,lat2,b1,b2,mcg,gflag

   	   !--------- compute the expected number of aftershocks of each eq. using ETAS
 		write(*,*) '--- computes rho and d for each EQ'
		! --- for each EQ
        do j=1,N1		! for each "new" EQ
			ix=(lon(j)  + 0.0001 -lonmin)/l+1
			iy=(lat(j)  + 0.0001 -latmin)/l+1
			if (ix.gt.0 .and. ix.le.LX .and. iy.gt.0 .and. iy.le.LY) then
				bj=b(ix,iy)
				if (bj.le.0) bj=b0
			else
				bj=b0
			endif
			if (gflag(ix,iy).eq.1) then  
				! special correction for the geysers
				call get_pm_GRb1b2(Pm,Md,Mmax,Mmin,Mmax,mcg,b1,b2)
			else
				Pm=10.**(-bj*(Mmin-Md)) ! <=1, fraction of learning EQs  with mag > mmin
			endif
		    d(j)=fd*0.01*10.**(0.5*M(j)) + 0.5    ! characteristic size of aftershock zone + loc. accuracy 0.5km
  		    rho(j)=K*10.**(alpha*(M(j)-Md))*Pm	 ! number of aft. >Mmin over the infinite space
        enddo  
		end
!---------------------------------------------------------------
      subroutine predphixy(Kmeans,ix,iy,tp)
!---------------------------------------------------------------
!       evaluate the number of eathquakes Kmeans in cell (ix,iy) with m>Mmin
!       and times tp<t<tp+TF triggered by all N past eqs. and to the background mu
        implicit none
        integer nlmax,LX,LY,LMAX,NM
		parameter (nlmax=200000)   ! nlmax = maximum number of events in the catalog     
        parameter (LMAX=500)         ! max number of cells =LMAX*LMAX
        parameter (NM=100)        ! max number of 'mainshocks' (M>=MBIG) 
        integer i,j,N1,ix,iy,IMS(NM),IAS(nlmax),im
		real Md,Mmin,Mmax,Mcorner,dm,MM,MBIG,rc,f,mub,alpha,K,p,c,br,fd,b0,theta ! ETAS param.
        double precision  t(nlmax),dt,tp,TFmin,TFmax,TF
		real mubg(LMAX,LMAX),dKxy(LMAX,LMAX,NM),TAFT,Kmeans,w
        real latmin,latmax,lonmin,lonmax,zmax,l
        real M(nlmax),lon(nlmax),lat(nlmax),rho(nlmax),d(nlmax)
        real lonc,latc,daft,feval
		character*2 kr  
		common /ETAS/mub,alpha,K,p,c,br,fd  ! ETAS param.   
        common /kernelaft/kr    
        common /cat/t,lon,lat,rho,d,M,N1
        common /grid/latmin,latmax,lonmin,lonmax,zmax,l,LX,LY 
		common /par/TFmin,TFmax,TF,Mmin,Mmax,Mcorner,dm,Md,b0    
  		common /MU/mubg
        common /BIG/dKxy,IMS,IAS
        common /BIGPAR/TAFT,daft,MBIG     
		lonc=lonmin+(ix-0.5)*l  ! longitude of center of cell ix,iy
        latc=latmin+(iy-0.5)*l 
        Kmeans=mubg(ix,iy)      ! background rate >Mmin for a time window TF 
		i=1
		theta=p-1.
		do while (t(i).lt.tp  .and. i.le.N1)	              ! for each past event
		    w=(((tp-t(i)+c)/c)**(-theta)-((tp+TF-t(i)+c)/c)**(-theta))*sign(1.,theta)
           if(M(i).gt.MBIG) then 
              if (tp-t(i)<TAFT) then  ! recent big event
                 ! compute the spatial distribution of aftershocks of event i incell ix,iy
                 ! using all events that occurred between t(i) and tp 
                 call bigaftfxy(f,i,ix,iy,tp) 
              else   ! use the spatial distribution already computed with TAFT
                 im=IAS(i)! mainshock index 
                 f=dKxy(ix,iy,im)
              endif   
           else
			 ! compute integral of spatial kernel of width di
			 ! centered at lon(i),lat(i)  over the cell ix,iy
			  f=feval(lon(i),lat(i),d(i),lonc,latc,l,kr);
            endif
            Kmeans=Kmeans+rho(i)*f*w  
			if (Kmeans.lt.0. .or. Kmeans.ne.Kmeans) then
				write(*,*) '!! PPxy: ix,iy,tp,i,rho,f,w,Kmeans:',ix,iy,tp,i,rho(i),f,w,Kmeans
				Kmeans=mubg(ix,iy)
				! pause
			endif
		i=i+1			! next event i
        enddo !  while t<tp loop
        end 
!---------------------------------------------------------------
      subroutine bigaft(n,naft) 
!     spatial distribution of aftershocks of large mainshocks
!     = sum of the kernels of past aft.  with d= const. =daft
!     of all "aftershocks" that occured within TAFT days
!     at a distance less than dmax=2*d(mainshock)
!     and of the mainshock (with d=d(mainshock))
!     n: integer, index of the aftershock sequence (3rd column in dKxy)
        implicit none
        integer nlmax,LX,LY,LMAX,NM
      	parameter (nlmax=200000)    ! nlmax = maximum number of events in the catalog     
        parameter (LMAX=500)        ! number of cells =LMAX*LMAX
        parameter (NM=100)          ! max number of 'mainshocks '(M>=MGIG)       
        integer ix,iy,i,n,N1,im,IMS(NM),IAS(nlmax),naft,iaft 
		real latmin,latmax,lonmin,lonmax,zmax,l,dKs,TAFT
        real MBIG,dmax,daftms,daft,tim,f,feval,distance,lonc,latc,lonm,latm,di
        double precision  t(nlmax) 
        real dKxy(LMAX,LMAX,NM)  
    	character*2 kr
		real M(nlmax),lon(nlmax),lat(nlmax),rho(nlmax),d(nlmax) ! seismicity catalog 
		common /grid/latmin,latmax,lonmin,lonmax,zmax,l,LX,LY  
        common /cat/t,lon,lat,rho,d,M,N1
        common /BIG/dKxy,IMS,IAS  
        common /BIGPAR/TAFT,daft,MBIG 
        common /kernelaft/kr        
        im=IMS(n)  ! mainshock index in the catalog
        tim=t(im) 		
		lonm=lon(im)
		latm=lat(im)          
        dmax=2.*0.01*10.**(0.5*M(im))   ! max distance between ms and aft. ~ 2 rupture lengths
        do ix=1,LX              ! for each grid point
           do iy=1,LY  
              dKxy(ix,iy,n)=0.   ! average rate above Mmin:
           enddo  
        enddo
        iaft=im
        naft=0
        do while(t(iaft).le.tim+TAFT) ! for each aft. within a time TAFT of the mainshock i
 		   daftms=distance(lonm,latm,lon(iaft),lat(iaft))
           if (daftms.lt.dmax) then
               naft=naft+1   ! number of aftershocks
              if (iaft.eq.im) then
                 di=d(im)   ! for the mainshock use d = mainshock rupture length
              else
                 di=daft      ! use kernel  smoothing distance d =1 km for the aftershocks
              endif
			  do ix=1,LX    ! for each cell
				 lonc=lonmin+(ix-0.5)*l   ! longitude of center of cell ix,iy
                 do iy=1,LY 
					latc=latmin+(iy-0.5)*l 
					! compute integral of spatial kernel of width di 
					! centered at lon(i),lat(i) over the cell ix,iy
					f=feval(lon(iaft),lat(iaft),di,lonc,latc,l,kr);
					dKxy(ix,iy,n)=dKxy(ix,iy,n)+f
                 enddo
              enddo 
           endif    ! end. loop if distance to mainshock < dmax
           iaft=iaft+1
        enddo  ! next aftershock iaft
		write(*,*) 'n=',n,' t=',t(im),' m=',M(im),' naft=',naft
		
        f=0.                    ! normalisation dK=sum(dK)
        do ix=1,LX           ! for each grid point
              do iy=1,LY  
                 dKxy(ix,iy,n)=dKxy(ix,iy,n)/naft
                 f=f+ dKxy(ix,iy,n)
              enddo  
        enddo       
	    end
!---------------------------------------------------------------
      subroutine bigaftfxy(fxy,im,ix,iy,tp)
!     density  of aftershocks of mainshock im in cell ix,iy
!     = sum of the kernels  of past aftershocks with d=daft
!     for all "aftershocks" that occured within TAFT days
!     at a distance less than dmax=2*d(mainshock)
!     and of the mainshock (with d=d(mainshock))
        implicit none
        integer nlmax,ix,iy,i,im,iaft,n,N1,naft,LX,LY
     	parameter (nlmax=200000) ! nlmax = maximum number of events in the catalog     
        real fxy,f,f2,dx,dy,di,x1,y1,daft,dmax,rc,rmax,daftms,tim,lonm,latm 
		real latmin,latmax,lonmin,lonmax,zmax,l,lonc,latc,distance,feval,TAFT,MBIG
        double precision  t(nlmax),tp
    	character*2 kr
		real M(nlmax),lon(nlmax),lat(nlmax),rho(nlmax),d(nlmax) ! seismicity catalog 
        common /grid/latmin,latmax,lonmin,lonmax,zmax,l,LX,LY 
        common /cat/t,lon,lat,rho,d,M,N1
        common /kernelaft/kr         
        common /BIGPAR/TAFT,daft,MBIG             
        lonc=lonmin+(ix-0.5)*l   ! longitude of center of cell ix,iy
        latc=latmin+(iy-0.5)*l 
        tim=t(im)             
 		lonm=lon(im)
		latm=lat(im)          
        dmax=2.*0.01*10.**(0.5*M(im))   ! max distance between ms and aft. ~ 2 rupture lengths
        fxy=0.   ! average rate above Mmin:
        iaft=im
        naft=0
		! for each aft. within a time TAFT of the mainshock i
		! and before foercast time tp
        do while(t(iaft).le.min(tp,tim+TAFT)) 
		   daftms=distance(lonm,latm,lon(iaft),lat(iaft))
           if (daftms.lt.dmax) then
              naft=naft+1   ! number of aftershocks
              if (iaft.eq.im) then
                 di=d(im)   ! for the mainshock use d = mainshock rupture length
              else
                 di=daft      ! use kernel  smoothing distance d =1 km for the aftershocks
              endif    
			  ! compute integral of spatial kernel of width di 
			  ! centered at lon(i),lat(i) over the cell ix,iy
			  f=feval(lon(iaft),lat(iaft),di,lonc,latc,l,kr);
              fxy=fxy+f
           endif    ! end. loop if distance to mainshock < dmax
           iaft=iaft+1
        enddo  ! next aftershock iaft
		fxy=fxy/naft
        end
!----------------------------------------
        subroutine evalbigaft    
!----------------------------------------
		! estimate the aftershock density of all large m>MBIG events
        implicit none
        integer nlmax,LX,LY,LMAX,NM,N,N1,is
		parameter (nlmax=200000) ! nlmax = maximum number of events in the catalog     
        parameter (LMAX=500)        ! max number of cells =LX*LY
        parameter (NM=100)        ! max number of 'mainshocks '(M>=MBIG)      
        integer IMS(NM),IAS(nlmax),naft,ix,iy
        real dKxy(LMAX,LMAX,NM)
        double precision t(nlmax)
 		real M(nlmax),lon(nlmax),lat(nlmax),rho(nlmax),d(nlmax) ! seismicity catalog
 		real latmin,latmax,lonmin,lonmax,zmax,l,daft,TAFT,MBIG 
    	common /cat/t,lon,lat,rho,d,M,N1
        common /BIG/dKxy,IMS,IAS
        common /BIGPAR/TAFT,daft,MBIG
 	    common /grid/latmin,latmax,lonmin,lonmax,zmax,l,LX,LY   	
		write(*,*) '--- Compute the spatial distribution of aftershocks of M>',MBIG,' mainshocks'        
		!-------- spatial distribution of aftershocks of large events
        N=1     ! number of past events ti<=t
        is=0    ! index of aftershock sequence
120     format(a,i6,a,f9.1,2(a,f8.2),a,f4.1,a,i6)    !  Npred. mean must be <10^4
        do while (N.le.N1) ! for each earthquake
			if (M(N).ge.MBIG) then  
				is=is+1		  ! index of aftershock sequence
				IMS(is)=N       ! mainshock index in the catalog
				IAS(N)=is       ! aftershock sequence index of mainshock N			  
		        !--- compute dKxy(ix,iy,is) 
				call bigaft(is,naft) 
			endif
			N=N+1
        enddo 
        end
!-------------------------------------------------------
       subroutine mapnpred
!      compute the map of predicted number  with m>Mmin in each cell and write output file
!-------------------------------------------------------
       implicit none
       integer LX,LY,LMAX,opt,ix,iy,i,j,nflag
       parameter (LMAX=500)       ! max number of cells =LMAX*LMAX
       double precision TFmin,TFmax,TF
       real Npxy(LMAX,LMAX),Np
	   real Mmin,Mmax,Mcorner,dm,Md,b0
	   real latmin,latmax,lonmin,lonmax,zmax,l
	   integer flagxy(LMAX,LMAX)
	   common /flag/flagxy,nflag
	   common /par/TFmin,TFmax,TF,Mmin,Mmax,Mcorner,dm,Md,b0
	   common /grid/latmin,latmax,lonmin,lonmax,zmax,l,LX,LY   	
	   common /Rxy/Npxy
 	   Np=0.
       !------- filename for the results (LL and predicted number at each time step)
  		write(*,*) '--- MAP: computing rate in all cells'
       do iy=1,LY  
		  if (mod(iy,10).eq.0) write(*,*) '--- MAP: iy=',iy,' LY=',LY
          do ix=1,LX	
			Npxy(ix,iy)=0.  
            if (flagxy(ix,iy).eq.1) then
				call predphixy(Npxy(ix,iy),ix,iy,TFmin)
				Np=Np+Npxy(ix,iy)
			endif
          enddo
       enddo
	   write (*,*)  ' --- MAP: TFmin=',TFmin,' TF=',TF,' Np=',Np
       end		
!-----------------------------------------------------------------------------
        subroutine readpar(inputfile)	  ! read input parameters and seismic catalog
!-----------------------------------------------------------------------------
        implicit none      
        integer LM,LMAX,LX,LY,s
        parameter (LM=100)		! number of magnitude bins
		parameter (LMAX=500)	! max grid size
		character *200 datafile,outdir,mufile,bfile,parfile,inputfile,catfile,csepfile,flagfile
        character *200 commoninputdir, scratchdir 
        character*2 kr
		real alpha,K,p,mu0,br,fd,M1,Md,b0,MM,c,Mmin,Mmax,Mcorner,dm,mub
		real w,sec,pm(LM),TAFT,daft,MBIG 
        double precision TFmin,TFmax,TF,T0
        integer i,j,im,ix,iy,yr,mo,day,ho,mn,seci
        real latmin,latmax,lonmin,lonmax,zmax,l,llat,llon,lon,lat
		integer flagxy(LMAX,LMAX),nflag,gflag(LMAX,LMAX)
		real b(LMAX,LMAX),lon1,lon2,lat1,lat2,b1,b2,mcg 
		logical dir_exist		
		common /bmap/b
		common /catfile/catfile
        common /par/TFmin,TFmax,TF,Mmin,Mmax,Mcorner,dm,Md,b0    
        common /grid/latmin,latmax,lonmin,lonmax,zmax,l,LX,LY   	
        common /ETAS/mub,alpha,K,p,c,br,fd
        common /DATE/sec,yr,mo,day,ho,mn 
        common /BIGPAR/TAFT,daft,MBIG      
        common /kernelaft/kr
		common /flag/flagxy,nflag
		common /csep/csepfile
		common /geysers/lon1,lon2,lat1,lat2,b1,b2,mcg,gflag 

	    open(12,file=inputfile)		! csep input file
		! --- time interval of forecast: from TFmin until TFmax		
		read(12,'(18x,i4,1x,i2,1x,i2,1x,i2,1x,i2,1x,i2)') yr,mo,day,ho,mn,seci
		sec=real(seci)
		call CDAY(TFmin)			   ! convert yr,mo,day,ho,mn,sec into decimal days since 1/1/1900
		read(12,'(16x,i4,1x,i2,1x,i2,1x,i2,1x,i2,1x,i2,3x)') yr,mo,day,ho,mn,seci
		sec=real(seci)
		call CDAY(TFmax)			   ! convert yr,mo,day,ho,mn,sec into decimal days since 1/1/1900
		read(12,'(17x,a)') datafile
		read(12,'(19x,a)') parfile
		read(12,'(19x,a)') outdir
		read(12,'(15x,a)') commoninputdir		
!		read(12,'(14x,a)') scratchdir		
		close(12)		
		
		! --- input and output files
!		catfile=trim(scratchdir)//'cat'
		csepfile=trim(outdir)
		TF = TFmax - TFmin		! time duration in days
		write(*,*) 'forecastStartDate= ',TFmin, 'forecastEndDate= ',TFmax,' TF= ',TF
		write(*,*) 'inputCatalogFile= ',trim(datafile)
		write(*,*) 'OutputDir= ',trim(outdir)
		write(*,*) 'Input parameter file= ',trim(parfile)
		write(*,*) 'Output csep xml file= ',trim(csepfile)
!		write(*,*) 'Output earthquake catalog= ',trim(catfile)
		! --- test if output directory exists, and create it if it does not exist
!		inquire( file=outdir, exist=dir_exist )
!		if ( .not. dir_exist ) then
!			call system('mkdir '//trim(outdir))	! you can comment this line if needed ...
!		endif
		! --- test if sratch output directory exists, and create it if it does not exist
!		inquire( file=scratchdir, exist=dir_exist )
!		if ( .not. dir_exist ) then
!			call system('mkdir '//trim(scratchdir))	! you can comment this line if needed ...
!		endif
			
		! --- read input parameter file. Input file "ETAS.par" must be present in the output directory
		open(11,file=parfile)
        read(11,*) mufile   	   ! background rate	
		mufile=trim(commoninputdir)//trim(mufile)
		read(11,*) flagfile		   ! flag for each cell, 0 or 1	
		read(11,*) bfile		   ! name of file containng the b map b(ix,iy) or 'X' to use b everywhere 
        read(11,*) kr              ! spatial kernel, power law ('pl') or gaussian('gs')
        read(11,*) daft			   ! smoothing distance (km) for aftershocks of large mainshocks
        read(11,*) TAFT			   ! time window (days) used to select aft. of M>MBIG EQ
        read(11,*) MBIG			   ! compute aftershock spatial distribution for m>MBIG EQs using early aftershcoks
		read(11,*) Md			   ! mag. detection threshold
        read(11,*) b0              ! exponent GR law for null hypothesis, and for model if no b map is given
        read(11,*) latmin,latmax   ! boundary of the grid
        read(11,*) lonmin,lonmax   ! 
		read(11,*) zmax			   ! max depth
		read(11,*) l			   ! grid size, degree   
        read(11,*) Mmin			   ! mag. min.prediction
        read(11,*) Mmax			   ! mag. min.prediction
        read(11,*) Mcorner		   ! corner magnitude (GR P(m) with an exponential cutoff)
        read(11,*) dm			   ! mag. step
        read(11,*) alpha	       ! increase of aftershock productivity with magnitude
		read(11,*) p			   ! Omori exponent
		read(11,*) K			   ! aftershock productivity
		read(11,*) mub			   ! background rate : number of m>Mmin EQs inside testing area for a time window TF
		read(11,*) fd			   ! relative size of aftershock zone
		read(11,*) c			   ! short-times cut-off of Omori's law (days)	
		read(11,*) yr,mo,day,ho,mn,sec ! select earthquakes between T0 and TFmin  
        call CDAY(T0)			       ! convert yr,mo,day,ho,mn,sec into decimal days since 1/1/1900
		read(11,*) lon1,lon2,lat1,lat2,b1,b2,mcg ! mag pdf in the Geysers area : b=b1 for m<mcg and b2 for m>mcg       
 		close(11)	
 

        !--- write summary of input parameters
  	    write(*,*) '% Md= ',Md,' b0= ',b0
        write(*,*) '% Mmin=',Mmin,' Mmax=',Mmax,' Mcorner=',Mcorner,' dm=',dm
        write(*,*) '% latmin=',latmin,' latmax=',latmax
        write(*,*) '% lonmin=',lonmin,' lonmax=',lonmax    
		write(*,*) '% zmax=',zmax,' l=',l
        write(*,*) '% kernel type:',kr  
        write(*,*) '% T0=',T0,' TFmin=',TFmin,' TFmax=',TFmax,' TF=',TF        
		write(*,'(a,a)') ' % background file: ',trim(mufile)
		write(*,'(a,a)') ' % bfile: ',trim(bfile)
        write(*,*) '% BIGAFT: d=',daft,' M=',MBIG,' T=',TAFT 
		write(*,*) '% ETAS parameters: alpha=',alpha, ' p=',p,' c=',c,' fd=',fd,' mub=',mub,' K=',K
     	
		!--- grid size
		LX=nint((lonmax-lonmin)/l) !LX in longitude
		LY=nint((latmax-latmin)/l) !LY is number of cells in latitude
	        write(*,*) 'LX=',LX,' LY=',LY
	    !--- flag : =1 for cells within the testing area, 0 outside
		nflag=0
		if (flagfile(1:1).eq.'X') then	   ! set flag =1 for all cells
			write(*,*) '--- using flag=1 for all cells'
			do ix=1,LX                     ! longitude						
				do iy=1,LY		           ! latitude						
					flagxy(ix,iy)=1		   ! =1 to compute density in cell i,j
					nflag=nflag+1
				enddo							
			enddo	
		else						
			write(*,*) '--- reading flag file: ',trim(flagfile)
			open(13,file=flagfile)		   ! read flag file (0 or 1 for each cell)
			do ix=1,LX                     ! longitude						
				do iy=1,LY		           ! latitude						
					read(13,*) flagxy(ix,iy) ! =1 to compute density in cell i,j
					if (flagxy(ix,iy).eq.1) nflag=nflag+1
				enddo							
			enddo							
			close(13)
		endif
		!--- flag for Geysers area
		lon=lonmin+l/2.
		s=0
		do ix=1,LX                     ! longitude		
			lat=latmin+l/2.				
			do iy=1,LY		           ! latitude		
				if (lon.ge.lon1 .and. lon.le.lon2 .and. lat.ge.lat1 .and. lat.le.lat2) then					
					gflag(ix,iy)=1
					write(*,*) '--- cell ix=',ix,' iy=',iy,' lat=',lat,' lon=',lon,' inside Geysers area' 
					s=s+1
				else
					gflag(ix,iy)=0
				endif
				lat=lat+l  
			enddo	
			lon=lon+l					! longitude center of cell			
		enddo
		if (s.gt.0) write(*,*) '--- found ',s,' cells inside special Geysers area'
		!--- b-value
		if (bfile(1:1).eq.'X') then			! set b=b0 for all cells
			write(*,*) '--- use b=',b0,' for all cells'	
			do ix=1,LX                     ! longitude						
				do iy=1,LY		           ! latitude	
					b(ix,iy)=b0					
				enddo							
			enddo							
		else		! read b map from file
			write(*,*) '--- reading bfile ',trim(bfile)	
			! one b-value per line, no header
			! values <0 are replaced by b0
			open(14,file=bfile)		       ! read b-value file
			do ix=1,LX                     ! longitude						
				do iy=1,LY		           ! latitude						
					read(14,*) b(ix,iy)		
					if (b(ix,iy).lt.0.) 	b(ix,iy)=b0			
				enddo							
			enddo							
			close(14)
 		endif
		
		!--- read background map
		call readbg(mufile)
		
		!--- read earthquake catalog
		call readcat(datafile,outdir,T0)	
		end
		
!-----------------------------------------------------------------------------		
		subroutine readbg(mufile)
!		read background map
!		mub : background rate of m>=Mmin for a time window TF (in days) inside testing area	
!		mu = rate of target events in each cell >=Mmin per time window TF (in days)
!-----------------------------------------------------------------------------		
        implicit none      
        integer i,j,LMAX,LX,LY
 		parameter (LMAX=500)
		character *200 mufile
        real latmin,latmax,lonmin,lonmax,zmax,l
		real mu(LMAX,LMAX),mubg(LMAX,LMAX),muyx,s
		real mub,alpha,K,p,c,br,fd,Np
		integer flagxy(LMAX,LMAX),nflag
        common /grid/latmin,latmax,lonmin,lonmax,zmax,l,LX,LY   
		common /ETAS/mub,alpha,K,p,c,br,fd  ! ETAS param. 
		common /flag/flagxy,nflag
		common /MU/mubg
		! mub: background rate of M>Mmin earthquakes inside testing area for a time window TF in days
		open(10,file=mufile)
        do i=1,4						! skip first 4 lines (comments)
           read(10,*) 
        enddo  
		s=0.
        do  i=1,LX
           do j=1,LY
              read(10,*) muyx      ! background 
			  if (flagxy(i,j).eq.1) then
				s=s+muyx
				mu(i,j)=muyx
			   else
				mu(i,j)=0.
			  endif
           enddo
        enddo
		! normalisation
		do  i=1,LX
			do j=1,LY
				mubg(i,j)=mu(i,j)*mub/s ! background rate of M>Mmin earthquakes in cell i,j per time window TF in days
			enddo
        enddo		
		close(10)
		end
		
!-----------------------------------------------------------------------------
        subroutine readcat(datafile,outdir,T0)	  ! read EQ catalog
!-----------------------------------------------------------------------------
        implicit none      
        integer i,nlmax,N1,yr,mo,day,ho,mn,LX,LY
        parameter (nlmax=200000) ! nlmax = maximum number of events       
		character *200 datafile,outdir,mufile,bfile,filemap,catfile,csepfile
        double precision t(nlmax),st(nlmax),tt,T0,TFmin,TFmax,TF,yrf
        real M(nlmax),lon(nlmax),lat(nlmax),rho(nlmax),d(nlmax),mm,z
 		real latmin,latmax,lonmin,lonmax,zmax,l,llat,llon,sec
  		real x,mof,dayf,hof,mnf,Mmin,Mmax,Mcorner,dm,Md,b0
        common /grid/latmin,latmax,lonmin,lonmax,zmax,l,LX,LY    	
		common /cat/t,lon,lat,rho,d,M,N1
		common /par/TFmin,TFmax,TF,Mmin,Mmax,Mcorner,dm,Md,b0
        common /DATE/sec,yr,mo,day,ho,mn        
		!------ reading "old" catalog (with values of mc, rho, Ir...) if it exists 
		i=0
		!------ read earthquake catalog in ZMAP format          
		open(10,file=datafile)
		write(*,*) '--- reading EQ catalog: ',trim(datafile),' i=',i
        open(10,file=datafile)
		do while (.true.)
		   read(10,*, end = 2) llon,llat,yrf,mof,dayf,mm,z,hof,mnf,sec
		   yr=int(yrf) ; mo=int(mof); day=int(dayf); ho=int(hof); mn=int(mnf)
		   call CDAY(tt)  ! convert yr/mo/day/ho/mn/sec into days
		   ! ---- select M>m=d, t>=T0, and within 1 degree from the grid boundary
           if (tt.lt.TFmin .and. tt.ge.T0 .and. mm.ge.md.and.
     &		   llat.ge.latmin-1.and.llat.le.latmax+1.and.
     &		   llon.ge.lonmin-1.and.llon.le.lonmax+1. .and. z.le.zmax  .and.tt.gt.t(i)) then 
               i=i+1
               M(i)=mm
               t(i)=tt
               lon(i)=llon
               lat(i)=llat
            endif    
		enddo
2       close(10)
		N1=i
 	    write(*,*) '--- readcat: m>',Md,' N=',N1,' t(1)=',t(1),' t(end)=',t(N1)
        end     
!----------------------------------------------------- 		
		subroutine write_output_cat
!----------------------------------------------------- 
		implicit none        
        integer nlmax,i,N1
        parameter (nlmax=200000) ! nlmax = maximum number of events				
        double precision t(nlmax)
        real M(nlmax),lon(nlmax),lat(nlmax),d(nlmax),rho(nlmax)
		character *200 catfile
		common /catfile/catfile
		common /cat/t,lon,lat,rho,d,M,N1
117     format(f12.6,2(1x,f9.3),1x,f5.2,1x,f8.2,1x,g10.4)   
!		write(*,*) '--- Write_output_cat : ',trim(catfile)
!		open(17,file=catfile) 
!		write(17,*) '%   t          lat        lon     mag    d     rho '
!		do i=1,N1
!			write(17,117) t(i),lat(i),lon(i),M(i),d(i),rho(i)
!		enddo
!		close(17)
		end
!-----------------------------------------------------------------------------
        subroutine csep_template(inputfile)	  ! fills out CSEP template
!-----------------------------------------------------------------------------
        implicit none  
		integer LX,LY,LM,LMAX,ix,iy,im,nflag
		parameter (LMAX=500)		! max grid size
		real Rxy(LMAX,LMAX),Mmin,Mmax,Mcorner,dm,Md,b0,mu
		real latmin,latmax,lonmin,lonmax,zmax,l,lon,lat,mag,pm,f
		double precision TFmin,TFmax,TF
		character *200 file,inputfile
		character*22 forecastStartDate, forecastEndDate
		integer flagxy(LMAX,LMAX)
		common /csep/file
		common /grid/latmin,latmax,lonmin,lonmax,zmax,l,LX,LY    	
		common /Rxy/Rxy
		common /flag/flagxy,nflag	
	    common /par/TFmin,TFmax,TF,Mmin,Mmax,Mcorner,dm,Md,b0  
		! --- issue date (local time?):
		character(8)  :: da
		character(10) :: ti
		character(5)  :: zone
		integer,dimension(8) :: values
		call date_and_time(da,ti,zone,values)
		! --------- header
		open(18,file=file) 
		write(18,'(a,a,a)') '<?xml version=''1.0'' encoding=''UTF-8''?>'
		write(18,'(a)') '<CSEPForecast xmlns=''http://www.scec.org/xml-ns/csep/forecast/0.1''>'
		write(18,'(a)') '  <forecastData publicID=''smi:org.scec/csep/forecast/1''>'
		write(18,'(3(a,f3.1),a)') '    <modelName>ETAS_Md',Md,'_Mmin',Mmin,'_T',TF,' </modelName>'
		write(18,'(a)') '    <version>1.0</version>'
		write(18,'(a)') '    <author>Helmstetter and Werner</author>'
		write(18,'(13a)') '    <issueDate>',da(1:4),'-',da(5:6),'-',da(7:8),'T',ti(1:2),':',ti(3:4),':',ti(5:6),'L</issueDate>'
		open(12,file=inputfile)
		read(unit=12,fmt = "(18x,22a)") forecastStartDate
		write(18,'(a,20a,a)') '    <forecastStartDate>',trim(forecastStartDate),'</forecastStartDate>'
		read(unit=12,fmt = "(16x,22a)") forecastEndDate
		write(18,'(a,20a,a)') '    <forecastEndDate>',trim(forecastEndDate),'</forecastEndDate>'
		close(12)
		write(18,'(2(a,f4.2),a)') '    <defaultCellDimension latRange=''',l,''' lonRange=''',l,'''/>'
		write(18,'(a,f4.2,a)') '    <defaultMagBinDimension>',dm,'</defaultMagBinDimension>'
		write(18,'(a)') '    <lastMagBinOpen>1</lastMagBinOpen>'
		write(18,'(2(a,f5.1),a)') '    <depthLayer max=''',zmax,''' min=''',0.0,'''>'
		! --------- forecasted rate per bin
		LM=int((Mmax-Mmin)/dm) +1
		lon=lonmin+l/2
		do ix=1,LX
			lat=latmin+l/2
			do iy=1,LY
				if (flagxy(ix,iy).eq.1) then
					write(18,'(a,f7.3,a,f8.3,a)')'     <cell lat=''',lat,''' lon=''',lon,'''>'
					mag=Mmin+dm/2.
					do im=1,LM
						call get_pm(im,ix,iy,pm)
						mu=Rxy(ix,iy)*pm
						write(18,'(a,f3.1,a,g9.3,a)') '        <bin m=''',mag,'''>',mu,'</bin>'
						mag=mag+dm
					enddo
					write(18,'(a)') '      </cell>'
				endif
				lat=lat+l
			enddo
			lon=lon+l
		enddo
		write(18,'(a)') '    </depthLayer>'
		write(18,'(a)') '  </forecastData>'
		write(18,'(a)') '</CSEPForecast>'
		close(18)
		end
