! written by Agnes Helmstetter
! last modified June 2012
!
! Compilation (g95,g77 or gfortran) 
! g95  F/csepK2.f F/common.f -o csepK2 -lm  -ffixed-line-length-132 -fimplicit-none
!
! Execution:
!	 csepK2 INPUT/csep-input-file-K2
!--------------------------------------------------------
        program cespK2
!--------------------------------------------------------
        implicit none
		character *200 inputfile
		call getarg(1, inputfile)

		!------ read input parameters in inputfile. Also read EQ catalog and background rate
		call readpar(inputfile)         
	 
		! ---- estimate smoothing time sd and distance sd
		call get_stsd
		
		! ---- Estimate the completeness magnitude from map mc(x,y) + transcient increase after m>5.5 EQs
		call evalMceq
		
		! ---- weigth associated with each EQ to account for Mmin # Mc(t,r)
		call get_wm		
			
		! ---- write output catalog and kernel bandwidth		
		call write_output_cat	    ! write EQ catalogs and smoothing time and distance	
		
		!------ compute the predicted rate per cell
		call mapnpred

		!------ csep output xml file
	    call csep_template(inputfile)				
		end		

!------------------------------------------------------
		subroutine get_stsd
!		Estimate the smoothing time st and distance sd for each earthquake
!-------------------------------------------
        implicit none
        integer nlmax,i,j,l,N1,k,n,p,q
 		parameter (nlmax=200000) ! nlmax = maximum number of events in the catalog  
		real Nmin,fN,alpha,loni,lati
        double precision st(nlmax),t(nlmax),dt,s1,s2,ti
        real M(nlmax),Mc(nlmax),lon(nlmax),lat(nlmax)		! seismicity catalog
		real sd(nlmax),wm(nlmax)				! smoothing distance
		real h,hm,d1(nlmax),d,dk,dm,distance,dmin
		external distance
		common /modpar/fN,Nmin,alpha,dmin,k
  		common /cat/t,lon,lat,st,sd,wm,M,Mc,N1   
		st(1)=k*(t(N1)-t(1))/N1
		sd(1)=k*distance(lon(2),lat(2),lon(1),lat(1))
		do i=2,N1
			loni=lon(i)
			lati=lat(i)
			ti=t(i)
			j=i-1
			n=1
			do n=1,k+1
				d1(n)=1e10
			enddo
			! ---- first k events before event i
			do while (j.ge.1 .and. n.le.k)
				d=distance(loni,lati,lon(j),lat(j))
				! sort d1 values by ascending order and keep first k values
				p=n
				do while (d1(p).gt. d .and. p.gt.0) 
					p=p-1 
				enddo
				p=p+1
				do q=n+1,p,-1
					d1(q)=d1(q-1)
				enddo
				d1(p)=d
				j=j-1
				n=n+1
			enddo
			l=j-1
			dt=ti-t(j+1)
			dk=d1(n-1)
			dm=dk
			hm=dt+alpha*dm
			! write(*,*) '* i=',i,' j=',j+1,' dt=',dt,' dm=',dm,' dk=',dk,' hm=',hm
			! --- previous events: 
			do while (j.ge.1 .and. dt.lt.hm)
				dt=ti-t(j)
				d=distance(loni,lati,lon(j),lat(j))
				! sort d1 values by ascending order and keep first k values		
				p=k
				do while (d1(p).gt. d .and. p.gt.0) 
					p=p-1 
				enddo
				p=p+1
				do q=k+1,p,-1
					d1(q)=d1(q-1)
				enddo
				d1(p)=d
				dk=d1(k)
				h=dt+alpha*dk
				if (h.lt.hm) then
					l=j
					hm=h
					dm=dk
				endif
				j=j-1
			enddo 
			st(i)=ti-t(l)
			sd(i)=dm
			sd(i)=max(dmin,sd(i))	! to avoid sd=zero and pb of location resolution and location error
			if (mod(i,10000).eq.0) then
				write(*,*) 'Gst: i=',i,' n=',n,' s=',st(i),' d=',sd(i),' h=',hm,' dt=',dt
			endif
		enddo
		end
!-------------------------------------------------------------  
        subroutine evalMceq
!-------------------------------------------------------------        
       ! evaluate the magnitude of completeness of the catalog at the time of each earthquake	
        implicit none        
        integer nlmax,i,j,N1
        parameter (nlmax=200000) ! nlmax = maximum number of events       
        double precision t(nlmax),st(nlmax),tau,tM,TFmin,TFmax,TF
        real M(nlmax),Mc(nlmax),lon(nlmax),lat(nlmax)
		real sd(nlmax),wm(nlmax)
		real Md,Mmin,Mmax,Mcorner,dm,b0,MA,MM
 		common /cat/t,lon,lat,st,sd,wm,M,Mc,N1  
		common /par/TFmin,TFmax,TF,Mmin,Mmax,Mcorner,dm,Md,b0
		do i=1,N1
			Mc(i)=Md
			! set Mc(i) equal to interpolated mcxy from map
			call interp2(lon(i),lat(i),Mc(i))
        enddo
		! transient increase of mc after large earthquakes (see Helmstetter et al 2005,2006,2007)
        do i=1,N1-1
           if(M(i).ge.5.) then
				 tM=t(i)
                 MM=M(i)
                 j=i+1
                 tau=t(j)-tM
                 call Mcomp(tau,MM,MA)  ! MA = completeness mag. at t(j)
                 Mc(j)=max(MA,Mc(j))
                 do while(MA.gt.min(Mmin,Md).and.j.lt.N1) ! for each aftershock
                     j=j+1
                     tau=t(j)-tM
                     call Mcomp(tau,MM,MA)
                     Mc(j)=max(MA,Mc(j))
                 enddo
            endif
        enddo
        end

!-------------------------------------------------------
		subroutine  get_wm
!		estimate weigth for each EQ, to account for variations of completeness magnitude	
!-------------------------------------------------------
		implicit none
        integer nlmax,j,N1,LMAX,ix,iy,LX,LY
		parameter (nlmax=200000)  ! nlmax = maximum number of events in the catalog     
		parameter (LMAX=500)      ! max grid size  
		real Mmin,Mmax,Mcorner,dm,Md,b0
        double precision t(nlmax),st(nlmax),TFmin,TFmax,TF
        real M(nlmax),Mc(nlmax),lon(nlmax),lat(nlmax),sd(nlmax),wm(nlmax)
		real b(LMAX,LMAX),bj		    ! GR b-value
		real latmin,latmax,lonmin,lonmax,zmax,l
		real lon1,lon2,lat1,lat2,b1,b2,mcg,P				! geysers. b=b1 if m<mc and b2 if m>mc
		integer gflag(LMAX,LMAX)
		common /geysers/lon1,lon2,lat1,lat2,b1,b2,mcg,gflag
		common /par/TFmin,TFmax,TF,Mmin,Mmax,Mcorner,dm,Md,b0
  	    common /cat/t,lon,lat,st,sd,wm,M,Mc,N1  
        common /grid/latmin,latmax,lonmin,lonmax,zmax,l,LX,LY		
		common /bmap/b      
		do j=1,N1
			ix=(lon(j)  + 0.0001 -lonmin)/l+1
			iy=(lat(j)  + 0.0001 -latmin)/l+1
			if (ix.gt.0 .and. ix.le.LX .and. iy.gt.0 .and. iy.le.LY) then
				bj=b(ix,iy)
				if (bj.le.0) bj=b0
			else
				bj=b0
			endif
			wm(j)=1.
			! --- correction to account for missing events in the learning catalog
			if (M(j).ge.Mc(j) .and. Mc(j).gt.Md) then
				wm(j)=10.**(bj*(Mc(j)-Md))  ! weigth >1, increases with mc
			endif
			! ---- special correction for the geysers
			if (gflag(ix,iy).eq.1) then  
				call get_pm_GRb1b2(P,Md,Mmax,Mmin,Mmax,mcg,b1,b2)
			else
				P=10.**(-bj*(Mmin-Md)) ! <=1, fraction of learning EQs  with mag > mmin
			endif
			wm(j)=wm(j)*P
			! --- correction to account for missing events in the target catalog
			if (Mmin.le. Mc(j)) then
				wm(j)=wm(j)*10.**(-bj*(Mc(j)-Mmin))  ! weigth <1, decreases with mc
		    endif
			
			! --- for events below completeness, set weigth to 0
			! if (M(j).lt.Mc(j)) then; wm(j)=0.; endif
	    enddo   
		end

!---------------------------------------------------------------
       subroutine predphixytp(dK,ix,iy,tp)
!---------------------------------------------------------------
!       evaluate the rate of aftershocks dK in cell (ix,iy) with m>Mmin per day
!       at time tp due to all N1 past eqs. and to background mu
        implicit none
        integer nlmax,LX,LY,LMAX,NM
		parameter (nlmax=200000)   ! nlmax = maximum number of events in the catalog     
        parameter (LMAX=500)         ! max number of cells =LMAX*LMAX
        parameter (NM=1000)        ! max number of 'mainshocks' (M>=MBIG) 
		integer i,j,k,N,N1,ix,iy,IMS(NM),IAS(nlmax),im
		character*2 kr
        double precision st(nlmax),t(nlmax),dt,tp
		real f,feval,lonc,latc
		real fN,Nmin,alpha,dmin
        real dKxy(LMAX,LMAX,NM),mu(LMAX,LMAX),dK,w,sq2pi
        real latmin,latmax,lonmin,lonmax,zmax,l
        real M(nlmax),Mc(nlmax),lon(nlmax),lat(nlmax),sd(nlmax),wm(nlmax)
		common /modpar/fN,Nmin,alpha,dmin,k
		common /cat/t,lon,lat,st,sd,wm,M,Mc,N1        
        common /grid/latmin,latmax,lonmin,lonmax,zmax,l,LX,LY
		common /bg/mu
		common /kernel/kr		 ! spatial kernel, 'pl' or 'gs'	
		lonc=lonmin+(ix-0.5)*l   ! longitude of center of cell ix,iy
        latc=latmin+(iy-0.5)*l 
		dK=Nmin*mu(ix,iy)			  ! stationnary rate per cell for M>Min per time unit
		sq2pi=sqrt(2.*3.14159265)/2.  ! normalizing factor for a half Gaussian kernel
		i=1
		do while (t(i).lt.tp  .and. i .le. N1)	 ! for each past event
            dt=tp-t(i)			 ! time between prediction time tp and earthquake i
			if (dt .le. 5. * st(i) ) then
				w=exp(-dt**2/2./st(i)**2)/st(i)/sq2pi	! gaussian kernel
				if (w.gt.0.) then
					f=feval(lon(i),lat(i),sd(i),lonc,latc,l,kr) 
					dK=dK+f*w*wm(i)		  ! rate in the cell ix,iy at time tp due to event i	
				endif
			endif
			i=i+1				! next event i
       enddo				
	   dK=dK*fN				    ! corrective factor
	   ! write(*,*) tp,ix,iy,dK,Nmin*mu(ix,iy),f,w
	   end       
!-------------------------------------------------------
       subroutine mapnpred
!      compute the map of predicted rate/day at time tp with m>Mmin in each cell
!-------------------------------------------------------
       implicit none
       integer LX,LY,LMAX,nflag,im,ix,iy
       parameter (LMAX=500)       ! max number of cells in each direction X or Y
       real Npxy(LMAX,LMAX),Np
	   real Mmin,Mmax,Mcorner,dm,Md,b0
	   double precision TFmin,TFmax,TF
	   integer flagxy(LMAX,LMAX)
	   real latmin,latmax,lonmin,lonmax,zmax,l
 	   common /par/TFmin,TFmax,TF,Mmin,Mmax,Mcorner,dm,Md,b0
	   common /grid/latmin,latmax,lonmin,lonmax,zmax,l,LX,LY 	
	   common /flag/flagxy,nflag
	   common /Rxy/Npxy
       Np=0.
      !------- filename for the results (LL and predicted number at each time step)
		write(*,*) '--- MAP: computing rate in all cells'
		! computes expected rate in all space and magnitude bins ix,iy,im
        do iy=1,LY  
		  if (mod(iy,10).eq.0) write(*,*)'--- MAP: iy=',iy,' LY=',LY,' Np=',Np
          do ix=1,LX		 
			Npxy(ix,iy)=0.  
			if (flagxy(ix,iy).eq.1) then	
				call predphixytp(Npxy(ix,iy),ix,iy,TFmin)
				Np=Np+Npxy(ix,iy)
			 endif
          enddo
       enddo
	   write (*,*)  '--- MAP: TFmin=',TFmin,' TF=',TF,' Np=',Np
       end	   
!----------------------------------------------------- 		
		subroutine write_output_cat
!----------------------------------------------------- 
		implicit none        
        integer nlmax,i,N1
        parameter (nlmax=200000) ! nlmax = maximum number of events				
        double precision st(nlmax),t(nlmax)
        real M(nlmax),Mc(nlmax),lon(nlmax),lat(nlmax),wm(nlmax)
        real sd(nlmax)		! smoothing time and distance
		character*200 catfile
		common /catfile/catfile
		common /cat/t,lon,lat,st,sd,wm,M,Mc,N1
117     format(f12.6,2(1x,f9.3),2(1x,f5.2),3(1x,g12.4))   
!		write(*,*) '--- Write_output_cat : ',trim(catfile)
!		open(17,file=catfile) 
!		write(17,*) '%   t         lat       lon       mag    mc	  st          sd           wm'
!		do i=1,N1
!			write(17,117) t(i),lat(i),lon(i),M(i),Mc(i),st(i),sd(i),wm(i)
!		enddo
!		close(17)
		end
		
!-----------------------------------------------------------------------------
        subroutine readpar(inputfile)	  ! read input parameters and seismic catalog
!-----------------------------------------------------------------------------
        implicit none      
        integer LX,LY,LMAX,LM,NM,k,Nxmc,Nymc
		parameter (LMAX=500)		! grid size
		parameter (LM=100)			! max number of magnitude bins
		character*200 datafile,catfile,flagfile,outdir,csepfile
		character*200 mufile,mcfile,bfile,parfile,inputfile
                character *200 commoninputdir, scratchdir
   		character*2 kr				! spatial kernel
		logical dir_exist
 		real Md,b0,dm,Mmin,Mmax,Mcorner,sec,dmin
		real Nmin,fN,alpha			! model parameters
		double precision T0,TFmin,TFmax,TF
        integer i,j,s,im,ix,iy,yr,mo,day,ho,mn,seci
		integer gflag(LMAX,LMAX),flagxy(LMAX,LMAX),nflag
        real latmin,latmax,lonmin,lonmax,zmax,l,lon,lat
		real lonmc(LMAX),latmc(LMAX),mcxy(LMAX,LMAX),dxmc,dymc,latmcmin,lonmcmin ! map Mc(x,y)
		real b(LMAX,LMAX),lon1,lon2,lat1,lat2,b1,b2,mcg
		common /csep/csepfile
		common /par/TFmin,TFmax,TF,Mmin,Mmax,Mcorner,dm,Md,b0    
        common /grid/latmin,latmax,lonmin,lonmax,zmax,l,LX,LY 	
		common /geysers/lon1,lon2,lat1,lat2,b1,b2,mcg,gflag 
		common /catfile/catfile
		common /DATE/ sec, yr, mo, day, ho,mn 
		common /mcmap/lonmc,latmc,mcxy,Nxmc,Nymc	! used in interp2
		common /bmap/b
		common /flag/flagxy,nflag
		common /modpar/fN,Nmin,alpha,dmin,k
		common /kernel/kr		     ! spatial kernel, 'pl' or 'gs'	
		
		open(12,file=inputfile)		! csep input file
		! --- time interval of forecast: from TFmin until TFmax		
		read(12,'(18x,i4,1x,i2,1x,i2,1x,i2,1x,i2,1x,i2)') yr,mo,day,ho,mn,seci
		sec=real(seci)
		call CDAY(TFmin)			   ! convert yr,mo,day,ho,mn,sec into decimal days since 1/1/1900
		read(12,'(16x,i4,1x,i2,1x,i2,1x,i2,1x,i2,1x,i2,3x)') yr,mo,day,ho,mn,seci
		sec=real(seci)
		call CDAY(TFmax)			   ! convert yr,mo,day,ho,mn,sec into decimal days since 1/1/1900
		read(12,'(17x,a)') datafile
		read(12,'(19x,a)') parfile
		read(12,'(19x,a)') outdir
		read(12,'(15x,a)') commoninputdir		
!		read(12,'(14x,a)') scratchdir		
		close(12)		
		
		! --- input and output files
!		catfile=trim(outdir)//'cat'
		csepfile=trim(outdir)
		TF=(TFmax - TFmin)	! time duration in days 
		write(*,*) 'forecastStartDate= ',TFmin, 'forecastEndDate= ',TFmax,' TF= ',TF,' days'
		write(*,*) 'inputCatalogFile= ',trim(datafile)
		write(*,*) 'OutputDir= ',trim(outdir)
		write(*,*) 'Input parameter file= ',trim(parfile)
		write(*,*) 'Output csep xml file= ',trim(csepfile)
!		write(*,*) 'Output earthquake catalog= ',trim(catfile)

		! --- test if output directory exists, and create it if it does not exist
!		inquire( file=outdir, exist=dir_exist )
!		if ( .not. dir_exist ) then
!			call system('mkdir '//outdir)	! you can comment this line if needed ...
!		endif
!		inquire( file=scratchdir, exist=dir_exist )
!		if ( .not. dir_exist ) then
!			call system('mkdir '//trim(scratchdir))	! you can comment this line if needed ...
!		endif
		
		! --- read input parameter file. Input file "K3.par" must be present in the output directory
		open(11,file=parfile)
		read(11,*) mufile  	       ! background rate
		mufile=trim(commoninputdir)//trim(mufile)
		read(11,*) mcfile  	       ! background rate
		read(11,*) flagfile		   ! flag for each cell, 0 or 1	
		read(11,*) bfile		   ! flag for each cell, 0 or 1	
        read(11,*) kr              ! spatial kernel, power law ('pl') or gaussian('gs')
 		read(11,*) Md		       ! min mag. for input EQs
        read(11,*) b0              ! exponent GR law for null hypothesis, and for model if no b map is given
        read(11,*) latmin,latmax   ! grid boundary
        read(11,*) lonmin,lonmax   ! grid boundary
		read(11,*) zmax			   ! max depth
        read(11,*) l			   ! grid size, degree   
		read(11,*) dmin			   ! minimum value of smoothing distance, equal to localiszation error or resolution 	
        read(11,*) Mmin			   ! mag. min. prediction
        read(11,*) Mmax			   ! mag. min. prediction
        read(11,*) Mcorner		   ! corner magnitude (GR P(m) with an exponential cutoff)
        read(11,*) dm			   ! mag. step
 		read(11,*) k			   ! number of "near-neighbours"
		read(11,*) alpha		   ! relative importace of time and space 	
		read(11,*) Nmin			   ! stationnary rate f EQ >Mmin inside testing area /day
		read(11,*) fN			   ! correcting factor
		read(11,*) yr,mo,day,ho,mn,sec ! select earthquakes between T0 and TFmin  
        call CDAY(T0)			       ! convert yr,mo,day,ho,mn,sec into decimal days since 1/1/1900
        close(11)

       !--- write summary of input parameters
  	    write(*,*) '% Md= ',Md,' b0= ',b0
  	    write(*,*) '% Mmin=',Mmin,' Mmax=',Mmax,' Mcorner=',Mcorner,' dm=',dm     
        write(*,*) '% latmin=',latmin,' latmax=',latmax
        write(*,*) '% lonmin=',lonmin,' lonmax=',lonmax    
		write(*,*) '% zmax=',zmax,' l=',l
		write(*,*) '% kernel type:',kr  
        write(*,*) '% T0=',T0,' TFmin=',TFmin,' TFmax=',TFmax,' TF=',TF        
		write(*,'(a,a)') ' % background file: ',trim(mufile)
		write(*,'(a,a)') ' % bfile: ',trim(bfile)
 		write(*,*) '% K2 parameters : k=',k,' alpha=',alpha,' Nmin=',Nmin,' fN=',fN 
				
		!----- grid size
		LY=nint((latmax-latmin)/l) ! LY is number of cells in latitude
 		LX=nint((lonmax-lonmin)/l) ! LX in longitude
		      		
	    !---- flag : =1 for cells within the testing area, 0 outside
		nflag=0
		if (flagfile(1:1).eq.'X') then	   ! set flag =1 for all cells
		write(*,*) '-- using flag=1 for all cells'
			do ix=1,LX                     ! longitude						
				do iy=1,LY		           ! latitude						
					flagxy(ix,iy)=1		   ! =1 to compute density in cell i,j
					nflag=nflag+1
				enddo							
			enddo	
		else						
			write(*,*) '-- reading flag file: ',trim(flagfile)
			open(13,file=flagfile)		   ! read flag file (0 or 1 for each cell)
			do ix=1,LX                     ! longitude						
				do iy=1,LY		           ! latitude						
					read(13,*) flagxy(ix,iy)  ! =1 to compute density in cell i,j
					if (flagxy(ix,iy).eq.1) nflag=nflag+1
				enddo							
			enddo							
			close(13)
		endif	
		! ---- look for cells inside spacial zone, wo-ith a different magnitude distirubiton (ex: Geysers in Calif)
		lon=lonmin+l/2.
		s=0
		do ix=1,LX                     ! longitude		
			lat=latmin+l/2.				
			do iy=1,LY		           ! latitude		
				if (lon.ge.lon1 .and. lon.le.lon2 .and. lat.ge.lat1 .and. lat.le.lat2) then					
					gflag(ix,iy)=1
					write(*,*) '-- cell ix=',ix,' iy=',iy,' lat=',lat,' lon=',lon,' inside Geysers area' 
					s=s+1
				else
					gflag(ix,iy)=0
				endif
				lat=lat+l  
			enddo	
			lon=lon+l								
		enddo
		if (s.gt.0) then 
			write(*,*) '-- found ',s,' cells inside special magnitude area'
		endif
		! --- map of completeness magnitude
		if (mcfile(1:1).eq.'X') then			! set mc=md for all cells
			Nxmc=0
			mcxy(1,1)=Md	
			write(*,*) '-- use mc=',Md,' for all cells'	
		else		! read mc map from file
			open(21,file=mcfile)
			read(21,*)											! first line is filename_background_model
			read(21,*) lonmcmin,latmcmin,dxmc,dymc,Nxmc,Nymc	! cell size and number of rows and lines
			write(*,*) '-- reading mcfile ',mcfile,' NX=',Nxmc,' NY=',Nymc
			write(*,*) '    latmin=',latmcmin,' lonmin=',lonmcmin
			do i=1,Nxmc
				lonmc(i)=lonmcmin + (i-1)*dxmc
				do j=1,Nymc 
				  read(21,*) mcxy(i,j)
			   enddo
			enddo
			do j=1,Nymc 
				latmc(j)=latmcmin + (j-1)*dymc
			enddo
			close(21)	
		endif
		
		! --- b-value
		if (bfile(1:1).eq.'X') then			! set b=b0 for all cells
			write(*,*) '-- use b=',b0,' for all cells'	
			do ix=1,LX                     ! longitude						
				do iy=1,LY		           ! latitude	
					b(ix,iy)=b0					
				enddo							
			enddo							
		else		! read mc map form file
			write(*,*) '-- reading bfile ',bfile	
			! one b-value per line, no header
			! values <0 are replaced by b0
			open(14,file=bfile)		       ! read b-value file
			do ix=1,LX                     ! longitude						
				do iy=1,LY		           ! latitude						
					read(14,*) b(ix,iy)		
					if (b(ix,iy).lt.0.) 	b(ix,iy)=b0			
				enddo							
			enddo							
			close(14)
		endif
		
		!----  read background rate
		call readbg(mufile)					
	
		!--- read earthquake caatalogs
		call readcat(datafile,outdir,T0)      
		end
		
!-----------------------------------------------------------------------------
        subroutine readcat(datafile,outdir,T0)	  ! read EQ catalog
!-----------------------------------------------------------------------------
        implicit none      
        integer i,nlmax,N1,yr,mo,day,ho,mn,LX,LY
		parameter (nlmax=200000) ! nlmax = maximum number of events       
		character*200 datafile,outdir
        double precision t(nlmax),st(nlmax),tt,T0,TFmin,TFmax,TF,yrf
        real M(nlmax),Mc(nlmax),wm(nlmax),lon(nlmax),lat(nlmax),sd(nlmax)
 		real latmin,latmax,lonmin,lonmax,zmax,l,llat,llon,sec,Mmin,Mmax,Mcorner,Md,dm,b0
  		real x,mof,dayf,hof,mnf,mm,z
        common /grid/latmin,latmax,lonmin,lonmax,zmax,l,LX,LY    	
 		common /cat/t,lon,lat,st,sd,wm,M,Mc,N1  
		common /par/TFmin,TFmax,TF,Mmin,Mmax,Mcorner,dm,Md,b0
        common /DATE/sec,yr,mo,day,ho,mn        
		!------ reading "old" catalog (with values of mc, rho, Ir...) if it exists 
		i=0
		!------ read earthquake catalog for "new" events in ZMAP format         
		write(*,*) '--- reading EQ catalog: ',trim(datafile)
        open(10,file=datafile)
		do while (.true.)
			read(10,*, end = 2) llon,llat,yrf,mof,dayf,mm,z,hof,mnf,sec
			yr=int(yrf) ; mo=int(mof); day=int(dayf); ho=int(hof); mn=int(mnf)
		    call CDAY(tt)  ! convert yr/mo/day/ho/mn/sec into days
		    ! ---- full catalog, >md 
            if (tt.lt.TFmin  .and. tt.ge.T0 .and. mm.ge.md.and.
     &		   llat.ge.latmin-1.and.llat.le.latmax+1.and.
     &		   llon.ge.lonmin-1.and.llon.le.lonmax+1. .and. z.le.zmax .and.tt.gt.t(i)) then 
              !     select events larger than Md not too far from the boundary
               i=i+1
               M(i)=mm
               t(i)=tt
               lon(i)=llon
               lat(i)=llat
			   ! write(*,*) i,t(i),M(i),lat(i),lon(i),z
            endif    
		enddo
2       close(10)
		N1=i
 	    write(*,*) '--- readcat: m>',Md,' N=',N1,' t(1)=',t(1),' t(end)=',t(N1)
        end     
    
!!-----------------------------------------------------------------------------		
		subroutine readbg(mufile)
!		read background map
!-----------------------------------------------------------------------------		
        implicit none      
        integer i,j,LMAX,LX,LY
 		parameter (LMAX=500)
		character*200 mufile
        real latmin,latmax,lonmin,lonmax,zmax,l
		real mu(LMAX,LMAX),muyx,s
		integer flagxy(LMAX,LMAX),nflag
        common /grid/latmin,latmax,lonmin,lonmax,zmax,l,LX,LY   	
		common /flag/flagxy,nflag
		common /bg/mu
		if (mufile(1:1).eq.'X') then
			write(*,*) '--- set mu=1/nflag for all cells'
			do  i=1,LX
				do j=1,LY
					mu(i,j)=1./nflag	! long-term rate normalized to 1 inside testing area
				enddo
			enddo	
		else
			write(*,*) '--- reading map of stationnary rate: ',trim(mufile)
			open(10,file=mufile)
			do i=1,4		! skip first 4 lines (comments)
				read(10,*) 
			enddo  
			s=0.
			do  i=1,LX
				do j=1,LY
					read(10,*) muyx     
					if (flagxy(i,j).eq.1) then ! inside testing area
						s=s+muyx
						mu(i,j)=muyx
					else
						mu(i,j)=0.
					endif
				enddo
			enddo
			! normalisation
			do  i=1,LX
				do j=1,LY
					mu(i,j)=mu(i,j)/s	! long-term rate normalized to 1 inside testing area
				enddo
			enddo		
			close(10)
		endif
		end	
!-----------------------------------------------------------------------------
        subroutine csep_template(inputfile)	  ! fills out CSEP template
!-----------------------------------------------------------------------------
        implicit none  
		integer LX,LY,LM,LMAX,ix,iy,im,nflag
		parameter (LMAX=500)		! max grid size
		real Rxy(LMAX,LMAX),Mmin,Mmax,Mcorner,dm,Md,b0
		real latmin,latmax,lonmin,lonmax,zmax,l,lon,lat,mag,pm,f,mu
		double precision TFmin,TFmax,TF
		character*200 file,inputfile
		character*22 forecastStartDate, forecastEndDate
		integer flagxy(LMAX,LMAX)
		common /csep/file
		common /grid/latmin,latmax,lonmin,lonmax,zmax,l,LX,LY    	
		common /Rxy/Rxy
		common /flag/flagxy,nflag	
	    common /par/TFmin,TFmax,TF,Mmin,Mmax,Mcorner,dm,Md,b0  
		! --- issue date (local time?):
		character(8)  :: da
		character(10) :: ti
		character(5)  :: zone
		integer,dimension(8) :: values
		call date_and_time(da,ti,zone,values)
		! --------- header
		open(19,file=file) 
		write(19,'(a,a,a)') '<?xml version=''1.0'' encoding=''UTF-8''?>'
		write(19,'(a)') '<CSEPForecast xmlns=''http://www.scec.org/xml-ns/csep/forecast/0.1''>'
		write(19,'(a)') '  <forecastData publicID=''smi:org.scec/csep/forecast/1''>'
		write(19,'(3(a,f3.1),a)') '    <modelName>K2_Md',Md,'_Mmin',Mmin,'_T',TF,' </modelName>'
		write(19,'(a)') '    <version>1.0</version>'
		write(19,'(a)') '    <author>Helmstetter and Werner</author>'
		write(19,'(13a)') '    <issueDate>',da(1:4),'-',da(5:6),'-',da(7:8),'T',ti(1:2),':',ti(3:4),':',ti(5:6),'L</issueDate>'
		open(12,file=inputfile)
		read(unit=12,fmt = "(18x,22a)") forecastStartDate
		write(19,'(a,20a,a)') '    <forecastStartDate>',trim(forecastStartDate),'</forecastStartDate>'
		read(unit=12,fmt = "(16x,22a)") forecastEndDate
		write(19,'(a,20a,a)') '    <forecastEndDate>',trim(forecastEndDate),'</forecastEndDate>'
		close(12)
		write(19,'(2(a,f4.2),a)') '    <defaultCellDimension latRange=''',l,''' lonRange=''',l,'''/>'
		write(19,'(a,f4.2,a)') '    <defaultMagBinDimension>',dm,'</defaultMagBinDimension>'
		write(19,'(a)') '    <lastMagBinOpen>1</lastMagBinOpen>'
		write(19,'(2(a,f5.1),a)') '    <depthLayer max=''',zmax,''' min=''',0.0,'''>'
	
		! --------- forecasted rate per bin
		LM=int((Mmax-Mmin)/dm) +1
		lon=lonmin+l/2
		do ix=1,LX
			lat=latmin+l/2
			do iy=1,LY
				if (flagxy(ix,iy).eq.1) then
					write(19,'(a,f7.3,a,f8.3,a)')'     <cell lat=''',lat,''' lon=''',lon,'''>'
					mag=Mmin+dm/2.
					do im=1,LM
						call get_pm(im,ix,iy,pm)
						mu=Rxy(ix,iy)*pm*TF
						write(19,'(a,f3.1,a,g9.3,a)') '        <bin m=''',mag,'''>',mu,'</bin>'
						mag=mag+dm
					enddo
					write(19,'(a)') '      </cell>'
				endif
				lat=lat+l
			enddo
			lon=lon+l
		enddo
		write(19,'(a)') '    </depthLayer>'
		write(19,'(a)') '  </forecastData>'
		write(19,'(a)') '</CSEPForecast>'
		close(19)
		end
