package scratch.martinez.LossCurveSandbox.beans;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyVetoException;
import java.beans.VetoableChangeListener;
import java.beans.VetoableChangeSupport;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.Iterator;
import java.util.TreeMap;
import java.util.Vector;

import scratch.martinez.LossCurveSandbox.ui.BeanEditorAPI;
import scratch.martinez.LossCurveSandbox.ui.VulnerabilityBeanEditor;
import scratch.martinez.LossCurveSandbox.ui.gui.VulnerabilityBeanGuiEditor;
import scratch.martinez.LossCurveSandbox.vulnerability.DiscreteVulnerabilityFactory;
import scratch.martinez.LossCurveSandbox.vulnerability.IllegalVulnerabilityFormatException;
import scratch.martinez.LossCurveSandbox.vulnerability.UnsupportedDistributionException;
import scratch.martinez.LossCurveSandbox.vulnerability.VulnerabilityModel;

/**
 * <p>
 * A vulnerability bean is a widget component used to store information about
 * vulnerabilities. Vulnerabilities are loaded from input data files (some by
 * default, and others can be loaded from the user). Once the vulnerability is
 * loaded it is considered to be &ldquo;known&rdquo;. Additionally, there is a
 * concept of availability. In its initial implementation, all vulnerabilities
 * are available all the time, however in reality this makes less sense. Certain
 * vulnerabilities are only meant to be used with certain structures (this is a
 * many-to-many relationship). Once the concept of different structure types is
 * implemented then this class should be refactored to include that
 * functionality as well.
 * </p>
 * <p>
 * Some of this beans properties are bound and/or constrained and an interested
 * listener can be added via utility method to listen for a named property. The
 * handling of this is done using the
 * <code>java.beans.PropertyChangeSupport</code> and
 * <code>java.beans.VetoableChangeSupport</code> utility classes. See the Java
 * Bean specification for more details.
 * </p>
 * <p>
 * This bean supports the singleton design structure such that multiple editors
 * can be used to display this same information in a number of different
 * locations at once. Due to this shared-object structure, it is important to
 * note that all read/write methods contained within are synchronized to prevent
 * dual access.
 * </p>
 * 
 * @author <a href="mailto:emartinez@usgs.gov">Eric Martinez</a>
 *
 */
public class VulnerabilityBean extends AbstractBean 
		implements VetoableChangeListener {
	
	//---------------------------------------------------------------------------
	// Member Variables
	//---------------------------------------------------------------------------
	
	//---------------------------- Constant Members ---------------------------//

	// Implementation side-effect for serialization.
	private static final long serialVersionUID = 0x2F9E201;
	
	// Extension of file to look for when searching recursively for input files.
	private static final String INPUT_FILE_EXTENSION = ".xml";
	
	// Name of file to look  for when searching recursively for input files.
	private static final String INPUT_FILE_NAME =
			"vulnerability" + INPUT_FILE_EXTENSION;
	
	// Prompt for user when we find a file that *might* be an input file.
	private static final String ASK_USER_ABOUT_FILE = "The file (%s) may be a " +
			"valid input file but is not named \"" + INPUT_FILE_NAME + "\". " +
			"Would you like to try to parse it?";
	
	/**
	 * The name of the property for the current model. Useful when a listener
	 * wants to listen for changes to this property.
	 */
	public static final String CURRENT_MODEL_PROPERTY = 
		"VulnerabilityBean::currentModel";
	
	/**
	 * The name of the property listing known models.
	 */
	public static final String KNOWN_MODELS_PROPERTY = 
		"VulnerabilityBean::knownModels";
	
	//----------------------------  Static Members  ---------------------------//

	/**
	 * This is the singleton instance of this class. All callers will receive
	 * this instance when requesting <code>getInstance</code>. The singleton
	 * structure is used such that many editors can display the same information
	 * to users.
	 */
	private static VulnerabilityBean instance = null;
	
	//---------------------------- Instance Members ---------------------------//
	
	/**
	 * The currently selected vulnerability model.
	 */
	private VulnerabilityModel currentModel = null;
	
	/**
	 * A mapping of known models with their current respective availabilities. A
	 * listener may only veto the availability of a model, not the existence of
	 * the model itself. Listeners therefore may not object to putting a model
	 * with the availability of <code>false</code>. 
	 */
	private TreeMap<VulnerabilityModel, Boolean> knownModels = null;
	
	/**
	 * The editor used to manipulate values on this bean.
	 */
	private VulnerabilityBeanEditor editor = null;
	
	/**
	 * Name of the directory (internal to the jar file) where default
	 * vulnerability models can be found.
	 */
	private String defaultModelsDir = System.getProperty("user.dir") + 
			"/etc/vulnerabilities/";

	//---------------------------------------------------------------------------
	// Constructors/Initializers
	//---------------------------------------------------------------------------

	/**
	 * A default no-arg constructor. Initializes required variables such that
	 * subsequent call to this bean should not directly cause an exception. This
	 * constructor loads vulnerabilities from the default location and lists
	 * them as available to a caller.
	 */
	private VulnerabilityBean() {
		this(null);
	}
	
	/**
	 * Creates a vulnerability bean and loads in vulnerability models from
	 * vulnerability input data files found by recursively searching the given
	 * <code>loadVulnsFromDir</code> directory.
	 * 
	 * @param loadVulnsFromDir A top-level directory to search for vulnerability
	 * input files from. Can be <code>null</code>.
	 */
	private VulnerabilityBean(String loadVulnsFromDir) {
		knownModels = new TreeMap<VulnerabilityModel, Boolean>();
		
		// Instantiate the property manager
		propertyChanger = new PropertyChangeSupport(this);
		
		// Instantiate the change manager
		vetoableChanger = new VetoableChangeSupport(this);

		// Load up the default models...
		if(loadVulnsFromDir  != null && (new File(loadVulnsFromDir)).exists()) {
			defaultModelsDir = loadVulnsFromDir;
		}
		loadDefaultModels();
		
		// Reset to null...
		currentModel = null;
		
		// This bean will listen to changes to its current model.
		addVetoableChangeListener(CURRENT_MODEL_PROPERTY, this); // Listen
		
	}
	
	/**
	 * The preferred way to instantiate a vulnerability bean. Use of this method
	 * ensures the singleton design architecture and allows many editors to
	 * display the same information to the user.
	 * 
	 * @param vulnDir The name of a top-level directory to search recursively
	 * for vulnerability input files to use as a default set of models. Can be
	 * <code>null</code>.
	 * @return The singleton instance of this class.
	 */
	public static synchronized VulnerabilityBean getSharedInstance(
			String vulnDir) {
		
		if(instance == null) {
			instance = new VulnerabilityBean(vulnDir);
		}
		return instance;
	}
	
	/**
	 * The preferred way to instantiate a vulnerability bean. Use of this method
	 * ensures the singleton design architecture and allows many editors to
	 * display the same information to the user.
	 * 
	 * @return The singleton instance of this class.
	 */
	public static synchronized VulnerabilityBean getSharedInstance() {
		return getSharedInstance(null);
	}
	
	//---------------------------------------------------------------------------
	// Public Methods
	//---------------------------------------------------------------------------
	
	//---------------------- Public Setter Methods ----------------------------//
	
	/**
	 * Sets the <code>currentModel</code> to the given <code>model</code>.
	 * Interested listeners should listen to this bean's
	 * <code>CURRENT_MODEL_PROPERTY</code> property to be informed of changes.
	 * 
	 * @param model The vulnerability model to set as current.
	 * @return The name of the model that is currently active.
	 */
	public synchronized void setCurrentModel( VulnerabilityModel model) {
		// Hold for later...
		VulnerabilityModel oldModel = currentModel;
		
		try {
			// Allow for a veto
			fireVetoableChange(CURRENT_MODEL_PROPERTY, oldModel, model);
			// Make the change if not vetoed
			currentModel = model;
			// Notify of the change
			firePropertyChange(CURRENT_MODEL_PROPERTY, oldModel, currentModel);
		} catch (PropertyVetoException e) {
			// Change was vetoed, "change" back to normal.
			setCurrentModel(currentModel);
		}
	}

	/**
	 * </p>
	 * Attempts to parse  the given <code>inputFile</code> to create a 
	 * <code>VulnerabilityModel</code>. If parsing succeeds, then this model
	 * is added to the list of <code>knownModels</code> and depending on the
	 * given value of <code>enabled</code> is optionally set as available.
	 * </p>
	 * <p>
	 * If the <code>inputFile</code> is a directory rather than an actual file
	 * then the directory is recursively searched for a file named
	 * <code>INPUT_FILE_NAME</code>, or for and file ending with
	 * <code>INPUT_FILE_EXTENSION</code>. If the exact name is found, then it
	 * is silently loaded and parse errors will cause a halt in the runtime
	 * execution. If only the extension is matched, then the user is given the
	 * choice to try to parse the file and upon parse failure is asked if runtime
	 * execution should halt execution or ignore it and continue.
	 * </p>
	 * 
	 * @param inputFile The input file (or directory) to parse.
	 * @param enabled <code>true</code> if the resulting model should currently
	 * be considered enabled, <code>false</code> otherwise.
	 * @throws FileNotFoundException If the specified <code>inputFile</code> is
	 * not found.
	 * @throws IOException If an I/O error occurs while reading the
	 * <code>inputFile</code>.
	 * @throws UnsupportedDistributionException If the <code>inputFile</code>
	 * specifies a probabilistically distributed model but requests an unknown
	 * probability distribution function.
	 * @throws IllegalVulnerabilityFormatException If the <code>inputFile</code>
	 * is poorly formatted XML or does not follow the specification for a
	 * vulnerability model input file.
	 */
	public synchronized void updateModels(File inputFile,
			boolean enabled, boolean ignoreName) throws FileNotFoundException,
			IOException, UnsupportedDistributionException,
			IllegalVulnerabilityFormatException {
		
		if(inputFile.isDirectory()) {
			// Loop over all sub directories looking for an input file
			File[] fileList = inputFile.listFiles();
			for(int i = 0; i < fileList.length; ++i) {
				updateModels(fileList[i], enabled, false);
			}
		} else {
			// This is (or may be) an input file. Do it (or ask user first).
			String inputFileName = inputFile.getName();
			if(ignoreName | INPUT_FILE_NAME.equalsIgnoreCase(inputFileName) ||
					(
							inputFileName.endsWith(INPUT_FILE_EXTENSION) &&
							editor.boolPrompt(String.format(ASK_USER_ABOUT_FILE, 
									inputFileName), true)
					)
				) {
				// Try to create the model and add it
				updateModels(new FileInputStream(inputFile), enabled);
			}
		}
	}
	
	/**
	 * Attempts to read the given input stream <code>in</code> and parse a
	 * vulnerability model out of it. If parsing succeeds, then the resulting
	 * model is added to the list of <code>knownModels</code> and its current
	 * availability is set according to the value of <code>enabled</code>.
	 * 
	 * @param in The input stream to read.
	 * @param enabled <code>true</code> if the resulting model should currently
	 * be considered available, <code>false</code> otherwise.
	 * @throws IOException If an I/O error occurs while reading the input stream.
	 * @throws UnsupportedDistributionException If the input stream defines a
	 * probabilistically distributed vulnerability model but requests the user
	 * of a probability distribution that is not  known.
	 * @throws NullPointerException If the resulting model after parsing the
	 * input is <code>null</code>.
	 * @throws IllegalVulnerabilityFormatException If the input specifies poorly
	 * formatted XML or does not follow the specification for vulnerability
	 * model input.
	 */
	public synchronized void updateModels(InputStream in, boolean enabled)
			throws IOException, UnsupportedDistributionException,
			NullPointerException, IllegalVulnerabilityFormatException {
		VulnerabilityModel model = null;
		
	//	try {
			// Try a discrete vulnerability first...
			model = DiscreteVulnerabilityFactory.createVulnerability(in);
	//	} catch (IllegalVulnerabilityFormatException e) {
			// Continuous factory is not implemented yet...
			
			// This may also result in an exception which is why we declare one to
			// be thrown in the method signature.
			
			//model = ContinuousVulnerabilityFactory.createVulnerability(in);
	//	}
		
		setAvailability(model, enabled);
		if(isAvailable(model)) {
			setCurrentModel(model);
		}
	}
	
	/**
	 * Updates the given <code>model</code>'s availability based on the given
	 * value of <code>enabled</code>. If the model is not currently known, it is
	 * added to this list so long as it is not <code>null</code>. Interested 
	 * listeners should listen to this bean's <code>KNOWN_MODELS_PROPERTY</code>
	 * property to be informed of changes. This property is not constrained and
	 * changes here cannot be vetoed.
	 * 
	 * @param model The model to update (or add).
	 * @param enabled <code>true</code> if the model is currently available,
	 * <code>false</code> otherwise.
	 * @throws NullPointerException If the <code>model</code> is
	 * <code>null</code>.
	 */
	public synchronized void setAvailability(VulnerabilityModel model,
			boolean enabled) throws NullPointerException {
		
		// Check for null, throw exception if so...
		if(model == null) {
			String value = "";
			if(enabled) { value = "enable"; } else { value = "disable"; }
			NullPointerException npx = new NullPointerException("Attempted to " +
					value + " a null vulnerability model.");
			npx.fillInStackTrace();
			throw npx;
		}
		
		try {
			// Build a candidate new tree
			TreeMap<VulnerabilityModel, Boolean> oldModels = 
				new TreeMap<VulnerabilityModel, Boolean>();
			TreeMap<VulnerabilityModel, Boolean> newModels = 
				new TreeMap<VulnerabilityModel, Boolean>();
			
			oldModels.putAll(knownModels);
			newModels.putAll(knownModels);
			
			newModels.put(model, new Boolean(enabled));
			
			// Allow for a veto
			fireVetoableChange(KNOWN_MODELS_PROPERTY, knownModels, newModels);
			
			// Update the list if not vetoed
			knownModels = newModels;
			
			// Notify of the change
			firePropertyChange(KNOWN_MODELS_PROPERTY, oldModels, newModels);
			
		} catch (PropertyVetoException e) {
			// Listeners may not object to this but this is proper style.
			setAvailability(model, new Boolean(!enabled));
		}
	}
	
	/**
	 * Set the editor used to modify this bean. Interested listeners should
	 * listen to this bean's <code>EDITOR_PROPERTY</code> property to be informed
	 * of changes. This property is not constrained and changes here cannot be
	 * vetoed. If multiple editors are set in succession, only the most recent
	 * is used to alert the user of something, however all editors may still
	 * accept user input and change values.
	 * 
	 * @param editor The new editor used.
	 * @throws ClassCastException If some other editor than a
	 * <code>VulnerabilityBeanEditor</code> is given as the
	 * <code>newEditor</code>.
	 */
	public synchronized void setEditor(BeanEditorAPI newEditor)
			throws ClassCastException {
		// Ignore calls with a null editor...
		if(newEditor == null) { return; }
		
		// Save for later
		VulnerabilityBeanEditor oldEditor = editor;
		
		// This is not constrained, so don't allow for a veto.
		editor = (VulnerabilityBeanGuiEditor) newEditor;
		
		// Notify of the change
		firePropertyChange(EDITOR_PROPERTY, oldEditor, editor);
	}
	
	//---------------------- Public Getter Methods ----------------------------//
	
	/**
	 * @return The currently selected vulnerability model.
	 */
	public synchronized VulnerabilityModel getCurrentModel() {
		return currentModel;
	}
	
	/**
	 * @return A list of available currently vulnerability models.
	 */
	public synchronized Vector<VulnerabilityModel> getAvailableModels() {
		Vector<VulnerabilityModel> availableModels =
			new Vector<VulnerabilityModel>();
		
		Iterator<VulnerabilityModel> iter = knownModels.keySet().iterator();
		
		while(iter.hasNext()) {
			VulnerabilityModel model = iter.next();
			if(knownModels.get(model).booleanValue()) {
				availableModels.add(model);
			}
			
		}
		
		return availableModels;
	}
	
	/**
	 * @return A list of known vulnerability models with their current
	 * corresponding availability.
	 */
	public synchronized TreeMap<VulnerabilityModel, Boolean> getKnownModels() {
		return knownModels;
	}
	
	/**
	 * Checks whether or not a specific vulnerability is currently available.
	 * 
	 * @param model The vulnerability to look for.
	 * @return <code>true</code> if the vulnerability is available, 
	 * <code>false</code> otherwise.
	 */
	public synchronized boolean isAvailable(VulnerabilityModel model) {
		Boolean availability = knownModels.get(model);
		if(availability == null) {
			return false;
		} else {
			return availability.booleanValue();
		}
	}
	
	/**
	 * @return The editor used to manipulate this bean.
	 */
	public synchronized BeanEditorAPI getBeanEditor() {
		return editor;
	}
	
	//---------------------- Public Utility Methods ---------------------------//
	
	/**
	 * Method to implement the <code>VetoableChangeListener</code> interface.
	 * Checks to  ensure the model we are trying to set as the
	 * <code>currentModel</code> is currently available. The
	 * <code>currentModel</code> is the only constrained bean property.
	 */
	public void vetoableChange(PropertyChangeEvent evt) 
			throws PropertyVetoException {
		
		if(CURRENT_MODEL_PROPERTY.equals(evt.getPropertyName()) ) {
			// Make sure the requested property is currently available
			if(	!isAvailable( (VulnerabilityModel) evt.getNewValue()) ) {
				throw new PropertyVetoException("The requested model is not " + 
						"currently available.", evt);
			}
			// The change seems to be okay so do not throw exception.
			
		}
	}
	
	/**
	 * Wrapper method to allow external objects (namely an editor) to fire
	 * property change events on this bean's properties. Note that this method
	 * should be triggered <em>after</em> the internal state of the bean is
	 * changed.
	 * 
	 * @param propertyName The name of the property that changed.
	 * @param oldValue The old value of the property.
	 * @param newValue The new value of the property.
	 */
	public void firePropertyChange(String propertyName, Object oldValue,
			Object newValue) {
		propertyChanger.firePropertyChange(propertyName, oldValue, newValue);
	}
	
	/**
	 * Wrapper method to allow external objects (namely an editor) to fire
	 * veto-able change events on this beans constrained properties. Note that
	 * this method is triggered <em>before</em> the internal state of the bean is
	 * changed, thus allowing a listener to veto the change via throwing a
	 * <code>PropertyVetoException</code>.
	 * 
	 * @param propertyName The name of the property we want to change.
	 * @param oldValue The old (current) value of the property.
	 * @param newValue The (potential) new value of the property. 
	 * @throws PropertyVetoException If a listener vetoes the prospective change.
	 */
	public void fireVetoableChange(String propertyName, Object oldValue,
			Object newValue) throws PropertyVetoException {
		vetoableChanger.fireVetoableChange(propertyName, oldValue, newValue);
	}
	
	//---------------------------------------------------------------------------
	// Private Methods
	//---------------------------------------------------------------------------
	
	/**
	 * Loads a default set of known vulnerability models for use in the parent
	 * application. If the <code>loadVulnsFromDir</code> is <code>null</code>
	 * then we try a default location within the parent application's jar file.
	 * In any case, the bean supports loading models after the fact, so failure
	 * should always allow for program execution to proceed.
	 * 
	 */
	private void loadDefaultModels() {
		try {
			updateModels(new File(defaultModelsDir), true, true);
		} /*catch (URISyntaxException e) {
			// This should not happen. We can't continue if it does...
			e.printStackTrace();
		} */catch (FileNotFoundException e) {
			System.err.println(e.getMessage());
		} catch (IOException e) {
			System.err.println(e.getMessage());
		} catch (UnsupportedDistributionException e) {
			System.err.println(e.getMessage());
		} catch (IllegalVulnerabilityFormatException e) {
			System.err.println(e.getMessage());
		}
	}
}
