package scratch.martinez.LossCurveSandbox.vulnerability;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;

import org.apache.commons.math.MathException;
import org.apache.commons.math.distribution.Distribution;

import scratch.martinez.LossCurveSandbox.math.distribution.AdvDistributionFactory;

/**
 * <p>
 * This class is used to produce Damage Exceedance Matrices (DEMs) for use when 
 * computing risk/loss using discrete vulnerabilities. DEMs can be evenly
 * spaced, arbitrarily spaced, follow some probabilistic distribution, or be
 * empirically determined (the latter is not producible here).
 * </p>
 * </p>
 * A DEM is a convenient way to display probabilities of exceeding a certain
 * damage factor (DF) for a given intensity measure level (IML). A DEM can be
 * represented as:
 * </p>
 * <pre>
 *   \      IML0      IML1      IML2      ...      IMLn
 *    \--------------------------------------------------------
 * DF0 |    P(0,0)    P(1,0)    P(2,0)    ...       P(n,0)
 *     |
 * DF1 |    P(0,1)    P(1,1)    P(2,1)    ...       P(n,1)
 *     |
 * DF2 |    P(0,2)    P(1,2)    P(2,2)    ...       P(n,2)
 *     |
 *  .  |     .         .         .        ...        .
 *  .  |     .         .         .        ...        .
 *  .  |     .         .         .        ...        .
 *     |
 * DFm |    P(0,m)    P(1,m)    P(2,m)    ...       P(n,m)
 *     |
 * </pre>
 * <p>
 * Where P(i,j) = P [ DF > df<sub>j</sub> | IML = iml<sub>i</sub> ] <br />
 * Initial implementation returns a DEM as a simple 2-dimensional array of
 * <code>double</code> values.
 * </p>
 * <p>
 * One should recall that since these are probabilities that:
 * 0.0 &le; P(i,j) &le; 1.0. If this constraint is ignored then odd output can
 * be observed on computing risk/loss.
 * </p>
 * 
 * @author   
 * <a href="mailto:emartinez@usgs.gov?subject=NSHMP%20Application%20Question">
 * Eric Martinez
 * </a>
 */
public class DEMFactory {
	
	//---------------------------------------------------------------------------
	// Member Variables (Incl. Static)
	//---------------------------------------------------------------------------
	
	/**
	 * Factory used to create distributions.
	 */
	private AdvDistributionFactory distFactory = null;
	
	//---------------------------------------------------------------------------
	// Constructors (and such)
	//---------------------------------------------------------------------------
	
	/**
	 * The constructor is made private such that instantiation should happen from
	 * a call to the <code>newInstance</code> method. This is a factory
	 * convention and implementation is as such for no other reason.
	 */
	private DEMFactory() {
		distFactory = AdvDistributionFactory.newInstance();
	}
	
	/**
	 * Creates a new <code>DEMFactory</code> and returns it.
	 * @return A new instance of a <code>DEMFactory</code>.
	 */
	public static DEMFactory newInstance() {
		return new DEMFactory();
	}
	
	//---------------------------------------------------------------------------
	// Public Methods
	//---------------------------------------------------------------------------
	
	/**
	 * Creates a damage exceedance matrix from a distribution.
	 * 
	 * @param distName The name of the distribution to use.
	 * @param params The list of parameters with values defined at agreed-upon
	 * discreet intensity measure levels.
	 * @param dfs A discreet set of damage factors used to constrain the DEM.
	 * @return A damage exceedance matrix as specified in this class's
	 * declaration.
	 * 
	 * @throws IllegalArgumentException If a value in the <code>params</code>
	 * list is used to instantiate a probability distribution and that value is
	 * not valid for the distribution.
	 * @throws NoSuchMethodException If the <code>AdvDistributionFactory</code>
	 * can't find a suitable &ldquo;create&rdquo; method to create the requested
	 * distribution.
	 * @throws IllegalAccessException If the <code>AdvDistributionFactory</code>
	 * finds a &ldquo;create&rdquo; method, but that method is not available due
	 * to Java language access control properties.
	 * @throws InvocationTargetException If the &ldquo;create&rdquo; method
	 * throws and exception.
	 * @throws MathException If the distribution fails to converge for a
	 * requested value.
	 */
	public Double[][] createDEM(String distName, ArrayList<Object[]> params,
			Double[] dfs) throws IllegalArgumentException, NoSuchMethodException,
			IllegalAccessException, InvocationTargetException, MathException {
		// There should be a set of parameters for each IML;
		int paramsLength = params.get(0).length;
		int paramsCount  = params.size();
		int dfCount      = dfs.length;
		
		// Instantiate the resulting DEM. Its default values are undefined by
		// default.
		Double [][] dem = new Double[dfCount][paramsLength];
		for(int i = 0; i < paramsLength; ++i) {
			
			// Build an array of distribution parameters. Namely the Ith set of
			// each parameter value.
			ArrayList<Object> distParam = new ArrayList<Object>();
			for(int j = 0; j < paramsCount; ++j) {
				distParam.add(params.get(j)[i]);
			} // END: for(paramsCount)

			// Get the distribution to use for the Ith column of the DEM
			Distribution pDist = getDistribution(distName, distParam);
			
			// Fill in the value of the Kth row of the Ith column...
			for(int k = 0; k < dfCount; ++k) {
				dem[k][i]  = 1.0 - pDist.cumulativeProbability(dfs[k]);
			} // END: for(dfCount)
			
		} // END: for(paramsLength)
		
		return dem;
	}
	
	//---------------------------------------------------------------------------
	// Private Methods (Incl. Utilities)
	//---------------------------------------------------------------------------
	
	/**
	 * Creates the distribution identified by the <code>distName</code> with
	 * parameters contained in the <code>params</code> list. This is done by
	 * invoking the corresponding method on the
	 * <code>AdvDistributionFactory</code>.
	 * 
	 * @param distName The name of the distribution to create. This should be
	 * camel-cased as appropriate.
	 * @param  params An ordered list of parameters to provide the factory when
	 * creating the distribution.
	 * @return A new instance of the requested distribution with the given
	 * parameters.
	 * 
	 * @throws IllegalArgumentException If a value in the <code>params</code>
	 * list is used to instantiate a probability distribution and that value is
	 * not valid for the distribution.
	 * @throws NoSuchMethodException If the <code>AdvDistributionFactory</code>
	 * can't find a suitable &ldquo;create&rdquo; method to create the requested
	 * distribution.
	 * @throws IllegalAccessException If the <code>AdvDistributionFactory</code>
	 * finds a &ldquo;create&rdquo; method, but that method is not available due
	 * to Java language access control properties.
	 * @throws InvocationTargetException If the &ldquo;create&rdquo; method
	 * throws and exception.
	 */
	private Distribution getDistribution(String distName, 
			ArrayList<Object> params) throws IllegalArgumentException, 
			NoSuchMethodException, IllegalAccessException,
			InvocationTargetException {
		String methodName = "create" + distName + "Distribution";
		Class<?>[] classList = getParamClasses(params);
		Method creator = getFactoryMethod(methodName, classList);
		return (Distribution) creator.invoke(distFactory, params.toArray());
	}
	
	/**
	 * Inspects the factory to find the proper &ldquo;create&rdquo; method for
	 * the requested <code>methodName</code> and <code>classList</code>.
	 * 
	 * @param methodName The name of the method to find.
	 * @param classList The list of classes (in order) that this method will
	 * expect as arguments.
	 * @return The factory method for creating a distribution.
	 * 
	 * @throws SecurityException If a security manager,s, is present and any of
	 * the following conditions is met:
	 * <ul>
	 * <li>Invocation of <code>s.checkMemberAccess(this, Member.PUBLIC)</code>
	 * denies access to the method.</li>
	 * <li>The caller's class loader is not the same as or an ancestor of the
	 * class loader for the current class and invocation of
	 * <code>s.checkPackageAccess()</code> denies access to the package of this
	 * class.</li>
	 * </ul>
	 * @throws NoSuchMethodException If the matching method is not found or if
	 * the name is &ldquo;&lt;init&gt;&rdquo; or &ldquo;&lt;clinit&gt;&rdquo;.
	 */
	private Method getFactoryMethod(String methodName, Class<?>[] classList)
			throws SecurityException, NoSuchMethodException {
		return distFactory.getClass().getMethod(methodName, classList);
	}
	
	/**
	 * Creates an array of classes of the objects stored in <code>params</code>.
	 * This new array's entries will correspond in order to the objects in the
	 * given <code>params</code> list.
	 * 
	 * @param params The list of objects whose classes we want.
	 * @return An ordered array of classes corresponding to the objects in the
	 * <code>params</code> list.
	 */
	private Class<?>[] getParamClasses(ArrayList<Object> params) {
		int pSize = params.size();
		Class<?>[] classes = new Class[pSize];
		for(int i = 0; i < pSize; ++i) {
			classes[i] = params.get(i).getClass();
		}
		
		return classes;
	}
}
