package scratch.martinez.LossCurveSandbox.vulnerability;

/**
 * <p>
 * This class defines the methods a class must implement to be a discrete
 * vulnerability model. Discrete vulnerability models calculate loss using a
 * damage exceendance matrix (DEM). A damage exceedance matrix is defined at
 * discrete intensity measure levels (IMLs) and corresponding damage factors 
 * (DF). Entries to the DEM define the probability of exceeding a DF level 
 * for a given IML. That is:
 * </p>
 * <pre>
 * DEM(i,j) = P[DF > dfi | IML = imlj]
 * </pre>
 * <p>
 * These probabilities may follow a certain probabilistic distribution or could
 * possibly be empirically determined. There is no restriction on how the DEM is
 * defined or what IML/DF values are used to constrain it. However if one 
 * expects any degree of accuracy from the model then proper engineering
 * principles must be applied.
 * </p>
 * 
 * @author   
 * <a href="mailto:emartinez@usgs.gov?subject=NSHMP%20Application%20Question">
 * Eric Martinez
 * </a>
 * @see DiscreteFactory
 */
public abstract class DiscreteVulnerability extends AbstractVulnerability {
	
	/**
	 * <p>
	 * These are default Damage Factor (DF) values to use for defining the
	 * Damage exceedance matrix for a discrete vulnerability model. An
	 * implementation can certainly override this however vulnerability designers
	 * who specify vulnerability in terms of a probability distribution may not
	 * consider to provide these value and they are required.
	 * </p>
	 * <p>
	 * These values are evenly spaced in log space from roughly 0.0 to 1.0. Here
	 * we are considering 1.0E-6 to be &ldquo;close enough&rdquo; to zero. There
	 * are 76 value in this array.
	 * </p>
	 */
	protected static final Double [] DEFAULT_DF_VALUES =
	{
		1.00E-006, 1.20E-006, 1.44E-006, 1.73E-006, 2.07E-006, 2.49E-006,
		2.99E-006, 3.58E-006, 4.30E-006, 5.16E-006, 6.19E-006, 7.43E-006,
		8.92E-006, 1.07E-005, 1.28E-005, 1.54E-005, 1.85E-005, 2.22E-005,
		2.66E-005, 3.19E-005, 3.83E-005, 4.60E-005, 5.52E-005, 6.62E-005,
		7.95E-005, 9.54E-005, 1.14E-004, 1.37E-004, 1.65E-004, 1.98E-004,
		2.37E-004, 2.85E-004, 3.42E-004, 4.10E-004, 4.92E-004, 5.91E-004,
		7.09E-004, 8.51E-004, 1.02E-003, 1.22E-003, 1.47E-003, 1.76E-003,
		2.12E-003, 2.54E-003, 3.05E-003, 3.66E-003, 4.39E-003, 5.27E-003,
		6.32E-003, 7.58E-003, 9.10E-003, 1.09E-002, 1.31E-002, 1.57E-002,
		1.89E-002, 2.26E-002, 2.72E-002, 3.26E-002, 3.91E-002, 4.70E-002,
		5.63E-002, 6.76E-002, 8.11E-002, 9.74E-002, 1.17E-001, 1.40E-001,
		1.68E-001, 2.02E-001, 2.42E-001, 2.91E-001, 3.49E-001, 4.19E-001,
		5.02E-001, 6.03E-001, 7.23E-001, 8.68E-001
	};
	
	/**
	 * @return The intensity measure levels (IMLs) at which the DEM for this
	 * discrete vulnerability model is defined.
	 */
	public abstract Double [] getIMLValues();
	
	/**
	 * @return The damage factors (DFs) at which the DEM for this discrete
	 * vulnerability model is defined.
	 */
	public abstract Double [] getDFValues();
	
	/**
	 * @return The damage exceedance matrix (DEM) used by this discrete
	 * vulnerability model to compute risk/loss.
	 */
	public abstract Double [][] getDEM();
}
