package scratch.martinez.LossCurveSandbox.vulnerability;

import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.TreeMap;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.w3c.dom.CharacterData;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

/**
 * <p>
 * This class is used to produce discrete vulnerability models from input files.
 * The only actual implemented class discrete model is only an abstract
 * definition of what it means to be a discrete vulnerability model. One should
 * use this class to generate new vulnerability models from well-defined input
 * files. All methods in this class can be accessed statically so there is no
 * need to ever instantiate a factory.
 * </p>
 * 
 * @author   
 * <a href="mailto:emartinez@usgs.gov?subject=NSHMP%20Application%20Question">
 * Eric Martinez
 * </a>
 * @see DiscreteVulnerability
 * @see VulnerabilityModel
 */
public class DiscreteVulnerabilityFactory {

	//---------------------------------------------------------------------------
	// Class Member Variables
	//---------------------------------------------------------------------------
	
	/**
	 * The XML document builder used to parse input files.
	 */
	private static DocumentBuilder docBuilder = null;

	/**
	 * A mapping of known probabilistic distributions with their required
	 * parameter names. K = Distribution name; V = Required parameter names.
	 */
	private static TreeMap<String, String[]> distributions = null;
	
	/**
	 * A factory class used to create DEMs from probability distributions.
	 */
	private static DEMFactory demFactory = null;
	
	/**
	 * The type of vulnerability this factory expects.
	 */
	private static final String VULN_TYPE_PARAM_VALUE = "discrete";
	
	//----------------- XML Document Element Tag Names ------------------------//
	private static final String NAME_TAG                 = "name";
	private static final String DISPLAY_NAME_TAG         = "displayName";
	private static final String DESCRIPTION_TAG          = "description";
	private static final String DISTRIBUTION_TAG         = "distribution";
	private static final String PARAM_TAG                = "param";
	private static final String ITEM_TAG                 = "item";
	
	//--------------- XML Document Element Attribute Names --------------------//
	private static final String TYPE_ATTR        = "type";
	private static final String NAME_ATTR        = "name";
	private static final String INDEX_ATTR       = "index";
	private static final String VALUE_ATTR       = "value";
	
	//--------------- XML Document Element Attribute Values -------------------//
	private static final String IML_VAL         = "imls";
	private static final String SUPT_STRUCT_VAL = "supportedStructures";
	private static final String DF_VAL          = "damageFactors";
	
	// This is returned when a requested text node is note found.
	private static final String DEFAULT_TEXT_NODE_VALUE = "UNDEFINED";
	
	
	//---------------------------------------------------------------------------
	// Initialization/Constructors
	//---------------------------------------------------------------------------
	
	/**
	 * This block performs some static initializations that are required before
	 * a vulnerability can be built from an input file. Namely it instantiates
	 * the <code>DocumentBuilder</code>.
	 */
	static {
		DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
		try {
			// Instantiate a document builder
			docBuilder = factory.newDocumentBuilder();
			
			// Define the required parameters for know distributions
			distributions = new TreeMap<String, String[]>();
			
			// Add the log normal distribution here
			distributions.put("LogNormal", new String [] {"mean",
					"covariance"});
			
			// Add more distributions...we currently only use log normal, so...
			
			
			// Instantiate the DEM factory
			demFactory = DEMFactory.newInstance();
			
		} catch (ParserConfigurationException pcx) {
			// If this fails then continuing is hopeless. Die.
			pcx.printStackTrace(System.err);
			System.exit(-1);
		}
	}
	
	//---------------------------------------------------------------------------
	// Public Static Methods
	//---------------------------------------------------------------------------
	
	/**
	 * Generates a new discrete vulnerability by reading the given input file.
	 * @param fileName The name of the input file to read.
	 * @return A new discrete vulnerability as defined by the input file.
	 * @throws IOException If an I/O error occurs while reading the input file.
	 * @throws UnsupportedDistributionException If the input file tries to
	 * specify the vulnerability via a parameterized probability distribution
	 * that we are not aware of.
	 * @throws IllegalVulnerabilityFormatException If the input stream does not
	 * provide properly formatted data.
	 */
	public static synchronized DiscreteVulnerability createVulnerability(
			String fileName) throws IOException, UnsupportedDistributionException,
			IllegalVulnerabilityFormatException {
		
		return createVulnerability(new FileInputStream(fileName));
	}
	
	/**
	 * Generates a new discrete vulnerability by reading the given input stream.
	 * @param in The stream to read.
	 * @return A new discrete vulnerability as defined by the input stream.
	 * @throws IOException If an I/O error occurs while reading the input stream.
	 * @throws UnsupportedDistributionException If the input stream tries to
	 * specify the vulnerability via a parameterized probability distribution
	 * that we are not aware of.
	 * @throws IllegalVulnerabilityFormatException If the input stream does not
	 * provide properly formatted data.
	 */
	public static synchronized DiscreteVulnerability createVulnerability(
			InputStream in) throws IOException,	UnsupportedDistributionException,
			IllegalVulnerabilityFormatException {

		Element vulnInfo = null;
		try {
			vulnInfo = docBuilder.parse(in).
					getDocumentElement();
		} catch (SAXException sxe) {
			throw new IllegalVulnerabilityFormatException("Failed to parse the " +
					"XML from the specified input.", sxe);
		}
		
		String type = vulnInfo.getAttribute(TYPE_ATTR);
		if(!VULN_TYPE_PARAM_VALUE.equals(type)) {
			throw new IllegalVulnerabilityFormatException("This factory only " +
					"knows how to produce discrete vulnerabilities. You requested " +
					"a vulnerability of type: " + type);
		}
		
		String name        = getTextNodeValue(vulnInfo, NAME_TAG);
		String displayName = getTextNodeValue(vulnInfo, DISPLAY_NAME_TAG);
		String description = getTextNodeValue(vulnInfo, DESCRIPTION_TAG);
		
		ArrayList<Object> holder = null;
		String [] structs = null;
		Double [] imls    = null;
		Double [] dfs     = null;
		
		try {
			holder   = getNamedParameterItems(vulnInfo, SUPT_STRUCT_VAL);
			structs  = (String[]) holder.toArray(new String[holder.size()]);
			holder   = getNamedParameterItems(vulnInfo, IML_VAL);
			imls     = (Double[]) holder.toArray(new Double[holder.size()]);
			holder             = getNamedParameterItems(vulnInfo, DF_VAL);
			if(holder != null) {
				// If specified in XML, fine, use them...
				dfs = (Double[]) holder.toArray(new Double[holder.size()]);
			} else {
				// ... otherwise use the default values.
				dfs = DiscreteVulnerability.DEFAULT_DF_VALUES;
			}
		} catch (NullPointerException npx) {
			throw new IllegalVulnerabilityFormatException(npx);
		}
		
		Double [][] dem    = null;
		
		Node distNode = vulnInfo.getElementsByTagName(DISTRIBUTION_TAG).item(0);		
		if(distNode != null && distNode.getNodeType() == Node.ELEMENT_NODE) {
			// If we are here, then this discrete vulnerability is defined by a
			// probability distribution, so lets build it as such...
			
			// Get name of distribution (it must be known to us)
			String distName = distNode.getAttributes().
					getNamedItem(NAME_ATTR).getNodeValue();
			
			// Make sure we know about this probability distribution
			if(!distributions.containsKey(distName)) {
				throw new UnsupportedDistributionException();
			}

			// Get the required parameters vs. provided parameters
			ArrayList<Object[]> paramValuesList = 
					getDistributionParameterValues((Element) distNode, distName);
			
			try {
				dem = demFactory.createDEM(distName, paramValuesList, dfs);
			} catch (Exception e) {
				// Wrap any of these exceptions as an unsupported distribution
				e.printStackTrace(System.err);
				throw new UnsupportedDistributionException(e);
			}
			
		} else {
			// If we are here then this discrete vulnerability is defined by a
			// statically specifying the Damage Exceedance Matrix values along with
			// the Intensity Measure Levels and Damage Factors used to parameterize
			// the DEM.  Let's build it...
			
		} // END: if( /* Building vulnerability by Distribution or DEM */ )
		
			
		return new DiscreteVulnerabilityImpl(name, displayName, description,
				structs, imls, dfs, dem);
	}
	
	//---------------------------------------------------------------------------
	// Private Utility Stuff
	//---------------------------------------------------------------------------

	/**
	 * Looks for a tag named <code>parentTagName</code> within this scope of the
	 * given <code>scope</code>. The first matching tag is used and the value of
	 * its first child that is either a non-empty TEXT node, or CDATA node is
	 * returned.
	 * 
	 * @param scope The &ldquo;root&rdquo; element to search from.
	 * @param parentTagName The name of the tag whose text we are interested in.
	 * @return The text of the first non-empty TEXT or CDATA node of the first
	 * tag that matches <code>parentTagName</code>.
	 */
	private static String getTextNodeValue(
			Element scope, String parentTagName) {
		try {
			NodeList children = scope.getElementsByTagName(parentTagName).
				item(0).getChildNodes();
			for(int i = 0; i < children.getLength(); ++i) {
				Node child = children.item(i);
				if(child.getNodeType() == Node.TEXT_NODE &&
						!child.getNodeValue().matches("^\\s*$")) {
					return child.getNodeValue().trim();
				} else if (child.getNodeType() == Node.CDATA_SECTION_NODE) {
					// CDATA can only have a single child, namely, the text data.
					return ( (CharacterData) child).getData().trim();
				}
			}
			// If we didn't find a good value, then return a default string...
			return DEFAULT_TEXT_NODE_VALUE;
		} catch (NullPointerException npx) {
			// If the named parentTag is not found or has no children then a
			// null pointer exception might get thrown so we'll return a default
			// string rather than die.
			return DEFAULT_TEXT_NODE_VALUE;
		}
	}
	
	/**
	 * Creates a list of an array of parameters that are at an agreed-upon set of
	 * discrete intensity measure levels. The parameters it looks for are as
	 * specified for the given <code>distName</code> in the
	 * <code>distributions</code> mapping.
	 * 
	 * @param distNode The XML node defining the distribution
	 * @param distName The name of the distribution we are creating.
	 * @return A list of arrays of parameters.
	 * @throws UnsupportedDistributionException If the current XML does not
	 * specify a parameter that is required per the <code>distributions</code>
	 * mapping.
	 */
	private static ArrayList<Object[]> getDistributionParameterValues(
			Element distNode, String distName)
			throws UnsupportedDistributionException {
		ArrayList<Object[]> params = new ArrayList<Object[]>();
		
		String [] requiredParams = distributions.get(distName);
		for(int i = 0; i < requiredParams.length; ++i) {
			String paramName = requiredParams[i];
			ArrayList<Object> paramVals = getNamedParameterItems(distNode, 
					paramName);
			if(paramVals == null) {
				throw new UnsupportedDistributionException("The requested " +
						"distribution requires a parameter named \"" + paramName +
						"\" but this parameter was not found in the provided data.");
			}
			// Add the parameter values...
			params.add(paramVals.toArray(new Object[paramVals.size()]));
		}
		
		return params;
	}
	
	/**
	 * Searches the <code>scope</code> for the parameter named <code>name</code>
	 * and returns a list of that parameter's items.
	 * 
	 * @param scope The XML sub-tree to search.
	 * @param name The name of the parameter to look for.
	 * @return A list of items defined for that parameter.
	 */
	private static ArrayList<Object> getNamedParameterItems(Element scope,
			String name) {
		NodeList possibleParams = scope.getElementsByTagName(PARAM_TAG);
	
		for(int i = 0; i < possibleParams.getLength(); ++i) {
			String paramName = possibleParams.item(i).getAttributes().
					getNamedItem(NAME_ATTR).getNodeValue();
			
			if(name.equals(paramName)) {
				// We have found the named parameter, so get the values
				return getParameterItems( (Element) possibleParams.item(i));
			}
		}
		
		// If the named parameter is not found, return null
		return null;
	}
	
	/**
	 * Creates a list of items from the XML <code>parameter</code> node.
	 * @param parameter The XML node containing the items of interest
	 * @return A list of the items.
	 */
	private static ArrayList<Object> getParameterItems(Element parameter) {
		ArrayList<Object> items = new ArrayList<Object>();
		
		NodeList levelNodes = parameter.getElementsByTagName(ITEM_TAG);
		int itemCount = levelNodes.getLength();
		
		for(int i = 0; i < itemCount; ++i) {
			Node level = levelNodes.item(i);
			int index = Integer.parseInt(
					level.getAttributes().getNamedItem(INDEX_ATTR).getNodeValue());
			String valHolder = level.getAttributes().getNamedItem(VALUE_ATTR).
					getNodeValue();
			Object value = null;
			
			// We have to do some magic to ensure correct types; we might get
			// double values, integer values, or simple strings, let's play...
			try {
				value = Double.parseDouble(valHolder);
			} catch (NumberFormatException nfxDbl) { 
				// Okay, it was not a double, how about an integer?
				try {
					value = Integer.parseInt(valHolder);
				} catch (NumberFormatException nfxInt) {
					// Okay, it was not an integer, let's hope a string works...
					value = valHolder;
				}
			}
			
			if(index != i) {
				// Declared index does not match display order. Print a warning.
				System.err.println("Declared index (" + index + ") does not " +
						"match the displayed index order (" + i + "). Using the " +
						"displayed index order.");
			}
			
			items.add(value);
		}
		
		return items;
	}
	
	//---------------------------------------------------------------------------
	// This is a private inner class the implements the
	// <code>DiscreteVulnerability</code> abstract class.
	//---------------------------------------------------------------------------
	
	/**
	 * This class is an implementation of a <code>DiscreteVulnerability</code>.
	 * All parameters for this class must be specified to the constructor else
	 * an exception is thrown.
	 */
	private static class DiscreteVulnerabilityImpl 
			extends DiscreteVulnerability {
		//------------------------------------------------------------------------
		// Member Variables
		//------------------------------------------------------------------------
		private Double [][] dem = null;
		private Double [] dfs = null;
		private Double [] imls = null;
		private String displayName = null;
		private String name = null;
		private String [] supportedStructs = null;
		private String description;
		

		//------------------------------------------------------------------------
		// Constructor(s)
		//------------------------------------------------------------------------
		
		/**
		 * The constructor for this implementation. All values are as expected.
		 * 
		 * @param name The short name/id of this vulnerability.
		 * @param displayName The human-readable name of this vulnerability.
		 * @param description The description of this vulnerability.
		 * @param supportedStructs A list of structure types this vulnerability
		 * can be used with.
		 * @param imls A discrete set of intensity measure levels.
		 * @param dfs A discrete set of damage factors.
		 * @param dem A damage exceedance matrix with probabilities defined for
		 * the given IMLs and DFs.
		 * 
		 * @throws IllegalArgumentException If any of the parameters are null.
		 */
		public DiscreteVulnerabilityImpl(String name, String displayName,
				String description, String [] supportedStructs, Double [] imls, 
				Double [] dfs, Double [][] dem) {
			
			if(name == null || displayName == null || description == null ||
					supportedStructs == null || imls == null || dfs == null ||
					dem == null) {
				IllegalArgumentException iax = new IllegalArgumentException(
						"One or more parameters passed to the consctructor were " +
						"null. Please check your parameters and try again."
					);
				iax.fillInStackTrace();
				throw iax;
			}
			
			this.name             = name;
			this.displayName      = displayName;
			this.description      = description;
			this.supportedStructs = supportedStructs;
			this.imls             = imls;
			this.dfs              = dfs;
			this.dem              = dem;
		}
		

		//------------------------------------------------------------------------
		// Public Methods
		//------------------------------------------------------------------------

		//---------------------- Public Getter Methods -------------------------//
		
		/**
		 * @return The DEM for this vulnerability.
		 */
		public Double[][] getDEM() { return dem; }
		/**
		 * @return The DF values for this vulnerability.
		 */
		public Double[] getDFValues() { return dfs; }
		/**
		 * @return The IML values for this vulnerability.
		 */
		public Double[] getIMLValues() { return imls; }
		/**
		 * @return The human-readable name for this vulnerability.
		 */
		public String getDisplayName() { return displayName; }
		/**
		 * @return The name/id for this vulnerability.
		 */
		public String getName() { return name; }
		/**
		 * @return The list of structure types for this vulnerability.
		 */
		public String[] getSupportedStructures() { return supportedStructs; }
		/**
		 * @return The description for this vulnerability.
		 */
		public String getDescription() { return description; }
		
		//------------------------- No Setter Methods --------------------------//
		
		//----------------------- Public Utility Methods -----------------------//
		
		/**
		 * @return A nice, human-readable display exposing all the parameters
		 * associated with this vulnerability. Output can be read in by Matlab.
		 */
		public String toString() {
			StringBuffer buf = new StringBuffer();
			buf.append("Vulnerability_Name = '" + getDisplayName() + "'\n");
			buf.append("Vulnerability_ID = '" + getName() + "'\n");
			
			buf.append("Supported_Structures = ...\n{\n");
			for(int i = 0; i < supportedStructs.length; ++i) {
				buf.append("   '" + supportedStructs[i] + "'\n");
			}
			buf.append("}\n\n");
			
			buf.append("Description = ");
			buf.append("'" + getDescription() + "'\n");
			
			buf.append("Intensity_Measure_Levels = ...\n[\n");
			for(int i = 0; i < imls.length;) {
				buf.append("   " + String.format("%5.4f", imls[i]));
				if( (++i%8) == 0) { buf.append(" ...\n"); }
			}
			buf.append("\n]\n\n");
			
			buf.append("Damage_Factors = ...\n[\n");
			for(int i = 0; i < dfs.length;) {
				buf.append("   " + String.format("%4.3e", dfs[i]));
				if( (++i%6) == 0) { buf.append(" ...\n"); }
			}
			buf.append("\n]'\n\n");
			
			buf.append("Damage_Exceedance_Matrix = ...\n[\n");
			buf.append(matrixString(dem));
			buf.append("\n]\n\n");
			
			return buf.toString().replace("\n\n", "\n");
		}

		//------------------------------------------------------------------------
		// Private Methods
		//------------------------------------------------------------------------
		
		/**
		 * @return A nice, human-readable display of a matrix
		 */
		private String matrixString(Double[][] dem) {
			int rowCount = dem.length;
			if(rowCount == 0) { /* No Rows */ return ""; }
			
			int colCount = dem[0].length;
			StringBuffer buf = new StringBuffer();
			
			for(int i = 0; i < rowCount; ++i) {
				buf.append(String.format(" [ ", i));
				for(int j = 0; j < colCount;) {
					buf.append("   " + String.format("%5.4e", dem[i][j]));
					// Add nice line breaks for 80-character terminal
					if( (++j%5) == 0) { buf.append(" ...\n"); }
				} // END: for(colCount)
				
				buf.append(" ]\n");
			} // END: for(rowCount)
			
			return buf.toString().replace("[   ", "[");
		}
		
	} // END: DiscreteVulnerabilityImpl
	
} // END: DiscreteVulnerabilityFactory
