package org.opensha.sra.vulnerability;

import java.util.ArrayList;
import java.util.Arrays;

import org.opensha.commons.data.function.ArbitrarilyDiscretizedFunc;
import org.opensha.commons.data.function.DiscretizedFunc;
import org.opensha.commons.param.ParameterList;
import org.opensha.sha.imr.param.IntensityMeasureParams.PGA_Param;
import org.opensha.sha.imr.param.IntensityMeasureParams.SA_Param;
import org.opensha.sra.asset.AssetType;
import org.opensha.sra.asset.LossType;


/**
 * Represents a probabilistic relationship between an intensity measure 
 * level (IML) and damage factor (DF) for a given structure type.  Flexible
 * enough to be used for modeling vulnerability for earthquake, wind, 
 * flood, or other hazards.
 * 
 * 
 * @author Keith Porter
 * @author Eric Martinez
 * @author Ned Field
 * @author Peter Powers
 */
public abstract class AbstractVulnerability implements Vulnerability {
	
	private String name = "Unnamed Vulnerability";
	private String shortName = "unnamed";
	private String im_type;
	private double[] im_levels;
	
	// NOTE: the Trackable implementation was removed, this may cause 
	// problems with this ported implementation
	
//public abstract class AbstractVulnerability implements Trackable {
//	/* These variables are useful for implementing the Trackable Interface **/
//	private static ArrayList<Trackable> registeredTypes = new ArrayList<Trackable>();
//	private static TreeMap<String, ArrayList<Trackable>> tracker = new TreeMap<String, ArrayList<Trackable>>();

//	// A list of fully qualified class names that the implementing Vulnerability Supports (i.e. IMTs, StructureTypes)
//	protected ArrayList<String> supportedTypes = null;
	
	/* These variables are useful for implementing VulnerabilityFunctions */
	public static final String SA = SA_Param.NAME;
	public static final String PGA = PGA_Param.NAME;
	private double period = 0.0;
	
	protected double ADF;
	protected double BDF;
	//protected int NIML;
	
	/**
	 * Convenience constructor for use by implementors.

	 * @param name of the <code>Vulnerability</code> relationship
	 * @param im_levels the IML values to use
	 * @throws NullPointerException if <code>im_levels</code> is 
	 * 		<code>null</code> 
	 * @throws IllegalArgumentException if <code>im_levels</code> is empty
	 */
	protected AbstractVulnerability(
			String name,
			String shortName, 
			String im_type,
			double[] im_levels) {
		
		// check imt
		// TODO : IMT's MUST BE DEFINED AS AN ENUM. This is a case in point as
		// to why representing each IMT as a separate parameter makes for more
		// work. In this instance, we'd have to check that the passed in String
		// equals SA_Param.NAME etc. Furthermore, if a new IMT is defined, we
		// potentially have to revisit this code and adjust the argument checks
		
		// check for null and empty imls
		if (im_levels == null) {
			throw new NullPointerException("IM Levels array is null");
		} else if (im_levels.length == 0) {
			throw new IllegalArgumentException("IM Levels array is empty");
		}
		
		// check for negative iml values
		for (double level : im_levels) {
			if (level < 0) {
				throw new IllegalArgumentException(
						"IM Levels array contains negative values");
			}
		}
		this.name = name;
		this.shortName = shortName;
		this.im_type = im_type;
		// create independent copy and sort IMLs ascending
		this.im_levels = im_levels.clone();
		Arrays.sort(this.im_levels);
	}

	
//	////////////////////////////////////////////////////////////////////////////////
//	//                    Minimum Functions to Implement Trackable                //
//	////////////////////////////////////////////////////////////////////////////////
//	/**
//	 * See the generic contract in Trackable.
//	 * @see Trackable
//	 */
//	public ArrayList<Trackable> getRegisteredTypes() {
//		return registeredTypes;
//	}
//
//	/**
//	 * See the generic contract in Trackable.
//	 * @see Trackable
//	 */
//	public ArrayList<Trackable> getSupportedTypes(Trackable obj) {
//		String tId = obj.getTrackableId();
//		ArrayList<Trackable> rtn = null;
//		if(tracker.containsKey(tId)) {
//			rtn = tracker.get(tId);
//		}
//		return rtn;
//	}
//
//	/**
//	 * See the generic contract in Trackable.
//	 * @see Trackable
//	 */
//	public String getTrackableId() {
//		String str = (this.getClass()).toString();
//		return str.substring(6);
//	}
//
//	/**
//	 * See the generic contract in Trackable.
//	 * @see Trackable
//	 */
//	public void register(ArrayList<String> types) {
//		Trackable t = (Trackable) this;
//		
//		// Add this to the list of known registered types.
//		if(registeredTypes.indexOf(t) == -1)
//			registeredTypes.add(t);
//		
//		// Add all the supported types to the tracker
//		for(int i = 0; i < types.size(); ++i) {
//			String type = types.get(i);
//			ArrayList<Trackable> supported = tracker.get(type);
//			if(supported == null)
//				supported = new ArrayList<Trackable>();
//			supported.add(t);
//			tracker.put(type, supported);
//		}
//	}
//	
//	/**
//	 * See the generic contract in Trackable.
//	 * @see Trackable
//	 */
//	public boolean equals(Object o) {
//		if(o == null) return false;
//		if(! (o instanceof StructureType) ) return false;
//		String tId = this.getTrackableId();
//		String oId = (o.getClass().toString()).substring(6);
//		return tId.equals(oId);
//	}
	
	
	////////////////////////////////////////////////////////////////////////////////
	//                               Public Functions                             //
	////////////////////////////////////////////////////////////////////////////////
	
//  not used	
//	/**
//	 * @return An <code>ArrayList</code> of currently registered <code>VulnerabilityModel</code>s
//	 */
//	public ArrayList<StructureType> getLibraryOfTypes() {
//		ArrayList<StructureType> st = new ArrayList<StructureType>();
//		for(int i = 0; i < registeredTypes.size(); ++i) {
//			st.add( ( (StructureType) registeredTypes.get(i) ) );
//		}
//		return st;
//	}
	
//  not used	
//	/**
//	 * @return An <code>ArrayList</code> of types supported by one or more of the
//	 * currently registered <code>VulnerabilityModels</code>
//	 */
//	public ArrayList<String> getSupportedTypes() {
//		return supportedTypes;
//	}
	
	
	/**
	 * @return The lower bound of the Damage Factor values associated with the implementing model
	 */
	public double getADF() {
		return ADF;
	}
	
	/**
	 * @return The upper bound of the Damage Factor values associated with the implementing model
	 */
	public double getBDF() {
		return BDF;
	}

	/**
	 * @return The return period for the implementing model.
	 */
	public double getPeriod() {
		return period;
	}

	/**
	 * Sets the return period to <code>period</code>
	 * @param period The new return period (in years).
	 */
	public void setPeriod(double period) {
		this.period = period;
	}
	
//	/** TODO clean
//	 * @return The cannonical name used to describe this Vulnerability Model.
//	 */
//	public abstract String getDisplayName();

	
	/**
	 * Gets the IML values associated with the current VulnerabilityModel.  And puts
	 * them into the X-values of a new </code>DiscretizedFunc</code> object.  The 
	 * Y-values are all initialized to zero.
	 * 
	 * @return A new <code>DiscretizedFunc</code> with X-values representing the IML
	 * values for the current Vulnerability Model, and Y-Values all set to Zero.
	 */
	public DiscretizedFunc getHazardTemplate() {
		// updated to use imls now stored internally in this class - ppowers
		// TODO clean
//		ArrayList<Double> IMLs = getIMLValues();
//		DiscretizedFunc rtn = new ArbitrarilyDiscretizedFunc();
//		
//		for(int i = 0; i < im_levels.length; ++i)
//			rtn.set(IMLs.get(i), 0.0);
//		return rtn;
		DiscretizedFunc rtn = new ArbitrarilyDiscretizedFunc();
		for(double level : im_levels) {
			rtn.set(level, 0.0);
		}
		return rtn;
	}
	
	/**
	 * Fetches the DF values out of the DF Lookup Table.
	 * @return An <code>ArrayList</code> of doubles representing the
	 * DF values for the current Vulnerability Model.
	 */
	public ArrayList<Double> getDFVals() {
		ArrayList<double[]> tmp = getDFTable();
		ArrayList<Double> rtn = new ArrayList<Double>();
		
		for(int i = 0; i < tmp.size(); ++i) {
			double[] entry = tmp.get(i);
			rtn.add(entry[1]);
		}
		return rtn;
	}
	
//	/** TODO clean
//	 * @return The number of Intensity Measure Levels for the current Vulnerability Model.
//	 */
//	public double getNIML() {
//		return NIML;
//	}
	
	/* implementation */
	public String getName() {
		return name;
	}
	
	public String getShortName() {
		return shortName;
	}

	/* implementation */
	public ParameterList getParameters() {
		return null;
	}
	
	/* implementation */
	public void setParameter(String name, Object value) {
		// TODO
	};

//	/**
//	 * old javadoc: TODO clean
//	 * Fetches the IML values out of the DF Lookup Table.
//	 * @return An <code>ArrayList</code> of doubles representing the
//	 * IML values for the current Vulnerability Model.
//	 */
//	/* implementation */
//	public ArrayList<Double> getIMLValues() {
//		ArrayList<double[]> tmp = getDFTable();
//		ArrayList<Double> rtn = new ArrayList<Double>();
//		
//		for(int i = 0; i < tmp.size(); ++i) {
//			double[] entry = tmp.get(i);
//			rtn.add(entry[0]);
//		}
//		return rtn;
//	}

	/* implementation */
	public double[] getIMLValues() {
		return im_levels.clone();
	}

	/* implementation */
	public double getMinIMLVal() {
		return im_levels[0];
	}
	
	/* implementation */
	public double getMaxIMLVal() {
		return im_levels[im_levels.length-1];
	}
	
	public double getMeanLoss() {
		// TODO 
		return -1;
	}
	
	/* implementation */
	public double getLossStdDev() {
		// TODO 
		return -1;
	}
	
	/* implementation */
	public double getLossAtExceedProb() {
		// TODO 
		return -1;
	}
	
	/* implementation */
	public double getExceedProb(double loss, double iml) {
		// TODO 
		return -1;
	}
	
	/* implementation */
	public AssetType getAssetType() { return null; }
	
	/* implementation */
	public LossType getLossType() { return null; }
	
	/**
	 * original javadoc: TODO clean
	 * Gets the IML values and DF values and maps them into the X,Y
	 * values (respectively) of a new <code>DiscretizedFunc</code>.
	 * 
	 * @return A <code>DiscretizedFunc</code> with X,Y values corresponding
	 * to the IML,DF values respectively.  The function will have NIML points.
	 */
	/* implementation */
	public DiscretizedFunc getVulnerabilityFunc() {
		//ArrayList<Double> IMLs = getIMLValues();
		ArrayList<Double> DFs = getDFVals();
		DiscretizedFunc func = new ArbitrarilyDiscretizedFunc();
//		String modelID = (getClass()).toString();
//			return str.substring(6);

		String name = getClass().getSimpleName();//getTrackableId().replaceAll("/^.*\\./", "");
		String info = getClass().getSimpleName() + "," + getIMT() + "," + im_levels.length;
		
		func.setName(name);
		func.setInfo(info);
		for(int i = 0; i < im_levels.length; i++)
			func.set(im_levels[i], DFs.get(i));
		
		return func;
	}

	/* implementation */
	public String getIMT() {
		return im_type;
	}
	
	////////////////////////////////////////////////////////////////////////////////
	//                             Abstract Functions                             //
	////////////////////////////////////////////////////////////////////////////////
	
	
	/**
	 * The DF Lookup Table has dimension NIML x 3 and holds the IML, MDF, and COV
	 * values associated with the current Vulnerability Model.  The structure of
	 * this table is as follows:<br />
	 * <pre>
	 * 			\________________________________________________
	 * 		IML  |  IML1	IML2	IML3	....	IML(NIML)
	 * 		MDF  |	MDF1	MDF2	MDF3	....	MDF(NIML)
	 * 		COV  |	COV1	COV2	COV3	....	COV(NIML
	 * </pre>
	 * @return The DF lookup table used with the current Vulnerability Model.
	 */
	public abstract ArrayList<double[]> getDFTable();
	
	/**
	 * Interpolates the MDF values found in the DFTable for the given IML.
	 * 
	 * @param IML The desired Intensity Measure Level.
	 * @return The mean Damage Factor for the given IML.
	 */
	public abstract double getDF(double IML);

	/**
	 * Gets or creates the Damage Exceedance Matrix associated with the
	 * current vulnerability model.  This can either be precomputed and
	 * stored in memory (faster runtime, larger memory requirements), or
	 * it can be computed on the fly with an (assumed/given) distribution
	 * by using the MDF and COVDF.
	 * 
	 * @return The Damage Exceedance Matrix
	 */
	public abstract double[][] getDEMMatrix();
	
	/**
	 * Gets or computes the DF values that correspond the the Damage Exceedance
	 * Matrix.  This may or may not be the same as the regular MDF values depending
	 * on implementation.
	 * 
	 * @return The Damage Factor values associated with the DEM.
	 */
	public abstract double[] getDEMDFVals();

		
}
