package org.opensha.sra.vulnerability;

import java.util.List;

import org.opensha.commons.data.function.DiscretizedFunc;
import org.opensha.commons.param.Parameter;
import org.opensha.commons.param.ParameterList;
import org.opensha.sra.asset.Asset;
import org.opensha.sra.asset.AssetType;
import org.opensha.sra.asset.LossType;

/**
 * This class represents a <code>Vulnerability</code> relationship. A 
 * <code>Vulnerability</code> relationship gives the pdf of loss as a function
 * of (scalar) Intensity measure Level (IML). It is associated with one among 
 * a predefined set of <code>LossType</code>s. The <code>Vulnerability</code> 
 * relationship for a given <code>Asset</code> must be associated with the same
 * <code>LossType</code> as the <code>Asset</code>. If the <code>Asset</code>
 * is a building, all the possible <code>Vulnerability</code> relationships 
 * must give building repair cost normalized by replacement cost. If the
 * <code>Asset</code> is contents, all the possible <code>Vulnerability</code> 
 * relationships must give content repair normalized by replacement cost. 
 * If the <code>Asset</code> is people, all the possible <code>Vulnerability
 * </code> relationships must give number of people injured normalized by 
 * number of occupants.  If the <code>Asset</code> is economic output, all the
 * possible <code>Vulnerability</code> relationships must give number of days
 * of lost use.<br/>
 * <br/>
 * Four subclasses of vulnerability relationship have been defined:
 * <ul>
 * <li><em>Damage exceedance matrix</em>. Columns: IMLs; rows: fraction of value
 * 		lost; cells: probability of exceeding fraction of value lost given IML. 
 * 		Implemented by Martinez.</li>
 * <li><em>Damage probability matrix</em>. Columns: IMLs; rows: MECE range of 
 * 		fraction of value lost from 0 to 100%; cells: probability of 
 * 		experiencing fraction of value lost given IML.</li>
 * <li><em>Discrete vulnerability function and COV</em>: column 1: IML; column 
 * 		2: expected value of loss as fraction of value exposed, given IML; 
 * 		column 3: coefficient of variation of loss given IML. Vulnerability 
 * 		function also has a prescribed parametric conditional distribution of
 * 		loss, given IML.</li>
 * <li><em>Parametric vulnerability relationship</em> like Wesson et al. 
 * 		(2004).</li>
 * 
 * @author Keith Porter
 * @author Ned Field
 * @author Peter Powers
 * @version $Id:$
 * @see Asset
 * @see AssetType
 * @see LossType
 */
public interface Vulnerability {

	/**
	 * Returns the name of this <code>Vulnerability</code> relationship.
	 * @return the name of this <code>Vulnerability</code>
	 */
	public String getName();
	
	/**
	 * Returns the short name of this <code>Vulnerability</code> relationship.
	 * @return the short name of this <code>Vulnerability</code>
	 */
	public String getShortName();
	
	/**
	 * Returns the list of adjustable parameters associated with this
	 * <code>Vulnerability</code> relationship. These <code>Parameter</code>s
	 * are implementation specific.
	 * 
	 * @return the list of adjustable <code>Parameter</code>s
	 */
	public ParameterList getParameters();
	
	/**
	 * Sets an adjustable parameter
	 * @param name of the <code>Parameter</code> to set
	 * @param value
	 */
	public void setParameter(String name, Object value);

	/**
	 * Returns a sorted (ascending) array of discrete IMLs at which the 
	 * <code>Vulnerability</code> relationship is defined.
	 */
	public double[] getIMLValues();
	
	/**
	 * Returns minimum IML supported by the <code>Vulnerability</code>
	 * relationship. Nonnegative real.
	 * 
	 * @return the minimum IML value
	 */
	public double getMinIMLVal();
	
	/**
	 * Returns maximum IML supported by the <code>Vulnerability</code>
	 * relationship. Nonnegative real. (Can be infinity.)
	 * 
	 * @return the maximum IML value
	 */
	public double getMaxIMLVal();
	
	/**
	 * Returns the mean loss (potentially normalized by value exposed) given
	 * an IML. Non-negative real (double).
	 * 
	 * @return the mean loss for a given IML
	 */
	public double getMeanLoss();
	
	
	/**
	 * Returns the standard deviation of loss given a value of IML. 
	 * Non-negative real (double).
	 * 
	 * @return the stadard deviation of loss for a given IML
	 */
	public double getLossStdDev();
	
	/**
	 * Returns the loss with probability P of being exceeded given P and an 
	 * IML. Non-negative real (double).
	 * 
	 * @return the loss for a given probability and IML
	 */
	public double getLossAtExceedProb();
	
	/**
	 * Returns the probability P of loss L being exceeded given L and IML.
	 * (i.e. the method returns P[LM ≥ L|L, IML]).
	 * Real between 0 and 1 (double).
	 * 
	 * @return the probability of a loss being exceeded for a given loss 
	 * 		and IML.
	 */
	public double getExceedProb(double loss, double iml);
	
	/**
	 * Returns the <code>AssetType</code> for which this <code>Vulnerability
	 * </code> relationship is applicable.
	 * 
	 * @return the associated <code>AssetType</code>
	 */
	public AssetType getAssetType();
	
	/**
	 * Returns the <code>LossType</code> for which this <code>Vulnerability
	 * </code> relationship is applicable.
	 * 
	 * @return the associated <code>LossType</code>
	 */
	public LossType getLossType();
	
	/**
	 * Returns the mean value of loss at each IML value. The values are
	 * returned as a function.
	 * 
	 * @return the underlying vulnerability function
	 */
	public DiscretizedFunc getVulnerabilityFunc();
	
	/**
	 * Returns the Intensity Measure Type (IMT) associated with this 
	 * <code>Vulnerability</code> relationship.
	 * 
	 * @return the <code>String</code> representation of the IMT
	 */
	public String getIMT();

	
}
