
!STRUCTURE: procedure         PURPOSE: Compute UTM grid from GP           
!LANGUAGE : Power Basic       AUTHOR : B. W. Drew                         
!Wolftrap Run Road      Vienna, VA 22182      (703) 356-9454  
!                           DOCUMENTATION                                 
!FUNCTIONAL DESCRIPTION:  Converts geographic (latitude, longitude)
!coordinates to grid coordinates (zone, northing, easting) on the universal
!transverse Mercator (UTM) projection. ACCURACY NOTE: Terms T4, T5, T8 & T9
!may not be needed in applications requiring less accuracy.

!INPUT ARGUMENTS:
!Variable Name      Type  Description
!lat              dbl   latitude, degrees
!lon             dbl   longitude, degrees
!iutm              int   UTM zone number
!fix              bool  Flag, = 0 compute UTM zone, = 1 use input zone

!OUTPUT ARGUEMENTS:
!Variable Name      Type  Description
!iutm              int   UTM zone number (see input arguements)
!y#                dbl   UTM northing, meters
!x#                dbl   UTM easting, meters


	subroutine GP2UTM(latr,lonr,iutm,x,y,fix)

	implicit none
	
	integer fix,iutm

	real*4 latr,lonr,x,y

	real*8 a_axis,b_axis,flattening
	real*8 FE,OK,RECF,ES,EBS,SPHI,SLAM
	real*8 TN,AP,BP,CP,DP,EP,DLAM,OLAM
	real*8 T1,T2,T3,T4,T5,T6,T7,T8,T9
	real*8 S,C,T,ETA,SN,TMD,NFN
	real*8 DegToRad

	parameter (a_axis=6378137.)
	parameter (flattening=1./298.2572236)
	parameter (FE = 500000.,OK = .9996)	
	DegToRad=datan(1.D+00)/45.D+00

!***********************************
!	Checking the longitude value
!***********************************

	if (lonr.gt.180.) then
	   write(6,*) 'Warning!!!'
	   write(6,*) 'Longitude not in [-180;180]'
	   write(6,*) 'The program will stop'
	   stop
	endif

!************************************************
!****   DERIVE OTHER ELLIPSOID PARAMTERS    *****
!****         FROM INPUT VARIABLES          *****
!****    A = SEMI-MAJOR AXIS OF ELLIPSOID   *****
!**** RECF = RECIPROCAL OF FLATTENING (1/F) *****
!************************************************

	SPHI=dble(latr)*DegToRad
	SLAM=dble(lonr)*DegToRad

!* SEMI MAJOR AXIS - b_axis **
	RECF = 1. / flattening
	b_axis = a_axis*( RECF -1. )/RECF

!* ECCENTRICITY SQUARED **
	ES = ( a_axis**2 -b_axis**2 )/a_axis**2

!* SECOND ECCENTRICITY SQUARED **
	EBS = ( a_axis**2 -b_axis**2 )/b_axis**2

!* TRUE MERIDIONAL DISTANCE CONSTANTS **
	TN = ( a_axis -b_axis )/( a_axis + b_axis )
	AP = a_axis*( 1. -TN +5.*( TN**2 -TN**3)/4. &
     		+81.*( TN**4 -TN**5 )/64.)
	BP = 3.*a_axis*( TN -TN**2  +7.*( TN**3 -TN**4)/8. & 
     		+55.*TN**5/64.)/2.
	CP = 15.*a_axis*( TN**2 -TN**3 +3.*( TN**4 &
     		-TN**5)/4.)/16.
	DP = 35.*a_axis*( TN**3 -TN**4 +11.*TN**5/16.)/48.
	EP = 315.*a_axis*( TN**4 -TN**5 )/512.

!**** ZONE - CENTRAL MERIDIAN *****

!** HOLD FIXED IF fix IS SET TO ONE
!** DETERMINE ZONE NUMBER IF fix IS ZERO
!	if (fix.eq.0) iutm = 31 + idint(SLAM/DegToRad/6.D+0)
	if (fix.eq.0) iutm =1+idint(30.D+0 +SLAM/DegToRad/6.D+0)

! ZONE TRAP - AT HEMISPHERE LIMITS **
	if (iutm.gt.60) iutm = 60
	if (iutm.lt.1)  iutm = 1

!** CENTRAL MERIDIAN ***
	OLAM = (iutm*6 -183 )*DegToRad

!** DELTA LONGITUDE ***
!** DIFFERENCE BETWEEN LONGITUDE AND CENTRAL MERIDIAN ***
	DLAM = SLAM -OLAM

!** OTHER COMMON TERMS ***
	S = sin( SPHI )
	C = cos( SPHI )
	T = S/C
	ETA = EBS*C**2

!* RADIUS OF CURVATURE IN PRIME VERTICAL **
	SN  = a_axis/sqrt(1.- ES*sin(SPHI)**2)

!* TRUE MERIDIONAL DISTANCE **
	TMD = AP*SPHI-BP*sin(2.*SPHI)+CP*sin(4.*SPHI)- &	
     	DP*sin(6.*SPHI)+EP*sin(8.*SPHI)

!**** NORTHING *****

	T1 = TMD*OK
	T2 = SN*S*C*OK/2.
	T3 = SN*S*C**3*OK*( 5. -T**2 +9.*ETA +4.*ETA**2 )/24.
	T4 = SN*S*C**5*OK*( 61. -58.*T**2 +T**4 +270.*ETA &
     		-330.*T**2*ETA +445.*ETA**2 +324.*ETA**3  &
     		-680.*T**2*ETA**2 +88.*ETA**4 -600.*T**2*ETA**3  &
     		-192.*T**2*ETA**4 )/720.
	T5 = SN*S*C**7*OK*( 1385. -3111.*T**2 +543.*T**4 &
     		-T**6 )/40320.

!* FALSE NORTHING **
	nfn = 0.
!	if (SPHI.lt.0.) nfn = 10000000.

	Y = sngl(nfn +T1 +DLAM**2*T2 +DLAM**4*T3 +DLAM**6*T4 &
     		+DLAM**8*T5)

!**** EASTING *****

	T6 = SN*C*OK
	T7 = SN*C**3*OK*( 1. -T**2 +ETA )/6.
	T8 = SN*C**5*OK*( 5. -18.*T**2 +T**4 +14.*ETA &
     		-58.*T**2*ETA +13.*ETA**2 +4.*ETA**3 &
     		-64.*T**2*ETA**2 -24.*T**2*ETA**3 )/120.
	T9 = SN*C**7*OK*( 61. -479.*T**2 +179.*T**4  &
    		-T**6)/5040.

	X = sngl(FE +DLAM*T6 +DLAM**3*T7 +DLAM**5*T8 &
     		+DLAM**7*T9)

	end
