"""
Module DarfieldForecastsTests

This module invokes acceptance tests for all installed forecasts models
within New Zealand testing center for the Darfield testing region.
"""

__version__ = "$Revision$"
__revision__ = "$Id$"


import os, unittest, shutil, datetime
from Environment import *
import CSEPFile, CSEPGeneric, Forecast, GeographicalRegions, SlipModels

from CSEPTestCase import CSEPTestCase
from CSEPPropertyFile import CSEPPropertyFile

from ForecastFactory import ForecastFactory
from ForecastHandlerFactory import ForecastHandlerFactory
from GeoNetNZDataSource import GeoNetNZDataSource
from ETAS_HWOneDayModel import ETAS_HWOneDayModel
from ETAS_HWOneMonthModel import ETAS_HWOneMonthModel
from ETAS_HWOneYearModel import ETAS_HWOneYearModel
from K2_HWOneDayModel import K2_HWOneDayModel
from K2_HWOneMonthModel import K2_HWOneMonthModel
from K2_HWOneYearModel import K2_HWOneYearModel
from K3_HWOneDayModel import K3_HWOneDayModel
from K3_HWOneMonthModel import K3_HWOneMonthModel
from K3_HWOneYearModel import K3_HWOneYearModel
from HainzlModel import HainzlModel
from HainzlOneDayModel import HainzlOneDayModel
from HainzlOneMonthModel import HainzlOneMonthModel
from HainzlOneYearModel import HainzlOneYearModel
from CattaniaModel import CattaniaModel, HASH_DIR_ENV
from CattaniaOneDayModel import CattaniaOneDayModel
from CattaniaOneMonthModel import CattaniaOneMonthModel
from CattaniaOneYearModel import CattaniaOneYearModel
from STEPCoulombModel import STEPCoulombModel
from STEPCoulombOneDayModel import STEPCoulombOneDayModel
from STEPCoulombOneMonthModel import STEPCoulombOneMonthModel
from STEPCoulombOneYearModel import STEPCoulombOneYearModel
from SUPCanterburyOneDayModel import SUPCanterburyOneDayModel
from SUPCanterburyOneMonthModel import SUPCanterburyOneMonthModel
from SUPCanterburyOneYearModel import SUPCanterburyOneYearModel


#--------------------------------------------------------------------
#
# Test forecast generation for registered models in the NZ
# testing center for the Darfield testing region.
#
class DarfieldForecastsTests (CSEPTestCase):

    # Static data of the class
    
    # Reference directory
    __refDir = os.path.join(Environment.Variable[CENTER_CODE_ENV],
                            'src',
                            'SCECModels',
                            'NewZealand',
                            'test',
                            'data',
                            'models')

    # Test date used by Max Werner and Agnes Helmstetter forecasts models:
    # ETAS, K2 and K3 (2010-09-03T16:35:42Z)
    __testDate = datetime.datetime(2010, 9, 3, 16, 35, 42)
    
    __CattaniaTestDate = datetime.datetime(2010, 9, 13, 16, 35, 41)
    
    __K2T1_HWTestDate = datetime.datetime(2010, 9, 12, 16, 35, 42)
    
    __STEPCoulombTestDate = datetime.datetime(2012, 5, 20, 5, 6, 0)
    
    __SUPTestDate = datetime.datetime(2011, 12, 7, 0, 0, 0)
    
    __HWRefDir = os.path.join(__refDir, 
                              'HelmstetterWernerModels')
                                          
    __HainzlRefDir = os.path.join(__refDir, 
                                  'HainzlModels')

    __CattaniaRefDir = os.path.join(__refDir, 
                                    'CattaniaModels')

    __STEPCoulombRefDir = os.path.join(__refDir, 
                                       'STEPCoulombModel')

    __SUPRefDir = os.path.join(__refDir, 
                               'SUPCanterbury')


    #---------------------------------------------------------------------------
    #
    # Invoke ETAS_HW one-day model for Darfield testing region, and verify 
    # the results.
    #
    def testETASHWOneDayModel(self):
        """ Generate ETAS_HW one-day forecast for Darfield testing region and
verify results."""

        # Setup test name
        CSEPTestCase.setTestName(self, self.id())

        # Use the same test directory for input catalog and result forecast files
        args = [CSEPTestCase.TestDirPath]
        
        # Instantiate forecast object
        hw_model = ForecastFactory().object(ETAS_HWOneDayModel.Type, 
                                            args)

        # Copy reference input catalog data file to the test directory
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__HWRefDir, 
                                     "Canterbury_M2_1980_2012.txt"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     hw_model.inputCatalogFilename()))    
        
        # Change directory to the test location to guarantee that all model
        # configuration and scratch files are placed there.
        cwd = os.getcwd() 
        os.chdir(CSEPTestCase.TestDirPath)

        try:            
           
           hw_model.create(DarfieldForecastsTests.__testDate, 
                           CSEPTestCase.TestDirPath)
   
           
           # Reference file with test results to compare against
           reference_file = "E_m3M4_T1_pl_forecast_2010-09-03T16:35:42Z_v2.xml"
           
           shutil.copyfile(os.path.join(DarfieldForecastsTests.__HWRefDir, 
                                        reference_file),
                           os.path.join(CSEPTestCase.TestDirPath, 
                                        reference_file))
           
   
           error_msg = "ETAS_HW one-day forecast generation for 0.1degree \
Global testing region failed."
           _ref_file = ForecastHandlerFactory().CurrentHandler.XML(reference_file)
           _mod_file = ForecastHandlerFactory().CurrentHandler.XML(hw_model.filename())
           
           self.failIf(CSEPFile.compare(_ref_file.toASCII(), 
                                        _mod_file.toASCII(),
                                        precision = 1.0,
                                        use_percent_diff=True) is False,
                       error_msg)
           
           # Check for existence of metadata file for the forecast
           error_msg = "Metadata file for ETAS_HW one-day forecast 0.1degree \
Global forecast is missing."
           self.failIf(os.path.exists(hw_model.filename() + 
                                      CSEPPropertyFile.Metadata.Extension) is False, 
                       error_msg)

        finally:
            os.chdir(cwd)


    #---------------------------------------------------------------------------
    #
    # Invoke ETAS_HW one-month model for Darfield testing region, and verify 
    # the results.
    #
    def testETASHWOneMonthModel(self):
        """ Generate ETAS_HW one-month forecast for NZ Darfield testing region 
and verify results."""
        
        # Setup test name
        CSEPTestCase.setTestName(self, self.id())

        # Use the same test directory for input catalog and result forecast files
        args = [CSEPTestCase.TestDirPath]
        
        # Instantiate forecast object
        hw_model = ForecastFactory().object(ETAS_HWOneMonthModel.Type, 
                                            args)

        # Copy reference input catalog data file to the test directory
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__HWRefDir, 
                                     "Canterbury_M2_1980_2012.txt"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     hw_model.inputCatalogFilename()))    
        
        # Change directory to the test location to guarantee that all model
        # configuration and scratch files are placed there.
        cwd = os.getcwd() 
        os.chdir(CSEPTestCase.TestDirPath)

        try:            
           
           hw_model.create(DarfieldForecastsTests.__testDate, 
                           CSEPTestCase.TestDirPath)
   
           
           # Reference file with test results to compare against
           reference_file = "E_m3M4_T30_pl_forecast_2010-09-03-16:35:42.xml"
           
           shutil.copyfile(os.path.join(DarfieldForecastsTests.__HWRefDir, 
                                        reference_file),
                           os.path.join(CSEPTestCase.TestDirPath, 
                                        reference_file))
           
   
           error_msg = "ETAS_HW one-month forecast generation for 0.1degree \
Global testing region failed."
           _ref_file = ForecastHandlerFactory().CurrentHandler.XML(reference_file)
           _mod_file = ForecastHandlerFactory().CurrentHandler.XML(hw_model.filename())
           
           self.failIf(CSEPFile.compare(_ref_file.toASCII(), 
                                        _mod_file.toASCII(),
                                        precision = 1.0,
                                        use_percent_diff=True) is False,
                       error_msg)
           
           # Check for existence of metadata file for the forecast
           error_msg = "Metadata file for ETAS_HW one-month forecast 0.1degree \
Global forecast is missing."
           self.failIf(os.path.exists(hw_model.filename() + 
                                      CSEPPropertyFile.Metadata.Extension) is False, 
                       error_msg)

        finally:
            os.chdir(cwd)
            

    #---------------------------------------------------------------------------
    #
    # Invoke ETAS_HW one-year model for Darfield testing region, and verify 
    # the results.
    #
    def testETASHWOneYearModel(self):
        """ Generate ETAS_HW one-year forecast for Darfield testing region 
and verify results."""
        
        # Setup test name
        CSEPTestCase.setTestName(self, self.id())

        # Use the same test directory for input catalog and result forecast files
        args = [CSEPTestCase.TestDirPath]
        
        # Instantiate forecast object
        hw_model = ForecastFactory().object(ETAS_HWOneYearModel.Type, 
                                            args)

        # Copy reference input catalog data file to the test directory
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__HWRefDir, 
                                     "Canterbury_M2_1980_2012.txt"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     hw_model.inputCatalogFilename()))    
        
        # Change directory to the test location to guarantee that all model
        # configuration and scratch files are placed there.
        cwd = os.getcwd() 
        os.chdir(CSEPTestCase.TestDirPath)

        try:            
           
           hw_model.create(DarfieldForecastsTests.__testDate, 
                           CSEPTestCase.TestDirPath)
   
           
           # Reference file with test results to compare against
           reference_file = "E_m3M4_T365_pl_forecast_2010-09-03-16:35:42.xml"
           
           shutil.copyfile(os.path.join(DarfieldForecastsTests.__HWRefDir, 
                                        reference_file),
                           os.path.join(CSEPTestCase.TestDirPath, 
                                        reference_file))
           
   
           error_msg = "ETAS_HW one-year forecast generation for 0.1degree \
Global testing region failed."
           _ref_file = ForecastHandlerFactory().CurrentHandler.XML(reference_file)
           _mod_file = ForecastHandlerFactory().CurrentHandler.XML(hw_model.filename())
           
           self.failIf(CSEPFile.compare(_ref_file.toASCII(), 
                                        _mod_file.toASCII(),
                                        precision = 1.0,
                                        use_percent_diff=True) is False,
                       error_msg)
           
           # Check for existence of metadata file for the forecast
           error_msg = "Metadata file for ETAS_HW one-year forecast 0.1degree \
Global forecast is missing."
           self.failIf(os.path.exists(hw_model.filename() + 
                                      CSEPPropertyFile.Metadata.Extension) is False, 
                       error_msg)

        finally:
            os.chdir(cwd)


    #---------------------------------------------------------------------------
    #
    # Invoke K2_HW one-day model for Darfield testing region, and verify 
    # the results.
    #
    def testK2HWOneDayModel(self):
        """ Generate K2_HW one-day forecast for Darfield testing region and
verify results."""
        
        # Setup test name
        CSEPTestCase.setTestName(self, self.id())

        # Use the same test directory for input catalog and result forecast files
        args = [CSEPTestCase.TestDirPath]
        
        # Instantiate forecast object
        hw_model = ForecastFactory().object(K2_HWOneDayModel.Type, 
                                            args)

        # Copy reference input catalog data file to the test directory
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__HWRefDir, 
                                     "Canterbury_M2_1980_2012.txt"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     hw_model.inputCatalogFilename()))    
        
        # Change directory to the test location to guarantee that all model
        # configuration and scratch files are placed there.
        cwd = os.getcwd() 
        os.chdir(CSEPTestCase.TestDirPath)

        try:            
           
           hw_model.create(DarfieldForecastsTests.__testDate, 
                           CSEPTestCase.TestDirPath)
   
           
           # Reference file with test results to compare against
           reference_file = "K2N_m3M4_T1_pl_forecast_2010-09-12T16:35:42Z.xml"
           
           shutil.copyfile(os.path.join(DarfieldForecastsTests.__HWRefDir, 
                                        reference_file),
                           os.path.join(CSEPTestCase.TestDirPath, 
                                        reference_file))
           
   
           error_msg = "K2_HW one-day forecast generation for 0.1degree \
Global testing region failed."
           _ref_file = ForecastHandlerFactory().CurrentHandler.XML(reference_file)
           _mod_file = ForecastHandlerFactory().CurrentHandler.XML(hw_model.filename())
           
           self.failIf(CSEPFile.compare(_ref_file.toASCII(), 
                                        _mod_file.toASCII(),
                                        precision = 1.0,
                                        use_percent_diff=True) is False,
                       error_msg)
           
           # Check for existence of metadata file for the forecast
           error_msg = "Metadata file for K2_HW one-day forecast 0.1degree \
Global forecast is missing."
           self.failIf(os.path.exists(hw_model.filename() + 
                                      CSEPPropertyFile.Metadata.Extension) is False, 
                       error_msg)

        finally:
            os.chdir(cwd)


    #---------------------------------------------------------------------------
    #
    # Invoke K2_HW one-month model for Darfield testing region, and verify 
    # the results.
    #
    def testK2HWOneMonthModel(self):
        """ Generate K2_HW one-month forecast for Darfield testing region and
verify results."""
        
        # Setup test name
        CSEPTestCase.setTestName(self, self.id())

        # Use the same test directory for input catalog and result forecast files
        args = [CSEPTestCase.TestDirPath]
        
        # Instantiate forecast object
        hw_model = ForecastFactory().object(K2_HWOneMonthModel.Type, 
                                            args)

        # Copy reference input catalog data file to the test directory
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__HWRefDir, 
                                     "Canterbury_M2_1980_2012.txt"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     hw_model.inputCatalogFilename()))    
        
        # Change directory to the test location to guarantee that all model
        # configuration and scratch files are placed there.
        cwd = os.getcwd() 
        os.chdir(CSEPTestCase.TestDirPath)

        try:            
           
           hw_model.create(DarfieldForecastsTests.__testDate, 
                           CSEPTestCase.TestDirPath)
   
           
           # Reference file with test results to compare against
           reference_file = "K2N_m3M4_T30_pl_forecast_2010-09-03-16:35:42.xml"
           
           shutil.copyfile(os.path.join(DarfieldForecastsTests.__HWRefDir, 
                                        reference_file),
                           os.path.join(CSEPTestCase.TestDirPath, 
                                        reference_file))
           
   
           error_msg = "K2_HW one-month forecast generation for 0.1degree \
Global testing region failed."
           _ref_file = ForecastHandlerFactory().CurrentHandler.XML(reference_file)
           _mod_file = ForecastHandlerFactory().CurrentHandler.XML(hw_model.filename())
           
           self.failIf(CSEPFile.compare(_ref_file.toASCII(), 
                                        _mod_file.toASCII(),
                                        precision = 1.0,
                                        use_percent_diff=True) is False,
                       error_msg)
           
           # Check for existence of metadata file for the forecast
           error_msg = "Metadata file for K2_HW one-month forecast 0.1degree \
Global forecast is missing."
           self.failIf(os.path.exists(hw_model.filename() + 
                                      CSEPPropertyFile.Metadata.Extension) is False, 
                       error_msg)

        finally:
            os.chdir(cwd)


    #---------------------------------------------------------------------------
    #
    # Invoke K2_HW one-year model for Darfield testing region, and verify 
    # the results.
    #
    def testK2HWOneYearModel(self):
        """ Generate K2_HW one-year forecast for Darfield testing region and
verify results."""
        
        # Setup test name
        CSEPTestCase.setTestName(self, self.id())

        # Use the same test directory for input catalog and result forecast files
        args = [CSEPTestCase.TestDirPath]
        
        # Instantiate forecast object
        hw_model = ForecastFactory().object(K2_HWOneYearModel.Type, 
                                            args)

        # Copy reference input catalog data file to the test directory
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__HWRefDir, 
                                     "Canterbury_M2_1980_2012.txt"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     hw_model.inputCatalogFilename()))    
        
        # Change directory to the test location to guarantee that all model
        # configuration and scratch files are placed there.
        cwd = os.getcwd() 
        os.chdir(CSEPTestCase.TestDirPath)

        try:            
           
           hw_model.create(DarfieldForecastsTests.__testDate, 
                           CSEPTestCase.TestDirPath)
   
           
           # Reference file with test results to compare against
           reference_file = "K2N_m3M4_T365_pl_forecast_2010-09-03-16:35:42.xml"
           
           shutil.copyfile(os.path.join(DarfieldForecastsTests.__HWRefDir, 
                                        reference_file),
                           os.path.join(CSEPTestCase.TestDirPath, 
                                        reference_file))
           
   
           error_msg = "K2_HW one-year forecast generation for 0.1degree \
Global testing region failed."
           _ref_file = ForecastHandlerFactory().CurrentHandler.XML(reference_file)
           _mod_file = ForecastHandlerFactory().CurrentHandler.XML(hw_model.filename())
           
           self.failIf(CSEPFile.compare(_ref_file.toASCII(), 
                                        _mod_file.toASCII(),
                                        precision = 1.0,
                                        use_percent_diff=True) is False,
                       error_msg)
           
           # Check for existence of metadata file for the forecast
           error_msg = "Metadata file for K2_HW one-year forecast 0.1degree \
Global forecast is missing."
           self.failIf(os.path.exists(hw_model.filename() + 
                                      CSEPPropertyFile.Metadata.Extension) is False, 
                       error_msg)

        finally:
            os.chdir(cwd)


    #---------------------------------------------------------------------------
    #
    # Invoke K3_HW one-day model for Darfield testing region, and verify 
    # the results.
    #
    def testK3HWOneDayModel(self):
        """ Generate K3_HW one-day forecast for Darfield testing region and
verify results."""
        
        # Setup test name
        CSEPTestCase.setTestName(self, self.id())

        # Use the same test directory for input catalog and result forecast files
        args = [CSEPTestCase.TestDirPath]
        
        # Instantiate forecast object
        hw_model = ForecastFactory().object(K3_HWOneDayModel.Type, 
                                            args)

        # Copy reference input catalog data file to the test directory
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__HWRefDir, 
                                     "Canterbury_M2_1980_2012.txt"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     hw_model.inputCatalogFilename()))    
        
        # Change directory to the test location to guarantee that all model
        # configuration and scratch files are placed there.
        cwd = os.getcwd() 
        os.chdir(CSEPTestCase.TestDirPath)

        try:            
           
           hw_model.create(DarfieldForecastsTests.__testDate, 
                           CSEPTestCase.TestDirPath)
   
           
           # Reference file with test results to compare against
           reference_file = "K3m2M4T1NKMp_forecast_2010-09-03T16:35:42Z.xml"
           
           shutil.copyfile(os.path.join(DarfieldForecastsTests.__HWRefDir, 
                                        reference_file),
                           os.path.join(CSEPTestCase.TestDirPath, 
                                        reference_file))
           
   
           error_msg = "K3_HW one-day forecast generation for 0.1degree \
Global testing region failed."
           _ref_file = ForecastHandlerFactory().CurrentHandler.XML(reference_file)
           _mod_file = ForecastHandlerFactory().CurrentHandler.XML(hw_model.filename())
           
           self.failIf(CSEPFile.compare(_ref_file.toASCII(), 
                                        _mod_file.toASCII(),
                                        precision = 1.0,
                                        use_percent_diff=True) is False,
                       error_msg)
           
           # Check for existence of metadata file for the forecast
           error_msg = "Metadata file for K3_HW one-day forecast 0.1degree \
Global forecast is missing."
           self.failIf(os.path.exists(hw_model.filename() + 
                                      CSEPPropertyFile.Metadata.Extension) is False, 
                       error_msg)

        finally:
            os.chdir(cwd)


    #---------------------------------------------------------------------------
    #
    # Invoke K3_HW one-month model for Darfield testing region, and verify 
    # the results.
    #
    def testK3HWOneMonthModel(self):
        """ Generate K3_HW one-month forecast for Darfield testing region and
verify results."""
        
        # Setup test name
        CSEPTestCase.setTestName(self, self.id())

        # Use the same test directory for input catalog and result forecast files
        args = [CSEPTestCase.TestDirPath]
        
        # Instantiate forecast object
        hw_model = ForecastFactory().object(K3_HWOneMonthModel.Type, 
                                            args)

        # Copy reference input catalog data file to the test directory
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__HWRefDir, 
                                     "Canterbury_M2_1980_2012.txt"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     hw_model.inputCatalogFilename()))    
        
        # Change directory to the test location to guarantee that all model
        # configuration and scratch files are placed there.
        cwd = os.getcwd() 
        os.chdir(CSEPTestCase.TestDirPath)

        try:            
           
           hw_model.create(DarfieldForecastsTests.__testDate, 
                           CSEPTestCase.TestDirPath)
   
           
           # Reference file with test results to compare against
           reference_file = "K3m2M4T30NKMp_forecast_2010-09-03-16:35:42.xml"
           
           shutil.copyfile(os.path.join(DarfieldForecastsTests.__HWRefDir, 
                                        reference_file),
                           os.path.join(CSEPTestCase.TestDirPath, 
                                        reference_file))
           
   
           error_msg = "K3_HW one-month forecast generation for 0.1degree \
Global testing region failed."
           _ref_file = ForecastHandlerFactory().CurrentHandler.XML(reference_file)
           _mod_file = ForecastHandlerFactory().CurrentHandler.XML(hw_model.filename())
           
           self.failIf(CSEPFile.compare(_ref_file.toASCII(), 
                                        _mod_file.toASCII(),
                                        precision = 1.0,
                                        use_percent_diff=True) is False,
                       error_msg)
           
           # Check for existence of metadata file for the forecast
           error_msg = "Metadata file for K3_HW one-month forecast 0.1degree \
Global forecast is missing."
           self.failIf(os.path.exists(hw_model.filename() + 
                                      CSEPPropertyFile.Metadata.Extension) is False, 
                       error_msg)

        finally:
            os.chdir(cwd)


    #---------------------------------------------------------------------------
    #
    # Invoke K3_HW one-year model for Darfield testing region, and verify 
    # the results.
    #
    def testK3HWOneYearModel(self):
        """ Generate K2_HW one-year forecast for Darfield testing region and
verify results."""
        
        # Setup test name
        CSEPTestCase.setTestName(self, self.id())

        # Use the same test directory for input catalog and result forecast files
        args = [CSEPTestCase.TestDirPath]
        
        # Instantiate forecast object
        hw_model = ForecastFactory().object(K3_HWOneYearModel.Type, 
                                            args)

        # Copy reference input catalog data file to the test directory
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__HWRefDir, 
                                     "Canterbury_M2_1980_2012.txt"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     hw_model.inputCatalogFilename()))    
        
        # Change directory to the test location to guarantee that all model
        # configuration and scratch files are placed there.
        cwd = os.getcwd() 
        os.chdir(CSEPTestCase.TestDirPath)

        try:            
           
           hw_model.create(DarfieldForecastsTests.__testDate, 
                           CSEPTestCase.TestDirPath)
   
           
           # Reference file with test results to compare against
           reference_file = "K3m2M4T365NKMp_forecast_2010-09-03-16:35:42.xml"
           
           shutil.copyfile(os.path.join(DarfieldForecastsTests.__HWRefDir, 
                                        reference_file),
                           os.path.join(CSEPTestCase.TestDirPath, 
                                        reference_file))
           
   
           error_msg = "K3_HW one-year forecast generation for 0.1degree \
Global testing region failed."
           _ref_file = ForecastHandlerFactory().CurrentHandler.XML(reference_file)
           _mod_file = ForecastHandlerFactory().CurrentHandler.XML(hw_model.filename())
           
           self.failIf(CSEPFile.compare(_ref_file.toASCII(), 
                                        _mod_file.toASCII(),
                                        precision = 1.0,
                                        use_percent_diff=True) is False,
                       error_msg)
           
           # Check for existence of metadata file for the forecast
           error_msg = "Metadata file for K3_HW one-year forecast 0.1degree \
Global forecast is missing."
           self.failIf(os.path.exists(hw_model.filename() + 
                                      CSEPPropertyFile.Metadata.Extension) is False, 
                       error_msg)

        finally:
            os.chdir(cwd)



    def __slipModelList (self, 
                         list_filename):
        """ Replace slip model files with absolute file paths specific to the
            runtime environment."""
            
        list_path, list_file = os.path.split(list_filename)
        out_list_filename = os.path.join(CSEPTestCase.TestDirPath,
                                         list_file)

        fhandle = CSEPFile.openFile(out_list_filename, 
                                    CSEPFile.Mode.WRITE)
        
        with CSEPFile.openFile(list_filename) as fhandle_in:
            for each_row in fhandle_in:
                
                row_elems = each_row.split()
                if len(row_elems) > 1 and \
                   len(row_elems[0]) and \
                   row_elems[0] != '#' :
                    
                    # Copy file to the test directory
                    shutil.copyfile(os.path.join(list_path, 
                                                row_elems[-1]),
                                    os.path.join(CSEPTestCase.TestDirPath, 
                                                 row_elems[-1]))

                    # Replace last element with an absolute path to provided file
                    row_elems[-1] = os.path.join(CSEPTestCase.TestDirPath, 
                                                 row_elems[-1])
                
                fhandle.write(' '.join(row_elems) + '\n') 
        
        fhandle.close()


    #---------------------------------------------------------------------------
    #
    # Invoke ETAS0_HAINZL one-day model for Darfield testing region, and verify 
    # the results.
    #
    def testETAS0HAINZLOneDayModel(self):
        """ Generate ETAS0_HAINZL one-day forecast for Darfield testing region and
verify results."""

        
        # Setup test name
        CSEPTestCase.setTestName(self, self.id())

        model_list_file = 'list_of_slip_models.dat'
        
        # Use the same test directory for input catalog and result forecast files
        args = [CSEPTestCase.TestDirPath, 
                '%s=ETAS,%s=0,%s=%s' %(HainzlModel.ModelOption,
                                       HainzlModel.MainshockProbMapsOption,
                                       HainzlModel.SlipModelFileListOption,
                                       os.path.join(CSEPTestCase.TestDirPath,
                                                    model_list_file))]
        
        # Instantiate forecast object
        model = ForecastFactory().object(HainzlOneDayModel.Type, 
                                         args)

        # Copy reference input catalog data file to the test directory
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__HainzlRefDir, 
                                     "examplecatalog.dat"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     model.inputCatalogFilename()))    
        
        # Change directory to the test location to guarantee that all model
        # configuration and scratch files are placed there.
        cwd = os.getcwd() 
        os.chdir(CSEPTestCase.TestDirPath)

        try:            

           # Copy slip model list file into local directory for the run
           self.__slipModelList(os.path.join(DarfieldForecastsTests.__HainzlRefDir,
                                                   model_list_file))
           
           model.create(DarfieldForecastsTests.__testDate, 
                        CSEPTestCase.TestDirPath)
   
           
           # Reference file with test results to compare against
           reference_file = "csepETAS0.dat"
           
           shutil.copyfile(os.path.join(DarfieldForecastsTests.__HainzlRefDir, 
                                        reference_file),
                           os.path.join(CSEPTestCase.TestDirPath, 
                                        reference_file))
           
   
           error_msg = "ETAS0_HAINZL one-day forecast generation for Darfield \
testing region failed."
           
           self.failIf(CSEPFile.compare(reference_file, 
                                        model.filename()) is False,
                       error_msg)
           
           # Check for existence of metadata file for the forecast
           error_msg = "Metadata file for ETAS0_HAINZL one-day forecast for Darfield \
testing region is missing."
           self.failIf(os.path.exists(model.filename() + 
                                      CSEPPropertyFile.Metadata.Extension) is False, 
                       error_msg)

        finally:
            os.chdir(cwd)


    #---------------------------------------------------------------------------
    #
    # Invoke ETAS0_HAINZL one-day model for Darfield testing region using
    # SlipModels Python module for automatic detection of available slip models,
    # and verify the results.
    #
    def testETAS0HAINZLOneDayModelWithSlipModelsModule(self):
        """ Generate ETAS0_HAINZL one-day forecast for Darfield testing region and
verify results."""

        
        # Setup test name
        CSEPTestCase.setTestName(self, self.id())

        os.environ[SlipModels.SLIP_MODELS_ENV] = os.path.join(DarfieldForecastsTests.__refDir,
                                                              'slipModels')
        os.environ[SlipModels.SLIP_MODELS_LAG_ENV] = '0'
        
        # Use the same test directory for input catalog and result forecast files
        args = [CSEPTestCase.TestDirPath, 
                '%s=ETAS,%s=0' %(HainzlModel.ModelOption,
                                 HainzlModel.MainshockProbMapsOption)]
        
        # Instantiate forecast object
        model = ForecastFactory().object(HainzlOneDayModel.Type, 
                                         args)

        # Copy reference input catalog data file to the test directory
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__HainzlRefDir, 
                                     "examplecatalog.dat"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     model.inputCatalogFilename()))    
        
        # Change directory to the test location to guarantee that all model
        # configuration and scratch files are placed there.
        cwd = os.getcwd() 
        os.chdir(CSEPTestCase.TestDirPath)

        try:            

           model.create(DarfieldForecastsTests.__testDate, 
                        CSEPTestCase.TestDirPath)
   
           
           # Reference file with test results to compare against
           reference_file = "csepETAS0.dat"
           
           shutil.copyfile(os.path.join(DarfieldForecastsTests.__HainzlRefDir, 
                                        reference_file),
                           os.path.join(CSEPTestCase.TestDirPath, 
                                        reference_file))
           
   
           error_msg = "ETAS0_HAINZL one-day forecast generation for Darfield \
testing region failed."
           
           self.failIf(CSEPFile.compare(reference_file, 
                                        model.filename()) is False,
                       error_msg)
           
           # Check for existence of metadata file for the forecast
           error_msg = "Metadata file for ETAS0_HAINZL one-day forecast for Darfield \
testing region is missing."
           self.failIf(os.path.exists(model.filename() + 
                                      CSEPPropertyFile.Metadata.Extension) is False, 
                       error_msg)

        finally:
            os.chdir(cwd)
            
            del os.environ[SlipModels.SLIP_MODELS_ENV]


    #---------------------------------------------------------------------------
    #
    # Invoke ETAS1_HAINZL one-day model for Darfield testing region, and verify 
    # the results.
    #
    def testETAS1HAINZLOneDayModel(self):
        """ Generate ETAS1_HAINZL one-day forecast for Darfield testing region and
verify results."""

        # Setup test name
        CSEPTestCase.setTestName(self, self.id())

        model_list_file = model_list_file = 'list_of_slip_models.dat'
        
        # Use the same test directory for input catalog and result forecast files
        args = [CSEPTestCase.TestDirPath, 
                '%s=ETAS,%s=1,%s=%s' %(HainzlModel.ModelOption,
                                       HainzlModel.MainshockProbMapsOption,
                                       HainzlModel.SlipModelFileListOption,
                                       os.path.join(CSEPTestCase.TestDirPath,
                                                    model_list_file))]
        
        # Instantiate forecast object
        model = ForecastFactory().object(HainzlOneDayModel.Type, 
                                         args)

        # Copy reference input catalog data file to the test directory
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__HainzlRefDir, 
                                     "examplecatalog.dat"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     model.inputCatalogFilename()))    
        
        # Change directory to the test location to guarantee that all model
        # configuration and scratch files are placed there.
        cwd = os.getcwd() 
        os.chdir(CSEPTestCase.TestDirPath)

        try:            

           # Copy slip model list file into local directory for the run
           self.__slipModelList(os.path.join(DarfieldForecastsTests.__HainzlRefDir,
                                                   model_list_file))
           
           model.create(DarfieldForecastsTests.__testDate, 
                        CSEPTestCase.TestDirPath)
   
           
           # Reference file with test results to compare against
           reference_file = "csepETAS1.dat"
           
           shutil.copyfile(os.path.join(DarfieldForecastsTests.__HainzlRefDir, 
                                        reference_file),
                           os.path.join(CSEPTestCase.TestDirPath, 
                                        reference_file))
           
   
           error_msg = "ETAS1_HAINZL one-day forecast generation for Darfield \
testing region failed."
           
           self.failIf(CSEPFile.compare(reference_file, 
                                        model.filename()) is False,
                       error_msg)
           
           # Check for existence of metadata file for the forecast
           error_msg = "Metadata file for ETAS1_HAINZL one-day forecast for Darfield \
testing region is missing."
           self.failIf(os.path.exists(model.filename() + 
                                      CSEPPropertyFile.Metadata.Extension) is False, 
                       error_msg)

        finally:
            os.chdir(cwd)


    #---------------------------------------------------------------------------
    #
    # Invoke ETAS2_HAINZL one-day model for Darfield testing region, and verify 
    # the results.
    #
    def testETAS2HAINZLOneDayModel(self):
        """ Generate ETAS2_HAINZL one-day forecast for Darfield testing region and
verify results."""

        # Setup test name
        CSEPTestCase.setTestName(self, self.id())

        model_list_file = 'list_of_slip_models.dat'
        
        # Use the same test directory for input catalog and result forecast files
        args = [CSEPTestCase.TestDirPath, 
                '%s=ETAS,%s=2,%s=%s' %(HainzlModel.ModelOption,
                                       HainzlModel.MainshockProbMapsOption,
                                       HainzlModel.SlipModelFileListOption,
                                       os.path.join(CSEPTestCase.TestDirPath,
                                                    model_list_file))]
        
        # Instantiate forecast object
        model = ForecastFactory().object(HainzlOneDayModel.Type, 
                                         args)

        # Copy reference input catalog data file to the test directory
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__HainzlRefDir, 
                                     "examplecatalog.dat"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     model.inputCatalogFilename()))    
        
        # Change directory to the test location to guarantee that all model
        # configuration and scratch files are placed there.
        cwd = os.getcwd() 
        os.chdir(CSEPTestCase.TestDirPath)

        try:            

           # Copy slip model list file into local directory for the run
           self.__slipModelList(os.path.join(DarfieldForecastsTests.__HainzlRefDir,
                                                   model_list_file))
           
           model.create(DarfieldForecastsTests.__testDate, 
                        CSEPTestCase.TestDirPath)
   
           
           # Reference file with test results to compare against
           reference_file = "csepETAS2.dat"
           
           shutil.copyfile(os.path.join(DarfieldForecastsTests.__HainzlRefDir, 
                                        reference_file),
                           os.path.join(CSEPTestCase.TestDirPath, 
                                        reference_file))
           
   
           error_msg = "ETAS2_HAINZL one-day forecast generation for Darfield \
testing region failed."
           
           self.failIf(CSEPFile.compare(reference_file, 
                                        model.filename()) is False,
                       error_msg)
           
           # Check for existence of metadata file for the forecast
           error_msg = "Metadata file for ETAS2_HAINZL one-day forecast for Darfield \
testing region is missing."
           self.failIf(os.path.exists(model.filename() + 
                                      CSEPPropertyFile.Metadata.Extension) is False, 
                       error_msg)

        finally:
            os.chdir(cwd)


    #---------------------------------------------------------------------------
    #
    # Invoke RETAS0_HAINZL one-day model for Darfield testing region, and verify 
    # the results.
    #
    def testRETAS0HAINZLOneDayModel(self):
        """ Generate RETAS0_HAINZL one-day forecast for Darfield testing region and
verify results."""

        # Setup test name
        CSEPTestCase.setTestName(self, self.id())

        model_list_file = 'list_of_slip_models.dat'
        
        # Use the same test directory for input catalog and result forecast files
        args = [CSEPTestCase.TestDirPath, 
                '%s=RETAS,%s=0,%s=%s' %(HainzlModel.ModelOption,
                                       HainzlModel.MainshockProbMapsOption,
                                       HainzlModel.SlipModelFileListOption,
                                       os.path.join(CSEPTestCase.TestDirPath,
                                                    model_list_file))]
        
        # Instantiate forecast object
        model = ForecastFactory().object(HainzlOneDayModel.Type, 
                                         args)

        # Copy reference input catalog data file to the test directory
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__HainzlRefDir, 
                                     "examplecatalog.dat"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     model.inputCatalogFilename()))    
        
        # Change directory to the test location to guarantee that all model
        # configuration and scratch files are placed there.
        cwd = os.getcwd() 
        os.chdir(CSEPTestCase.TestDirPath)

        try:            

           # Copy slip model list file into local directory for the run
           self.__slipModelList(os.path.join(DarfieldForecastsTests.__HainzlRefDir,
                                                   model_list_file))
           
           model.create(DarfieldForecastsTests.__testDate, 
                        CSEPTestCase.TestDirPath)
   
           
           # Reference file with test results to compare against
           reference_file = "csepRETAS0.dat"
           
           shutil.copyfile(os.path.join(DarfieldForecastsTests.__HainzlRefDir, 
                                        reference_file),
                           os.path.join(CSEPTestCase.TestDirPath, 
                                        reference_file))
           
   
           error_msg = "RETAS0_HAINZL one-day forecast generation for Darfield \
testing region failed."
           
           self.failIf(CSEPFile.compare(reference_file, 
                                        model.filename()) is False,
                       error_msg)
           
           # Check for existence of metadata file for the forecast
           error_msg = "Metadata file for RETAS0_HAINZL one-day forecast for Darfield \
testing region is missing."
           self.failIf(os.path.exists(model.filename() + 
                                      CSEPPropertyFile.Metadata.Extension) is False, 
                       error_msg)

        finally:
            os.chdir(cwd)


    #---------------------------------------------------------------------------
    #
    # Invoke RETAS1_HAINZL one-day model for Darfield testing region, and verify 
    # the results.
    #
    def testRETAS1HAINZLOneDayModel(self):
        """ Generate RETAS1_HAINZL one-day forecast for Darfield testing region and
verify results."""

        # Setup test name
        CSEPTestCase.setTestName(self, self.id())

        model_list_file = 'list_of_slip_models.dat'
        
        # Use the same test directory for input catalog and result forecast files
        args = [CSEPTestCase.TestDirPath, 
                '%s=RETAS,%s=1,%s=%s' %(HainzlModel.ModelOption,
                                       HainzlModel.MainshockProbMapsOption,
                                       HainzlModel.SlipModelFileListOption,
                                       os.path.join(CSEPTestCase.TestDirPath,
                                                    model_list_file))]
        
        # Instantiate forecast object
        model = ForecastFactory().object(HainzlOneDayModel.Type, 
                                         args)

        # Copy reference input catalog data file to the test directory
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__HainzlRefDir, 
                                     "examplecatalog.dat"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     model.inputCatalogFilename()))    
        
        # Change directory to the test location to guarantee that all model
        # configuration and scratch files are placed there.
        cwd = os.getcwd() 
        os.chdir(CSEPTestCase.TestDirPath)

        try:            

           # Copy slip model list file into local directory for the run
           self.__slipModelList(os.path.join(DarfieldForecastsTests.__HainzlRefDir,
                                                   model_list_file))
           
           model.create(DarfieldForecastsTests.__testDate, 
                        CSEPTestCase.TestDirPath)
   
           
           # Reference file with test results to compare against
           reference_file = "csepRETAS1.dat"
           
           shutil.copyfile(os.path.join(DarfieldForecastsTests.__HainzlRefDir, 
                                        reference_file),
                           os.path.join(CSEPTestCase.TestDirPath, 
                                        reference_file))
           
   
           error_msg = "RETAS1_HAINZL one-day forecast generation for Darfield \
testing region failed."
           
           self.failIf(CSEPFile.compare(reference_file, 
                                        model.filename()) is False,
                       error_msg)
           
           # Check for existence of metadata file for the forecast
           error_msg = "Metadata file for RETAS1_HAINZL one-day forecast for Darfield \
testing region is missing."
           self.failIf(os.path.exists(model.filename() + 
                                      CSEPPropertyFile.Metadata.Extension) is False, 
                       error_msg)

        finally:
            os.chdir(cwd)


    #---------------------------------------------------------------------------
    #
    # Invoke RETAS2_HAINZL one-day model for Darfield testing region, and verify 
    # the results.
    #
    def testRETAS2HAINZLOneDayModel(self):
        """ Generate RETAS2_HAINZL one-day forecast for Darfield testing region and
verify results."""

        # Setup test name
        CSEPTestCase.setTestName(self, self.id())

        model_list_file = 'list_of_slip_models.dat'
        
        # Use the same test directory for input catalog and result forecast files
        args = [CSEPTestCase.TestDirPath, 
                '%s=RETAS,%s=2,%s=%s' %(HainzlModel.ModelOption,
                                       HainzlModel.MainshockProbMapsOption,
                                       HainzlModel.SlipModelFileListOption,
                                       os.path.join(CSEPTestCase.TestDirPath,
                                                    model_list_file))]
        
        # Instantiate forecast object
        model = ForecastFactory().object(HainzlOneDayModel.Type, 
                                         args)

        # Copy reference input catalog data file to the test directory
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__HainzlRefDir, 
                                     "examplecatalog.dat"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     model.inputCatalogFilename()))    
        
        # Change directory to the test location to guarantee that all model
        # configuration and scratch files are placed there.
        cwd = os.getcwd() 
        os.chdir(CSEPTestCase.TestDirPath)

        try:            

           # Copy slip model list file into local directory for the run
           self.__slipModelList(os.path.join(DarfieldForecastsTests.__HainzlRefDir,
                                                   model_list_file))
           
           model.create(DarfieldForecastsTests.__testDate, 
                        CSEPTestCase.TestDirPath)
   
           
           # Reference file with test results to compare against
           reference_file = "csepRETAS2.dat"
           
           shutil.copyfile(os.path.join(DarfieldForecastsTests.__HainzlRefDir, 
                                        reference_file),
                           os.path.join(CSEPTestCase.TestDirPath, 
                                        reference_file))
           
   
           error_msg = "RETAS2_HAINZL one-day forecast generation for Darfield \
testing region failed."
           
           self.failIf(CSEPFile.compare(reference_file, 
                                        model.filename()) is False,
                       error_msg)
           
           # Check for existence of metadata file for the forecast
           error_msg = "Metadata file for RETAS2_HAINZL one-day forecast for Darfield \
testing region is missing."
           self.failIf(os.path.exists(model.filename() + 
                                      CSEPPropertyFile.Metadata.Extension) is False, 
                       error_msg)

        finally:
            os.chdir(cwd)


    #---------------------------------------------------------------------------
    #
    # Invoke ETAS0_HAINZL one-month model for Darfield testing region, and verify 
    # the results.
    #
    def testETAS0HAINZLOneMonthModel(self):
        """ Generate ETAS0_HAINZL one-month forecast for Darfield testing region and
verify results."""

        # Setup test name
        CSEPTestCase.setTestName(self, self.id())

        model_list_file = 'list_of_slip_models.dat'
        
        # Use the same test directory for input catalog and result forecast files
        args = [CSEPTestCase.TestDirPath, 
                '%s=ETAS,%s=0,%s=%s' %(HainzlModel.ModelOption,
                                       HainzlModel.MainshockProbMapsOption,
                                       HainzlModel.SlipModelFileListOption,
                                       os.path.join(CSEPTestCase.TestDirPath,
                                                    model_list_file))]
        
        # Instantiate forecast object
        model = ForecastFactory().object(HainzlOneMonthModel.Type, 
                                         args)

        # Copy reference input catalog data file to the test directory
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__HainzlRefDir, 
                                     "examplecatalog.dat"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     model.inputCatalogFilename()))    
        
        # Change directory to the test location to guarantee that all model
        # configuration and scratch files are placed there.
        cwd = os.getcwd() 
        os.chdir(CSEPTestCase.TestDirPath)

        try:            

           # Copy slip model list file into local directory for the run
           self.__slipModelList(os.path.join(DarfieldForecastsTests.__HainzlRefDir,
                                                   model_list_file))
           
           model.create(DarfieldForecastsTests.__testDate, 
                        CSEPTestCase.TestDirPath)
   
           
           # Reference file with test results to compare against
           reference_file = "csepETAS0_OneMonth.dat"
           
           shutil.copyfile(os.path.join(DarfieldForecastsTests.__HainzlRefDir, 
                                        reference_file),
                           os.path.join(CSEPTestCase.TestDirPath, 
                                        reference_file))
           
   
           error_msg = "ETAS0_HAINZL one-month forecast generation for Darfield \
testing region failed."
           
           self.failIf(CSEPFile.compare(reference_file, 
                                        model.filename()) is False,
                       error_msg)
           
           # Check for existence of metadata file for the forecast
           error_msg = "Metadata file for ETAS0_HAINZL one-month forecast for Darfield \
testing region is missing."
           self.failIf(os.path.exists(model.filename() + 
                                      CSEPPropertyFile.Metadata.Extension) is False, 
                       error_msg)

        finally:
            os.chdir(cwd)


    #---------------------------------------------------------------------------
    #
    # Invoke ETAS1_HAINZL one-month model for Darfield testing region, and verify 
    # the results.
    #
    def testETAS1HAINZLOneMonthModel(self):
        """ Generate ETAS1_HAINZL one-month forecast for Darfield testing region and
verify results."""

        # Setup test name
        CSEPTestCase.setTestName(self, self.id())

        model_list_file = 'list_of_slip_models.dat'
        
        # Use the same test directory for input catalog and result forecast files
        args = [CSEPTestCase.TestDirPath, 
                '%s=ETAS,%s=1,%s=%s' %(HainzlModel.ModelOption,
                                       HainzlModel.MainshockProbMapsOption,
                                       HainzlModel.SlipModelFileListOption,
                                       os.path.join(CSEPTestCase.TestDirPath,
                                                    model_list_file))]
        
        # Instantiate forecast object
        model = ForecastFactory().object(HainzlOneMonthModel.Type, 
                                         args)

        # Copy reference input catalog data file to the test directory
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__HainzlRefDir, 
                                     "examplecatalog.dat"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     model.inputCatalogFilename()))    
        
        # Change directory to the test location to guarantee that all model
        # configuration and scratch files are placed there.
        cwd = os.getcwd() 
        os.chdir(CSEPTestCase.TestDirPath)

        try:            

           # Copy slip model list file into local directory for the run
           self.__slipModelList(os.path.join(DarfieldForecastsTests.__HainzlRefDir,
                                                   model_list_file))
           
           model.create(DarfieldForecastsTests.__testDate, 
                        CSEPTestCase.TestDirPath)
   
           
           # Reference file with test results to compare against
           reference_file = "csepETAS1_OneMonth.dat"
           
           shutil.copyfile(os.path.join(DarfieldForecastsTests.__HainzlRefDir, 
                                        reference_file),
                           os.path.join(CSEPTestCase.TestDirPath, 
                                        reference_file))
           
   
           error_msg = "ETAS1_HAINZL one-month forecast generation for Darfield \
testing region failed."
           
           self.failIf(CSEPFile.compare(reference_file, 
                                        model.filename()) is False,
                       error_msg)
           
           # Check for existence of metadata file for the forecast
           error_msg = "Metadata file for ETAS1_HAINZL one-month forecast for Darfield \
testing region is missing."
           self.failIf(os.path.exists(model.filename() + 
                                      CSEPPropertyFile.Metadata.Extension) is False, 
                       error_msg)

        finally:
            os.chdir(cwd)


    #---------------------------------------------------------------------------
    #
    # Invoke ETAS2_HAINZL one-month model for Darfield testing region, and verify 
    # the results.
    #
    def testETAS2HAINZLOneMonthModel(self):
        """ Generate ETAS2_HAINZL one-month forecast for Darfield testing region and
verify results."""

        # Setup test name
        CSEPTestCase.setTestName(self, self.id())

        model_list_file = 'list_of_slip_models.dat'
        
        # Use the same test directory for input catalog and result forecast files
        args = [CSEPTestCase.TestDirPath, 
                '%s=ETAS,%s=2,%s=%s' %(HainzlModel.ModelOption,
                                       HainzlModel.MainshockProbMapsOption,
                                       HainzlModel.SlipModelFileListOption,
                                       os.path.join(CSEPTestCase.TestDirPath,
                                                    model_list_file))]
        
        # Instantiate forecast object
        model = ForecastFactory().object(HainzlOneMonthModel.Type, 
                                         args)

        # Copy reference input catalog data file to the test directory
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__HainzlRefDir, 
                                     "examplecatalog.dat"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     model.inputCatalogFilename()))    
        
        # Change directory to the test location to guarantee that all model
        # configuration and scratch files are placed there.
        cwd = os.getcwd() 
        os.chdir(CSEPTestCase.TestDirPath)

        try:            

           # Copy slip model list file into local directory for the run
           self.__slipModelList(os.path.join(DarfieldForecastsTests.__HainzlRefDir,
                                                   model_list_file))
           
           model.create(DarfieldForecastsTests.__testDate, 
                        CSEPTestCase.TestDirPath)
   
           
           # Reference file with test results to compare against
           reference_file = "csepETAS2_OneMonth.dat"
           
           shutil.copyfile(os.path.join(DarfieldForecastsTests.__HainzlRefDir, 
                                        reference_file),
                           os.path.join(CSEPTestCase.TestDirPath, 
                                        reference_file))
           
   
           error_msg = "ETAS2_HAINZL one-month forecast generation for Darfield \
testing region failed."
           
           self.failIf(CSEPFile.compare(reference_file, 
                                        model.filename()) is False,
                       error_msg)
           
           # Check for existence of metadata file for the forecast
           error_msg = "Metadata file for ETAS2_HAINZL one-month forecast for Darfield \
testing region is missing."
           self.failIf(os.path.exists(model.filename() + 
                                      CSEPPropertyFile.Metadata.Extension) is False, 
                       error_msg)

        finally:
            os.chdir(cwd)


    #---------------------------------------------------------------------------
    #
    # Invoke RETAS0_HAINZL one-month model for Darfield testing region, and verify 
    # the results.
    #
    def testRETAS0HAINZLOneMonthModel(self):
        """ Generate RETAS0_HAINZL one-month forecast for Darfield testing region and
verify results."""

        # Setup test name
        CSEPTestCase.setTestName(self, self.id())

        model_list_file = 'list_of_slip_models.dat'
        
        # Use the same test directory for input catalog and result forecast files
        args = [CSEPTestCase.TestDirPath, 
                '%s=RETAS,%s=0,%s=%s' %(HainzlModel.ModelOption,
                                       HainzlModel.MainshockProbMapsOption,
                                       HainzlModel.SlipModelFileListOption,
                                       os.path.join(CSEPTestCase.TestDirPath,
                                                    model_list_file))]
        
        # Instantiate forecast object
        model = ForecastFactory().object(HainzlOneMonthModel.Type, 
                                         args)

        # Copy reference input catalog data file to the test directory
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__HainzlRefDir, 
                                     "examplecatalog.dat"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     model.inputCatalogFilename()))    
        
        # Change directory to the test location to guarantee that all model
        # configuration and scratch files are placed there.
        cwd = os.getcwd() 
        os.chdir(CSEPTestCase.TestDirPath)

        try:            

           # Copy slip model list file into local directory for the run
           self.__slipModelList(os.path.join(DarfieldForecastsTests.__HainzlRefDir,
                                                   model_list_file))
           
           model.create(DarfieldForecastsTests.__testDate, 
                        CSEPTestCase.TestDirPath)
   
           
           # Reference file with test results to compare against
           reference_file = "csepRETAS0_OneMonth.dat"
           
           shutil.copyfile(os.path.join(DarfieldForecastsTests.__HainzlRefDir, 
                                        reference_file),
                           os.path.join(CSEPTestCase.TestDirPath, 
                                        reference_file))
           
   
           error_msg = "RETAS0_HAINZL one-month forecast generation for Darfield \
testing region failed."
           
           self.failIf(CSEPFile.compare(reference_file, 
                                        model.filename()) is False,
                       error_msg)
           
           # Check for existence of metadata file for the forecast
           error_msg = "Metadata file for RETAS0_HAINZL one-month forecast for Darfield \
testing region is missing."
           self.failIf(os.path.exists(model.filename() + 
                                      CSEPPropertyFile.Metadata.Extension) is False, 
                       error_msg)

        finally:
            os.chdir(cwd)


    #---------------------------------------------------------------------------
    #
    # Invoke RETAS1_HAINZL one-month model for Darfield testing region, and verify 
    # the results.
    #
    def testRETAS1HAINZLOneMonthModel(self):
        """ Generate RETAS1_HAINZL one-month forecast for Darfield testing region and
verify results."""

        # Setup test name
        CSEPTestCase.setTestName(self, self.id())

        model_list_file = 'list_of_slip_models.dat'
        
        # Use the same test directory for input catalog and result forecast files
        args = [CSEPTestCase.TestDirPath, 
                '%s=RETAS,%s=1,%s=%s' %(HainzlModel.ModelOption,
                                       HainzlModel.MainshockProbMapsOption,
                                       HainzlModel.SlipModelFileListOption,
                                       os.path.join(CSEPTestCase.TestDirPath,
                                                    model_list_file))]
        
        # Instantiate forecast object
        model = ForecastFactory().object(HainzlOneMonthModel.Type, 
                                         args)

        # Copy reference input catalog data file to the test directory
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__HainzlRefDir, 
                                     "examplecatalog.dat"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     model.inputCatalogFilename()))    
        
        # Change directory to the test location to guarantee that all model
        # configuration and scratch files are placed there.
        cwd = os.getcwd() 
        os.chdir(CSEPTestCase.TestDirPath)

        try:            

           # Copy slip model list file into local directory for the run
           self.__slipModelList(os.path.join(DarfieldForecastsTests.__HainzlRefDir,
                                                   model_list_file))
           
           model.create(DarfieldForecastsTests.__testDate, 
                        CSEPTestCase.TestDirPath)
   
           
           # Reference file with test results to compare against
           reference_file = "csepRETAS1_OneMonth.dat"
           
           shutil.copyfile(os.path.join(DarfieldForecastsTests.__HainzlRefDir, 
                                        reference_file),
                           os.path.join(CSEPTestCase.TestDirPath, 
                                        reference_file))
           
   
           error_msg = "RETAS1_HAINZL one-month forecast generation for Darfield \
testing region failed."
           
           self.failIf(CSEPFile.compare(reference_file, 
                                        model.filename()) is False,
                       error_msg)
           
           # Check for existence of metadata file for the forecast
           error_msg = "Metadata file for RETAS1_HAINZL one-month forecast for Darfield \
testing region is missing."
           self.failIf(os.path.exists(model.filename() + 
                                      CSEPPropertyFile.Metadata.Extension) is False, 
                       error_msg)

        finally:
            os.chdir(cwd)


    #---------------------------------------------------------------------------
    #
    # Invoke RETAS2_HAINZL one-month model for Darfield testing region, and verify 
    # the results.
    #
    def testRETAS2HAINZLOneMonthModel(self):
        """ Generate ERTAS2_HAINZL one-month forecast for Darfield testing region and
verify results."""

        # Setup test name
        CSEPTestCase.setTestName(self, self.id())

        model_list_file = 'list_of_slip_models.dat'
        
        # Use the same test directory for input catalog and result forecast files
        args = [CSEPTestCase.TestDirPath, 
                '%s=RETAS,%s=2,%s=%s' %(HainzlModel.ModelOption,
                                       HainzlModel.MainshockProbMapsOption,
                                       HainzlModel.SlipModelFileListOption,
                                       os.path.join(CSEPTestCase.TestDirPath,
                                                    model_list_file))]
        
        # Instantiate forecast object
        model = ForecastFactory().object(HainzlOneMonthModel.Type, 
                                         args)

        # Copy reference input catalog data file to the test directory
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__HainzlRefDir, 
                                     "examplecatalog.dat"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     model.inputCatalogFilename()))    
        
        # Change directory to the test location to guarantee that all model
        # configuration and scratch files are placed there.
        cwd = os.getcwd() 
        os.chdir(CSEPTestCase.TestDirPath)

        try:            

           # Copy slip model list file into local directory for the run
           self.__slipModelList(os.path.join(DarfieldForecastsTests.__HainzlRefDir,
                                                   model_list_file))
           
           model.create(DarfieldForecastsTests.__testDate, 
                        CSEPTestCase.TestDirPath)
   
           
           # Reference file with test results to compare against
           reference_file = "csepRETAS2_OneMonth.dat"
           
           shutil.copyfile(os.path.join(DarfieldForecastsTests.__HainzlRefDir, 
                                        reference_file),
                           os.path.join(CSEPTestCase.TestDirPath, 
                                        reference_file))
           
   
           error_msg = "RETAS2_HAINZL one-month forecast generation for Darfield \
testing region failed."
           
           self.failIf(CSEPFile.compare(reference_file, 
                                        model.filename()) is False,
                       error_msg)
           
           # Check for existence of metadata file for the forecast
           error_msg = "Metadata file for RETAS2_HAINZL one-month forecast for Darfield \
testing region is missing."
           self.failIf(os.path.exists(model.filename() + 
                                      CSEPPropertyFile.Metadata.Extension) is False, 
                       error_msg)

        finally:
            os.chdir(cwd)


    #---------------------------------------------------------------------------
    #
    # Invoke ETAS0_HAINZL one-year model for Darfield testing region, and verify 
    # the results.
    #
    def testETAS0HAINZLOneYearModel(self):
        """ Generate ETAS0_HAINZL one-year forecast for Darfield testing region and
verify results."""

        # Setup test name
        CSEPTestCase.setTestName(self, self.id())

        model_list_file = 'list_of_slip_models.dat'
        
        # Use the same test directory for input catalog and result forecast files
        args = [CSEPTestCase.TestDirPath, 
                '%s=ETAS,%s=0,%s=%s' %(HainzlModel.ModelOption,
                                       HainzlModel.MainshockProbMapsOption,
                                       HainzlModel.SlipModelFileListOption,
                                       os.path.join(CSEPTestCase.TestDirPath,
                                                    model_list_file))]
        
        # Instantiate forecast object
        model = ForecastFactory().object(HainzlOneYearModel.Type, 
                                         args)

        # Copy reference input catalog data file to the test directory
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__HainzlRefDir, 
                                     "examplecatalog.dat"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     model.inputCatalogFilename()))    
        
        # Change directory to the test location to guarantee that all model
        # configuration and scratch files are placed there.
        cwd = os.getcwd() 
        os.chdir(CSEPTestCase.TestDirPath)

        try:            

           # Copy slip model list file into local directory for the run
           self.__slipModelList(os.path.join(DarfieldForecastsTests.__HainzlRefDir,
                                                   model_list_file))
           
           model.create(DarfieldForecastsTests.__testDate, 
                        CSEPTestCase.TestDirPath)
   
           
           # Reference file with test results to compare against
           reference_file = "csepETAS0_OneYear.dat"
           
           shutil.copyfile(os.path.join(DarfieldForecastsTests.__HainzlRefDir, 
                                        reference_file),
                           os.path.join(CSEPTestCase.TestDirPath, 
                                        reference_file))
           
   
           error_msg = "ETAS0_HAINZL one-year forecast generation for Darfield \
testing region failed."
           
           self.failIf(CSEPFile.compare(reference_file, 
                                        model.filename()) is False,
                       error_msg)
           
           # Check for existence of metadata file for the forecast
           error_msg = "Metadata file for ETAS0_HAINZL one-year forecast for Darfield \
testing region is missing."
           self.failIf(os.path.exists(model.filename() + 
                                      CSEPPropertyFile.Metadata.Extension) is False, 
                       error_msg)

        finally:
            os.chdir(cwd)


    #---------------------------------------------------------------------------
    #
    # Invoke ETAS1_HAINZL one-year model for Darfield testing region, and verify 
    # the results.
    #
    def testETAS1HAINZLOneYearModel(self):
        """ Generate ETAS1_HAINZL one-year forecast for Darfield testing region and
verify results."""

        # Setup test name
        CSEPTestCase.setTestName(self, self.id())

        model_list_file = 'list_of_slip_models.dat'
        
        # Use the same test directory for input catalog and result forecast files
        args = [CSEPTestCase.TestDirPath, 
                '%s=ETAS,%s=1,%s=%s' %(HainzlModel.ModelOption,
                                       HainzlModel.MainshockProbMapsOption,
                                       HainzlModel.SlipModelFileListOption,
                                       os.path.join(CSEPTestCase.TestDirPath,
                                                    model_list_file))]
        
        # Instantiate forecast object
        model = ForecastFactory().object(HainzlOneYearModel.Type, 
                                         args)

        # Copy reference input catalog data file to the test directory
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__HainzlRefDir, 
                                     "examplecatalog.dat"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     model.inputCatalogFilename()))    
        
        # Change directory to the test location to guarantee that all model
        # configuration and scratch files are placed there.
        cwd = os.getcwd() 
        os.chdir(CSEPTestCase.TestDirPath)

        try:            

           # Copy slip model list file into local directory for the run
           self.__slipModelList(os.path.join(DarfieldForecastsTests.__HainzlRefDir,
                                                   model_list_file))
           
           model.create(DarfieldForecastsTests.__testDate, 
                        CSEPTestCase.TestDirPath)
   
           
           # Reference file with test results to compare against
           reference_file = "csepETAS1_OneYear.dat"
           
           shutil.copyfile(os.path.join(DarfieldForecastsTests.__HainzlRefDir, 
                                        reference_file),
                           os.path.join(CSEPTestCase.TestDirPath, 
                                        reference_file))
           
   
           error_msg = "ETAS1_HAINZL one-year forecast generation for Darfield \
testing region failed."
           
           self.failIf(CSEPFile.compare(reference_file, 
                                        model.filename()) is False,
                       error_msg)
           
           # Check for existence of metadata file for the forecast
           error_msg = "Metadata file for ETAS1_HAINZL one-year forecast for Darfield \
testing region is missing."
           self.failIf(os.path.exists(model.filename() + 
                                      CSEPPropertyFile.Metadata.Extension) is False, 
                       error_msg)

        finally:
            os.chdir(cwd)


    #---------------------------------------------------------------------------
    #
    # Invoke ETAS2_HAINZL one-year model for Darfield testing region, and verify 
    # the results.
    #
    def testETAS2HAINZLOneYearModel(self):
        """ Generate ETAS2_HAINZL one-year forecast for Darfield testing region and
verify results."""

        # Setup test name
        CSEPTestCase.setTestName(self, self.id())

        model_list_file = 'list_of_slip_models.dat'
        
        # Use the same test directory for input catalog and result forecast files
        args = [CSEPTestCase.TestDirPath, 
                '%s=ETAS,%s=2,%s=%s' %(HainzlModel.ModelOption,
                                       HainzlModel.MainshockProbMapsOption,
                                       HainzlModel.SlipModelFileListOption,
                                       os.path.join(CSEPTestCase.TestDirPath,
                                                    model_list_file))]
        
        # Instantiate forecast object
        model = ForecastFactory().object(HainzlOneYearModel.Type, 
                                         args)

        # Copy reference input catalog data file to the test directory
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__HainzlRefDir, 
                                     "examplecatalog.dat"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     model.inputCatalogFilename()))    
        
        # Change directory to the test location to guarantee that all model
        # configuration and scratch files are placed there.
        cwd = os.getcwd() 
        os.chdir(CSEPTestCase.TestDirPath)

        try:            

           # Copy slip model list file into local directory for the run
           self.__slipModelList(os.path.join(DarfieldForecastsTests.__HainzlRefDir,
                                                   model_list_file))
           
           model.create(DarfieldForecastsTests.__testDate, 
                        CSEPTestCase.TestDirPath)
   
           
           # Reference file with test results to compare against
           reference_file = "csepETAS2_OneYear.dat"
           
           shutil.copyfile(os.path.join(DarfieldForecastsTests.__HainzlRefDir, 
                                        reference_file),
                           os.path.join(CSEPTestCase.TestDirPath, 
                                        reference_file))
           
   
           error_msg = "ETAS2_HAINZL one-year forecast generation for Darfield \
testing region failed."
           
           self.failIf(CSEPFile.compare(reference_file, 
                                        model.filename()) is False,
                       error_msg)
           
           # Check for existence of metadata file for the forecast
           error_msg = "Metadata file for ETAS2_HAINZL one-year forecast for Darfield \
testing region is missing."
           self.failIf(os.path.exists(model.filename() + 
                                      CSEPPropertyFile.Metadata.Extension) is False, 
                       error_msg)

        finally:
            os.chdir(cwd)


    #---------------------------------------------------------------------------
    #
    # Invoke RETAS0_HAINZL one-year model for Darfield testing region, and verify 
    # the results.
    #
    def testRETAS0HAINZLOneYearModel(self):
        """ Generate RETAS0_HAINZL one-year forecast for Darfield testing region and
verify results."""

        # Setup test name
        CSEPTestCase.setTestName(self, self.id())

        model_list_file = 'list_of_slip_models.dat'
        
        # Use the same test directory for input catalog and result forecast files
        args = [CSEPTestCase.TestDirPath, 
                '%s=RETAS,%s=0,%s=%s' %(HainzlModel.ModelOption,
                                       HainzlModel.MainshockProbMapsOption,
                                       HainzlModel.SlipModelFileListOption,
                                       os.path.join(CSEPTestCase.TestDirPath,
                                                    model_list_file))]
        
        # Instantiate forecast object
        model = ForecastFactory().object(HainzlOneYearModel.Type, 
                                         args)

        # Copy reference input catalog data file to the test directory
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__HainzlRefDir, 
                                     "examplecatalog.dat"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     model.inputCatalogFilename()))    
        
        # Change directory to the test location to guarantee that all model
        # configuration and scratch files are placed there.
        cwd = os.getcwd() 
        os.chdir(CSEPTestCase.TestDirPath)

        try:            

           # Copy slip model list file into local directory for the run
           self.__slipModelList(os.path.join(DarfieldForecastsTests.__HainzlRefDir,
                                                   model_list_file))
           
           model.create(DarfieldForecastsTests.__testDate, 
                        CSEPTestCase.TestDirPath)
   
           
           # Reference file with test results to compare against
           reference_file = "csepRETAS0_OneYear.dat"
           
           shutil.copyfile(os.path.join(DarfieldForecastsTests.__HainzlRefDir, 
                                        reference_file),
                           os.path.join(CSEPTestCase.TestDirPath, 
                                        reference_file))
           
   
           error_msg = "RETAS0_HAINZL one-year forecast generation for Darfield \
testing region failed."
           
           self.failIf(CSEPFile.compare(reference_file, 
                                        model.filename()) is False,
                       error_msg)
           
           # Check for existence of metadata file for the forecast
           error_msg = "Metadata file for RETAS0_HAINZL one-year forecast for Darfield \
testing region is missing."
           self.failIf(os.path.exists(model.filename() + 
                                      CSEPPropertyFile.Metadata.Extension) is False, 
                       error_msg)

        finally:
            os.chdir(cwd)


    #---------------------------------------------------------------------------
    #
    # Invoke RETAS1_HAINZL one-year model for Darfield testing region, and verify 
    # the results.
    #
    def testRETAS1HAINZLOneYearModel(self):
        """ Generate RETAS1_HAINZL one-year forecast for Darfield testing region and
verify results."""

        # Setup test name
        CSEPTestCase.setTestName(self, self.id())

        model_list_file = 'list_of_slip_models.dat'
        
        # Use the same test directory for input catalog and result forecast files
        args = [CSEPTestCase.TestDirPath, 
                '%s=RETAS,%s=1,%s=%s' %(HainzlModel.ModelOption,
                                       HainzlModel.MainshockProbMapsOption,
                                       HainzlModel.SlipModelFileListOption,
                                       os.path.join(CSEPTestCase.TestDirPath,
                                                    model_list_file))]
        
        # Instantiate forecast object
        model = ForecastFactory().object(HainzlOneYearModel.Type, 
                                         args)

        # Copy reference input catalog data file to the test directory
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__HainzlRefDir, 
                                     "examplecatalog.dat"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     model.inputCatalogFilename()))    
        
        # Change directory to the test location to guarantee that all model
        # configuration and scratch files are placed there.
        cwd = os.getcwd() 
        os.chdir(CSEPTestCase.TestDirPath)

        try:            

           # Copy slip model list file into local directory for the run
           self.__slipModelList(os.path.join(DarfieldForecastsTests.__HainzlRefDir,
                                                   model_list_file))
           
           model.create(DarfieldForecastsTests.__testDate, 
                        CSEPTestCase.TestDirPath)
   
           
           # Reference file with test results to compare against
           reference_file = "csepRETAS1_OneYear.dat"
           
           shutil.copyfile(os.path.join(DarfieldForecastsTests.__HainzlRefDir, 
                                        reference_file),
                           os.path.join(CSEPTestCase.TestDirPath, 
                                        reference_file))
           
   
           error_msg = "RETAS1_HAINZL one-year forecast generation for Darfield \
testing region failed."
           
           self.failIf(CSEPFile.compare(reference_file, 
                                        model.filename()) is False,
                       error_msg)
           
           # Check for existence of metadata file for the forecast
           error_msg = "Metadata file for RETAS1_HAINZL one-year forecast for Darfield \
testing region is missing."
           self.failIf(os.path.exists(model.filename() + 
                                      CSEPPropertyFile.Metadata.Extension) is False, 
                       error_msg)

        finally:
            os.chdir(cwd)


    #---------------------------------------------------------------------------
    #
    # Invoke RETAS2_HAINZL one-year model for Darfield testing region, and verify 
    # the results.
    #
    def testRETAS2HAINZLOneYearModel(self):
        """ Generate ERTAS2_HAINZL one-year forecast for Darfield testing region and
verify results."""

        # Setup test name
        CSEPTestCase.setTestName(self, self.id())

        model_list_file = 'list_of_slip_models.dat'
        
        # Use the same test directory for input catalog and result forecast files
        args = [CSEPTestCase.TestDirPath, 
                '%s=RETAS,%s=2,%s=%s' %(HainzlModel.ModelOption,
                                       HainzlModel.MainshockProbMapsOption,
                                       HainzlModel.SlipModelFileListOption,
                                       os.path.join(CSEPTestCase.TestDirPath,
                                                    model_list_file))]
        
        # Instantiate forecast object
        model = ForecastFactory().object(HainzlOneYearModel.Type, 
                                         args)

        # Copy reference input catalog data file to the test directory
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__HainzlRefDir, 
                                     "examplecatalog.dat"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     model.inputCatalogFilename()))    
        
        # Change directory to the test location to guarantee that all model
        # configuration and scratch files are placed there.
        cwd = os.getcwd() 
        os.chdir(CSEPTestCase.TestDirPath)

        try:            

           # Copy slip model list file into local directory for the run
           self.__slipModelList(os.path.join(DarfieldForecastsTests.__HainzlRefDir,
                                                   model_list_file))
           
           model.create(DarfieldForecastsTests.__testDate, 
                        CSEPTestCase.TestDirPath)
   
           
           # Reference file with test results to compare against
           reference_file = "csepRETAS2_OneYear.dat"
           
           shutil.copyfile(os.path.join(DarfieldForecastsTests.__HainzlRefDir, 
                                        reference_file),
                           os.path.join(CSEPTestCase.TestDirPath, 
                                        reference_file))
           
   
           error_msg = "RETAS2_HAINZL one-year forecast generation for Darfield \
testing region failed."
           
           self.failIf(CSEPFile.compare(reference_file, 
                                        model.filename()) is False,
                       error_msg)
           
           # Check for existence of metadata file for the forecast
           error_msg = "Metadata file for RETAS2_HAINZL one-year forecast for Darfield \
testing region is missing."
           self.failIf(os.path.exists(model.filename() + 
                                      CSEPPropertyFile.Metadata.Extension) is False, 
                       error_msg)

        finally:
            os.chdir(cwd)


    #---------------------------------------------------------------------------
    #
    # Invoke CATTANIAOneDayCRS0 one-day model for Darfield testing region, and verify 
    # the results.
    #
    def testCATTANIAOneDayCRS0Model(self):
        """ Generate CATTANIAOneDayCRS0 one-day forecast for Darfield testing region and
verify results."""

        
        # Setup test name
        CSEPTestCase.setTestName(self, self.id())

        model_list_file = 'exampleInputListSlipModels.dat'
        
        # Use the same test directory for input catalog and result forecast files
        args = [CSEPTestCase.TestDirPath, 
                '%s=0,%s=-37284630,%s=%s,%s=%s' %(CattaniaModel.ConfigOption,
                                                  CattaniaModel.RandomSeedOption,
                                                  CattaniaModel.SlipModelFileListOption,
                                                  os.path.join(CSEPTestCase.TestDirPath,
                                                               model_list_file),
                                                  CattaniaModel.ParamFilePathOption,
                                                  DarfieldForecastsTests.__CattaniaRefDir)]
        
        # Instantiate forecast object
        model = ForecastFactory().object(CattaniaOneDayModel.Type, 
                                         args)

        # Copy reference input catalog data file to the test directory
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__CattaniaRefDir, 
                                     "CatalogCanterbury_M2_1980_2012.txt"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     model.inputCatalogFilename()))    

        # Copy reference focal mechanism input catalog data file to the test directory
        start_date_focal_mechanism = "%s%s" %(DarfieldForecastsTests.__CattaniaTestDate.strftime("%Y_%m_%d_"),
                                              GeoNetNZDataSource.ProcessedFocalMechanismFile)
        
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__CattaniaRefDir, 
                                     "GeoNet_CMT_solutions.dat"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     start_date_focal_mechanism))    
        
        # Change directory to the test location to guarantee that all model
        # configuration and scratch files are placed there.
        cwd = os.getcwd() 
        os.chdir(CSEPTestCase.TestDirPath)

        try:            

           # Copy slip model list file into local directory for the run
           self.__slipModelList(os.path.join(DarfieldForecastsTests.__CattaniaRefDir,
                                                   model_list_file))
           
           model.create(DarfieldForecastsTests.__CattaniaTestDate, 
                        CSEPTestCase.TestDirPath)
   
           
           # Reference file with test results to compare against
           reference_file = "model0_1d_foremap.dat"
           
           shutil.copyfile(os.path.join(DarfieldForecastsTests.__CattaniaRefDir, 
                                        reference_file),
                           os.path.join(CSEPTestCase.TestDirPath, 
                                        reference_file))
   
           error_msg = "CattaniaCRS0 one-day forecast generation for Darfield \
testing region failed."
           
           self.failIf(CSEPFile.compare(reference_file, 
                                        model.filename()) is False,
                       error_msg)
           
           # Check for existence of metadata file for the forecast
           error_msg = "Metadata file for CattaniaCRS0 one-day forecast for Darfield \
testing region is missing."
           self.failIf(os.path.exists(model.filename() + 
                                      CSEPPropertyFile.Metadata.Extension) is False, 
                       error_msg)

        finally:
            os.chdir(cwd)


    #---------------------------------------------------------------------------
    #
    # Invoke CATTANIAOneDayCRS0 one-day model for Darfield testing region, and verify 
    # the results.
    #
    def testCATTANIAOneDayCRS0ModelWithSlipModelsModule(self):
        """ Generate CATTANIAOneDayCRS0 one-day forecast for Darfield testing region and
verify results."""

        
        # Setup test name
        CSEPTestCase.setTestName(self, self.id())
        
        os.environ[SlipModels.SLIP_MODELS_ENV] = os.path.join(DarfieldForecastsTests.__refDir,
                                                              'slipModels')
        os.environ[SlipModels.SLIP_MODELS_LAG_ENV] = '0'
        
        # Use the same test directory for input catalog and result forecast files
        args = [CSEPTestCase.TestDirPath, 
                '%s=0,%s=-37284630,%s=%s' %(CattaniaModel.ConfigOption,
                                                  CattaniaModel.RandomSeedOption,
                                                  CattaniaModel.ParamFilePathOption,
                                                  DarfieldForecastsTests.__CattaniaRefDir)]
        
        # Instantiate forecast object
        model = ForecastFactory().object(CattaniaOneDayModel.Type, 
                                         args)

        # Copy reference input catalog data file to the test directory
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__CattaniaRefDir, 
                                     "CatalogCanterbury_M2_1980_2012.txt"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     model.inputCatalogFilename()))    

        # Copy reference focal mechanism input catalog data file to the test directory
        start_date_focal_mechanism = "%s%s" %(DarfieldForecastsTests.__CattaniaTestDate.strftime("%Y_%m_%d_"),
                                              GeoNetNZDataSource.ProcessedFocalMechanismFile)
        
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__CattaniaRefDir, 
                                     "GeoNet_CMT_solutions.dat"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     start_date_focal_mechanism))    
        
        # Change directory to the test location to guarantee that all model
        # configuration and scratch files are placed there.
        cwd = os.getcwd() 
        os.chdir(CSEPTestCase.TestDirPath)

        try:            

           model.create(DarfieldForecastsTests.__CattaniaTestDate, 
                        CSEPTestCase.TestDirPath)
   
           
           # Reference file with test results to compare against
           reference_file = "model0_1d_foremap.dat"
           
           shutil.copyfile(os.path.join(DarfieldForecastsTests.__CattaniaRefDir, 
                                        reference_file),
                           os.path.join(CSEPTestCase.TestDirPath, 
                                        reference_file))
   
           # Only check for existence of forecast file (other than forecast model
           # provided slip models were used to test SlipModule functionality
           # in this test)
           error_msg = "CattaniaCRS0 one-day forecast generation with SlipModule \
for Darfield testing region failed."
           
           self.failIf(os.path.exists(model.filename()) is False,
                       error_msg)
           
           # Check for existence of metadata file for the forecast
           error_msg = "Metadata file for CattaniaCRS0 one-day forecast for Darfield \
testing region is missing."
           self.failIf(os.path.exists(model.filename() + 
                                      CSEPPropertyFile.Metadata.Extension) is False, 
                       error_msg)

        finally:
            os.chdir(cwd)
            del os.environ[SlipModels.SLIP_MODELS_ENV]


    #---------------------------------------------------------------------------
    #
    # Invoke CATTANIAOneDayCRS0 one-day model for Darfield testing region, and verify 
    # the results.
    #
    def testCATTANIAOneDayCRS0ModelWithHash(self):
        """ Generate CATTANIAOneDayCRS0 one-day forecast for Darfield testing region and
verify results."""

        # Setup test name
        CSEPTestCase.setTestName(self, self.id())

        os.environ[HASH_DIR_ENV] = os.path.join(CSEPTestCase.TestDirPath,
                                                'CattaniaHash')

        model_list_file = 'exampleInputListSlipModels.dat'
        
        # Use the same test directory for input catalog and result forecast files
        args = [CSEPTestCase.TestDirPath, 
                '%s=0,%s=-37284630,%s=%s,%s=%s' %(CattaniaModel.ConfigOption,
                                                  CattaniaModel.RandomSeedOption,
                                                  CattaniaModel.SlipModelFileListOption,
                                                  os.path.join(CSEPTestCase.TestDirPath,
                                                               model_list_file),
                                                  CattaniaModel.ParamFilePathOption,
                                                  DarfieldForecastsTests.__CattaniaRefDir)]
        
        # Instantiate forecast object
        model = ForecastFactory().object(CattaniaOneDayModel.Type, 
                                         args)

        # Copy reference input catalog data file to the test directory
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__CattaniaRefDir, 
                                     "CatalogCanterbury_M2_1980_2012.txt"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     model.inputCatalogFilename()))    

        # Copy reference focal mechanism input catalog data file to the test directory
        start_date_focal_mechanism = "%s%s" %(DarfieldForecastsTests.__CattaniaTestDate.strftime("%Y_%m_%d_"),
                                              GeoNetNZDataSource.ProcessedFocalMechanismFile)
        
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__CattaniaRefDir, 
                                     "GeoNet_CMT_solutions.dat"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     start_date_focal_mechanism))    
        
        # Change directory to the test location to guarantee that all model
        # configuration and scratch files are placed there.
        cwd = os.getcwd() 
        os.chdir(CSEPTestCase.TestDirPath)

        try:            

           # Copy slip model list file into local directory for the run
           self.__slipModelList(os.path.join(DarfieldForecastsTests.__CattaniaRefDir,
                                                   model_list_file))
           
           model.create(DarfieldForecastsTests.__CattaniaTestDate, 
                        CSEPTestCase.TestDirPath)
   
           
           # Reference file with test results to compare against
           reference_file = "model0_1d_foremap.dat"
           
           shutil.copyfile(os.path.join(DarfieldForecastsTests.__CattaniaRefDir, 
                                        reference_file),
                           os.path.join(CSEPTestCase.TestDirPath, 
                                        reference_file))
   
           error_msg = "CattaniaCRS0 one-day forecast generation for Darfield \
testing region failed."
           
           self.failIf(CSEPFile.compare(reference_file, 
                                        model.filename()) is False,
                       error_msg)
           
           # Check for existence of metadata file for the forecast
           error_msg = "Metadata file for CattaniaCRS0 one-day forecast for Darfield \
testing region is missing."
           self.failIf(os.path.exists(model.filename() + 
                                      CSEPPropertyFile.Metadata.Extension) is False, 
                       error_msg)

        finally:
            os.chdir(cwd)
            
            del os.environ[HASH_DIR_ENV]


    #---------------------------------------------------------------------------
    #
    # Invoke CATTANIAOneDayCRS1 one-day model for Darfield testing region, and verify 
    # the results.
    #
    def testCATTANIAOneDayCRS1Model(self):
        """ Generate CATTANIAOneDayCRS1 one-day forecast for Darfield testing region and
verify results."""

        # Setup test name
        CSEPTestCase.setTestName(self, self.id())

        model_list_file = 'exampleInputListSlipModels.dat'
        
        # Use the same test directory for input catalog and result forecast files
        args = [CSEPTestCase.TestDirPath, 
                '%s=1,%s=-37284630,%s=%s,%s=%s' %(CattaniaModel.ConfigOption,
                                                  CattaniaModel.RandomSeedOption,
                                                  CattaniaModel.SlipModelFileListOption,
                                                  os.path.join(CSEPTestCase.TestDirPath,
                                                               model_list_file),
                                                  CattaniaModel.ParamFilePathOption,
                                                  DarfieldForecastsTests.__CattaniaRefDir)]
        
        # Instantiate forecast object
        model = ForecastFactory().object(CattaniaOneDayModel.Type, 
                                         args)

        # Copy reference input catalog data file to the test directory
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__CattaniaRefDir, 
                                     "CatalogCanterbury_M2_1980_2012.txt"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     model.inputCatalogFilename()))    

        # Copy reference focal mechanism input catalog data file to the test directory
        start_date_focal_mechanism = "%s%s" %(DarfieldForecastsTests.__CattaniaTestDate.strftime("%Y_%m_%d_"),
                                              GeoNetNZDataSource.ProcessedFocalMechanismFile)
        
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__CattaniaRefDir, 
                                     "GeoNet_CMT_solutions.dat"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     start_date_focal_mechanism))    
        
        # Change directory to the test location to guarantee that all model
        # configuration and scratch files are placed there.
        cwd = os.getcwd() 
        os.chdir(CSEPTestCase.TestDirPath)

        try:            

           # Copy slip model list file into local directory for the run
           self.__slipModelList(os.path.join(DarfieldForecastsTests.__CattaniaRefDir,
                                                   model_list_file))
           
           model.create(DarfieldForecastsTests.__CattaniaTestDate, 
                        CSEPTestCase.TestDirPath)
   
           
           # Reference file with test results to compare against
           reference_file = "model1_1d_foremap.dat"
           
           shutil.copyfile(os.path.join(DarfieldForecastsTests.__CattaniaRefDir, 
                                        reference_file),
                           os.path.join(CSEPTestCase.TestDirPath, 
                                        reference_file))
   
           error_msg = "CattaniaCRS1 one-day forecast generation for Darfield \
testing region failed."
           
           self.failIf(CSEPFile.compare(reference_file, 
                                        model.filename()) is False,
                       error_msg)
           
           # Check for existence of metadata file for the forecast
           error_msg = "Metadata file for CattaniaCRS1 one-day forecast for Darfield \
testing region is missing."
           self.failIf(os.path.exists(model.filename() + 
                                      CSEPPropertyFile.Metadata.Extension) is False, 
                       error_msg)

        finally:
            os.chdir(cwd)


    #---------------------------------------------------------------------------
    #
    # Invoke CATTANIAOneDayCRS2 one-day model for Darfield testing region, and verify 
    # the results.
    #
    def testCATTANIAOneDayCRS2Model(self):
        """ Generate CATTANIAOneDayCRS2 one-day forecast for Darfield testing region and
verify results."""

        # Setup test name
        CSEPTestCase.setTestName(self, self.id())

        model_list_file = 'exampleInputListSlipModels.dat'
        
        # Use the same test directory for input catalog and result forecast files
        args = [CSEPTestCase.TestDirPath, 
                '%s=2,%s=-37284630,%s=%s,%s=%s' %(CattaniaModel.ConfigOption,
                                                  CattaniaModel.RandomSeedOption,
                                                  CattaniaModel.SlipModelFileListOption,
                                                  os.path.join(CSEPTestCase.TestDirPath,
                                                               model_list_file),
                                                  CattaniaModel.ParamFilePathOption,
                                                  DarfieldForecastsTests.__CattaniaRefDir)]
        
        # Instantiate forecast object
        model = ForecastFactory().object(CattaniaOneDayModel.Type, 
                                         args)

        # Copy reference input catalog data file to the test directory
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__CattaniaRefDir, 
                                     "CatalogCanterbury_M2_1980_2012.txt"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     model.inputCatalogFilename()))    

        # Copy reference focal mechanism input catalog data file to the test directory
        start_date_focal_mechanism = "%s%s" %(DarfieldForecastsTests.__CattaniaTestDate.strftime("%Y_%m_%d_"),
                                              GeoNetNZDataSource.ProcessedFocalMechanismFile)
        
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__CattaniaRefDir, 
                                     "GeoNet_CMT_solutions.dat"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     start_date_focal_mechanism))    
        
        # Change directory to the test location to guarantee that all model
        # configuration and scratch files are placed there.
        cwd = os.getcwd() 
        os.chdir(CSEPTestCase.TestDirPath)

        try:            

           # Copy slip model list file into local directory for the run
           self.__slipModelList(os.path.join(DarfieldForecastsTests.__CattaniaRefDir,
                                                   model_list_file))
           
           model.create(DarfieldForecastsTests.__CattaniaTestDate, 
                        CSEPTestCase.TestDirPath)
   
           
           # Reference file with test results to compare against
           reference_file = "model2_1d_foremap.dat"
           
           shutil.copyfile(os.path.join(DarfieldForecastsTests.__CattaniaRefDir, 
                                        reference_file),
                           os.path.join(CSEPTestCase.TestDirPath, 
                                        reference_file))
   
           error_msg = "CattaniaCRS2 one-day forecast generation for Darfield \
testing region failed."
           
           self.failIf(CSEPFile.compare(reference_file, 
                                        model.filename()) is False,
                       error_msg)
           
           # Check for existence of metadata file for the forecast
           error_msg = "Metadata file for CattaniaCRS2 one-day forecast for Darfield \
testing region is missing."
           self.failIf(os.path.exists(model.filename() + 
                                      CSEPPropertyFile.Metadata.Extension) is False, 
                       error_msg)

        finally:
            os.chdir(cwd)


    #---------------------------------------------------------------------------
    #
    # Invoke CATTANIAOneDayCRS3 one-day model for Darfield testing region, and verify 
    # the results.
    #
    def testCATTANIAOneDayCRS3Model(self):
        """ Generate CATTANIAOneDayCRS3 one-day forecast for Darfield testing region and
verify results."""

        # Setup test name
        CSEPTestCase.setTestName(self, self.id())

        model_list_file = 'exampleInputListSlipModels.dat'
        
        # Use the same test directory for input catalog and result forecast files
        args = [CSEPTestCase.TestDirPath, 
                '%s=3,%s=-37284630,%s=%s,%s=%s' %(CattaniaModel.ConfigOption,
                                                  CattaniaModel.RandomSeedOption,
                                                  CattaniaModel.SlipModelFileListOption,
                                                  os.path.join(CSEPTestCase.TestDirPath,
                                                               model_list_file),
                                                  CattaniaModel.ParamFilePathOption,
                                                  DarfieldForecastsTests.__CattaniaRefDir)]
        
        # Instantiate forecast object
        model = ForecastFactory().object(CattaniaOneDayModel.Type, 
                                         args)

        # Copy reference input catalog data file to the test directory
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__CattaniaRefDir, 
                                     "CatalogCanterbury_M2_1980_2012.txt"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     model.inputCatalogFilename()))    

        # Copy reference focal mechanism input catalog data file to the test directory
        start_date_focal_mechanism = "%s%s" %(DarfieldForecastsTests.__CattaniaTestDate.strftime("%Y_%m_%d_"),
                                              GeoNetNZDataSource.ProcessedFocalMechanismFile)
        
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__CattaniaRefDir, 
                                     "GeoNet_CMT_solutions.dat"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     start_date_focal_mechanism))    
        
        # Change directory to the test location to guarantee that all model
        # configuration and scratch files are placed there.
        cwd = os.getcwd() 
        os.chdir(CSEPTestCase.TestDirPath)

        try:            

           # Copy slip model list file into local directory for the run
           self.__slipModelList(os.path.join(DarfieldForecastsTests.__CattaniaRefDir,
                                                   model_list_file))
           
           model.create(DarfieldForecastsTests.__CattaniaTestDate, 
                        CSEPTestCase.TestDirPath)
   
           
           # Reference file with test results to compare against
           reference_file = "model3_1d_foremap.dat"
           
           shutil.copyfile(os.path.join(DarfieldForecastsTests.__CattaniaRefDir, 
                                        reference_file),
                           os.path.join(CSEPTestCase.TestDirPath, 
                                        reference_file))
   
           error_msg = "CattaniaCRS3 one-day forecast generation for Darfield \
testing region failed."
           
           self.failIf(CSEPFile.compare(reference_file, 
                                        model.filename()) is False,
                       error_msg)
           
           # Check for existence of metadata file for the forecast
           error_msg = "Metadata file for CattaniaCRS3 one-day forecast for Darfield \
testing region is missing."
           self.failIf(os.path.exists(model.filename() + 
                                      CSEPPropertyFile.Metadata.Extension) is False, 
                       error_msg)

        finally:
            os.chdir(cwd)


    #---------------------------------------------------------------------------
    #
    # Invoke CATTANIAOneDayCRS4 one-day model for Darfield testing region, and verify 
    # the results.
    #
    def testCATTANIAOneDayCRS4Model(self):
        """ Generate CATTANIAOneDayCRS3 one-day forecast for Darfield testing region and
verify results."""

        # Setup test name
        CSEPTestCase.setTestName(self, self.id())

        model_list_file = 'exampleInputListSlipModels.dat'
        
        # Use the same test directory for input catalog and result forecast files
        args = [CSEPTestCase.TestDirPath, 
                '%s=4,%s=-37284630,%s=%s,%s=%s' %(CattaniaModel.ConfigOption,
                                                  CattaniaModel.RandomSeedOption,
                                                  CattaniaModel.SlipModelFileListOption,
                                                  os.path.join(CSEPTestCase.TestDirPath,
                                                               model_list_file),
                                                  CattaniaModel.ParamFilePathOption,
                                                  DarfieldForecastsTests.__CattaniaRefDir)]
        
        # Instantiate forecast object
        model = ForecastFactory().object(CattaniaOneDayModel.Type, 
                                         args)

        # Copy reference input catalog data file to the test directory
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__CattaniaRefDir, 
                                     "CatalogCanterbury_M2_1980_2012.txt"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     model.inputCatalogFilename()))    

        # Copy reference focal mechanism input catalog data file to the test directory
        start_date_focal_mechanism = "%s%s" %(DarfieldForecastsTests.__CattaniaTestDate.strftime("%Y_%m_%d_"),
                                              GeoNetNZDataSource.ProcessedFocalMechanismFile)
        
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__CattaniaRefDir, 
                                     "GeoNet_CMT_solutions.dat"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     start_date_focal_mechanism))    
        
        # Change directory to the test location to guarantee that all model
        # configuration and scratch files are placed there.
        cwd = os.getcwd() 
        os.chdir(CSEPTestCase.TestDirPath)

        try:            

           # Copy slip model list file into local directory for the run
           self.__slipModelList(os.path.join(DarfieldForecastsTests.__CattaniaRefDir,
                                                   model_list_file))
           
           model.create(DarfieldForecastsTests.__CattaniaTestDate, 
                        CSEPTestCase.TestDirPath)
   
           
           # Reference file with test results to compare against
           reference_file = "model4_1d_foremap.dat"
           
           shutil.copyfile(os.path.join(DarfieldForecastsTests.__CattaniaRefDir, 
                                        reference_file),
                           os.path.join(CSEPTestCase.TestDirPath, 
                                        reference_file))
   
           error_msg = "CattaniaCRS4 one-day forecast generation for Darfield \
testing region failed."
           
           self.failIf(CSEPFile.compare(reference_file, 
                                        model.filename()) is False,
                       error_msg)
           
           # Check for existence of metadata file for the forecast
           error_msg = "Metadata file for CattaniaCRS4 one-day forecast for Darfield \
testing region is missing."
           self.failIf(os.path.exists(model.filename() + 
                                      CSEPPropertyFile.Metadata.Extension) is False, 
                       error_msg)

        finally:
            os.chdir(cwd)


    #---------------------------------------------------------------------------
    #
    # Invoke CATTANIAOneMonthCRS0 model for Darfield testing region, and verify 
    # the results.
    #
    def testCATTANIAOneMonthCRS0Model(self):
        """ Generate CATTANIAOneMonthCRS0 forecast for Darfield testing region and
verify results."""

        
        # Setup test name
        CSEPTestCase.setTestName(self, self.id())

        model_list_file = 'exampleInputListSlipModels.dat'
        
        # Use the same test directory for input catalog and result forecast files
        args = [CSEPTestCase.TestDirPath, 
                '%s=0,%s=-37284630,%s=%s,%s=%s' %(CattaniaModel.ConfigOption,
                                                  CattaniaModel.RandomSeedOption,
                                                  CattaniaModel.SlipModelFileListOption,
                                                  os.path.join(CSEPTestCase.TestDirPath,
                                                               model_list_file),
                                                  CattaniaModel.ParamFilePathOption,
                                                  DarfieldForecastsTests.__CattaniaRefDir)]
        
        # Instantiate forecast object
        model = ForecastFactory().object(CattaniaOneMonthModel.Type, 
                                         args)

        # Copy reference input catalog data file to the test directory
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__CattaniaRefDir, 
                                     "CatalogCanterbury_M2_1980_2012.txt"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     model.inputCatalogFilename()))    

        # Copy reference focal mechanism input catalog data file to the test directory
        start_date_focal_mechanism = "%s%s" %(DarfieldForecastsTests.__CattaniaTestDate.strftime("%Y_%m_%d_"),
                                              GeoNetNZDataSource.ProcessedFocalMechanismFile)
        
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__CattaniaRefDir, 
                                     "GeoNet_CMT_solutions.dat"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     start_date_focal_mechanism))    
        
        # Change directory to the test location to guarantee that all model
        # configuration and scratch files are placed there.
        cwd = os.getcwd() 
        os.chdir(CSEPTestCase.TestDirPath)

        try:            

           # Copy slip model list file into local directory for the run
           self.__slipModelList(os.path.join(DarfieldForecastsTests.__CattaniaRefDir,
                                                   model_list_file))
           
           model.create(DarfieldForecastsTests.__CattaniaTestDate, 
                        CSEPTestCase.TestDirPath)
   
           
           # Reference file with test results to compare against
           reference_file = "model0_1m_foremap.dat"
           
           shutil.copyfile(os.path.join(DarfieldForecastsTests.__CattaniaRefDir, 
                                        reference_file),
                           os.path.join(CSEPTestCase.TestDirPath, 
                                        reference_file))
   
           error_msg = "CattaniaCRS0 one-month forecast generation for Darfield \
testing region failed."
           
           self.failIf(CSEPFile.compare(reference_file, 
                                        model.filename()) is False,
                       error_msg)
           
           # Check for existence of metadata file for the forecast
           error_msg = "Metadata file for CattaniaCRS0 one-month forecast for Darfield \
testing region is missing."
           self.failIf(os.path.exists(model.filename() + 
                                      CSEPPropertyFile.Metadata.Extension) is False, 
                       error_msg)

        finally:
            os.chdir(cwd)


    #---------------------------------------------------------------------------
    #
    # Invoke CATTANIAOneMonthCRS1 model for Darfield testing region, and verify 
    # the results.
    #
    def testCATTANIAOneMonthCRS1Model(self):
        """ Generate CATTANIAOneMonthCRS1 forecast for Darfield testing region and
verify results."""

        # Setup test name
        CSEPTestCase.setTestName(self, self.id())

        model_list_file = 'exampleInputListSlipModels.dat'
        
        # Use the same test directory for input catalog and result forecast files
        args = [CSEPTestCase.TestDirPath, 
                '%s=1,%s=-37284630,%s=%s,%s=%s' %(CattaniaModel.ConfigOption,
                                                  CattaniaModel.RandomSeedOption,
                                                  CattaniaModel.SlipModelFileListOption,
                                                  os.path.join(CSEPTestCase.TestDirPath,
                                                               model_list_file),
                                                  CattaniaModel.ParamFilePathOption,
                                                  DarfieldForecastsTests.__CattaniaRefDir)]
        
        # Instantiate forecast object
        model = ForecastFactory().object(CattaniaOneMonthModel.Type, 
                                         args)

        # Copy reference input catalog data file to the test directory
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__CattaniaRefDir, 
                                     "CatalogCanterbury_M2_1980_2012.txt"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     model.inputCatalogFilename()))    

        # Copy reference focal mechanism input catalog data file to the test directory
        start_date_focal_mechanism = "%s%s" %(DarfieldForecastsTests.__CattaniaTestDate.strftime("%Y_%m_%d_"),
                                              GeoNetNZDataSource.ProcessedFocalMechanismFile)
        
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__CattaniaRefDir, 
                                     "GeoNet_CMT_solutions.dat"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     start_date_focal_mechanism))    
        
        # Change directory to the test location to guarantee that all model
        # configuration and scratch files are placed there.
        cwd = os.getcwd() 
        os.chdir(CSEPTestCase.TestDirPath)

        try:            

           # Copy slip model list file into local directory for the run
           self.__slipModelList(os.path.join(DarfieldForecastsTests.__CattaniaRefDir,
                                                   model_list_file))
           
           model.create(DarfieldForecastsTests.__CattaniaTestDate, 
                        CSEPTestCase.TestDirPath)
   
           
           # Reference file with test results to compare against
           reference_file = "model1_1m_foremap.dat"
           
           shutil.copyfile(os.path.join(DarfieldForecastsTests.__CattaniaRefDir, 
                                        reference_file),
                           os.path.join(CSEPTestCase.TestDirPath, 
                                        reference_file))
   
           error_msg = "CattaniaCRS1 one-month forecast generation for Darfield \
testing region failed."
           
           self.failIf(CSEPFile.compare(reference_file, 
                                        model.filename()) is False,
                       error_msg)
           
           # Check for existence of metadata file for the forecast
           error_msg = "Metadata file for CattaniaCRS1 one-month forecast for Darfield \
testing region is missing."
           self.failIf(os.path.exists(model.filename() + 
                                      CSEPPropertyFile.Metadata.Extension) is False, 
                       error_msg)

        finally:
            os.chdir(cwd)


    #---------------------------------------------------------------------------
    #
    # Invoke CATTANIAOneMonthCRS2 model for Darfield testing region, and verify 
    # the results.
    #
    def testCATTANIAOneMonthCRS2Model(self):
        """ Generate CATTANIAOneMonthCRS2 forecast for Darfield testing region and
verify results."""

        # Setup test name
        CSEPTestCase.setTestName(self, self.id())

        model_list_file = 'exampleInputListSlipModels.dat'
        
        # Use the same test directory for input catalog and result forecast files
        args = [CSEPTestCase.TestDirPath, 
                '%s=2,%s=-37284630,%s=%s,%s=%s' %(CattaniaModel.ConfigOption,
                                                  CattaniaModel.RandomSeedOption,
                                                  CattaniaModel.SlipModelFileListOption,
                                                  os.path.join(CSEPTestCase.TestDirPath,
                                                               model_list_file),
                                                  CattaniaModel.ParamFilePathOption,
                                                  DarfieldForecastsTests.__CattaniaRefDir)]
        
        # Instantiate forecast object
        model = ForecastFactory().object(CattaniaOneMonthModel.Type, 
                                         args)

        # Copy reference input catalog data file to the test directory
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__CattaniaRefDir, 
                                     "CatalogCanterbury_M2_1980_2012.txt"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     model.inputCatalogFilename()))    

        # Copy reference focal mechanism input catalog data file to the test directory
        start_date_focal_mechanism = "%s%s" %(DarfieldForecastsTests.__CattaniaTestDate.strftime("%Y_%m_%d_"),
                                              GeoNetNZDataSource.ProcessedFocalMechanismFile)
        
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__CattaniaRefDir, 
                                     "GeoNet_CMT_solutions.dat"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     start_date_focal_mechanism))    
        
        # Change directory to the test location to guarantee that all model
        # configuration and scratch files are placed there.
        cwd = os.getcwd() 
        os.chdir(CSEPTestCase.TestDirPath)

        try:            

           # Copy slip model list file into local directory for the run
           self.__slipModelList(os.path.join(DarfieldForecastsTests.__CattaniaRefDir,
                                                   model_list_file))
           
           model.create(DarfieldForecastsTests.__CattaniaTestDate, 
                        CSEPTestCase.TestDirPath)
   
           
           # Reference file with test results to compare against
           reference_file = "model2_1m_foremap.dat"
           
           shutil.copyfile(os.path.join(DarfieldForecastsTests.__CattaniaRefDir, 
                                        reference_file),
                           os.path.join(CSEPTestCase.TestDirPath, 
                                        reference_file))
   
           error_msg = "CattaniaCRS2 one-month forecast generation for Darfield \
testing region failed."
           
           self.failIf(CSEPFile.compare(reference_file, 
                                        model.filename()) is False,
                       error_msg)
           
           # Check for existence of metadata file for the forecast
           error_msg = "Metadata file for CattaniaCRS2 one-month forecast for Darfield \
testing region is missing."
           self.failIf(os.path.exists(model.filename() + 
                                      CSEPPropertyFile.Metadata.Extension) is False, 
                       error_msg)

        finally:
            os.chdir(cwd)


    #---------------------------------------------------------------------------
    #
    # Invoke CATTANIAOneMonthCRS3 model for Darfield testing region, and verify 
    # the results.
    #
    def testCATTANIAOneMonthCRS3Model(self):
        """ Generate CATTANIAOneMonthCRS3 forecast for Darfield testing region and
verify results."""

        # Setup test name
        CSEPTestCase.setTestName(self, self.id())

        model_list_file = 'exampleInputListSlipModels.dat'
        
        # Use the same test directory for input catalog and result forecast files
        args = [CSEPTestCase.TestDirPath, 
                '%s=3,%s=-37284630,%s=%s,%s=%s' %(CattaniaModel.ConfigOption,
                                                  CattaniaModel.RandomSeedOption,
                                                  CattaniaModel.SlipModelFileListOption,
                                                  os.path.join(CSEPTestCase.TestDirPath,
                                                               model_list_file),
                                                  CattaniaModel.ParamFilePathOption,
                                                  DarfieldForecastsTests.__CattaniaRefDir)]
        
        # Instantiate forecast object
        model = ForecastFactory().object(CattaniaOneMonthModel.Type, 
                                         args)

        # Copy reference input catalog data file to the test directory
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__CattaniaRefDir, 
                                     "CatalogCanterbury_M2_1980_2012.txt"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     model.inputCatalogFilename()))    

        # Copy reference focal mechanism input catalog data file to the test directory
        start_date_focal_mechanism = "%s%s" %(DarfieldForecastsTests.__CattaniaTestDate.strftime("%Y_%m_%d_"),
                                              GeoNetNZDataSource.ProcessedFocalMechanismFile)
        
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__CattaniaRefDir, 
                                     "GeoNet_CMT_solutions.dat"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     start_date_focal_mechanism))    
        
        # Change directory to the test location to guarantee that all model
        # configuration and scratch files are placed there.
        cwd = os.getcwd() 
        os.chdir(CSEPTestCase.TestDirPath)

        try:            

           # Copy slip model list file into local directory for the run
           self.__slipModelList(os.path.join(DarfieldForecastsTests.__CattaniaRefDir,
                                                   model_list_file))
           
           model.create(DarfieldForecastsTests.__CattaniaTestDate, 
                        CSEPTestCase.TestDirPath)
   
           
           # Reference file with test results to compare against
           reference_file = "model3_1m_foremap.dat"
           
           shutil.copyfile(os.path.join(DarfieldForecastsTests.__CattaniaRefDir, 
                                        reference_file),
                           os.path.join(CSEPTestCase.TestDirPath, 
                                        reference_file))
   
           error_msg = "CattaniaCRS3 one-month forecast generation for Darfield \
testing region failed."
           
           self.failIf(CSEPFile.compare(reference_file, 
                                        model.filename()) is False,
                       error_msg)
           
           # Check for existence of metadata file for the forecast
           error_msg = "Metadata file for CattaniaCRS3 one-month forecast for Darfield \
testing region is missing."
           self.failIf(os.path.exists(model.filename() + 
                                      CSEPPropertyFile.Metadata.Extension) is False, 
                       error_msg)

        finally:
            os.chdir(cwd)


    #---------------------------------------------------------------------------
    #
    # Invoke CATTANIAOneMonthCRS4 model for Darfield testing region, and verify 
    # the results.
    #
    def testCATTANIAOneMonthCRS4Model(self):
        """ Generate CATTANIAOneMonthCRS4 forecast for Darfield testing region and
verify results."""

        # Setup test name
        CSEPTestCase.setTestName(self, self.id())

        model_list_file = 'exampleInputListSlipModels.dat'
        
        # Use the same test directory for input catalog and result forecast files
        args = [CSEPTestCase.TestDirPath, 
                '%s=4,%s=-37284630,%s=%s,%s=%s' %(CattaniaModel.ConfigOption,
                                                  CattaniaModel.RandomSeedOption,
                                                  CattaniaModel.SlipModelFileListOption,
                                                  os.path.join(CSEPTestCase.TestDirPath,
                                                               model_list_file),
                                                  CattaniaModel.ParamFilePathOption,
                                                  DarfieldForecastsTests.__CattaniaRefDir)]
        
        # Instantiate forecast object
        model = ForecastFactory().object(CattaniaOneMonthModel.Type, 
                                         args)

        # Copy reference input catalog data file to the test directory
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__CattaniaRefDir, 
                                     "CatalogCanterbury_M2_1980_2012.txt"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     model.inputCatalogFilename()))    

        # Copy reference focal mechanism input catalog data file to the test directory
        start_date_focal_mechanism = "%s%s" %(DarfieldForecastsTests.__CattaniaTestDate.strftime("%Y_%m_%d_"),
                                              GeoNetNZDataSource.ProcessedFocalMechanismFile)
        
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__CattaniaRefDir, 
                                     "GeoNet_CMT_solutions.dat"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     start_date_focal_mechanism))    
        
        # Change directory to the test location to guarantee that all model
        # configuration and scratch files are placed there.
        cwd = os.getcwd() 
        os.chdir(CSEPTestCase.TestDirPath)

        try:            

           # Copy slip model list file into local directory for the run
           self.__slipModelList(os.path.join(DarfieldForecastsTests.__CattaniaRefDir,
                                                   model_list_file))
           
           model.create(DarfieldForecastsTests.__CattaniaTestDate, 
                        CSEPTestCase.TestDirPath)
   
           
           # Reference file with test results to compare against
           reference_file = "model4_1m_foremap.dat"
           
           shutil.copyfile(os.path.join(DarfieldForecastsTests.__CattaniaRefDir, 
                                        reference_file),
                           os.path.join(CSEPTestCase.TestDirPath, 
                                        reference_file))
   
           error_msg = "CattaniaCRS4 one-month forecast generation for Darfield \
testing region failed."
           
           self.failIf(CSEPFile.compare(reference_file, 
                                        model.filename()) is False,
                       error_msg)
           
           # Check for existence of metadata file for the forecast
           error_msg = "Metadata file for CattaniaCRS4 one-month forecast for Darfield \
testing region is missing."
           self.failIf(os.path.exists(model.filename() + 
                                      CSEPPropertyFile.Metadata.Extension) is False, 
                       error_msg)

        finally:
            os.chdir(cwd)


    #---------------------------------------------------------------------------
    #
    # Invoke CATTANIAOneYearCRS0 model for Darfield testing region, and verify 
    # the results.
    #
    def testCATTANIAOneYearCRS0Model(self):
        """ Generate CATTANIAOneYearCRS0 forecast for Darfield testing region and
verify results."""

        # Setup test name
        CSEPTestCase.setTestName(self, self.id())

        model_list_file = 'exampleInputListSlipModels.dat'
        
        # Use the same test directory for input catalog and result forecast files
        args = [CSEPTestCase.TestDirPath, 
                '%s=0,%s=-37284630,%s=%s,%s=%s' %(CattaniaModel.ConfigOption,
                                                  CattaniaModel.RandomSeedOption,
                                                  CattaniaModel.SlipModelFileListOption,
                                                  os.path.join(CSEPTestCase.TestDirPath,
                                                               model_list_file),
                                                  CattaniaModel.ParamFilePathOption,
                                                  DarfieldForecastsTests.__CattaniaRefDir)]
        
        # Instantiate forecast object
        model = ForecastFactory().object(CattaniaOneYearModel.Type, 
                                         args)

        # Copy reference input catalog data file to the test directory
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__CattaniaRefDir, 
                                     "CatalogCanterbury_M2_1980_2012.txt"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     model.inputCatalogFilename()))    

        # Copy reference focal mechanism input catalog data file to the test directory
        start_date_focal_mechanism = "%s%s" %(DarfieldForecastsTests.__CattaniaTestDate.strftime("%Y_%m_%d_"),
                                              GeoNetNZDataSource.ProcessedFocalMechanismFile)
        
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__CattaniaRefDir, 
                                     "GeoNet_CMT_solutions.dat"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     start_date_focal_mechanism))    
        
        # Change directory to the test location to guarantee that all model
        # configuration and scratch files are placed there.
        cwd = os.getcwd() 
        os.chdir(CSEPTestCase.TestDirPath)

        try:            

           # Copy slip model list file into local directory for the run
           self.__slipModelList(os.path.join(DarfieldForecastsTests.__CattaniaRefDir,
                                                   model_list_file))
           
           model.create(DarfieldForecastsTests.__CattaniaTestDate, 
                        CSEPTestCase.TestDirPath)
   
           
           # Reference file with test results to compare against
           reference_file = "model0_1y_foremap.dat"
           
           shutil.copyfile(os.path.join(DarfieldForecastsTests.__CattaniaRefDir, 
                                        reference_file),
                           os.path.join(CSEPTestCase.TestDirPath, 
                                        reference_file))
   
           error_msg = "CattaniaCRS0 one-year forecast generation for Darfield \
testing region failed."
           
           self.failIf(CSEPFile.compare(reference_file, 
                                        model.filename()) is False,
                       error_msg)
           
           # Check for existence of metadata file for the forecast
           error_msg = "Metadata file for CattaniaCRS0 one-year forecast for Darfield \
testing region is missing."
           self.failIf(os.path.exists(model.filename() + 
                                      CSEPPropertyFile.Metadata.Extension) is False, 
                       error_msg)

        finally:
            os.chdir(cwd)


    #---------------------------------------------------------------------------
    #
    # Invoke CATTANIAOneYearCRS1 model for Darfield testing region, and verify 
    # the results.
    #
    def testCATTANIAOneYearCRS1Model(self):
        """ Generate CATTANIAOneYearCRS1 forecast for Darfield testing region and
verify results."""

        # Setup test name
        CSEPTestCase.setTestName(self, self.id())

        model_list_file = 'exampleInputListSlipModels.dat'
        
        # Use the same test directory for input catalog and result forecast files
        args = [CSEPTestCase.TestDirPath, 
                '%s=1,%s=-37284630,%s=%s,%s=%s' %(CattaniaModel.ConfigOption,
                                                  CattaniaModel.RandomSeedOption,
                                                  CattaniaModel.SlipModelFileListOption,
                                                  os.path.join(CSEPTestCase.TestDirPath,
                                                               model_list_file),
                                                  CattaniaModel.ParamFilePathOption,
                                                  DarfieldForecastsTests.__CattaniaRefDir)]
        
        # Instantiate forecast object
        model = ForecastFactory().object(CattaniaOneYearModel.Type, 
                                         args)

        # Copy reference input catalog data file to the test directory
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__CattaniaRefDir, 
                                     "CatalogCanterbury_M2_1980_2012.txt"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     model.inputCatalogFilename()))    

        # Copy reference focal mechanism input catalog data file to the test directory
        start_date_focal_mechanism = "%s%s" %(DarfieldForecastsTests.__CattaniaTestDate.strftime("%Y_%m_%d_"),
                                              GeoNetNZDataSource.ProcessedFocalMechanismFile)
        
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__CattaniaRefDir, 
                                     "GeoNet_CMT_solutions.dat"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     start_date_focal_mechanism))    
        
        # Change directory to the test location to guarantee that all model
        # configuration and scratch files are placed there.
        cwd = os.getcwd() 
        os.chdir(CSEPTestCase.TestDirPath)

        try:            

           # Copy slip model list file into local directory for the run
           self.__slipModelList(os.path.join(DarfieldForecastsTests.__CattaniaRefDir,
                                                   model_list_file))
           
           model.create(DarfieldForecastsTests.__CattaniaTestDate, 
                        CSEPTestCase.TestDirPath)
   
           
           # Reference file with test results to compare against
           reference_file = "model1_1y_foremap.dat"
           
           shutil.copyfile(os.path.join(DarfieldForecastsTests.__CattaniaRefDir, 
                                        reference_file),
                           os.path.join(CSEPTestCase.TestDirPath, 
                                        reference_file))
   
           error_msg = "CattaniaCRS1 one-year forecast generation for Darfield \
testing region failed."
           
           self.failIf(CSEPFile.compare(reference_file, 
                                        model.filename()) is False,
                       error_msg)
           
           # Check for existence of metadata file for the forecast
           error_msg = "Metadata file for CattaniaCRS1 one-year forecast for Darfield \
testing region is missing."
           self.failIf(os.path.exists(model.filename() + 
                                      CSEPPropertyFile.Metadata.Extension) is False, 
                       error_msg)

        finally:
            os.chdir(cwd)


    #---------------------------------------------------------------------------
    #
    # Invoke CATTANIAOneYearCRS2 model for Darfield testing region, and verify 
    # the results.
    #
    def testCATTANIAOneYearCRS2Model(self):
        """ Generate CATTANIAOneYearCRS2 forecast for Darfield testing region and
verify results."""

        # Setup test name
        CSEPTestCase.setTestName(self, self.id())

        model_list_file = 'exampleInputListSlipModels.dat'
        
        # Use the same test directory for input catalog and result forecast files
        args = [CSEPTestCase.TestDirPath, 
                '%s=2,%s=-37284630,%s=%s,%s=%s' %(CattaniaModel.ConfigOption,
                                                  CattaniaModel.RandomSeedOption,
                                                  CattaniaModel.SlipModelFileListOption,
                                                  os.path.join(CSEPTestCase.TestDirPath,
                                                               model_list_file),
                                                  CattaniaModel.ParamFilePathOption,
                                                  DarfieldForecastsTests.__CattaniaRefDir)]
        
        # Instantiate forecast object
        model = ForecastFactory().object(CattaniaOneYearModel.Type, 
                                         args)

        # Copy reference input catalog data file to the test directory
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__CattaniaRefDir, 
                                     "CatalogCanterbury_M2_1980_2012.txt"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     model.inputCatalogFilename()))    

        # Copy reference focal mechanism input catalog data file to the test directory
        start_date_focal_mechanism = "%s%s" %(DarfieldForecastsTests.__CattaniaTestDate.strftime("%Y_%m_%d_"),
                                              GeoNetNZDataSource.ProcessedFocalMechanismFile)
        
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__CattaniaRefDir, 
                                     "GeoNet_CMT_solutions.dat"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     start_date_focal_mechanism))    
        
        # Change directory to the test location to guarantee that all model
        # configuration and scratch files are placed there.
        cwd = os.getcwd() 
        os.chdir(CSEPTestCase.TestDirPath)

        try:            

           # Copy slip model list file into local directory for the run
           self.__slipModelList(os.path.join(DarfieldForecastsTests.__CattaniaRefDir,
                                                   model_list_file))
           
           model.create(DarfieldForecastsTests.__CattaniaTestDate, 
                        CSEPTestCase.TestDirPath)
   
           
           # Reference file with test results to compare against
           reference_file = "model2_1y_foremap.dat"
           
           shutil.copyfile(os.path.join(DarfieldForecastsTests.__CattaniaRefDir, 
                                        reference_file),
                           os.path.join(CSEPTestCase.TestDirPath, 
                                        reference_file))
   
           error_msg = "CattaniaCRS2 one-year forecast generation for Darfield \
testing region failed."
           
           self.failIf(CSEPFile.compare(reference_file, 
                                        model.filename()) is False,
                       error_msg)
           
           # Check for existence of metadata file for the forecast
           error_msg = "Metadata file for CattaniaCRS2 one-year forecast for Darfield \
testing region is missing."
           self.failIf(os.path.exists(model.filename() + 
                                      CSEPPropertyFile.Metadata.Extension) is False, 
                       error_msg)

        finally:
            os.chdir(cwd)


    #---------------------------------------------------------------------------
    #
    # Invoke CATTANIAOneYearCRS3 model for Darfield testing region, and verify 
    # the results.
    #
    def testCATTANIAOneYearCRS3Model(self):
        """ Generate CATTANIAOneYearCRS3 forecast for Darfield testing region and
verify results."""

        # Setup test name
        CSEPTestCase.setTestName(self, self.id())

        model_list_file = 'exampleInputListSlipModels.dat'
        
        # Use the same test directory for input catalog and result forecast files
        args = [CSEPTestCase.TestDirPath, 
                '%s=3,%s=-37284630,%s=%s,%s=%s' %(CattaniaModel.ConfigOption,
                                                  CattaniaModel.RandomSeedOption,
                                                  CattaniaModel.SlipModelFileListOption,
                                                  os.path.join(CSEPTestCase.TestDirPath,
                                                               model_list_file),
                                                  CattaniaModel.ParamFilePathOption,
                                                  DarfieldForecastsTests.__CattaniaRefDir)]
        
        # Instantiate forecast object
        model = ForecastFactory().object(CattaniaOneYearModel.Type, 
                                         args)

        # Copy reference input catalog data file to the test directory
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__CattaniaRefDir, 
                                     "CatalogCanterbury_M2_1980_2012.txt"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     model.inputCatalogFilename()))    

        # Copy reference focal mechanism input catalog data file to the test directory
        start_date_focal_mechanism = "%s%s" %(DarfieldForecastsTests.__CattaniaTestDate.strftime("%Y_%m_%d_"),
                                              GeoNetNZDataSource.ProcessedFocalMechanismFile)
        
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__CattaniaRefDir, 
                                     "GeoNet_CMT_solutions.dat"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     start_date_focal_mechanism))    
        
        # Change directory to the test location to guarantee that all model
        # configuration and scratch files are placed there.
        cwd = os.getcwd() 
        os.chdir(CSEPTestCase.TestDirPath)

        try:            

           # Copy slip model list file into local directory for the run
           self.__slipModelList(os.path.join(DarfieldForecastsTests.__CattaniaRefDir,
                                                   model_list_file))
           
           model.create(DarfieldForecastsTests.__CattaniaTestDate, 
                        CSEPTestCase.TestDirPath)
   
           
           # Reference file with test results to compare against
           reference_file = "model3_1y_foremap.dat"
           
           shutil.copyfile(os.path.join(DarfieldForecastsTests.__CattaniaRefDir, 
                                        reference_file),
                           os.path.join(CSEPTestCase.TestDirPath, 
                                        reference_file))
   
           error_msg = "CattaniaCRS3 one-year forecast generation for Darfield \
testing region failed."
           
           self.failIf(CSEPFile.compare(reference_file, 
                                        model.filename()) is False,
                       error_msg)
           
           # Check for existence of metadata file for the forecast
           error_msg = "Metadata file for CattaniaCRS3 one-year forecast for Darfield \
testing region is missing."
           self.failIf(os.path.exists(model.filename() + 
                                      CSEPPropertyFile.Metadata.Extension) is False, 
                       error_msg)

        finally:
            os.chdir(cwd)


    #---------------------------------------------------------------------------
    #
    # Invoke CATTANIAOneYearCRS4 model for Darfield testing region, and verify 
    # the results.
    #
    def testCATTANIAOneYearCRS4Model(self):
        """ Generate CATTANIAOneYearCRS4 forecast for Darfield testing region and
verify results."""

        # Setup test name
        CSEPTestCase.setTestName(self, self.id())

        model_list_file = 'exampleInputListSlipModels.dat'
        
        # Use the same test directory for input catalog and result forecast files
        args = [CSEPTestCase.TestDirPath, 
                '%s=4,%s=-37284630,%s=%s,%s=%s' %(CattaniaModel.ConfigOption,
                                                  CattaniaModel.RandomSeedOption,
                                                  CattaniaModel.SlipModelFileListOption,
                                                  os.path.join(CSEPTestCase.TestDirPath,
                                                               model_list_file),
                                                  CattaniaModel.ParamFilePathOption,
                                                  DarfieldForecastsTests.__CattaniaRefDir)]
        
        # Instantiate forecast object
        model = ForecastFactory().object(CattaniaOneYearModel.Type, 
                                         args)

        # Copy reference input catalog data file to the test directory
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__CattaniaRefDir, 
                                     "CatalogCanterbury_M2_1980_2012.txt"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     model.inputCatalogFilename()))    

        # Copy reference focal mechanism input catalog data file to the test directory
        start_date_focal_mechanism = "%s%s" %(DarfieldForecastsTests.__CattaniaTestDate.strftime("%Y_%m_%d_"),
                                              GeoNetNZDataSource.ProcessedFocalMechanismFile)
        
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__CattaniaRefDir, 
                                     "GeoNet_CMT_solutions.dat"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     start_date_focal_mechanism))    
        
        # Change directory to the test location to guarantee that all model
        # configuration and scratch files are placed there.
        cwd = os.getcwd() 
        os.chdir(CSEPTestCase.TestDirPath)

        try:            

           # Copy slip model list file into local directory for the run
           self.__slipModelList(os.path.join(DarfieldForecastsTests.__CattaniaRefDir,
                                                   model_list_file))
           
           model.create(DarfieldForecastsTests.__CattaniaTestDate, 
                        CSEPTestCase.TestDirPath)
   
           
           # Reference file with test results to compare against
           reference_file = "model4_1y_foremap.dat"
           
           shutil.copyfile(os.path.join(DarfieldForecastsTests.__CattaniaRefDir, 
                                        reference_file),
                           os.path.join(CSEPTestCase.TestDirPath, 
                                        reference_file))
   
           error_msg = "CattaniaCRS4 one-year forecast generation for Darfield \
testing region failed."
           
           self.failIf(CSEPFile.compare(reference_file, 
                                        model.filename()) is False,
                       error_msg)
           
           # Check for existence of metadata file for the forecast
           error_msg = "Metadata file for CattaniaCRS4 one-year forecast for Darfield \
testing region is missing."
           self.failIf(os.path.exists(model.filename() + 
                                      CSEPPropertyFile.Metadata.Extension) is False, 
                       error_msg)

        finally:
            os.chdir(cwd)


    #---------------------------------------------------------------------------
    #
    # Invoke STEPCoulomb one-day model for Darfield testing region, and verify 
    # the results.
    #
    def testSTEPCoulombOneDayModel(self):
        """ Generate STEPCoulombOneDay forecast for Darfield testing region and
verify results."""

        # Setup test name
        CSEPTestCase.setTestName(self, self.id())

        model_list_file = 'slip_models.txt'
        
        # Use the same test directory for input catalog and result forecast files
        args = [CSEPTestCase.TestDirPath, 
                '%s=%s' %(STEPCoulombModel.SlipModelFileListOption,
                          os.path.join(CSEPTestCase.TestDirPath,
                                       model_list_file))]
        
        # Instantiate forecast object
        model = ForecastFactory().object(STEPCoulombOneDayModel.Type, 
                                         args)

        # Copy reference input catalog data file to the test directory
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__STEPCoulombRefDir, 
                                     "geonet_combined.04.07.14.txt"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     model.inputCatalogFilename()))    

        # Copy reference focal mechanism input catalog data file to the test directory
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__STEPCoulombRefDir, 
                                     "GeoNet_CMT_solutions.csv"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     GeoNetNZDataSource.ProcessedFocalMechanismFile))    
        
        # Change directory to the test location to guarantee that all model
        # configuration and scratch files are placed there.
        cwd = os.getcwd() 
        os.chdir(CSEPTestCase.TestDirPath)

        try:            

           # Copy slip model list file into local directory for the run
           self.__slipModelList(os.path.join(DarfieldForecastsTests.__STEPCoulombRefDir,
                                                   model_list_file))
           
           model.create(DarfieldForecastsTests.__STEPCoulombTestDate, 
                        CSEPTestCase.TestDirPath)
   
           
           # Reference file with test results to compare against
           reference_file = "out_STEP.dat"
           
           shutil.copyfile(os.path.join(DarfieldForecastsTests.__STEPCoulombRefDir, 
                                        reference_file),
                           os.path.join(CSEPTestCase.TestDirPath, 
                                        reference_file))
   
           error_msg = "Coulomb part of STEPCoulombOneDay forecast generation for Darfield \
testing region failed."
           
           self.failIf(CSEPFile.compare(reference_file, 
                                        model.CoulombOutput) is False,
                       error_msg)
           
           # Check for final STEPCoulomb forecast and it's metadata file
           reference_file = os.path.join(DarfieldForecastsTests.__STEPCoulombRefDir,
                                         "STEP_1d.txt")
           error_msg = "STEPCoulombOneDay forecast generation for Darfield \
testing region failed."
           
           self.failIf(CSEPFile.compare(reference_file, 
                                        model.filename()) is False,
                       error_msg)

           # Check for existence of metadata file for the forecast
           error_msg = "Metadata file for STEPCoulombOneDay forecast for Darfield \
testing region is missing."
           self.failIf(os.path.exists(model.filename() + 
                                      CSEPPropertyFile.Metadata.Extension) is False, 
                       error_msg)
           
        finally:
            os.chdir(cwd)


    #---------------------------------------------------------------------------
    #
    # Invoke STEPCoulomb one-month model for Darfield testing region, and verify 
    # the results.
    #
    def testSTEPCoulombOneMonthModel(self):
        """ Generate STEPCoulombOneMonth forecast for Darfield testing region and
verify results."""

        # Setup test name
        CSEPTestCase.setTestName(self, self.id())

        model_list_file = 'slip_models.txt'
        
        # Use the same test directory for input catalog and result forecast files
        args = [CSEPTestCase.TestDirPath, 
                '%s=%s' %(STEPCoulombModel.SlipModelFileListOption,
                          os.path.join(CSEPTestCase.TestDirPath,
                                       model_list_file))]
        
        # Instantiate forecast object
        model = ForecastFactory().object(STEPCoulombOneMonthModel.Type, 
                                         args)

        # Copy reference input catalog data file to the test directory
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__STEPCoulombRefDir, 
                                     "geonet_combined.04.07.14.txt"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     model.inputCatalogFilename()))    

        # Copy reference focal mechanism input catalog data file to the test directory
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__STEPCoulombRefDir, 
                                     "GeoNet_CMT_solutions.csv"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     GeoNetNZDataSource.ProcessedFocalMechanismFile))    
        
        # Change directory to the test location to guarantee that all model
        # configuration and scratch files are placed there.
        cwd = os.getcwd() 
        os.chdir(CSEPTestCase.TestDirPath)

        try:            

           # Copy slip model list file into local directory for the run
           self.__slipModelList(os.path.join(DarfieldForecastsTests.__STEPCoulombRefDir,
                                                   model_list_file))
           
           model.create(DarfieldForecastsTests.__STEPCoulombTestDate, 
                        CSEPTestCase.TestDirPath)
   
           
           # Reference file with test results to compare against
           reference_file = "out_STEP.dat"
           
           shutil.copyfile(os.path.join(DarfieldForecastsTests.__STEPCoulombRefDir, 
                                        reference_file),
                           os.path.join(CSEPTestCase.TestDirPath, 
                                        reference_file))
   
           error_msg = "STEPCoulomb one-month forecast generation for Darfield \
testing region failed."
           
           self.failIf(CSEPFile.compare(reference_file, 
                                        model.CoulombOutput) is False,
                       error_msg)
           
           # Check for final STEPCoulomb forecast and it's metadata file
           reference_file = os.path.join(DarfieldForecastsTests.__STEPCoulombRefDir,
                                         "STEP_30d.txt")
           error_msg = "STEPCoulombOneMonth forecast generation for Darfield \
testing region failed."
           
           self.failIf(CSEPFile.compare(reference_file, 
                                        model.filename()) is False,
                       error_msg)

           # Check for existence of metadata file for the forecast
           error_msg = "Metadata file for STEPCoulombOneMonth forecast for Darfield \
testing region is missing."
           self.failIf(os.path.exists(model.filename() + 
                                      CSEPPropertyFile.Metadata.Extension) is False, 
                       error_msg)
           
        finally:
            os.chdir(cwd)


    #---------------------------------------------------------------------------
    #
    # Invoke STEPCoulomb one-year model for Darfield testing region, and verify 
    # the results.
    #
    def testSTEPCoulombOneYearModel(self):
        """ Generate STEPCoulombOneYear forecast for Darfield testing region and
verify results."""

        # Setup test name
        CSEPTestCase.setTestName(self, self.id())

        model_list_file = 'slip_models.txt'
        
        # Use the same test directory for input catalog and result forecast files
        args = [CSEPTestCase.TestDirPath, 
                '%s=%s' %(STEPCoulombModel.SlipModelFileListOption,
                          os.path.join(CSEPTestCase.TestDirPath,
                                       model_list_file))]
        
        # Instantiate forecast object
        model = ForecastFactory().object(STEPCoulombOneYearModel.Type, 
                                         args)

        # Copy reference input catalog data file to the test directory
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__STEPCoulombRefDir, 
                                     "geonet_combined.04.07.14.txt"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     model.inputCatalogFilename()))    

        # Copy reference focal mechanism input catalog data file to the test directory
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__STEPCoulombRefDir, 
                                     "GeoNet_CMT_solutions.csv"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     GeoNetNZDataSource.ProcessedFocalMechanismFile))    
        
        # Change directory to the test location to guarantee that all model
        # configuration and scratch files are placed there.
        cwd = os.getcwd() 
        os.chdir(CSEPTestCase.TestDirPath)

        try:            

           # Copy slip model list file into local directory for the run
           self.__slipModelList(os.path.join(DarfieldForecastsTests.__STEPCoulombRefDir,
                                                   model_list_file))
           
           model.create(DarfieldForecastsTests.__STEPCoulombTestDate, 
                        CSEPTestCase.TestDirPath)
   
           
           # Reference file with test results to compare against
           reference_file = "out_STEP.dat"
           
           shutil.copyfile(os.path.join(DarfieldForecastsTests.__STEPCoulombRefDir, 
                                        reference_file),
                           os.path.join(CSEPTestCase.TestDirPath, 
                                        reference_file))
   
           error_msg = "STEPCoulombOneDay one-year forecast generation for Darfield \
testing region failed."
           
           self.failIf(CSEPFile.compare(reference_file, 
                                        model.CoulombOutput) is False,
                       error_msg)
           
           # Check for final STEPCoulomb forecast and it's metadata file
           reference_file = os.path.join(DarfieldForecastsTests.__STEPCoulombRefDir,
                                         "STEP_365d.txt")
           error_msg = "STEPCoulombOneYear forecast generation for Darfield \
testing region failed."
           
           self.failIf(CSEPFile.compare(reference_file, 
                                        model.filename()) is False,
                       error_msg)

           # Check for existence of metadata file for the forecast
           error_msg = "Metadata file for STEPCoulombOneYear forecast for Darfield \
testing region is missing."
           self.failIf(os.path.exists(model.filename() + 
                                      CSEPPropertyFile.Metadata.Extension) is False, 
                       error_msg)
           
        finally:
            os.chdir(cwd)


    #---------------------------------------------------------------------------
    #
    # Invoke SUP one-day model for Darfield testing region, and verify 
    # the results.
    #
    def testSUPOneDayModel(self):
        """ Generate SUPCanterburyOneDay forecast for Darfield testing region and
verify results."""

        # Setup test name
        CSEPTestCase.setTestName(self, self.id())

        # Use the same test directory for input catalog and result forecast files
        args = [CSEPTestCase.TestDirPath] 
        
        # Instantiate forecast object
        model = ForecastFactory().object(SUPCanterburyOneDayModel.Type, 
                                         args)

        # Copy reference input catalog data file to the test directory
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__SUPRefDir, 
                                     "earthquakesto20140703.ref"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     model.inputCatalogFilename()))    

        # Copy reference input catalog data file to the test directory
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__SUPRefDir, 
                                     "earthquakesto20140703.ref"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     CSEPFile.Name.extension(model.inputCatalogFilename(),
                                                             'ref')))    

        # Change directory to the test location to guarantee that all model
        # configuration and scratch files are placed there.
        cwd = os.getcwd() 
        os.chdir(CSEPTestCase.TestDirPath)

        try:            

           model.create(DarfieldForecastsTests.__SUPTestDate, 
                        CSEPTestCase.TestDirPath)
   
           
           # Reference file with test results to compare against
           reference_file = os.path.join(DarfieldForecastsTests.__SUPRefDir,
                                         "supmodeloutput2.grd")
           
           error_msg = "SUP one-day forecast generation for Darfield \
testing region failed."
           
           self.failIf(CSEPFile.compare(reference_file, 
                                        model.filename()) is False,
                       error_msg)
           
        finally:
            os.chdir(cwd)


    #---------------------------------------------------------------------------
    #
    # Invoke SUP one-month model for Darfield testing region, and verify 
    # the results.
    #
    def testSUPOneMonthModel(self):
        """ Generate SUPCanterburyOneMonth forecast for Darfield testing region and
verify results."""

        # Setup test name
        CSEPTestCase.setTestName(self, self.id())

        # Use the same test directory for input catalog and result forecast files
        args = [CSEPTestCase.TestDirPath] 
        
        # Instantiate forecast object
        model = ForecastFactory().object(SUPCanterburyOneMonthModel.Type, 
                                         args)

        # Copy reference input catalog data file to the test directory
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__SUPRefDir, 
                                     "earthquakesto20140703.ref"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     model.inputCatalogFilename()))    

        # Copy reference input catalog data file to the test directory
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__SUPRefDir, 
                                     "earthquakesto20140703.ref"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     CSEPFile.Name.extension(model.inputCatalogFilename(),
                                                             'ref')))    

        # Change directory to the test location to guarantee that all model
        # configuration and scratch files are placed there.
        cwd = os.getcwd() 
        os.chdir(CSEPTestCase.TestDirPath)

        try:            

           model.create(DarfieldForecastsTests.__SUPTestDate, 
                        CSEPTestCase.TestDirPath)
   
           
           # Reference file with test results to compare against
           reference_file = os.path.join(DarfieldForecastsTests.__SUPRefDir,
                                         "SUPCanterburyOneMonth_12_7_2011.dat")
           
           error_msg = "SUP one-month forecast generation for Darfield \
testing region failed."
           
           self.failIf(CSEPFile.compare(reference_file, 
                                        model.filename()) is False,
                       error_msg)
           
        finally:
            os.chdir(cwd)


    #---------------------------------------------------------------------------
    #
    # Invoke SUP one-year model for Darfield testing region, and verify 
    # the results.
    #
    def testSUPOneYearModel(self):
        """ Generate SUPCanterburyOneYear forecast for Darfield testing region and
verify results."""

        # Setup test name
        CSEPTestCase.setTestName(self, self.id())

        # Use the same test directory for input catalog and result forecast files
        args = [CSEPTestCase.TestDirPath] 
        
        # Instantiate forecast object
        model = ForecastFactory().object(SUPCanterburyOneYearModel.Type, 
                                         args)

        # Copy reference input catalog data file to the test directory
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__SUPRefDir, 
                                     "earthquakesto20140703.ref"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     model.inputCatalogFilename()))    

        # Copy reference input catalog data file to the test directory
        shutil.copyfile(os.path.join(DarfieldForecastsTests.__SUPRefDir, 
                                     "earthquakesto20140703.ref"),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     CSEPFile.Name.extension(model.inputCatalogFilename(),
                                                             'ref')))    

        # Change directory to the test location to guarantee that all model
        # configuration and scratch files are placed there.
        cwd = os.getcwd() 
        os.chdir(CSEPTestCase.TestDirPath)

        try:            

           model.create(DarfieldForecastsTests.__SUPTestDate, 
                        CSEPTestCase.TestDirPath)
   
           
           # Reference file with test results to compare against
           reference_file = os.path.join(DarfieldForecastsTests.__SUPRefDir,
                                         "SUPCanterburyOneYear_12_7_2011.dat")
           
           error_msg = "SUP one-year forecast generation for Darfield \
testing region failed."
           
           self.failIf(CSEPFile.compare(reference_file, 
                                        model.filename()) is False,
                       error_msg)
           
        finally:
            os.chdir(cwd)

# Invoke the module
if __name__ == '__main__':

    import logging
    from DataSourceFactory import DataSourceFactory
    from PostProcessFactory import PostProcessFactory
    
    if os.environ.has_key(CSEPTestCase.ResultsDirVar):
         
         CSEPTestCase.TestDirPath = os.path.join(os.environ[CSEPTestCase.ResultsDirVar],
                                                 'results')
    
    DataSourceFactory.composite().clear()
    DataSourceFactory().object(GeoNetNZDataSource.Type,
                               input_variables = CSEPTestCase.DataSourceArgs)
    
       
    # Set post-processing specific data 
    PostProcessFactory(os.path.join(Environment.Variable[CENTER_CODE_ENV],
                                    'src',
                                    'SCECModels',
                                    'NewZealand',
                                    'DarfieldPostProcessFactory.init.xml'))
    
    # Configuration file for installed forecast models within the center
    config_file = os.path.join(Environment.Variable[CENTER_CODE_ENV],
                               'src',
                               'SCECModels',
                               'NewZealand',
                               'DarfieldForecastFactory.init.xml')
    
    factory = ForecastFactory(config_file)
    
    GeographicalRegions.Region().set('Darfield')
    
    # Invoke all tests
    unittest.main()
    
    # Shutdown logging
    logging.shutdown()
       
        
# end of main
