"""
Module CatalogRetrievalTest
"""

__version__ = "$Revision$"
__revision__ = "$Id$"


import os, unittest, datetime

from CSEPTestCase import CSEPTestCase
from RELMCatalog import RELMCatalog
from GeoNetNZDataSource import GeoNetNZDataSource
from CatalogDataSource import CatalogDataSource


#--------------------------------------------------------------------------------
# 
# Year generator.
# 
# This generator function reads a specific entry from the file line that
# represents a year.
#
# Inputs:
#         filepath - Path to the catalog file.
# Output:
#         Integer representation of year from the line of file.     
#
def yearOfLine(filepath, line_of_words = str.split):
   """ Generator to return integer representation of decimal year from 
       each line of specified file."""
   
   # Decimal year position for the line entries
   __YEAR = 2
       
   fhandle = open(filepath)
   for line in fhandle:
      words = line_of_words(line)
      
      # Return year string
      yield int(float(words[__YEAR]))
   
   fhandle.close() 
       

#------------------------------------------------------------------------------
#
# Test retrieval of the catalog data.
#
class CatalogRetrievalTest (CSEPTestCase):
    

    #---------------------------------------------------------------------------
    #
    # Download GeoNet catalog with default start date of 1/1/1864 and confirm that
    # non-empty raw and pre-processed files are generated.
    #
    # Inputs: None.
    #    
    def testDownloadDefaultStartDate(self):
        """ Test retrieval of GeoNet catalog data in XML format and succeed. \
Download data for the previous month with default start date of 1863-1-1 \
and check for non-empty raw and pre-processed catalog file."""

        # Download data for the last month (use 31 days for a month)
        now = datetime.datetime.now()
        month_diff = datetime.timedelta(days=31)
        month_ago = now - month_diff

        # Narrow down the search to avoid long runtime for the test
        input_args = "depthLower=5"
        data_source = GeoNetNZDataSource(args = input_args)
        start_date = data_source.StartDate
        test_name = "GeoNetNZCatalogDownload-%s-%s" \
                    %(start_date.date(), month_ago.date())
        # Setup test name
        CSEPTestCase.setTestName(self, test_name)

   
        # Don't invoke any kind of post-processing for the catalog data.
        # Instantiating of catalog object takes care of generating a test directory
        catalog = RELMCatalog(CSEPTestCase.TestDirPath,
                              data_source)
                
        catalog.create(month_ago)


        # Check if raw catalog file exists and has non-zero size
        result_file = os.path.join(CSEPTestCase.TestDirPath, 
                                   CatalogDataSource._RawFile)
        
        self.failIf(os.path.exists(result_file) is False, 
                    'Download of GeoNet NZ catalog with start date of %s failed.' \
                    %start_date)
        self.failIf(os.path.getsize(result_file) == 0,
                    'Downloaded GeoNet NZ catalog file with start date of %s is empty.' \
                    %start_date)
        
                
        # Check if pre-processed catalog file exists and has non-zero size
        result_file = os.path.join(CSEPTestCase.TestDirPath, 
                                   CatalogDataSource.PreProcessedFile)
        
        self.failIf(os.path.exists(result_file) is False, 
                    'Pre-processing of GeoNet NZ catalog with start date of %s failed.' \
                    %start_date)
        self.failIf(os.path.getsize(result_file) == 0, 
                    'Pre-processed GeoNet NZ catalog file with start date of %s is empty.' \
                    %start_date)
        
        # Check that start and end years of the time period have been downloaded
        
        # Confirm that downloaded start and end years of the range are in final 
        # catalog file:
        years = [start_date.year, month_ago.year]
        self.__checkForYears(years)
        

    #----------------------------------------------------------------------------
    #
    # Download GeoNet NZ catalog with other than default start date (use 2008-1-1)
    # and confirm that non-empty raw and pre-processed files are generated.
    #
    # Inputs: None.
    #    
    def testDownload_2006_1_1_StartDate(self):
        """ Test retrieval of GeoNet NZ catalog data in XML format and succeed. \
Download data for the previous month with start date other than default \
(2006-1-1) and check for non-empty raw and pre-processed catalog file."""

        # Download data for the last month (use 31 days for a month)
        now = datetime.datetime.now()
        month_diff = datetime.timedelta(days=31)
        month_ago = now - month_diff

        start_date = datetime.datetime(2006, 1, 1)
        test_name = "GeoNetNZCatalogDownload-%s-%s" \
                    %(start_date.date(), month_ago.date())
        # Setup test name
        CSEPTestCase.setTestName(self, test_name)

   
        # Don't invoke any kind of post-processing for the catalog data.
        # Instantiating of catalog object takes care of generating a test directory
        # Narrow down the search to avoid long runtime for the test
        input_args = "depthLower=35, magnitudeUpper=3.5"
        data_source = GeoNetNZDataSource(start_date,
                                         args = input_args)
        
        catalog = RELMCatalog(CSEPTestCase.TestDirPath,
                              data_source)
                
        catalog.create(month_ago)


        # Check if raw catalog file exists and has non-zero size
        result_file = os.path.join(CSEPTestCase.TestDirPath, 
                                   CatalogDataSource._RawFile)
        
        self.failIf(os.path.exists(result_file) is False, 
                    'Download of GeoNet MZ catalog with start date of %s failed.' \
                    %start_date)
        self.failIf(os.path.getsize(result_file) == 0,
                    'Downloaded GeoNet NZ catalog file with start date %s is empty.' \
                    %start_date)
        
                
        # Check if pre-processed catalog file exists and has non-zero size
        result_file = os.path.join(CSEPTestCase.TestDirPath, 
                                   CatalogDataSource.PreProcessedFile)
        
        self.failIf(os.path.exists(result_file) is False, 
                    'Pre-processing of GeoNet NZ catalog with start date of %s failed.' \
                    %start_date)
        self.failIf(os.path.getsize(result_file) == 0, 
                    'Pre-processed GeoNet NZ catalog file with start date of %s is empty.' \
                    %start_date)

        # Confirm that downloaded start and end years of the range are in final 
        # catalog file:
        years = [start_date.year, month_ago.year]
        self.__checkForYears(years)


    #----------------------------------------------------------------------------
    #
    # Download GeoNet focal mechanism catalog along with GeoNet catalog
    # and confirm that non-empty raw and pre-processed files are generated for
    # focal mechanism catalog.
    #
    # Inputs: None.
    #    
    def testDownloadFocalMechanism(self):
        """ Test retrieval of GeoNet NZ focal mechanism catalog data and succeed. \
Check for non-empty raw and pre-processed catalog file."""

        # Download data for the last month (use 31 days for a month)
        now = datetime.datetime.now()
        month_diff = datetime.timedelta(days=31)
        month_ago = now - month_diff

        start_date = datetime.datetime(2006, 1, 1)

        # Setup test name
        CSEPTestCase.setTestName(self, self.id())

   
        # Don't invoke any kind of post-processing for the catalog data.
        # Instantiating of catalog object takes care of generating a test directory
        # Narrow down the search to avoid long runtime for the test
        input_args = "depthLower=35, magnitudeUpper=3.5,focalMechanism=True"
        data_source = GeoNetNZDataSource(start_date,
                                         args = input_args)
        
        catalog = RELMCatalog(CSEPTestCase.TestDirPath,
                              data_source)
                
        catalog.create(month_ago)


        # Check if raw catalog file exists and has non-zero size
        result_file = os.path.join(CSEPTestCase.TestDirPath, 
                                   CatalogDataSource._RawFile)
        
        self.failIf(os.path.exists(result_file) is False, 
                    'Download of GeoNet MZ catalog with start date of %s failed.' \
                    %start_date)
        self.failIf(os.path.getsize(result_file) == 0,
                    'Downloaded GeoNet NZ catalog file with start date %s is empty.' \
                    %start_date)
        
                
        # Check if pre-processed catalog file exists and has non-zero size
        result_file = os.path.join(CSEPTestCase.TestDirPath, 
                                   CatalogDataSource.PreProcessedFile)
        
        self.failIf(os.path.exists(result_file) is False, 
                    'Pre-processing of GeoNet NZ catalog with start date of %s failed.' \
                    %start_date)
        self.failIf(os.path.getsize(result_file) == 0, 
                    'Pre-processed GeoNet NZ catalog file with start date of %s is empty.' \
                    %start_date)

        # Check if raw focal mechanism catalog file exists and has non-zero size
        result_file = os.path.join(CSEPTestCase.TestDirPath, 
                                   GeoNetNZDataSource.RawFocalMechanismFile)
        
        self.failIf(os.path.exists(result_file) is False, 
                    'Download of GeoNet NZ focal mechanism catalog %s failed.' \
                    %result_file)
        self.failIf(os.path.getsize(result_file) == 0,
                    'Downloaded GeoNet NZ focal mechanism catalog file %s is empty.' \
                    %result_file)
        
        # Check if raw focal mechanism catalog file exists and has non-zero size
        result_file = os.path.join(CSEPTestCase.TestDirPath, 
                                   GeoNetNZDataSource.ProcessedFocalMechanismFile)
        
        self.failIf(os.path.exists(result_file) is False, 
                    'Pre-processing of GeoNet NZ focal mechanism catalog %s failed.' \
                    %result_file)
        self.failIf(os.path.getsize(result_file) == 0,
                    'Pre-processed GeoNet NZ focal mechanism catalog file %s is empty.' \
                    %result_file)


    #----------------------------------------------------------------------------
    #
    # Check that specified years are present in the result file.
    #
    # Inputs:
    #         years - List of years to check for. 
    #    
    def __checkForYears(self, years):
        """ Check for the presence of specified years in the catalog file."""
       
        # Build dictionary of found years from file that are only specified 
        # by 'years'
        found_years = []
        for year in yearOfLine(os.path.join(CSEPTestCase.TestDirPath,
                                            CatalogDataSource.PreProcessedFile)):
           if year in years:
              if year not in found_years:
                 found_years.append(year)

        years.sort()  # Sorted in place
        found_years.sort() # Sorted in place
        
        self.failIf(found_years != years,
                    "Expected to find %s years in pre-processed catalog file, \
found %s" %(years, found_years))
        

# Invoke the module
if __name__ == '__main__':

   # Invoke all tests - must have testXXX method in the class defined
   unittest.main()
   
# end of main
