"""
Module CattaniaModel
"""

__version__ = "$Revision: 4818 $"
__revision__ = "$Id: CattaniaModel.py 4818 2014-08-20 00:28:43Z liukis $"

import os, datetime

import Environment, CSEPFile, CSEP, CSEPLogging
from Forecast import Forecast
from CSEPInputParams import CSEPInputParams
from cseprandom import CSEPRandom
from GeoNetNZDataSource import GeoNetNZDataSource, FOCAL_MECHANISM_PATH_ENV
from DataSourceFactory import DataSourceFactory
from SlipModels import SlipModels

# Environment variable to store hash to (for retrospective runs only when
# input catalog does not change for sequential runs)
HASH_DIR_ENV = "CATTANIA_MODEL_HASH_DIR"


#-------------------------------------------------------------------------------
#
# CattaniaModel forecast model for New Zealand.
#
# This class is a helper to invoke any of Sebastian Hainzl forecasts models.
#
class CattaniaModel(object):

    # Static data of the class
    Type = 'CATTANIA'

    # Center code path for the model
    Path = os.path.join(Forecast.CodePath,
                        'NewZealand',
                        'src',
                        'CattaniaModels')

    # Option to specify which model configuration to invoke
    ConfigOption = 'config'

    # Allowed model configurations
    ConfigSet = ['0', '1', '2', '3', '4']

    # Option to specify random seed value
    RandomSeedOption = 'randomSeed'

    # Coulumb file list option
    CoulombFileOption = 'CoulombFile'

    # Coulumb file list option
    SlipModelFileListOption = 'slipModelList'

    # Location of parameter files for the models: to support short runtime
    # of acceptance tests only
    ParamFilePathOption = 'parameterPath'

    # Default options for the model
    __defaultArgs = {ConfigOption: None,
                     RandomSeedOption: None,
                     ParamFilePathOption: Path,
                     CoulombFileOption: os.path.join(Path,
                                                     'inCan.dat'),
                     SlipModelFileListOption: None}

    __executableFile = 'CRS_CSEP'


    #--------------------------------------------------------------------
    #
    # Initialization.
    #
    # Input:
    #        args - Optional arguments for the model. Default is None.
    #
    def __init__ (self, args=None):
        """ Initialization for CattaniaModel class"""
        
        # Input arguments for the model were provided:
        self.__args = CSEPInputParams.parse(CattaniaModel.__defaultArgs, 
                                            args)
        if self.__args[CattaniaModel.ConfigOption] not in \
           CattaniaModel.ConfigSet:
            msg = "One of %s models options is allowed, '%s' is provided" \
                        %(CattaniaModel.ConfigSet,
                          self.__args[CattaniaModel.ConfigOption])
             
            CSEPLogging.CSEPLogging.getLogger(CattaniaModel.__name__).error(msg)
            raise RuntimeError, msg
        

    #---------------------------------------------------------------------------
    #
    # Return sub-type keyword identifying the model: based on testing region.
    #
    # Input: None.
    #
    # Output:
    #           String identifying the sub-type
    #
    def subtype (self):
        """ Returns keyword identifying the forecast model sub-type."""

        # Specify selected type of forecast as the sub-type for the model        
        return 'CRS%s' %(self.__args[CattaniaModel.ConfigOption])


    #---------------------------------------------------------------------------
    #
    # Write input parameter file for the model in following format:
    #    #Compulsory arguments:
    #    OutputForecastFile=csepOutput/model0_1d
    #    InputModelParametersFile=input/parameter_files/model0.txt
    #    IssueDate=2010-09-13T16:35:41Z
    #    ForecastStartDate=2010-09-13T16:35:41Z
    #    ForecastEndDate=2010-09-14T16:35:41Z
    #    InputCatalogFile=input/catalogs/MaxWerner.Canterbury_M2_1980_2012.txt
    #    ForecastTemplate=input/other/darf_temp.txt
    #    RandomSeedValue=-37284630
    #    #InputCoulombFile:  the 'master' file in farfalle format.
    #    InputCoulombFile=input/slipmodels/inCan.dat
    #    #InputListSlipModels: file containing a of slip models, in farfalle format (see example).
    #    InputListSlipModels=input/slipmodels/exampleInputListSlipModels.dat
    #    #
    #    # Optional arguments:
    #    # Focal mechanisms are expected to be in CMT format.
    #    InputCatalogFocMecFile=input/catalogs/GeoNet_CMT_solutions.dat
    #    Logfile=csepOutput/model0_1d.log
    #    ExtendedOutput=0                #default value=0, only set to 1 for tests (in this case, extra output is produced).
    #    PropagateCalculations=0         #default value=0, only set to 1 if the calculations should be propagated between consecutive runs of the code.
    #        
    def writeFile (self, 
                   fhandle,
                   start_date,
                   end_date,
                   input_catalog,
                   result_forecast,
                   template_file,
                   log_file,
                   hash_dir):
        """ Format input parameter file for the model.
            Path to created input parameter file will be passed to the 
            model's executable."""

        fhandle.write("#Compulsory arguments:\n")

        line = "OutputForecastFile=%s\n" %result_forecast
        fhandle.write(line)

        model_param_file = 'model%s.txt' %self.__args[CattaniaModel.ConfigOption]
        line = "InputModelParametersFile=%s\n" %os.path.join(self.__args[CattaniaModel.ParamFilePathOption],
                                                             model_param_file)
        fhandle.write(line)
        
        # Has to be set to forecast's start date according to Camilla:
        # "...I use "IssueDate" to determine which aftershocks and which focal 
        # mechanisms I am allowed to use (only those up to IssueDate), and 
        # I expect for this experiment it will always be set to the same value 
        # as "ForecastStartDate". 
        line = "IssueDate=%sZ\n" \
               %start_date.strftime(CSEP.Time.ISO8601Format)
        fhandle.write(line)

        line = "ForecastStartDate=%sZ\n" \
               %start_date.strftime(CSEP.Time.ISO8601Format)
        fhandle.write(line)

        line = "ForecastEndDate=%sZ\n" \
               %end_date.strftime(CSEP.Time.ISO8601Format)
        fhandle.write(line)
        
        line = "InputCatalogFile=%s\n" %input_catalog
        fhandle.write(line)

        line = "ForecastTemplate=%s\n" %CSEPFile.Name.ascii(template_file)
        fhandle.write(line)

        # Model uses random numbers, provide the seed for the run
        seed = self.__args[CattaniaModel.RandomSeedOption]
        
        if self.__args[CattaniaModel.RandomSeedOption] is None:
            seed = CSEPRandom.createSeed()

        line = "RandomSeedValue=%s\n" %seed
        fhandle.write(line)
        
        # Use existing file if it's provided, otherwise identify available 
        # Coulomb files within file system
        coulomb_file = self.__args[CattaniaModel.CoulombFileOption]
        line = "InputCoulombFile=%s\n" %coulomb_file
        fhandle.write(line)

        local_path, local_file = os.path.split(input_catalog)
        
        slip_models_list = self.__args[CattaniaModel.SlipModelFileListOption]
        if slip_models_list is None:
            # Identify slip model files for the run
            slip_models_list = SlipModels().files(local_path,
                                                  start_date)
        
        line = "InputListSlipModels=%s\n" %slip_models_list
        fhandle.write(line)
        
        # Provide focal mechanism catalog
        data_path, data_file = os.path.split(DataSourceFactory().object(GeoNetNZDataSource.Type, 
                                                                        isObjReference=True).RawFile)
        
        if FOCAL_MECHANISM_PATH_ENV in os.environ:
            data_path = os.environ[FOCAL_MECHANISM_PATH_ENV]
        
        # Create or check for existing focal mechanism file cut up to start_date
        start_date_focal_mechanism = "%s%s" %(start_date.strftime("%Y_%m_%d_"),
                                              GeoNetNZDataSource.ProcessedFocalMechanismFile)
        start_date_focal_mechanism_path = os.path.join(local_path,
                                                       start_date_focal_mechanism)
        
        if not os.path.exists(start_date_focal_mechanism_path):
            
            fh_out = CSEPFile.openFile(start_date_focal_mechanism_path,
                                       CSEPFile.Mode.WRITE)
            
            with CSEPFile.openFile(os.path.join(data_path,
                                                GeoNetNZDataSource.ProcessedFocalMechanismFile)) as fh:
                for each_row in fh:
                    if each_row[0].isdigit():
                        # Parse out the data and filter by time
                        # Cut to start_time: "20030821121200" is catalog's format for date/time
                        time_stamp = each_row.split()[1]
                        
                        event_date = datetime.datetime.strptime(time_stamp,
                                                                "%Y%m%d%H%M%S")
                        if event_date < start_date:
                             fh_out.write(each_row)
                        
                    else:
                        fh_out.write(each_row)
                        
            fh_out.close()
            
        line = "InputCatalogFocMecFile=%s\n" %start_date_focal_mechanism_path
        fhandle.write(line)
        
        line = "Logfile=%s\n" %log_file
        fhandle.write(line)

        fhandle.write("ExtendedOutput=0\n")

        # Hash directory is used by the run, preserve hash for reproducibility
        local_hash_archive = None
        if HASH_DIR_ENV in os.environ:
            
            hash_dir_path = os.path.join(os.environ[HASH_DIR_ENV],
                                         hash_dir)

            if not os.path.exists(hash_dir_path):
                os.makedirs(hash_dir_path)
            
            local_hash_archive = os.path.join(os.getcwd(),
                                              hash_dir + CSEPFile.Extension.TARGZ)
            # Preserve existing hash directory for reproducibility
            archive = CSEPFile.GZIPArchive(local_hash_archive,
                                           CSEPFile.Mode.WRITE)
            archive.add(hash_dir_path)
            del archive
            
            fhandle.write("HashDirectory=%s\n" %hash_dir_path)
            
        
        return local_hash_archive


    #--------------------------------------------------------------------
    #
    # Invoke the model to generate forecast.
    #
    # Input: None
    #        
    def invokeModel (self, parameter_file):
        """ Invoke model."""

        # If modelers output some debug/progress info to stderr, ignore it - 
        # don't trigger an exception 
        ignore_model_errors = True
        
        # ( echo testinput/exampleinputfile.txt ) | ./runRETAS1.pl        
        command = "%s %s" %(os.path.join(CattaniaModel.Path,
                                         CattaniaModel.__executableFile),
                            parameter_file)
        Environment.invokeCommand(command,
                                  ignore_model_errors)
        

    #---------------------------------------------------------------------------
    #
    # Return commands that should be used to capture version of external
    # software packages the model is dependent on. 
    #
    # Input: None.
    #
    # Output:
    #           String identifying the type
    #
    @staticmethod
    def externalSoftwareVersions ():
        """ Returns dictionary of command to determine external software 
            version and flag if output of that command is redirected to the
            stderr (True) or not (False) (java -version, for example)."""
        
        return {"gcc --version" : False}

