"""
Module STEPCoulombOneMonthModel
"""

__version__ = "$Revision$"
__revision__ = "$Id$"


import os

import CSEPFile
from CSEPLogging import CSEPLogging
from Forecast import Forecast
from OneMonthForecast import OneMonthForecast
from ReproducibilityFiles import ReproducibilityFiles
from STEPCoulombModel import STEPCoulombModel


#-------------------------------------------------------------------------------
#
# STEPCoulombOneMonthModel one-day forecast model.
#
# This class is designed to invoke one-day STEPCoulombOneMonthModel forecast model 
# by Matt Gerstenberger and Sandy Steacy. It prepares input catalog data, 
# formats control file with model parameters, identifies list of available 
# Coulomb and slip models and provides it as an input to the model. It places 
# forecast file under user specified directory.
#
class STEPCoulombOneMonthModel (OneMonthForecast, STEPCoulombModel):

    # Static data of the class
    
    # Keyword identifying type of the class
    Type = STEPCoulombModel.Type + OneMonthForecast.Type
    
    
    #---------------------------------------------------------------------------
    #
    # Initialization.
    #
    # Input: 
    #        dir_path - Directory to store forecast file to.
    #        args - Optional arguments for the model. Default is None.
    # 
    def __init__ (self, dir_path, args = None):
        """ Initialization for STEPCoulombOneMonthModel class"""
        
        OneMonthForecast.__init__(self, 
                                  dir_path)
        STEPCoulombModel.__init__(self, 
                                  args)
        
        # Parameter file for the Coulomb part of the model
        self.CoulombParamFile = None
        

    #---------------------------------------------------------------------------
    #
    # Return keyword identifying the model.
    #
    # Input: None.
    #
    # Output:
    #           String identifying the type
    #
    def type (self):
        """ Returns keyword identifying the forecast model type."""
        
        return STEPCoulombOneMonthModel.Type
    
    
    #---------------------------------------------------------------------------
    #
    # Write input parameter file for the model
    #        
    def writeParameterFile (self, 
                            filename = None):
        """ Format input parameter file for the model.
            Path to created input parameter file will be passed to the 
            model's executable."""

        # Open STEPCoulomb param file
        fhandle = Forecast.writeParameterFile(self,
                                              filename)
        STEPCoulombModel.writeSTEPFile(self,
                                       fhandle,
                                       self.start_date,
                                       self.numDays(self.start_date,
                                                    self.end_date),
                                       os.path.join(self.catalogDir,
                                                    self.inputCatalogFilename()), 
                                       self.filename())

        fhandle.close()

        # Write param file to invoke Coulomb part of the model first
        param_path, param_file = os.path.split(self.parameterFile)
        
        self.CoulombParamFile = os.path.join(param_path,
                                             "CoulombPart_" + param_file)
        
        # To avoid inheritance from two classes that are derived from the same
        # base "Forecast" class, use this "pass all values" approach
        STEPCoulombModel.writeCoulombFile(self,
                                          self.CoulombParamFile,
                                          self.start_date,
                                          self.catalogDir)

        # Register input parameters file for reproducibility
        info_msg = "Input parameters file used by Coulomb part of %s model \
to generate '%s' forecast file for %s." %(self.type(), 
                                          self.rawFilename(), 
                                          self.start_date.date())

        # Record parameter file with reproducibility registry
        ReproducibilityFiles.add(self,
                                 self.CoulombParamFile, 
                                 info_msg, 
                                 CSEPFile.Extension.ASCII)
        

    #--------------------------------------------------------------------
    #
    # Invoke the model to generate forecast.
    #
    # Input: None
    #        
    def run (self):
        """ Invoke model."""

        coulomb_output = STEPCoulombModel.invokeModel(self,
                                                      self.CoulombParamFile,
                                                      self.parameterFile)
        
        # Move Coulomb file to the archive directory - should not be
        # evaluated
        if self.archive is not None:
           
            # Move file to the archive directory to avoid it
            # in evaluation tests 
            CSEPLogging.getLogger(STEPCoulombOneMonthModel.__name__).info("run(): moving %s to %s"
                                                                          %(coulomb_output,
                                                                            self.archive))

            new_path = os.path.join(self.archive, 
                                    os.path.basename(coulomb_output))
            os.renames(coulomb_output, 
                       new_path)
